//    
/*++
Dom::Fraction -- the domain of fractions over R

Dom::Fraction(R)

R - integral domain

Dom::Fractions are represented as new(this,n,d) where n and d are from R,
n is the numerator and d is the denominator.

Methods:
numer(x)       - returns the numerator of x
denom(x)       - returns the denominator of x
retract(x)     - returns numer(x)/denom(x) if this is an element of R and
	         FAIL otherwise
normalize(x,y) - returns the fraction x/y, 0/y is normalized to 0/1; in a
		 Cat::GcdDomain the gcd of x and y is removed.
normalizePrime(x,y) - returns the fraction x/y, 0/y is normalized to 0/1;
		 in a Cat::GcdDomain it is assumed that x and y are
		 relative prime, the gcd of x and y is not removed.
++*/

domain Dom::Fraction(R)
    
  inherits Dom::BaseDomain;
  category Cat::QuotientField(R),
    ( if R::hasProp(Cat::DifferentialRing) then
        Cat::DifferentialRing
      elif R::hasProp(Cat::PartialDifferentialRing) then
        Cat::PartialDifferentialRing
      end_if
     );

  axiom ( if R::hasProp(Ax::canonicalRep) and R::hasProp(Cat::GcdDomain) and
            R::hasProp(Ax::canonicalUnitNormal) then
            Ax::canonicalRep
          else
            Ax::normalRep
          end_if);

  coeffRing := R;
      
  characteristic := R::characteristic; // also correct if that is FAIL

      
  convert :=
   ( if R = Dom::Integer then
       proc(x)
       begin
         // we have to evaluate since x might be _divide(1, 2)
         x:= eval(x);
         case domtype(x)
           of dom do
             return(x)
           of DOM_INT do
             return(dom::normalizePrime(x, 1));
           of DOM_RAT do
             return(dom::normalizePrime(op(x,1), op(x,2)));
         end_case;
         FAIL
       end_proc
    else
    proc(x)
      local c, i;
    begin
      case type(x)
        of dom do
          return(x)
        of "_plus" do
          c:= [ dom::convert(op(x,i)) $ i=1..nops(x) ];
          if contains(c, FAIL) > 0 then break end_if;
          return(dom::_plus(op(c)));
        of "_mult" do
          c:= [ dom::convert(op(x,i)) $ i=1..nops(x) ];
		    if contains(c, FAIL) > 0 then break end_if;
		    return(dom::_mult(op(c)));
	     of "_power" do
		    if domtype(op(x,2)) <> DOM_INT then break end_if;
		    c:= dom::convert(op(x,1));
		    if c = FAIL then break end_if;
		    return(dom::_power(c,op(x,2)));
	     of "function" do
		    if R::hasProp(Cat::DifferentialRing) then
		      if op(x,0) <> hold(D) or nops(x) <> 1 then break end_if;
		      c:= dom::convert(op(x,1));
		      if c = FAIL then break end_if;
		      return(dom::D(c));
          elif R::hasProp(Cat::PartialDifferentialRing) then
		      if op(x,0) <> hold(D) or nops(x) <> 2 then break end_if;
		      c:= dom::convert(op(x,2));
		      if c = FAIL then break end_if;
		      if not testtype(op(x,1), Type::ListOf(Type::PosInt)) then
			     break
		      end_if;
		      return(dom::D(op(x,1), c));
		    end_if;
	    end_case;

	    c:= R::convert(x);
	    if c = FAIL then
         return(FAIL)
       end_if;
	    dom::normalize(c, R::one)
	end_proc
    end_if );

   expr :=
    proc(x)
    begin
      R::expr(extop(x,1)) / R::expr(extop(x,2))
    end_proc;

   print := x->generate::sortSums(expr(x));
      
   convert_to :=
   proc(x,T)
   begin
     case T
       of dom do return(x);
       of Dom::Expression do
       of Dom::ArithmeticalExpression do return(dom::expr(x));
     end_case;
     FAIL
   end_proc;

   TeX :=
   proc(x)
   begin
     "\\frac{".R::TeX(extop(x,1))."}{".R::TeX(extop(x,2))."}"
   end_proc;

   numer := proc(x) begin extop(x,1) end_proc;

   denom := proc(x) begin extop(x,2) end_proc;

   iszero := proc(x) begin R::iszero(extop(x,1)) end_proc;

   normalizePrime :=
   (if R::hasProp(Cat::GcdDomain) then
      proc(x, y)
        local d;
      begin
        if R::iszero(x) then
          dom::zero
        else
          d:= R::unitNormalRep(y);
          new(dom, R::_mult(x, d[2]), d[1])
        end_if
      end_proc
    else
      dom::normalize
    end_if);

    normalize :=
    (if R::hasProp(Cat::GcdDomain) then
       proc(x,y)
         local g,yy;
       begin
         if R::iszero(x) then
           dom::zero
         else
           g:= R::gcd(x,y);
           yy:= R::unitNormalRep(R::_divide(y,g));
           new(dom, R::_mult(R::_divide(x,g), yy[2]), yy[1])
         end_if
     end_proc
     else
     proc(x,y)
     begin
       if R::iszero(x) then
         dom::zero
       else
         new(dom, x, y)
       end_if
     end_proc
     end_if);

   factor :=
   proc(x)
     local i,numfac, denomfac;

   begin
     numfac:=factor(numer(x));
     numfac:= Factored::convert_to(numfac,DOM_LIST);
     denomfac:=factor(denom(x));
     denomfac:= Factored::convert_to(denomfac,DOM_LIST);
     numfac[1]:=dom(numfac[1])/dom(denomfac[1]);
     delete denomfac[1];
     for i from 2 to nops(numfac) step 2 do
       numfac[i]:=dom(numfac[i]);
     end_for;
     for i from 1 to nops(denomfac) step 2 do
       denomfac[i]:=dom(denomfac[i]);
       denomfac[i+1]:=-denomfac[i+1]
     end_for;
     Factored::create( numfac.denomfac,"irreducible",R )
   end_proc;
			    
    equal := (if not dom::hasProp(Ax::canonicalRep) then
	proc(x,y) begin
	    R::equal(R::_mult(extop(x,1), extop(y,2)),
		     R::_mult(extop(y,1), extop(x,2)))
	end_proc
    end_if);

    _less := ( if R::hasProp(Cat::OrderedSet) then
	proc(x,y) begin
	    R::_less(R::_mult(extop(x,1), extop(y,2)),
		     R::_mult(extop(y,1), extop(x,2)))
	end_proc
    end_if );

    zero := new(dom, R::zero, R::one);

    iszero := proc(x) begin R::iszero(extop(x,1)) end_proc;

    one := new(dom, R::one, R::one);

   _plus :=
    (if R::hasProp(Cat::GcdDomain) then
     proc(x,y) local g, d;
     begin
       case args(0)
         of 1 do
           return(x);
         of 2 do
           if map({args()}, domtype) <> {dom} then
             [x, y]:= map([x, y], dom::convert);
             if x = FAIL or y = FAIL then
               return(FAIL)
             end_if;
           end_if;
           g:= R::gcd(extop(x,2), extop(y,2));
           d:= R::_divide(extop(x,2), g);
		     return(dom::normalize
                  (
                   R::_plus(R::_mult(extop(x,1),
                                     R::_divide(extop(y,2), g)),
                            R::_mult(extop(y,1), d)),
                   R::_mult(extop(y,2), d)
                   )
                  );
        end_case;
        g:= args(0) div 2;
	     d:= _plus(args(1..g));
	     g:= _plus(args((g+1)..args(0)));
        if (domtype(d))::_plus <> FAIL then
          (domtype(d))::_plus(d, g)
        elif (domtype(g))::_plus <> FAIL then
          (domtype(g))::_plus(d, g)
	     else
          d + g
	     end_if
	   end_proc
    else
	proc(x,y)
     local g, i;
     begin
       case args(0)
         of 1 do
           return(x);
         of 2 do
           if map({args()}, domtype) <> {dom} then
             [x, y]:= map([x, y], dom::convert);
             if x = FAIL or y = FAIL then
               return(FAIL)
             end_if;
           end_if;
     return(dom::normalize(
		    R::_plus(R::_mult(extop(x,1), extop(y,2)),
			     R::_mult(extop(y,1), extop(x,2))),
		    R::_mult(extop(x,2), extop(y,2))));
	    end_case;
	    g:= args(0) div 2;
	    d:= _plus(args(1..g));
	    g:= _plus(args((g+1)..args(0)));
	    if (domtype(d))::_plus <> FAIL then (domtype(d))::_plus(d, g)
	    elif (domtype(g))::_plus <> FAIL then (domtype(g))::_plus(d, g)
	    else d + g
	    end_if
	end_proc
    end_if);

   _negate :=
   proc(x)
   begin
     dom::normalizePrime(R::_negate(extop(x,1)), extop(x,2))
   end_proc;

   _mult :=
    (if R::hasProp(Cat::GcdDomain) then
       if R::hasProp(Ax::systemRep) then
	
         proc(x,y)
           local p, q;
         begin
           case args(0)
             of 2 do
               if domtype(y) <> dom then
                 if domtype(y) = DOM_INT then
                   return(dom::intmult(args()))
                 elif (p:= dom::convert(y)) <> FAIL then
                   return(dom::_mult(x, p))
                 else
                   return((domtype(y))::_mult(x, y))
                 end_if;
               elif domtype(x) <> dom then
                  if domtype(x) = DOM_INT then
                    return(dom::intmult(y, x))
                  elif (p:=dom::convert(x)) <> FAIL then
                    return(dom::_mult(p, y))
                  else
                    return(FAIL)
                  end_if;
               end_if;
               p:= R::gcd(extop(x,1), extop(y,2));
		         q:= R::gcd(extop(y,1), extop(x,2));
               return(dom::normalizePrime
                      (
                       R::_mult(R::_divide(extop(x,1), p),
                                R::_divide(extop(y,1), q)),
                       R::_mult(R::_divide(extop(x,2), q),
                                R::_divide(extop(y,2), p))));
                of 1 do
                  return(x);
             end_case;
	          p:= args(0) div 2;
	          [p, q]:= [_mult(args(1..p)), _mult(args((p+1)..args(0)))];
	          if p::dom::_mult <> FAIL then
               p::dom::_mult(p, q)
             elif q::dom::_mult <> FAIL then
               q::dom::_mult(p, q)
	          else
               _mult(p, q)
             end_if
          end_proc
	
    	else
	
     proc(x,y)
       local p, q;
     begin
       case args(0)
         of 2 do
           if y::dom <> dom then
             case domtype(y)
               of DOM_INT do
                 return(dom::intmult(args()));
               of R do
                 return(dom::_mult(x, dom::normalize(y, R::one)));
             end_case;
             return(y::dom::_mult(x, y))
           elif x::dom <> dom then
             if domtype(x) = DOM_INT then
               return(dom::intmult(y, x))
             elif (p:=dom::convert(x)) <> FAIL then
               return(dom::_mult(p, y))
             else
               return(FAIL)
             end_if;
           end_if;
		p:= R::gcd(extop(x,1), extop(y,2));
		q:= R::gcd(extop(y,1), extop(x,2));
		return(dom::normalizePrime(
		    R::_mult(R::_divide(extop(x,1), p),
			     R::_divide(extop(y,1), q)),
		    R::_mult(R::_divide(extop(x,2), q),
			     R::_divide(extop(y,2), p))));
	    of 1 do return(x);
	    end_case;
	    p:= args(0) div 2;
	    _mult(args(1..p));
	    _mult(args((p+1)..args(0)));
	    if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
	    elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
	    else _mult(%2, %1)
	    end_if
	end_proc
	
    	end_if
    else
	if R::hasProp(Ax::systemRep) then
	
	proc(x,y)
     local p, q;
     begin
       case args(0)
         of 2 do
           if domtype(y) <> dom then
             if domtype(y) = DOM_INT then
               return(dom::intmult(args()))
             elif (p:=dom::convert(y)) <> FAIL then
               return(dom::_mult(x, p))
             end_if;
             return(FAIL);
           elif domtype(x) <> dom then
             return(dom::_mult(y,x));
		     end_if;
           return(dom::normalize
                  (
                   R::_mult(extop(args(1),1), extop(args(2),1)),
                   R::_mult(extop(args(1),2), extop(args(2),2)))
                  );
         of 1 do
           return(x);
	    end_case;
	    p:= args(0) div 2;
	    [p, q] := [_mult(args(1..p)), _mult(args((p+1)..args(0)))];
	    if p::dom::_mult <> FAIL then
         p::dom::_mult(p, q)
       elif q::dom::_mult <> FAIL then
         q::dom::_mult(p, q)
	    else
         _mult(p, q)
	    end_if
	end_proc
	
    	else
    	
	proc(x,y) local p, q; begin
	    case args(0)
	    of 2 do
		if domtype(y) <> dom then
		    case domtype(y)
		    of DOM_INT do
		    	return(dom::intmult(args()));
		    of R do
		        return(dom::_mult(x, dom::normalize(y, R::one)));
		    end_case;
		    return(FAIL);
		elif domtype(x) <> dom then
		    return(dom::_mult(y,x));
		end_if;
		return(dom::normalize(
			R::_mult(extop(args(1),1), extop(args(2),1)),
		        R::_mult(extop(args(1),2), extop(args(2),2))));
	    of 1 do return(x);
	    end_case;
	    p:= args(0) div 2;
     [p, q] := [_mult(args(1..p)), _mult(args((p+1)..args(0)))];
     if p::dom::_mult <> FAIL then
       p::dom::_mult(p, q)
     elif q::dom::_mult <> FAIL then
       q::dom::_mult(p, q)
     else
       _mult(p, q)
     end_if
     end_proc

    	end_if
    end_if);

    _invert :=
    proc(x: dom)
    begin
      if R::iszero(extop(x,1)) then
        error("division by zero")
      end_if;
      dom::normalizePrime(extop(x,2), extop(x,1))
    end_proc;

    intmult :=
    proc(x: dom, i:DOM_INT)
    begin
      dom::normalize(R::intmult(extop(x,1),i), extop(x,2))
    end_proc;

    _power :=
    proc(x, n: DOM_INT)
    begin
      if n > 0 then
        dom::normalizePrime(R::_power(extop(x,1), n),
                            R::_power(extop(x,2), n))
      elif n = 0 then
        dom::one
      else
        if R::iszero(extop(x,1)) then
          error("division by zero")
        end_if;
        dom::normalizePrime(R::_power(extop(x,2), -n),
                            R::_power(extop(x,1), -n))
      end_if
    end_proc;

    retract := (if R::hasProp(Cat::GcdDomain) then
	if R::hasProp(Ax::canonicalUnitNormal) then
	    proc(x) begin
		if R::equal(extop(x,2), R::one) then extop(x,1)
		else FAIL end_if
	    end_proc
	else
	    proc(x) local z; begin
		z:= R::_invert(extop(x,2));
		if z = FAIL then FAIL else R::_mult(extop(x,1), z) end_if
	    end_proc
	end_if
    else
	proc(x) begin R::_divide(extop(x,1), extop(x,2)) end_proc
    end_if);

    D := ( if R::hasProp(Cat::DifferentialRing) then
	proc(x) begin
	    if args(0) <> 1 then error("wrong no of args") end_if;
	    dom::normalize(
		R::_subtract(
		    R::_mult(R::D(extop(x,1)), extop(x,2)),
		    R::_mult(extop(x,1), R::D(extop(x,2)))),
		R::_power(extop(x,2),2))
	end_proc
    elif R::hasProp(Cat::PartialDifferentialRing) then
	proc(a,x) begin
	    if args(0) <> 2 then error("wrong no of args") end_if;
	    dom::normalize(
		R::_subtract(
		    R::_mult(R::D(a,extop(x,1)), extop(x,2)),
		    R::_mult(extop(x,1), R::D(a,extop(x,2)))),
		R::_power(extop(x,2),2))
	end_proc
    end_if );
    
    diff := ( if R::hasProp(Cat::PartialDifferentialRing) then
	proc(f,x) begin
	    case args(0)
	    of 2 do
		return(dom::normalize(
		    R::_subtract(
			R::_mult(R::diff(extop(f,1),x), extop(f,2)),
			R::_mult(extop(f,1), R::diff(extop(f,2),x))),
		    R::_power(extop(f,2),2)));
	    of 1 do return(f);
	    of 0 do error("wrong no of args");
	    end_case;
	    dom::diff(dom::diff(f,x), args(3..args(0)))
	end_proc
    end_if );

    random :=
    proc()
    begin
      R::random();
      if R::iszero(%) then
        dom::zero
      else
        dom::normalize(R::random(), %)
      end_if
    end_proc;


    // initialization of the domain
      
    begin
    if args(0) <> 1 then 
      error("wrong no of args") 
    end_if;
      if R::hasProp(Cat::IntegralDomain) <> TRUE then
        error("no integral domain")
      end_if
    end_domain:


// end of file 
