//   

/*++
Dom::ImageSet - domain of infinite sets, written as images
             of sets under mappings


An ImageSet represents the object {f(x1,..,xn); x1 \in S_1, ..., x_n \in S_n},
                  where S_1,..,S_n are sets and
                  f: S_1 x ... x S_n -> "the universe" is a mapping
                  (i.e. no set containing the range of f needs to be
                   known).

                  Operands:

                  0 - Dom::ImageSet
                  1 - f : DOM_EXPR
                  2 - lv: list of variables
                  3 - ls : list of sets, or multidimesional set


++*/

domain Dom::ImageSet
  local
  mapplus, mapmult, mapleftpower,
  avoidAliasProblem, // local methods
  identTable: DOM_TABLE;

  inherits Dom::BaseDomain;
  category Cat::Set;
    // no axioms

    // entries

    allowVectors:= FALSE;

    // slots

    expr:= M -> extop(M,1);

    variables:= M -> extop(M,2);

    sets:= M -> extop(M, 3);

    nvars:= M-> nops(extop(M,2));

    // methods:
        
    /* Creates elements without any type checking and conversions.
       Has to be changed when the internal representation changes. */
    create := () -> new(dom, args());
    
    new :=
    proc(f, x, S)
      local i,j, pos:DOM_INT,
      g, X, reduced,
      thisDom:DOM_DOMAIN, inds,
      newlist, newsets, cond: Type::Boolean;
      save MAXEFFORT;
    begin
      thisDom:= dom;
      // try conversion
      if args(0)=1 then
         dom::convert(f);
         if %=FAIL then
            error("Cannot convert to type ".expr2text(dom))
         else
            return(%)
         end_if
      end_if;

      // argument checking
      if testargs() then
        if args(0)<2 or args(0)>3 then
          error("Wrong number of args")
        end_if;
        if type(f) <> DOM_LIST and type(f) <> matrix and
          not testtype(f, Type::Arithmetical) then
          error("Wrong type of first argument")
        end_if;
        if not contains({DOM_IDENT, "_index", DOM_LIST}, type(x)) then
          error("Second argument must be identifier or list")
        end_if;
        if args(0)=3 then
          if type(x)=DOM_LIST then
            if type(S) = DOM_LIST then
              if nops(x)<>nops(S) then
                error("Both lists must have the same length")
              end_if;
              if testtype(S, Type::ListOf(Type::Set)) = FALSE then
                error("Third argument must consist of sets")
              end_if;
              if map({op(S)}, expr2text@type) intersect
                {"solvelib::cartesianPower"} <> {} then
                error("Third argument must not contain vector sets")
              end_if;
              if nops(x)<>nops({op(x)}) then
                error("Duplicate identifier")
              end_if
            else  //if testtype(S, Type::Set) = FALSE then
              error("Third argument is not a list")
            // elif S <> {} and solvelib::dimension(S) <> nops(x) then
            //  error("Third argument has wrong dimension")
            end_if;
          elif type(S)=DOM_LIST then
            error("Third argument must not be a list if the second is not")
          end_if;
        end_if;
        // test operands of S for being of "Type::Set" ?
      end_if; // arg-check

      if type(S) <> DOM_LIST then
        S:= [S]
      end_if;

      if type(x) <> DOM_LIST then
        x:= [x]
      end_if;
      
      // variable ranging over empty set?
      if contains(S, {}) > 0 then
        return({})
      end_if;


      if dom::allowVectors and type(f) <> DOM_LIST and type(f) <> matrix then
        return(Dom::ImageSet(f, x, S))
      end_if;

      if type(f) = DOM_LIST then
        f:= matrix(f)
      end_if;

      // handle special cases for f
      case expr2text(type(f))
        of "stdlib::Undefined" do
          return({})
        of "DOM_IDENT" do
          if (i:=contains(x, f)) > 0 then
            if nops(x) = 1 then
              return(S[1])
            else
              MAXEFFORT:= MAXEFFORT/nops(S);
              cond:= _or(solvelib::isEmpty(S[j]) $j=1..i-1,
                         solvelib::isEmpty(S[j]) $j=i+1..nops(S));
                     return(piecewise([cond, {}],
                                      [not cond, S[i]])
                            )
            end_if
          else
            MAXEFFORT:= MAXEFFORT/nops(S);
            cond:= _or(solvelib::isEmpty(S[i]) $i=1..nops(S));
            return(piecewise([cond, {}],
                           [not cond, {f}])
                 )
          end_if
        of "Dom::Matrix()" do
          if not dom::allowVectors then
            return(solvelib::VectorImageSet(f, x, S))
          end_if;
          g:= [op(f)];
          // prefer a cartesian power over {[x]; x in some_set}
          if nops(g) = 1 and (i:= contains(x, op(g, 1))) > 0 then
            return(solvelib::cartesianPower(S[i], 1))
          end_if;
          if g = x and nops({op(S)}) = 1 then
            // {[x, y, z] | x in A, y in A, z in A}
            // becomes A^3
            return(solvelib::cartesianPower(S[1], nops(S)))
          end_if;
          break
      // prefer a piecewise consisting of image sets over
      // an image set with piecewise first operand
        of "piecewise" do
          case nops((inds:= indets(f) intersect {op(x)}))
            of 0 do
              return(piecewise::extmap(f, dom::new, x, S))
            of 1 do
              // the conditions in the pw depend on one of the
              // variables
              // determine the position of that variable
              inds:= op(inds, 1);
              pos:= contains(x, inds );
              // f = expr1 if cond1(inds), expr2 if cond2(inds) etc
              // we write {f(inds); inds in S}
              // as {expr1(inds); inds in solve(cond1) intersect S}
              // union {expr2(inds); inds in solve(cond2) intersect S}
              // union ....
              MAXEFFORT:= MAXEFFORT/nops(f)/2;
              return
              (_union
               (
                Dom::ImageSet
                (piecewise::expression(f, i),
                 x,
                 subsop(S, pos=
                        solvelib::solve_intersect
                        (solve(piecewise::condition(f, i), inds), S[pos]))
                 )
                $i=1..nops(f)
                )
               )
              // otherwise: no special code to handle this
          end_case;

      end_case;


      if nops(S)=1 and type(S[1]) = piecewise then
        // at the moment, do not flatten out if there is more
        // than one operand because this would produce many branches
        return(piecewise::extmap(op(S,1), set -> Dom::ImageSet(f, x, [set])))
      end_if;


      if contains(S, undefined) > 0 then
        return(undefined)
      end_if;

      if contains(S, {}) > 0 then
        return({})
      end_if;

      for i from 1 to nops(S) do
        X:= op(x, i);
        
        case type(op(S, i))
        of Dom::Multiset do  
          S[i]:= coerce(S[i], DOM_SET);
          // fall through
        of DOM_SET do
          if nops(S) = 1 or nops(op(S, i)) = 1 then
            reduced:= {(if traperror((g:=(f | X = op(S, [i, j])))) = 0 then
                        g
                      end_if)
                      $j=1..nops(op(S, i))};
            if nops(S) = 1 then
              return(reduced)
            else
              delete x[i];
              delete S[i];
              return(_union(Dom::ImageSet(g, x, S) $g in reduced))
            end_if;
          else
            break
          end_if;  
          // NOT REACHED
          assert(FALSE)
        /*  
        of "_union" do
          // flatten out
          // this would give conflicts with flattenImageSet
          return(_union(Dom::ImageSet(f, x, subsop(S, i = op(S, [i, j]))) $j=1..nops(op(S, i)) ))
        */  
        of Dom::ImageSet do
          // X ranges over {f(y1, ..., yk); y1 in T1, ..., yk in Tk}
          // substitute for X, delete X and op(S, i), and add the yi and Ti to the list of sets
          S[i]:= solvelib::avoidAliasProblem(S[i], indets([op(S, 1..i-1), op(S, i+1..nops(S)), f].x));
          if traperror((g:= (f | X = expr(op(S, i))))) <> 0 then
            return({})
          else  
            newlist:= [op(x, 1..i-1), op(x, i+1..nops(x))] . extop(op(S, i), 2);
            newsets:= [op(S, 1..i-1), op(S, i+1..nops(S))] . extop(op(S, i), 3);
            return(Dom::ImageSet(g, newlist, newsets))
          end_if;
          // NOT REACHED          
          assert(FALSE)
        of piecewise do 
          return(piecewise::extmap(op(S, i),
          set -> thisDom::new(f, x, [op(S, 1..i-1), set, op(S, i+1..nops(S))])))
        end_case;        
      end_for;  
        
      if (i:= contains(S, Z_)) > 0 or (i:= contains(S, Q_)) > 0 then
        // we know that S = -S
        // do not do this for S=R_ (conflict with flattenImageSet)
        if not hastype(f, rationalize::hardToEval) and traperror((g:= evalAt(f, op(x, i)=-op(x, i)))) = 0 and
          Simplify::defaultValuation(g) < Simplify::defaultValuation(f)
        then
          f:= g
        end_if;
      end_if;  
      


      // remove unnecessary idents; but take care that the result
      // is the empty set if one of the S[i] is empty
      // (we call this condition cond)
      cond:= FALSE;
      for i from nops(x) downto 1 do
        if not has(f, x[i]) then
          cond:= cond or solvelib::isEmpty(S[i]);
          delete x[i];
          delete S[i]
        end_if;
      end_for;
      for i from nops(x) downto 1 do
        if domtype(S[i])=DOM_SET and nops(S[i])=1 then
          if traperror
            ((
              f:= evalAt(f, x[i] = op(S[i], 1))
              )) <> 0 then
            return({})
          end_if;
          delete x[i];
          delete S[i]
        end_if
      end_for;
      if nops(x)=0 then
        return(piecewise([cond, {}], [not cond, {f}]))
      else
        // nops(newlist) >= 1
        // substitute identifiers by appropriate ones
        newlist:= [0 $nops(x)];
        for i from 1 to nops(x) do
          newlist[i]:= solvelib::getIdent(S[i],
                                          (indets({f},All) minus indets(x))
                                          union {op(newlist, 1..i-1)}
                                          )
        end_for;
        // assert that we do not have duplicate identifiers
        assert(nops(newlist) = nops({op(newlist)}));
        return(piecewise
               ([cond, {}],
                [not cond,
                 dom::flattenImageSet(new(dom, subs(f, zip(x, newlist, _equal)
                                                    ), newlist, S))]
                ))
      end_if;
    end_proc;

    convert:=
    proc(x)
    begin
      if type(x)=dom then
        x
      else
        FAIL
      end_if
    end_proc;



    /* changevar(S, x, y) - replaces the variable x by y both
       in the expression and in the list of variables */

    // y does *not* implicitly get the same properties as x

    changevar:=
    proc(S:dom, x, y)
    begin
      extsubsop(S, 1=subs(extop(S,1), x=y), 2= subs(extop(S,2), x=y))
    end_proc;


    /* setvar(S, y) - sets the only variable of S to y */
    /* maps to set expressions */
    /* if S is not an ImageSet nor a set expression, nothing happens */

    // note that y gets no properties

    setvar:=
    proc(S, y: DOM_IDENT)

    begin
      case type(S)
        of dom do
          if dom::nvars(S)<>1 then
            error("Only allowed for image sets depending on one variable")
          end_if;
          return(dom::changevar(S, dom::variables(S)[1], y))
        of "_union" do
        of "_intersect" do
        of "_minus" do
          return(map(S, dom::setvar, y))
        otherwise
          S
      end_case
    end_proc;


    // avoidAliasProblem(iset, idents)
    // replaces idents in ImageSet by unused ones

    avoidAliasProblem:=
    proc(iset: dom, idents: DOM_SET): dom
      local
      ind: DOM_IDENT,
      newid: DOM_IDENT,
      toReplace: DOM_SET,
      toAvoid: DOM_SET,
      varlist: DOM_LIST,
      substitutions: DOM_LIST, 
      newsets
      ;
    begin
      newsets:= map(iset::dom::sets(iset), solvelib::avoidAliasProblem, idents);
      varlist:= iset::dom::variables(iset);
      toReplace:=  {op(varlist)} intersect (idents union indets(newsets, All));
      toAvoid:= {op(varlist)} union idents union indets(expr(iset),All) union
                indets(newsets, All);
      substitutions:= [];
      for ind in toReplace do
        newid:= solvelib::getIdent
        (newsets[contains(varlist, ind)], toAvoid);
        toAvoid:= toAvoid union {newid};
        substitutions:= substitutions.[ind = newid]
      end_for;
    
      extsubsop(iset,  1=subs(extop(iset,1), substitutions),
                2= subs(extop(iset,2), substitutions), 
                3= newsets
               )
    end_proc;




    _index:=
    proc(S)
      local
      indices: DOM_LIST,
      vars: DOM_LIST,
      substitutions: DOM_LIST,
      i: DOM_INT;
    begin
      indices:= [args(2..args(0))];
      if nops(indices) <> dom::nvars(S) then
        error("Number of indices does not match number of variables")
      end_if;
      for i from 1 to nops(indices) do
        if is(indices[i] in dom::sets(S)[i])
          = FALSE then
          error("Expression outside range of variable")
        end_if
      end_for;
      vars:= dom::variables(S);
      substitutions:=[(vars[i] = indices[i]) $i=1..nops(vars)];
      subs(dom::expr(S), substitutions,EvalChanges)
    end_proc;


    /* evaluation is done as follows:
       first, bound variables that have a value are substituted by new
       bound variables. After that, the expression and the sets are evaluated
       as usual. */

    evaluate:=
    proc(S: dom)
      local
        j: DOM_INT,
        i: DOM_INT,
        ausdr, ausdrev,
        l,
        X: DOM_IDENT,
        indices: DOM_LIST,
        vars: DOM_LIST,
        varsev: DOM_LIST,
        newvars: DOM_LIST,
        sets: DOM_LIST,
        setsev: DOM_LIST
      ;
    begin
      ausdr:= extop(S, 1);
      sets:= extop(S, 3);
      vars := extop(S, 2);
      indices:=[];
      for j from 1 to nops(vars) do
        if vars[j] <> eval(vars)[j] or property::hasprop(vars[j]) then
          indices:=indices.[j]
        end_if
      end_for;

      for j in indices do
        X:= solvelib::getIdent(sets[j], indets([ausdr, vars, sets],All));
        ausdr:= subs(ausdr, vars[j] = X);
        vars[j]:= X;
      end_for;
      varsev:= eval(vars);
      setsev :=  eval(sets);
      j:= 1;
      while j <= nops(varsev) do
        if setsev[j] = {} then
          return({})
        end_if;
        if domtype(varsev[j]) <> DOM_IDENT then
          // ignore variable that is not an identifier
          warning("Ignoring free variable ".expr2text(varsev[j]).
                  " because it is not an identifier");
          delete varsev[j];
          delete setsev[j]
        else
          if type(setsev[j]) <> RootOf then
            save varsev[j];
            traperror(assume(varsev[j] in setsev[j]))
          end_if;
          j:= j+1
        end_if;
      end_while;
      ausdrev:= eval(ausdr);

      
      // remove variables that do not occur
      map(varsev, unassume);
      newvars:={op(varsev)} intersect indets(ausdrev);
      if nops(newvars) < nops(vars) then
        if nops(newvars)=0 then
          // not an image set, the expression is constant
          return({ausdrev})
        end_if;
        // create list of index positions at which the variables
        // have survived
        l:=select([$1..nops(varsev)], i-> contains(newvars, varsev[i]));
        varsev:= [varsev[l[i]] $i=1..nops(l) ];
        setsev:= [setsev[l[i]] $i=1..nops(l)];
        dom::new(ausdrev, varsev, setsev)
      elif nops(setsev) = 1 and type(setsev[1]) = DOM_SET then
         // expand
        X:= op(varsev, 1);
        return({(if traperror((l:=(ausdrev | X=op(setsev, [1, i])))) = 0 then
                  l
                 end_if)
                 $i=1..nops(op(setsev, 1))})
      elif [ausdr, vars, sets] <> [ausdrev, varsev, setsev] then
        dom::new(ausdrev, varsev, setsev)
      else
        // original state may be kept
        dom::flattenImageSet(new(dom, ausdr, vars, sets))
      end_if;


    end_proc;



    // homogpointwise - create, for a given a law of composition
    // Operator on the complex numbers, a method that maps each tuple
    // of sets (A_1, ..., A_m) ---
    // where
    // Ai= {f_1(x_1i, x_2i, ..,x_{n_i}i); x_ki in S_ki for 1 <= k <= n_i}
    // to the set
    // { Operator(f_1(x_11, ...,x_{n_1}1), ..., f_m(x_1m, ...,x_{n_m}m));
    //   x_ki in S_ki for all (k,i) with 1<=i<=m and 1<=k <= n_i }
    // (pointwise application of a law of composition)

    homogpointwise:=
    proc(Operator)
      option escape;
      local d;
    begin
      d:=dom;

      proc()
        local arglist, vars, i, v, newid;

      begin
        arglist:=[args()];
        vars:={};
        for i from 1 to nops(arglist) do
          for v in {op(d::variables(arglist[i]))} intersect vars do
            /* v must be replaced by another variable to avoid
                               name conflicts */
            newid:= genident();
            if property::hasprop(v) then
              eval(hold(assume)(newid, getprop(v)))
            end_if;
            arglist[i]:=d::changevar(arglist[i], v, newid)
          end_for;
          vars:=vars union {op(d::variables(arglist[i]))};
        end_for;
        new(d, Operator(op(map(arglist,d::expr))),
            _concat(op(map(arglist, d::variables))),
            _concat(op(map(arglist, d::sets))))
      end_proc;
    end_proc;

    homog_plus:=dom::homogpointwise(_plus);

    homog_mult:=dom::homogpointwise(_mult);


// binpointwise(Operator)
/* create a function to compute Operator(imageset, otherset)
   where imageset is a Dom::ImageSet

   imageset = { expression(X1,..,Xn); X_i in S_i }

   that function returns

   { Operator(X, Y);  X in imageset, Y in otherset}
   =
   { Operator(expression(X1, .., Xn), Y); X_i in S_i, Y in otherset }

*/



    binpointwise:=
    proc(Operator)
      option escape;
      local d;
    begin
      d:= dom;
      
      // procedure to be returned
      
      proc(imageset: Dom::ImageSet, otherset)
        local Y;
      begin
        Y:= genident();
        d::new(Operator(d::expr(imageset), Y), d::variables(imageset).[Y],
               d::sets(imageset).[otherset])
      end_proc;

    end_proc;


// it is better to create fixed entries than to make
// binpointweise return one procedure per table entry
    pwplus := dom::binpointwise(_plus);
    pwmult := dom::binpointwise(_mult);



// bin_power ...



// bin_equal - try to simplify the expression S=T, where
// S and T are image sets
    bin_equal:=
    proc(S: dom, T:dom): Type::Boolean
      local varS, varT;
    begin
      varS:= dom::variables(S);
      varT:= dom::variables(T);
      if dom::sets(S) = dom::sets(T) and
        subs(dom::expr(S), zip(varS, varT, _equal)) = dom::expr(T) then
        TRUE
      else
        S = T
      end_if
    end_proc;

    // bin_union - compute {f(x) ; x \in A} union {g(y); y \in B}
    // this cannot contain more than a few heuristics

    bin_union:=
    proc(S: Dom::ImageSet, T: Dom::ImageSet)
      local SS, f, g, d, x;
    begin
      // do not try to handle more than one variable
      if dom::nvars(S)>1 or dom::nvars(T)>1 then
        return(FAIL)
      end_if;

      SS:=dom::setvar(S, dom::variables(T)[1]);
      if dom::expr(SS)=dom::expr(T) then
        // use the rule
        // { f(x); x \in S} union { f(x) ; x \in T } =
        // { f(x): x \in S \cup T }
        return(dom::new(dom::expr(T), dom::variables(T)[1],
                        dom::sets(T)[1] union dom::sets(SS)[1]))
      end_if;



      if dom::sets(SS)[1] = Z_ and dom::sets(T)[1]= Z_
        and dom::isLinear(SS) and dom::isLinear(T) then
        x:= dom::variables(SS)[1];
        f:= poly(dom::expr(SS), dom::variables(SS));
        g:= poly(dom::expr(T), dom::variables(T));
        d:= coeff(f, 0) - coeff(g, 0);

        // special case:
        //
        // the union of {a*n + b; n in Z} and {k*a*n + b; n in Z}
        // equals {a*n + b; n in Z}

        if iszero(d) then
          if type(coeff(f,1)/coeff(g,1)) = DOM_INT then
            return(T)
          elif type(coeff(g,1)/ coeff(f,1)) = DOM_INT then
            return(S)
          end_if
        end_if;

        // halving stepwidth (special case):
        // the union of
        //
        //  x       x       x       x
        //
        //  and
        //
        //      x       x       x       x
        //
        // equals
        //
        //  x   x   x   x   x   x   x   x
        //
        // that is:
        //  {ax+b; x in Z} union {ax+ (b-a)/2 ;  x\in Z} =
        //  { (a/2)x + b; x \in Z}


        if (coeff(f, 1) = coeff(g, 1) or coeff(f, 1)= -coeff(g, 1)) and
          (coeff(f, 1) = 2*d or coeff(f, 1) = -2*d) then
          return(Dom::ImageSet(d*x + coeff(f, 0), [x], [Z_]))
        end_if
      end_if;

      case S intersect T
        of S do
          return(T)
        of T do
          return(S)
      end_case;

      // else: heuristic did not succeed
      FAIL
    end_proc;

    /* This function intersects two sets of the type { a*Z_+b }
       with a,b being rational, or rational multiple of the same
       irrational value  
    */
    linZ_intersect :=
    proc( S, T )
      local A, B, a, b, c, d, f, g, h, v;
    begin
      if ( S=Z_) then
        A:=[1,0];
      elif domtype(S)=Dom::ImageSet and
        Dom::ImageSet::nvars(S)=1 and
        Dom::ImageSet::sets(S)=[Z_] then
        A := dom::isisetLinear( S )
      else
        return( hold(_intersect)(S, T) )
      end;

      if ( T=Z_) then
        B:=[1,0];
      elif ( domtype(T)=Dom::ImageSet and Dom::ImageSet::nvars(T)=1
        and Dom::ImageSet::sets(T)=[Z_] ) then
        B := dom::isisetLinear( T );
      else
        return( hold(_intersect)(S, T) )
      end;

      if ( A=FALSE) or (B=FALSE) then
        return( hold(_intersect)(S, T) )
      end_if;

      // A = {k*A[1] + A[2]; k in Z_}
      // B = {l*B[1] + B[2]; l in Z_}

      // if A[2] - B[2] is rational, we have to check
      // whether it can be written at all as a linear combination of
      // A[1] and B[1] over the rationals
      if is(A[2] - B[2] in Q_, Goal = TRUE) then
        case [is(A[1] in Q_), is(B[1] in Q_)]
          of [TRUE, FALSE] do
            // the only element of the intersection must satisfy l = 0 
            return({B[2]} intersect S)
          of [FALSE, TRUE] do  
          // k=0
            return({A[2]} intersect T)
        end_case
      end_if;
      
      // k*A[1] - l*B[1] = A[2] - B[2]  cannot hold 
      // if the left hand side is an integer while the right hand side is not
      if is(A[1] in Z_ and B[1] in Z_ and not (A[2]-B[2] in Z_), Goal = TRUE) then
        return({})
      end_if;  
      
      
      v := map
           ( [{op(A)}, {op(B)}],
            map,
            proc( X )
              local S;
            begin
              if testtype( X, "_mult" ) then
                S := split( X, testtype, Type::Rational );
                return( S[2] );
              elif X=0 then
                return( null() );
              elif testtype( X, Type::Rational ) then
                return( 1 );
              end_if;
            return( X );
            end_proc
            );

      if map(v, nops)=[1,1] and nops({op(v)})=2 and nops({op(v)} minus {{1}})=1
        and is(op({op(v)} minus {{1}}, [1,1]) in R_ minus Q_, Goal=TRUE) then
        case is(A[2]/A[1] in Z_ and B[2]/B[1] in Z_)
          of TRUE do
            return({0});
          of FALSE do
            return({});
        end_case;
      end_if;
      v := _union(op(v));
            
      if nops(v)>1 then
        return( hold(_intersect)(S, T) );
      elif nops(v)=1 then
        v := op(v);
        A[1] := A[1]/v;
        B[1] := B[1]/v;
        A[2] := A[2]/v;
        B[2] := B[2]/v;
      else
        v := 1;
      end_if;

      if not testtype(A[1],Type::Rational ) or
        not testtype(A[2],Type::Rational ) or
        not testtype(B[1],Type::Rational ) or
        not testtype(B[2],Type::Rational ) then
        return( hold(_intersect)(S, T) );
      end_if;

    // Now S intersect T are all rational numbers "q" which satisfy
    // q = A[1]*x + A[2] = B[1]*y + B[2]      with x,y in Z_, A/B[1,2] in Q_
    // We rewrite this to a*x = b*y + c         with a,b,c,x,y in Z_

      a := numer(A[1])*denom(B[2])*denom(A[2])*denom(B[1]);
      b := numer(B[1])*denom(A[1])*denom(B[2])*denom(A[2]);
      c := numer(B[2])*denom(A[1])*denom(B[1])*denom(A[2]) -
           numer(A[2])*denom(A[1])*denom(B[1])*denom(B[2]);

    // Now we search for f,g in Z_ such that
    // b*(f*y + g) +c (=bfy+bg+c) = 0 mod a     for all y in Z_
    // First we choose g, such that
    // bg+c = 0 mod a
    // Then we choose f, such that
    // bf = 0 mod a

      d := igcdex( a, b );
      if ( c mod d[1] <> 0 ) then
        return( {} );
      end_if;

      g :=-c/d[1]*d[3];
      f := a / d[1]:

    // Now all possible solutions are B[1]*(f*y + g) + B[2]   for all y in Z_
      f := f*B[1];
      g := g*B[1]+B[2];

    // At last simplify the expression
    // Z_/2 + 3/2 -> Z_/2
      h := ilcm( denom(f), denom(g) );
      g := ( g*h mod f*h )/h;

    // use symmetry Z_ = -Z_ to get a positive f
      Dom::ImageSet(specfunc::abs(f)*#k*v + g*v, #k, Z_)
      // ( Z_*specfunc::abs(f)+g )*v;
    end_proc;

    bin_intersect:=
    proc(S: Dom::ImageSet, T: Dom::ImageSet)
      local SS, SsetintersectTset, Sset, Tset, success: DOM_BOOL, A, B, r, s;
    begin
      if S = T then
          return(S)
      end_if;
      
      // do not try to handle more than one variable
      if dom::nvars(S)>1 or dom::nvars(T)>1 then
        return(FAIL)
      end_if;
      // special case
      SS:=dom::setvar(S, dom::variables(T)[1]);
      if dom::expr(SS)=dom::expr(T) then
        // in general, the rule
        //  { f(x); x \in S} intersect { f(x) ; x \in T } =
        // { f(x): x \in S intersect T }
        // does not hold!
        // but we check if S \subseteq T or T \subseteq S

        Sset:=dom::sets(SS)[1];
        Tset:=dom::sets(T)[1];
        if Sset = Tset then
          SsetintersectTset:= Sset;
          success:= TRUE
        else
          SsetintersectTset:=Sset intersect Tset;
          success:= (%=Sset or %=Tset)
        end_if;

        if success then
          return(dom::new(dom::expr(SS), dom::variables(SS)[1],
                          SsetintersectTset))
        end_if
      end_if;

      // special case
      // S and T are linear in C_
      if  ( (A := dom::isisetLinear(S) )<>FALSE ) and
        (dom::sets(S)[1] subset R_)=TRUE  and
        ( (B := dom::isisetLinear(T) )<>FALSE ) and
        (dom::sets(T)[1] subset  R_)=TRUE   then
        // parallel
        if iszero(A[1]) then
          return({dom::expr(S)} intersect T);
        end_if;
        if iszero(B[1]) then
          return({dom::expr(T)} intersect S);
        end_if;
        if ( is( A[1]/B[1], Type::Real )=TRUE ) then
          case is( (A[2]-B[2])/A[1], Type::Real )
            of TRUE do
              Sset:=dom::sets(S)[1];
              Tset:=dom::sets(T)[1];
              // special case: both sets depend on Z_
              if Sset=Z_ and Tset=Z_ then
                return( dom::linZ_intersect( S, T ) );
              end_if;
              // special case: boths sets depend on subsets of Z_
              if (Sset=Z_ or type(Sset)="_intersect" and contains({op(Sset)}, Z_)) and
                (Tset=Z_ or type(Tset)="_intersect" and contains({op(Tset)}, Z_)) then
                // we use the following rule:
                // {a*x + b | x in Z_ intersect A} = {a*x + b | x in Z_} intersect {a*x + b | x in A}
                // this conclusion works, since x -> a*x + b is injective
                SS := dom::linZ_intersect(extsubsop(S, 3=[Z_]), extsubsop(T, 3=[Z_]));
                if SS={} then return({}); end_if;
                if type(Sset)="_intersect" then
                  SS := SS intersect Dom::ImageSet(dom::expr(S), dom::variables(S), [_intersect(op({op(Sset)} minus {Z_}))]);
                end_if;
                if type(Tset)="_intersect" then
                  SS := SS intersect Dom::ImageSet(dom::expr(T), dom::variables(T), [_intersect(op({op(Tset)} minus {Z_}))]);
                end_if;
                return(SS);
              end_if;
              break;
            of FALSE do
              return( {} );
            end_case;
        elif ( is( A[1]/B[1], Type::Real )=FALSE ) then
          r := Im( (A[2]-B[2])/(A[1]) )/Im( B[1]/A[1] );
          s := Im( (B[2]-A[2])/(B[1]) )/Im( A[1]/B[1] );
//           assert( r*B[1]+B[2] = s*A[1]+A[2] );
          if is( r in dom::sets(T)[1], Goal=TRUE) and
            is( s in dom::sets(S)[1], Goal=TRUE) then
            return({ r*B[1]+B[2] })
          else
            return({})
          end_if;
        end_if;
      end_if;


      // special case: is one set contained in the other?
      // let S={f(x); x in S} and T={g(y); y in T}

      // if f(x) = g(y) has no solution, the intersection is empty
      // if f(x) = g(y) has a solution for y (whatever x is),
      // then every element of S
      // can be written as an element of T, hence S \subseteq T,
      // and their intersection is S

      // if f(x) = g(y) has a solution for x (whatever y is),
      // then every element of T
      // can be written as an element of S
      // hence the intersection is T

      /*

      empt:= solvelib::isEmpty(solve(expr(S)=expr(T), dom::variables(T)[1],
                               DontRewriteBySystem));
      if empt=FALSE then
        return(S)
      end_if;
      if not (empt=TRUE) then
        empt:= solvelib::isEmpty(solve(expr(S)=expr(T), dom::variables(S)[1],
                                 DontRewriteBySystem))
      end_if;

      case empt
        of TRUE do
          return({})
        of FALSE do
          return(T)
      end_case;
      // empt = UNKNOWN

*/
      FAIL
    end_proc;

    bin_minus:=
    proc(S: dom, T: dom)
      local linS, linT;
    begin
      // only simple cases can be settled here:
      if S=T then
        return({})
      end_if;

      // special case
      if dom::sets(S) = [Z_] and dom::sets(T) = [Z_] and
        (linS:= dom::isisetLinear(S)) <> FALSE and
        (linT:= dom::isisetLinear(T)) <> FALSE then
        // special case:
        // S = {  a*k + b; k in Z_}
        // T = {2*a*k + c; k in Z_}
        // if both sets have a point d in common, then
        // S minus T = {(2*a+1)*k + d; k in Z_}
        // to determine d, we have to solve
        // 2*ak + c = a*l + b
        // (2*k-l) = (b-c)/a
        // if this is solvable (i.e., the rhs is in Z_), we may take k=0 and
        // l = (c-b)/a; then c is a common point
        if linT[1]/linS[1] = 2 then
          case is((linS[2]-linT[2])/linS[1] in Z_)
            of FALSE do
              // no common point, S intersect T = {}
              return(S)
            of TRUE do  
              return(Dom::ImageSet(linS[1]*2*dom::variables(S)[1] +
                                   linT[2] + linS[1],
                                   dom::variables(S),
                                   dom::sets(S))
                     )
          end_case;
          // we do not know whether there is a point in the intersection -
          // give up
        end_if;
      end_if;
      
      case S intersect T
        of S do
          return({})
        of {} do
          return(S)
      end_case;

      FAIL
    end_proc;


    /*
     isisetLinear(A)
       returns a list [a, b] if A = { ax+b; x in S} for some S
       and FALSE in all other cases
    */
    isisetLinear :=
    proc( A : Dom::ImageSet )
      local p;
    begin
      if Dom::ImageSet::nvars( A )<>1 then
        return(FALSE)
      end_if;
      p := poly( dom::expr(A), dom::variables(A) );
      if p=FAIL then
        return(FALSE)
      end_if;
      if degree(p)>=2 then
        return(FALSE)
      end_if;
      p := [ coeff(p,1), coeff(p,0) ];
      if has(p, op(Dom::ImageSet::variables(A))) then
        return(FALSE)
      end_if;
      p
    end_proc;

    isisetPoly :=
    proc( A : Dom::ImageSet )
      local p;
    begin
      if Dom::ImageSet::nvars( A )<>1 then
        return( FALSE );
      end_if;
      p := poly( dom::expr(A), dom::variables(A) );
      if p=FAIL then
        return( FALSE );
      end_if;
      p := revert([coeff(p,All)]);
      if has( p, op(Dom::ImageSet::variables(A)) ) then
        return( FALSE );
      end_if;
      return( p );
    end_proc;

    isxprLinear :=
    proc( xpr, A : Dom::ImageSet, var=FAIL )
      local L, v, r;
    begin
      if var=FAIL then
        v := indets( xpr ) intersect { op(Dom::ImageSet::variables( A )) };
        if nops(v)<>1 then
          return( FALSE )
        end_if;
        v := op(v);
      else
        v := var;
      end_if;
      r := op(Dom::ImageSet::sets(A),contains( dom::variables( A ), v ) );
      if (r subset R_)<>TRUE then
        return( FALSE )
      else
        L := Type::Linear( xpr, [v] );
        if ( L=FALSE ) then
          return( FALSE )
        else
          return( [ op(L), v, r ] )
        end_if;
      end_if;
    end_proc;

    /* This functions intersects an ImageSet with R_.
     * I.e. it returns an as small as possible subset of iset
     * containing all real numbers of iset.
     * Warning: It cannot be assumed that the result is _only_ real!
   */
    intersectR_ :=
    proc( iset : Dom::ImageSet )
      local isReal, getZeroImPart, xpr, vars, sets, res, modified, T, v, i;
    begin
      if MAXEFFORT < 1 then return(iset) end_if;

      /* remove properties */
      
      vars := dom::variables( iset );
      sets:= dom::sets(iset);
      for i from 1 to nops(vars) do
        unassume(op(vars, i))
      end_for;

      // Check if 0 in Im(iset)
      T := Im(iset) assuming _and(op(vars, i) in op(sets, i) $i=1..nops(vars));
      if type(T) = Dom::ImageSet then
        v := dom::_simpexIn( 0, T, FALSE );
        if v=FALSE then
          return( {} )
        end_if;
      elif type(T) = DOM_SET and nops(T) = 1 then
        return(
               piecewise([op(T, 1) = 0, iset - I*op(T, 1)],
                         [op(T, 1) <> 0, {}]
                         )
               )
      else
        if is( 0 in T )=FALSE then return( {} ); end_if;
      end_if;

      /* This functions returns TRUE if xpr is a real number.
       * If this can't be determined the function returns FALSE.
       */
      isReal :=
      proc( xpr )
        local i;
      begin
        if testtype( xpr, Type::Real ) then
          return( TRUE )
        end_if;
        if ( i := is( Im(xpr)=0 ) )<>UNKNOWN then
          return( i );
        end_if;
        if testtype( xpr, DOM_IDENT ) and contains( vars, xpr ) and
          _subset( vars[ xpr ], R_ )=TRUE then
          return( TRUE )
        end_if;
        if testtype( xpr, "_plus" ) then
          return( _and( map( op(xpr), isReal ) ) )
        end_if;
        if testtype( xpr, "_mult" ) then
          return( _and( map( op(xpr), isReal ) ) )
        end_if;
        if testtype( xpr, "exp" ) then
          return( isReal(op(xpr)) );
        end_if;
        if testtype( xpr, "_power" ) and
          testtype( op(xpr,2), Type::PosInt ) then
          return( isReal(op(xpr,1) ) )
        end_if;
        FALSE;
      end_proc;

      /* This functions removes all parts that don't satisfy Im(xpr)=0
       * The result is so far equivalent to the input such that
       * xpr intersect R_ = getZeroImPart(xpr) intersect R_
       * The result is FAIL if Im(xpr) cannot be zero
       */
    getZeroImPart :=
    proc( xpr )
      local s, r, u, v;
      save MAXEFFORT;
    begin
      MAXEFFORT := MAXEFFORT*2/3;
      if nops( (v := freeIndets(xpr)) )=1 and contains( vars, op(v) )
      and
        (r := Type::Linear( xpr, [op(v)]) )<>FALSE then
        if is( Im(r[1])<>0, Goal=TRUE ) then
          v := op(v);
          vars[v] := vars[v] intersect {-Im(r[2])/Im(r[1])};
          if vars[v] = {} then
            return(FAIL)
          end_if;
          modified := TRUE;
        end_if;
      end_if;

      case type( xpr )
        of "_plus" do
          s := split( xpr, isReal );
          s[2] := s[2]+s[3];
          if s[2]=0 then
            break
          end_if;
          if not testtype( s[2], "_plus" ) and
            ( r:= Dom::ImageSet::isxprLinear( s[2], iset ) )<>FALSE
            and
            indets(s[1]) intersect indets(s[2]) ={} then
            if is( 0 in r[4] )=TRUE then
              return( s[1] )
            end_if;
            if is( 0 in r[4] )=FALSE then
              return( FAIL )
            end_if;
          end_if;
          if indets( s[2] )={} and is( Im(s[2])<>0 )=TRUE then
            return( FAIL )
          end_if;
          if s[1]<>0 then
            return( s[1] + getZeroImPart( s[2] ) );
          end_if;
          break;

        of "_mult" do
          s := split( xpr, isReal );
          s[2] := s[2]*s[3];
          /* special case: test if we have one (possibly) imaginary identifier */
          if testtype( s[2], DOM_IDENT ) and contains( vars, s[2] )
          and
            not testtype( ( v := vars[s[2]] intersect R_ ), "_intersect" ) then
            vars[ s[2] ] := v;
            modified := TRUE;
          end_if;

          /* If s[2] is purely imaginary then s[1] has to be 0 */
          if Re( s[2] )=0 then
            v := is( s[1]=0 );
            if v=FALSE then return( FAIL ); end_if;
            if v=TRUE then return( 0 ); end_if;
            v := select( indets( s[1] ), X-> contains( vars, X ) );
            if nops(v)=1 then
              v := op(v);
              MAXEFFORT := MAXEFFORT/10;
              r := solve( xpr, v );
              if not testtype( r, "solve" ) and
                not testtype( ( r := vars[ v ] intersect r ),
                             "_intersect" )
                then
                if r= {} then
                  return(FAIL)
                end_if;
                vars[ v ] := r;
                return( 0 );
              end_if;
            end;
          end_if;

          if s[2]=1 then
            return( s[1] )
          end_if;
          if not testtype( s[2], "_mult" ) then
            v := getZeroImPart( s[2] );
            if v=FAIL then
              r := {};
              if testtype( s[1], DOM_IDENT ) then
                r := { s[1] }
              end_if;
              if testtype( s[1], "_mult" ) then
                r := select( {op(s[1])},
                            X->( testtype( X, DOM_IDENT ) and
                                contains( vars, X ) ) );
              end_if;
              v := map( r, X-> _lazy_and(contains(vars, X), is( 0 in vars[X] )) );
              if contains( v, TRUE ) then
                return(0)
              end_if;
              return(xpr)
            end_if;
            return( s[1]*v )
          end_if;
          break;

        of "exp" do
          if (r:=Dom::ImageSet::isxprLinear( op(xpr), iset ) )<>FALSE
          and
            r[2]=0 then
            if r[1]=0 then break; end_if;
            if Re(r[1])=0 then
              s:= Dom::ImageSet(#X*Im(r[1]), #X, r[4]);
              u := solvelib::isEmpty( s intersect ( 2*PI*Z_ ) );
              v := solvelib::isEmpty( s intersect ( 2*PI*Z_+PI ) );
              if {u, v} minus {TRUE, FALSE} <> {} then break; end_if;
              if u then r := {} else r := {1}; end_if;
              if not v then r := r union {-1}; end_if;
              if nops( r ) = 1 then return( op(r) ); end_if;
              s:=genident();
              vars[ s ] := r;
              return( s );
            end_if;
          end_if;
          break;
      end_case;
      xpr;
    end_proc:

      modified := FALSE;
      xpr := iset::dom::expr( iset );
      vars := table( (zip( iset::dom::variables(iset),
      iset::dom::sets(iset),
                          _equal )));
      res := getZeroImPart( xpr );
      if res=FAIL then return( {} ); end_if;
      if modified or res <> xpr then
        // special case: the resulting ImageSet is of the type { x, x in A };
        if testtype(res, DOM_IDENT) and contains( vars, res ) then
          return( vars[res] );
        end_if;
        
        return( Dom::ImageSet( res, map( [op(vars)], op, 1), map(
        [op(vars)], op, 2) ) );
      end_if;
      return( iset );
    end_proc;

    homog_union :=
    proc()
      local i, j, operands;
      save MAXEFFORT;
    begin
      // test for correct types
      if not map( {args()}, testtype, Dom::ImageSet ) = {TRUE} then
        return( hold(_union)( args() ) );
      end_if;

      i := 1;
      operands := [ args() ];
      MAXEFFORT:= MAXEFFORT/ (nops(operands)^2 / 2);
      if MAXEFFORT > 5 then
        while i<nops( operands ) do
          j := i+1;
          while j<=nops( operands ) do
            if ( dom::_subset( operands[i], operands[j] )=TRUE ) then
              operands[i] := null();
            elif ( dom::_subset( operands[j], operands[i] )=TRUE ) then
              operands[i] := null();
            end_if;
            j := j+1;
          end_while;
          i := i+1;
        end_while;
      end_if;

      return( hold(_union)( op(operands) ) )
    end_proc;

   /* binary operations, inhomogenous case   */


    inhomog_union:=
    table
    (
     solvelib::BasicSet =
     proc( iset: Dom::ImageSet, S: solvelib::BasicSet )
     begin
       if Dom::ImageSet::_subset( iset, S )=TRUE then
         return( S )
       end_if;
       if Dom::ImageSet::_subset( S, iset )=TRUE then
         return( iset );
       end_if;
       hold(_union)( iset, S )
     end_proc,

     DOM_SET=
     proc(iset: Dom::ImageSet, s: DOM_SET)
       local r;
       save MAXEFFORT;
     begin
       if s={} then
         iset
       else
         MAXEFFORT:= MAXEFFORT/nops(s);
         r := select( s, X->is( X in iset )=TRUE );
         if r={} then
           return(FAIL)
         end_if;
         s := s minus r;
         if s={} then
           iset
         else
           hold(_union)(iset, s)
         end_if
       end_if
     end_proc,


     DOM_EXPR=
     proc(iset: Dom::ImageSet, a: DOM_EXPR)
       local l, i;
     begin
       case type(a)
         of "_minus" do
           if op(a, 1) subset iset = TRUE then
             return(iset)
           end_if;
           if op(a, 2) subset iset = TRUE then
             return(iset union op(a, 1))
           end_if;
           break
         of "_intersect" do
           // apply iset union (A intersect B intersect C ..) =
           // (iset union A) intersect (iset union B) ...
           // only if all of the unions can be computed
           l:= [FAIL $ nops(a)];
           for i from 1 to nops(l) do
             l[i]:= iset union op(a, i);
             if type(l[i]) = "_union" then
               break
             end_if;
           end_for;
           if contains(l, FAIL) = 0 then
             return(_intersect(op(l)))
           end_if;
           break
       end_case;
       FAIL
     end_proc,


     piecewise = piecewise::_union
     );


    inhomog_intersect:=
    table
    (
     DOM_SET=
     prog::remember(
     proc(iset:Dom::ImageSet, s:DOM_SET)
       local S, S1, yes, no, dontknow;
     begin

       [yes, no, dontknow]:= split(s, u -> is(u in iset));

       if dontknow = {} then
         return(yes)
       end_if;

       if nops(iset::dom::variables(iset))>1 then
         return(FAIL)
       end_if;

       iset:= iset::dom::avoidAliasProblem(iset, indets(s,All));
     // let iset= {f(x); x in A }
     // At first, get f^(-1)(s) and inters. with A
     // apply f
       S1:= solvelib::preImage(iset::dom::expr(iset),
                           iset::dom::variables(iset)[1], dontknow);                   
       S:= solvelib::solve_intersect
       (
        S1,
        iset::dom::sets(iset)[1]
        );
       S:= solvelib::avoidAliasProblem(S, {iset::dom::variables(iset)[1]});
       solvelib::substituteBySet(iset::dom::expr(iset),
                                 iset::dom::variables(iset)[1], S)
       union yes
     end_proc,
     property::depends, PreventRecursion,
                       ()-> hold(_intersect)(args())
     ),
     
     piecewise = piecewise::_intersect,


     Dom::Interval=
     proc(iset:Dom::ImageSet, iv:Dom::Interval)
       local preim, A, niv, result;
     begin
       if MAXEFFORT < 1 then return(FAIL) end_if;

       // local method fracsymb - the same mathematical meaning as frac,
       // but avoids floating point values
       // fracsymb:= x -> x - floor(x);
       
       iset := iset::dom::intersectR_( iset );
       if not testtype( iset, Dom::ImageSet ) then
         return( iv intersect iset )
       end_if;

       if nops(iset::dom::variables(iset))>1 then
         return(FAIL)
       end_if;

       if op(Dom::ImageSet::sets(iset),1) subset Z_ = TRUE then
         if (A := Dom::ImageSet::isisetLinear( iset )) <>FALSE then
           // iset = {A[1]*k + A[2]; k in Z_}
           if type( ( niv := ( iv-A[2] ) /A[1] ) )=Dom::Interval then
             niv := solvelib::BasicSet::inhomog_intersect["Dom::Interval"]
             ( Z_, niv  );
             if type(niv) = "_intersect" then
               niv := split( [op(niv)], X->type(X)=Dom::Interval );
               if nops(niv[1])=1 then
                 niv := op(niv,[1,1]) * A[1] + A[2];
                 if niv<>iv and type(niv)=Dom::Interval then
                   result:= iset intersect niv;
                   if type(result) <> "_intersect" or 
                      is(niv::dom::left(niv) > iv::dom::left(iv) or
                      iv::dom::right(niv) < iv::dom::right(iv), Goal = TRUE) then
                      return(result)
                   end_if
                 end_if;
               end_if;
             end_if; // type(niv) = "_intersect"
           end_if;
         end_if; // isisetLinear(iset)
       end_if;
     
     // let iset= {f(x); x \in A }
     // At first, get f^(-1)(s) and intersect with A
     // then apply f
       if has( iv, iset::dom::variables(iset)[1] ) then
         return( FAIL )
       end_if;
       unassume(iset::dom::variables(iset)[1]);
       preim:=solvelib::preImage(iset::dom::expr(iset),
                                 iset::dom::variables(iset)[1], iv);
       if type(preim) = Dom::ImageSet then
         // to avoid infinite recursion
         return(FAIL)
       end_if;
       solvelib::solve_intersect
       (
        preim,
        iset::dom::sets(iset)[1]
        );
       if type(%)="_intersect" then
         FAIL
       else
         solvelib::substituteBySet(iset::dom::expr(iset),
                                   iset::dom::variables(iset)[1], %)
       end_if
     end_proc,


     DOM_EXPR=
     proc(iset: Dom::ImageSet, xpr: DOM_EXPR)
       local l, i, preim;
     begin
       case type(xpr)
         of "_union" do
           l:=[op(xpr)];
           for i from 1 to nops(l) do
             l[i]:=iset intersect l[i];
             if type(%)="_intersect" then
               return(FAIL)
             end_if
           end_for;
           return(_union(op(l)))
         of "_minus" do
           // special case: iset intersect (C_ minus A) ->
           // iset minus A
           if op(xpr, 1) = C_ then
             return(iset minus op(xpr, 2))
           end_if;
           if iset::dom::nvars(iset)>1 then
             return(FAIL)
           else
             // delete properties from variable
             save op(iset::dom::variables(iset));
             // { f(X); X in S} intersect (A minus B)
             // get the set of all X in S s.t. f(X) is in A minus B

             preim:= solvelib::preImage(iset::dom::expr(iset),
                                 iset::dom::variables(iset)[1],
                                 xpr);
             if hastype(preim, Dom::ImageSet) and domtype(op(iset::dom::sets(iset),1)) = DOM_EXPR then
               // the following intersection is not easier to compute than the original problem
               return(FAIL)
             end_if;
             preim:= solvelib::solve_intersect
             (
              preim,
              op(iset::dom::sets(iset),1)
              );
             if type(preim)="_intersect" then
               return(FAIL)
             else
               // apply f
               return(solvelib::substituteBySet
                      (iset::dom::expr(iset),
                       op(iset::dom::variables(iset)), preim))
             end_if
           end_if
        end_case;
        FAIL
     end_proc
     );


    inhomogright_minus:=
    table
    (
     DOM_SET =
     proc(S: DOM_SET, iset: Dom::ImageSet)
       local yes, no, dontknow;
     begin
       [yes, no, dontknow]:= split(S, x-> is(x in iset));
       if nops(dontknow) = 0 then
         return(no)
       elif nops(yes) = 0 then
         return(FAIL)
       else
         return(hold(_union)(no, hold(_minus)(dontknow, iset)))
       end_if
     end_proc,

     DOM_EXPR =
     proc( X, iset: Dom::ImageSet )
       local i, hit, v, operands;
     begin
       if testtype( X, "_union" ) then
         i := 1;
         hit := TRUE;
         operands := [op(X)];
         while i<=nops(operands) do
           v := operands[i] minus iset;
           if not testtype( v, "_minus" )  then
             operands := subsop( operands, i=v )
           else
             hit := FALSE;
           end_if;
     i:=i+1;
     end_while;
     if hit then
       return( _union( op(operands) ) );
     end_if:
     X := _union( op(operands) );
     end;
     return( hold(_minus)( X, iset ) );
     end_proc
     );

    inhomogleft_minus:=
    table(
          solvelib::BasicSet = proc( iset: Dom::ImageSet, S: solvelib::BasicSet )
            local a;
          begin
            a := iset intersect S;
            if not testtype( a, Type::Union( "_intersect", solvelib::BasicSet ) ) and a<>S then return( iset minus a ); end_if;
            hold(_minus)(iset, S)
          end_proc,

          DOM_EXPR = proc( iset: Dom::ImageSet, X )
          local i, newiset, operands;
          begin
            if testtype( X, "_union" ) then
              i := 1;
              operands := [op(X)];
              while i<=nops(operands) do
                newiset:= iset minus op(operands, i);
                if type(newiset) = "_minus" then
                  i:= i+1
                else
                  iset:= newiset;
                  delete operands[i];
                end_if
              end_while;
              if nops(operands)=0 then
                return( iset )
              end_if;
              X := _union( op(operands) );
            end;

            return( hold(_minus)( iset, X ) );
          end_proc,

          DOM_SET=
          proc(iset: Dom::ImageSet, S: DOM_SET)
            local a, linear, linvec;
            save MAXEFFORT;
          begin
            linear := bool(Dom::ImageSet::nvars(iset)=1 and ( linvec := Type::Linear( Dom::ImageSet::expr(iset), Dom::ImageSet::variables(iset) ) )<>FALSE);
            if S={} then 
               return(iset)
            else
               MAXEFFORT:= MAXEFFORT/nops(S)
            end_if;
            for a in S do
              if is(a in iset) = FALSE then
                S:= S minus {a}
              else
                if linear and is(linvec[1]<>0, Goal=TRUE) then
                  iset := extsubsop( iset, 3=[ op(Dom::ImageSet::sets(iset)) minus {(a-linvec[2])/linvec[1]} ] );
                  S:= S minus {a}
                end_if;
              end_if
          end_for;
          if S={} then
            iset
          else
            hold(_minus)(iset, S)
          end_if
          end_proc
          );


    inhomog_plus:=
    table(
          DOM_INT=mapplus,
          DOM_RAT=mapplus,
          DOM_COMPLEX=mapplus,
          DOM_FLOAT=mapplus,
          DOM_IDENT=mapplus,
          stdlib::Infinity = mapplus,

          DOM_EXPR=
          proc(iset, a)
          begin
            if testtype(a, Type::Arithmetical) then
              mapplus(iset, a)
            elif type(a) = "_union" then
              map(a, _plus, iset)
            elif testtype(a, Type::Set) then
              Dom::ImageSet::pwplus(iset, a)
            else
              FAIL
            end_if
          end_proc,

          DOM_SET =
          proc(iset, S)
            local x;
          begin
            _union(iset + x $ x in S)
          end_proc,
          
          Dom::Interval = dom::pwplus
          );


    inhomog_mult:=
    table(
          DOM_INT=mapmult,
          DOM_RAT=mapmult,
          DOM_COMPLEX=mapmult,
          DOM_FLOAT=mapmult,
          DOM_IDENT=mapmult,
          stdlib::Infinity=mapmult,

          DOM_EXPR=
          proc(iset, a)
          begin
            if testtype(a, Type::Arithmetical) then
              mapmult(iset, a)
            elif type(a) = "_union" then
              map(a, _mult, iset)
            elif testtype(a, Type::Set) then
              Dom::ImageSet(expr(iset)*#X, extop(iset, 2).[#X],
                            extop(iset, 3).[a])
            else
              FAIL
            end_if
          end_proc,

         
          DOM_SET =
          proc(iset, S)
            local x;
          begin
            _union(iset * x $ x in S)
          end_proc,
          Dom::Interval = dom::pwmult
          );


    inhomogleft_power:=
    table(
          DOM_INT=mapleftpower,
          DOM_RAT=mapleftpower,
          DOM_COMPLEX=mapleftpower,
          DOM_FLOAT=mapleftpower,
          DOM_EXPR=mapleftpower,
          DOM_IDENT=mapleftpower
          );



    isEmpty:=
    proc(S:dom)
    begin
      // a set has an empty image iff it is empty itself
      map(dom::sets(S), solvelib::isEmpty);
      _or(op(%))
    end_proc;

    /* simpexIn(a, S) - expression equivalent to a \in S */

    simpexIn:=
    proc(a, S: Dom::ImageSet)
      local v, T, x, i, maxeffort;
      save MAXEFFORT;
    begin
/*
      // very simple case: expr(S) is linear
      if nops((x:= dom::variables(S))) = 1 then
         x:= op(x, 1);
         if type((v:= dom::expr(S) / x)) = DOM_INT then
		return(a/v in dom::sets(S)[1])
         end_if
      end_if;
*/
      if MAXEFFORT < 10 or 
         MAXEFFORT < 500 and freeIndets([a, S]) <> {} then
        return(hold(_in)(a, S) )
      end_if;

      /* some heuristics */
      MAXEFFORT:= MAXEFFORT/3;
      maxeffort:= MAXEFFORT;
      // one third for _subtract ...
      T := S-a;
      // two thirds for the rest: one third for if, one third for _simpexIn
      // in the if-clause, use one half for intersectR
      MAXEFFORT:= MAXEFFORT/2;
      if ( Dom::ImageSet::freeIndets(T) intersect indets(a) = {} ) then
        if testtype( T, Dom::ImageSet ) then T := Dom::ImageSet::intersectR_( T ) end_if;
        if testtype( T, Dom::ImageSet ) then
          MAXEFFORT:= MAXEFFORT/2;
          v := dom::_simpexIn( 0, T );
          if v=TRUE or v=FALSE then return( v ); end_if;
          if testtype( ( T:=Re(T) ), Dom::ImageSet  ) then
            v := dom::_simpexIn( 0, T );
            if v = FALSE then return( FALSE ); end_if;
          end_if;
        else
          v := is( 0 in T );
          if v<>UNKNOWN then return( v ); end_if;
        end_if;
      end_if;

      v := Dom::ImageSet::sets( S );
      for i from 1 to nops(v) do
        if testtype(op(v,i), DOM_SET ) then
          v := op(v,i);
          x := op( Dom::ImageSet::variables( S ), i );
          T := Dom::ImageSet::expr( S );
          return( _or(
            op( map( v, proc(X)
              local set;
            begin
              set := Dom::ImageSet( subs(T, x=X), dom::variables(S), dom::sets(S) );
              if testtype( set, Dom::ImageSet ) then
                dom::simpexIn( a, set );
              else
                is( a in set );
              end_if;
            end_proc
          ) ) ) );
        end_if;
      end_for;

      MAXEFFORT:= maxeffort;
      dom::_simpexIn( a, S )
    end_proc;

    _simpexIn :=
    proc( a, S : Dom::ImageSet, trySolve=TRUE )
      local x, eq, newx, xpr, v;
      save MAXEFFORT;
    begin
      if MAXEFFORT < 10 or MAXEFFORT < 1000 and freeIndets([a, S]) <> {} then
        return( hold(_in)(a, S) )
      end_if;
      /* special case x*y*.... = 0 */
      xpr := dom::expr( S );
      if a = 0 and testtype( xpr, "_mult" ) then
        MAXEFFORT:= MAXEFFORT/nops(xpr);
        x := _or( map( op(xpr), proc(X)
            local T;
          begin
            T := Dom::ImageSet( X, dom::variables(S), dom::sets(S) );
            if testtype( T, Dom::ImageSet ) then
              v := dom::_simpexIn( 0, T );
              if ( v<>TRUE ) and ( v<>FALSE ) then v := UNKNOWN; end_if;
            else
              v := is( 0 in T );
            end_if;
            v;
          end_proc ) );
          if x<>UNKNOWN then return( x ); end_if;
          return( hold(_in)(a, S) );
      end_if;

      if dom::nvars(S) = 1 and testtype(a, Type::Constant) then
        x:= op(dom::variables(S));
        eq:= expr(S);
        /*
        if has(a, x) then
          // avoid alias - problem
          newx:= genident();
          eq:= subs(eq, x= newx)
        else
          newx:= x
        end_if;
        */
        newx:= x;
        if trySolve and ( testtype(eq, Type::PolyExpr([newx])) or
          indets(eq) minus Type::ConstantIdents minus {newx} = {} ) then
          if solvelib::isEmpty(op(dom::sets(S), 1)) = TRUE then
            return(FALSE)
          end_if;
          save newx;
          assume(newx in op(dom::sets(S), 1));
          // one third of the effort for isEmpty, one third for solve,
          // one third for _intersect
          MAXEFFORT:= MAXEFFORT/3;
          v:= not solvelib::isEmpty(solvelib::solve_intersect
                                (
                                 solve(eq = a, newx),
                                op(dom::sets(S))
                                 )
                                );
          // if v = TRUE or v = FALSE then
          if length(v) < length(hold(_in)(a, S)) then
            return(v)
          end_if;
        end_if
      end_if;
      hold(_in)(a, S) // not yet implemented, would require an "exists"-
                        // quantifier
    end_proc;


    // Union : overloads solvelib::Union
    /* Union(S, x, M) - with S depending on the parameter x,
       return the set of all objects that can be obtained by substituting
       an element of M for x */

    Union:=
    proc(S:dom, x:DOM_IDENT, M /*: set */)
    begin
      if has(dom::variables(S), x) then
        error("Not a free parameter")
      end_if;
      if has(S::dom::sets(S), x) or hastype(M, "solve") then
        // cannot do this
        return(hold(solvelib::Union)(args()))
      end_if;
      S:= solvelib::avoidAliasProblem(S, indets(M));
      // in future: Union(subsBySet(S::dom::expr(S), x, M), x, M)
      simplify(dom::new(dom::expr(S), append(dom::variables(S), x),
               append(dom::sets(S), M)))
    end_proc;


    /* map(S,f, x1,...,xn) - computes the set of all
       f(y, x1, ..., xn), y Element of S
    */

    map:=
    proc(S:dom, f)
      local i, vars, sets, newex;
    begin
      vars:= dom::variables(S);
      sets:= dom::sets(S);
      for i from 1 to nops(vars) do
        save vars[i];
        assume(vars[i] in sets[i])
      end_for;
      newex:= f(dom::expr(S), args(3..args(0)));
      map(vars, unassume);
      dom::new(newex, vars, sets)
    end_proc;


    // operations defined by mapping

    normal:= S -> dom::map(S, normal);

    expand:= S-> dom::map(S, expand);


    subs:= S -> new(dom, subs(extop(S), args(2..args(0))));


    evalAt:=
    proc(S:dom, subst: Type::SetOf("_equal"))
      local f, vars, sets, fnew, setsnew;
    begin
      // avoid alias problem
      S:= solvelib::avoidAliasProblem(S, indets(map(subst, op, 2),All)
                                      union freeIndets(S)
                                      );
      // do not replace bound identifiers
      f:= dom::expr(S);
      vars:= dom::variables(S);
      sets:= dom::sets(S);
      subst:= select(subst, eq -> contains(vars, op(eq, 1))=0);
      fnew:= evalAt(f, subst);
      setsnew:= evalAt(sets, subst);
      if fnew <> f or setsnew <> sets then
        dom::new(fnew, vars, setsnew)
      else
        S
      end_if
    end_proc;

    // compute all objects that can be obtained by substituting
    // a Dom::ImageSet for x
    // the set of all a(x) with x in {g(y1, ..,yk) ; yi in S_i}
    // is { a(g(y1, ..., yk)) ; y_i in S_i }


    substituteBySet:=
    proc(a, x, S:dom)
      local e;
    begin
      if type(x) <> DOM_LIST then
        if solvelib::dimension(S) <> 1 then
          error("dimension does not match")
        else
          dom::new(subs(a, x=dom::expr(S),EvalChanges),
              dom::variables(S), dom::sets(S))
        end_if
      else
        e:= dom::expr(S);
        if nops(e) <> nops(x) then
          error("dimension does not match")
        end_if;
        dom::new(subs(a, zip(x, [op(e)], _equal),EvalChanges),
            dom::variables(S), dom::sets(S))
      end_if
    end_proc;




    // indets:= S -> indets(dom::expr(S)) minus {op(dom::variables(S))};

    freeIndets:= S -> (freeIndets(dom::expr(S), args(2..args(0))) union freeIndets(dom::sets(S), args(2..args(0))) ) 
                      minus {op(dom::variables(S))};


    has:= S -> has([extop(S)], args(2..args(0)));


    hastype:=
    proc(S: dom)
      local i: DOM_INT, sets;
    begin
      sets:= dom::sets(S);
      _lazy_or(hastype(extop(S, 1), args(2..args(0))),
               hastype(op(sets, i), args(2..args(0))) $i=1..nops(sets))
      or
      bool(args(2) = dom
           or _lazy_and(type(args(2)) = DOM_SET,
                        contains(args(2), dom)))
    end_proc;

    
    _set2sysprop := proc( S:dom )
      local props, A, i;
    begin
        props := [ 1, 0 $ 8 ];
        if dom::nvars(S)=1 then
          if  (A := dom::isisetLinear(S) )<>FALSE and _subset(dom::sets(S)[1], R_) = TRUE and is(A[1],Type::Real)=TRUE  and is(A[2],Type::Real)=TRUE then
            props[2] := 1; /* reell */
            if _subset( dom::sets(S)[1], Z_ )=TRUE then
              if type(A[1])=DOM_INT and type(A[2])=DOM_INT then
                props[ 6 ] := 1; /* ganzzahl */
              end_if;
              if A[2]<>0 then
                props[ 8 ] := 1; /* <> 0 */
              end_if;
            end_if;
          end_if;
          /* Wenn eine Menge ein _intersect ist, dann ist die Schnittmenge der einzelnen Bilder eine Obermenge */
          if type( (A := op(dom::sets(S))) )="_intersect" then
            for i in [op(A)] do
              props := zip( props, property::_set2sysprop(dom::new( dom::expr(S), dom::variables(S), [i]) ), max );
            end_for;
          end_if;
        end_if;
        return( props );
    end_proc;


    // evalParam(S, var1=value1, .., vark=valuek)
    // substitutes values for the parameters

    evalParam:=
    proc(S:dom)
      local xpr, varlist, setlist,i, subst, pos, substset;
    begin
      substset:= [args(2..args(0))];
      xpr:=dom::expr(S);
      varlist:=dom::variables(S);
      setlist:=dom::sets(S);
      for i from 1 to nops(substset) do
        if type((subst:=op(substset, i)))<>"_equal" then
          error("Equation expected")
        end_if;
        pos:=contains(varlist, lhs(subst));
        if pos=0 then
          next
        end_if;
        delete varlist[pos];
        delete setlist[pos];
        xpr:=subs(xpr, subst)
      end_for;
      if nops(varlist)=0 then
        xpr
      else
        new(dom, xpr, varlist, setlist)
      end_if
    end_proc;

    getElement:=
    proc(S:dom)
      local i, aux, l;
    begin
      // choose one element for each parameter set
      l:= [dom::variables(S)[i]=
           (aux:=solvelib::getElement(dom::sets(S)[i], args(2..args(0)));
            if aux=FAIL then
              userinfo(10, "Could not find value for parameter ".
                  expr2text(dom::variables(S)[i]));
              aux // should be return(aux) once the kernel bug has been fixed
            else
              aux
            end_if) $i=1..nops(dom::sets(S))];
      // substitute each chosen value into the expression
      if contains(map(l, op, 2), FAIL) = 0 and
        traperror((aux:= evalAt(dom::expr(S), op(l)))) = 0 then
        aux
      else
        FAIL
      end_if
    end_proc;

    // cannot decide whether the expression really
    // represents an *injective* function
    isFinite:= UNKNOWN;

    isLinear:=
    proc(S)
    begin
      if not testtype(dom::expr(S), Type::Arithmetical) then
        FALSE
      elif testtype(dom::expr(S), Type::PolyExpr(dom::variables(S))) then
        bool(degree(poly(dom::expr(S), dom::variables(S))) = 1)
      else
        FALSE
      end_if
    end_proc;

    simplify:=
    proc(S)
      local newvars, vars, sets, i, l, newexpressions, v, xpr, cond;
    begin
      sets:= dom::sets(S);
      vars:= dom::variables(S);

      // expand finite sets
      l:=contains(map(dom::sets(S), domtype), DOM_SET);
      if l>0 then
        newvars:= vars;
        v:= newvars[l];
        xpr:= S::dom::expr(S);
        newexpressions:= map(sets[l], el-> subs(xpr, v=el));
        delete newvars[l];
        delete sets[l];
        return(_union(op(map(newexpressions,
                             el -> simplify(Dom::ImageSet(el, newvars, sets))
                             ))))
      end_if;


      for i from 1 to nops(vars) do
        assume(vars[i] in sets[i])
      end_for;

      // first simplify the expression
      S:=extsubsop(S, 1=simplify(dom::expr(S)) );

      cond:= TRUE;
      
      for i from nops(vars) downto 1 do
        if not has(extop(S,1), vars[i]) then
          // {C; x in set} = {C} but only if set is nonempty
          cond:= cond and not solvelib::isEmpty(sets[i]);
          delete vars[i];
          delete sets[i]
        end_if;
      end_for;

      if nops(vars) = 0 then
        return(piecewise([cond, {extop(S, 1)}], [not cond, {}]))
      end_if;
      
      // simplify all sets in the list of sets

      S:= extsubsop(S, 2=vars, 3=map(sets, simplify));

      // simplify { x; x in F} to F

      if dom::nvars(S)=1 and op(dom::variables(S),1)=dom::expr(S) then
        return(piecewise([cond, op(dom::sets(S),1)], [not cond, {}]))
      end_if;

      // no further heuristisc at the moment
      piecewise([cond, S], [not cond, {}])
    end_proc;

    Re :=
    proc(S: Dom::ImageSet)
    begin
      Dom::ImageSet( Re( S::dom::expr(S) ),
                    S::dom::variables(S),
                    S::dom::sets(S)
                    )
    end;

    Im :=
    proc(S: Dom::ImageSet)
    begin
      Dom::ImageSet( Im( S::dom::expr(S) ),
                    S::dom::variables(S),
                    S::dom::sets(S) )
    end;


    diff:=
    proc(S: dom, x)
    begin
      if args(0) = 1 then
        S
      else
        // do nothing - consistent with diff of DOM_SETs
        hold(diff)(args())
      end_if
    end_proc;



    
    /* This functions simplifies an ImageSet.
     */
    flattenImageSet :=
    proc( S )
      local Vars, res, i, xpr, x, sets, mapSets;
    begin
      if not testtype( S, Dom::ImageSet ) then
        return(S);
      end_if;

      /* This functions maps the sets of S under the expression xpr.
       * blockedVars is a set of all variables that mustn't be touched.
       */
      mapSets :=
      proc( xpr, blockedVars )
        local i, j, k, v, inds, res, resSet, resVar, vars, tmpBlockedVars, sig,
        dependant, getRealPart, isReal, iMin, iMax, l, r, lo, ro;
      begin
        isReal :=
        proc( xpr )
          local i;
          begin
            if testtype( xpr, Type::Real ) then return( TRUE ); end_if;
            if ( i := is( Im(xpr)=0 ) )<>UNKNOWN then
              return( i )
            end_if;
            if testtype( xpr, DOM_IDENT ) and contains( sets, xpr ) and
              _subset( sets[ xpr ], R_ )=TRUE then
              return( TRUE )
            end_if;
            if testtype( xpr, "_mult" ) then
              return( _and( map( op(xpr), isReal ) ) )
            end_if;
            FALSE;
          end_proc;

          /* Returns Re(xpr)
           */
          getRealPart :=
          proc( xpr )
            local s, r;
          begin
            case type( xpr )
              of "_plus" do
                return( map( xpr, getRealPart ) );

              of "_power" do
                case domtype( op(xpr,2) )
                  of DOM_INT do
                    if isReal( op(xpr,1) )=TRUE then
                      return( xpr );
                    end_if;
                end_case;
                break;

              of "exp" do
                s := op(xpr,1);
                if isReal( s )=TRUE then return( xpr ); end_if;
                // Rewrite Re( exp(x*I) ) to cos(x)
                if testtype( s, "_mult" ) and testtype( (r:=op(s,nops(s))), Type::Imaginary ) and _and( map(op(s,1..(nops(s)-1)), isReal ) ) then
                  return( cos( subsop(s,nops(s)=Im(r) ) ) );
                end_if;
                break;

              of "_mult" do
                s := split( xpr, isReal );
                assert( s[3]=1 );
                if is( Re(s[2])=0 )=TRUE then return( 0 ); end_if;
                if s[1]<>1 then
                  return( getRealPart(s[2])*s[1] );
                end_if;
            end_case;
            Re(xpr);
          end_proc:

        // recursive call
        if testtype( xpr, DOM_EXPR ) then
          tmpBlockedVars := blockedVars;
          inds := {};
          for i from 1 to nops(xpr) do
            v := indets(op(xpr,i));
            tmpBlockedVars := tmpBlockedVars union ( inds intersect v );
            inds := inds union v;
          end_for;
          inds := map( [op(xpr)], freeIndets );
          for i from 1 to nops(xpr) do
            vars := select( sets, X->( contains( freeIndets(op(xpr,i)), op(X,1) ) ) );
            if ( nops(vars)<>0 ) then
              res := mapSets( op(xpr,i), tmpBlockedVars );
              xpr := extsubsop( xpr, i=res );
            end_if;
          end_for;
        end_if;

        dependant := null();

        case type(xpr)
          of "_mult" do
            xpr := split( xpr, testtype, Type::Constant );
            xpr := expand( xpr[1] ) * xpr[2];

            if type(xpr) = "_mult" then
              i:= [op(xpr)]
            else
              i:= [xpr]
            end_if;
            // at first extract all free set-variables
            i := split(i,
                       X -> ( testtype( X, DOM_IDENT ) and
                             contains( sets, X ) and
                             not contains( blockedVars, X ) )  );
            assert( i[3]= [] ); // there should be no unknown
            if (i[1]=[]) then
              break
            end_if;
            i[1]:= _mult(op(i[1]));
            i[2]:= _mult(op(i[2]));

            // then get a table of all sets
            v := table( map(op(i[1]), X->( X=sets[X] ) ) );

            // check if we have C_
            k := select( v, X->op(X,2)=C_ );
            if nops(k)>0 then return( op(k,[1,1]) ); end_if;

            // check if we can reduce sets
            k := select( v, X->type(op(X,2) )=solvelib::BasicSet );
            if nops(k)>1 then
              resSet :=Z_;
              for j in k do
                if op(j,2) = R_ then resVar := op(j,1); resSet := R_; break; end_if;
                if op(j,2) = Q_ then resVar := op(j,1); resSet := Q_; end_if;
                resVar := op(j,1);
              end_for;
              xpr := xpr / _mult( op( map( [ op(k) ], op, 1 ) ) ) * resVar;
              for j in k do
                if op(j,1)<>resVar then
                  delete( v[op(j,1)] );
                end_if;
              end_for;
            end_if;
            if nops(k)>=1 then
              if nops(k)=1 then resSet := op(k,[1,2]); end_if;
              if type(i[2])="_mult" then k := [op(i[2])]; else k := [i[2]]; end_if;
              case resSet
                of Z_ do
                  k := _mult( op( select( k, X->not contains( {1,-1}, X ) ) ) );
                  break;
                of Q_ do
                  k := _mult( op( select( k, X->not contains( {DOM_INT, DOM_RAT}, type(X) ) ) ) );
                  break;
                of R_ do
                  k := _mult( op( select( k, X->not contains( {DOM_INT, DOM_RAT, DOM_FLOAT}, type(X) ) ) ) );
                  break;
                of C_ do
                  k := _mult( op( select( k, X->not contains( {DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX}, type(X) ) ) ) );
                  break;
                otherwise
                  assert( FALSE );
              end_case;
              if k<>i[2] then xpr := i[1]*k; end_if;
            end_if;

            // check if we have R_
            k := select( v, X->op(X,2)=R_ );
            if nops(k)>0 then
              // j := freeVariable(R_) * variables which are no subset of R_
              j := _mult( op( map([op(select( v, X->_subset( op(X,2), R_ )<>TRUE ))], op, 1 ) ) )*op(k,[1,1]);
              if isReal( i[2] )=TRUE then return( j ); end_if;
              if testtype( i[2], "_mult" ) then
                i := select( i[2], X -> isReal( X )<>TRUE );
              else
                i := i[2];
              end_if;

              // remove unused signs
              if stdlib::hasmsign( i )=TRUE then
                i := -i;
              end_if;
              return( i*j );
            end_if;

            // check for intervals of R_
            k := select( v, X->testtype( op(X,2), Dom::Interval ) );
            if nops(k)>0 then
              /* check if we have SETS */
              l := select( v, X->testtype( op(X,2), DOM_SET ) );

              /* select all remaining multiplicands */
              if type(xpr) = "_mult" then
                i:= [op(xpr)]
              else
                i:= [xpr]
              end_if;
              i := select(i, X->( not contains( k , X ) ) and not contains( l , X ));
              i := _mult(op(i));
              if testtype( i, "_mult" ) then
                i := split( i, X ->( testtype( X, Type::Constant ) and is( X, Type::Real, Goal=TRUE ) ) );
              else
                if testtype( i, Type::Constant ) and is( i, Type::Real, Goal=TRUE  ) then
                  // since i is real we only need the real part of i
                  i := [ Re(i), 1, 1 ];
                else
                  i := [ 1, i, 1 ];
                end_if;
              end_if;

              // remove signs
              if stdlib::hasmsign( i[2] )=TRUE then
                i[1] := -i[1];
                i[2] := -i[2];
              end_if;

               v := _mult( op(map( [op(k)], op, 2 )) )*i[1];

               /* check for our SET-elements */
               for j in l do
                r := v*op(j,2);
                if not testtype( r, "_union" ) or nops( r )<=2 then
                  v := r;
                else
                  i[2] := i[2]*op(j,1);
                end_if;
               end_for;

               sets[ op(k,[1,1]) ] := v;
              return( op(k,[1,1])*i[2]*i[3] );
            end_if;
            break;

          of "Re" do
            i := op(xpr);
            if testtype( i, DOM_IDENT ) and contains( sets, i ) then
              if _subset( sets[i], R_ )=TRUE then
                return( i );
              end_if;
              if testtype( sets[i], DOM_SET ) then
                v := map( sets[i], Re );
                if not hastype( v, "Re" ) then
                  sets[i] := v;
                  return( i );
                end_if;
              end_if;
            end_if;
            return( getRealPart( i ) );
            break;

          of "Im" do
            i := op(xpr);
            if testtype( i, DOM_IDENT ) and contains( sets, i ) then
              if _subset( sets[i], R_ )=TRUE then
                return( 0 );
              end_if;
              if not contains( blockedVars, i ) and not testtype( (j := _intersect( sets[i], I*R_ ) ), "_intersect" ) then
//                sets[i] := -I*j;
//                return( i );
              end_if;
            end_if;
            break;

            i := op(xpr);
            if testtype( i, DOM_IDENT ) and contains( Vars, i ) and _subset( vars[i], R_ )=TRUE then
              return( 0 );
            end_if;
            break;

          of "_plus" do
            /* Special case: simplify expressions like
               { k+n | k in Z_ } -> Z_ for n integers
             */
            for i in select( indets(xpr) minus blockedVars,
                            X-> ( contains( sets, X ) and sets[X] = Z_ ) ) do
              if (res := Type::Linear( xpr, [i] ))<>FALSE and
                not iszero(res[1]) then
                k := res[2];
                if stdlib::hasmsign( res[1] )=TRUE then
                  res[1] := -res[1];
                end_if;
                l := [ 0, 0 ];
                if testtype( k, "_plus") then
                  k := [ op(k) ]
                else
                  k := [ k ]
                end_if;
                for j in k do
                  x := j/res[1];
                  if is(x in Z_, Goal = TRUE) then
                  elif testtype( x, DOM_RAT ) then
                    l[1] := l[1] + x;
                  else
                    l[2] := l[2] + j;
                  end_if;
                end_for;
                l[1] := ( numer(l[1]) mod denom(l[1]) ) / denom( l[1] );;
                xpr := res[1]*i + l[2] + l[1]*res[1];
              end_if;
            end_for;

            // Check xpr again since it has been changed.
            if type(xpr)<>"_plus" then
              break;
            end_if;

            /*
           Special case: test if we have images of intervals under polynomials
            */
            for i in select( indets(xpr) intersect Vars minus blockedVars,
                            X->( testtype( sets[X], Dom::Interval ) ) ) do
              res := split( xpr, X->indets(X) minus {i} = {} );
              assert( res[3] = 0 );
              if testtype( res[1], Type::PolyExpr(i, Type::Real) ) then
                j := solve( diff( res[1], i ), i, Domain=Dom::Real );
                if testtype( j, DOM_SET ) then
                  k := map( select( j, X->is( X in sets[i] ) ), X->subs( res[1], i=X) );
                  l := subs( res[1], i=Dom::Interval::left( sets[i] ) );
                  r := subs( res[1], i=Dom::Interval::right( sets[i] ) );
                  lo := Dom::Interval::isleftopen( sets[i] );
                  ro := Dom::Interval::isrightopen( sets[i] );
                  v := is(l>r);
                  if v<>TRUE and v<>FALSE then next; end_if;
                  if v=TRUE then
                    v := l; l := r; r := v;
                    v := lo; lo := ro; ro := v;
                  end_if;
                  if k<>{} then
                    iMin := min(k);
                    iMax := max(k);
                    if not testtype( iMin, Type::Real ) then next; end_if;
                    if not testtype( iMax, Type::Real ) then next; end_if;

                    v := is( iMin<l );
                    if v=UNKNOWN then next; end_if;
                    if v then
                    l := iMin; lo := FALSE;
                    end_if;

                    v := is (iMax>r);
                    if v=UNKNOWN then next; end_if;
                    if v then
                    r := iMax; ro := FALSE;
                    end_if;
                  end_if;
                  if not lo then l := [l]; end_if;
                  if not ro then r := [r]; end_if;
                  xpr := i + res[2];
                  sets[i] := Dom::Interval( l, r );
                end_if;
              end_if;
            end_for;

            res := 0;
            resSet := 0;
            for i in [ op(xpr) ] do
              if testtype( i, DOM_IDENT ) and
                contains( Vars minus blockedVars, i ) and
                Type::Linear( xpr, [i])<>FALSE then
                if ( dependant = null() ) then
                  dependant := i;
                  res := res + i;
                  resSet := resSet + sets[ i ];
                else
                  resSet := resSet + sets[ i ];
                end_if;
              elif testtype( i, DOM_IDENT ) and
                not contains( blockedVars, i ) and
                contains( sets, i ) and
                sets[i]=C_ then
                res := dependant := i;
                resSet := C_;
                break;
              elif indets(i) intersect {op(dom::variables(S))}={}  and
                is(i in R_, Goal=TRUE)  then
                resSet := resSet + i;
              else
                res := res + i;
              end_if;
            end_for;
            if ( dependant<> null() ) then
              // special case
              // test if we have C_
              // this is the case, if we have res := x+a*y with x in R_, y in R_ and Im(a)<>0
              if resSet = R_ and testtype( res, "_plus" ) then
                j := select( op(res), proc(X)
                  begin
                    if not testtype( X, "_mult" ) then return( FALSE ); end_if;
                    if not nops( X )=2 then return( FALSE ); end_if;
                    if not contains( Vars minus blockedVars, op(X,1) ) or op(X,1)=dependant or sets[op(X,1)]<>R_ then return( FALSE ); end_if;
                    return( is( Im(op(X,2))<>0 )=TRUE );
                  end_proc
                 );
                if nops(j)>0 then
                  Vars := Vars minus {dependant};
                  resSet := C_;
                  res := dependant;
                end_if;
              end_if;
              sets[ dependant ] := resSet;
              return( res );
            end_if;
            break;

            of "_power" do
              i := op(xpr,1);
              j := op(xpr,2);
              // limit to single not blocked identifiers
              if testtype( i, DOM_IDENT ) and contains( sets, i) and not contains( blockedVars, i ) and testtype( j, Type::Union( DOM_RAT, DOM_INT ) ) then
                if contains( Vars, i ) then
                  v := sets[i];
                  res := 1;
                  sig := sign(v);
                  if type(sig)=DOM_SET and contains(sig, -1) and not contains(sig,1) then
                    v := -v;
                    res := exp( I*PI*j );
                  end;
                  v := _power( v, j );
                  sets[ i ] := v;
                  return( res*i );
                elif testtype( j, Type::PosInt ) then
                  if sets[i]=C_ then return( i ); end_if;
                  if testtype( j, Type::Even ) then
                    // special case
                    // z^2 for z in A with A subset R is the same as z^2 for z in |A|
                    if testtype( sets[i], "_union" ) and map( {op(sets[i])}, testtype, Dom::Interval )={ TRUE } then
                      if traperror( (v := abs(sets[i])) )=0 then
                        if testtype( v, Dom::Interval ) then
                          v := _power( v, j );
                          xpr := i;
                        end_if;
                        sets[ i ] := v;
                      end_if;
                    end_if;
                  end_if;
                elif testtype( j, Type::NegInt ) then
                  // special case: 1/(R_ minus {0}), 1/(C_ minus {0})
                  if bool( sets[i]=R_ minus {0} ) or bool( sets[i]=C_ minus {0} ) then
                    xpr := i;
                  end_if;
                end_if;
              end_if;
              break;

              of "exp" do
                i := op(xpr,1);
                if ( testtype( i, DOM_IDENT ) and
                    contains( Vars minus blockedVars, i ) ) then
                  v := exp( sets[op(xpr,1)] );
                  if not testtype( v, Dom::ImageSet ) then
                    sets[i] := v;
                    xpr := i;
                  end_if;
                elif testtype( i, "_mult" ) or testtype( i, "_plus" )  then
                  if testtype( i, "_plus" ) then
                    k := split( i, isReal );
                    assert( k[3]=0 );
                  else
                    k := [ 0, i ];
                  end_if;

                  // Test for special case exp(PI*I*z) with z in Z_
                  if type(k[2]) = "_mult" then 
                    v := split( k[2], X->( testtype(X, DOM_IDENT ) and contains( sets, X ) and 
                    not contains( blockedVars union indets( k[1] ), X ) and sets[X] = Z_ ) );
                    if testtype( v[1], DOM_IDENT ) and testtype( v[2]/(PI*I/2), Type::Integer ) then
                      if testtype( v[2]/(PI*I), Type::Even ) then
                        sets[ v[1] ] := {1};
                      elif testtype( v[2]/(PI*I), Type::Odd ) then
                        sets[ v[1] ] := {-1, 1};
                      else
                        sets[ v[1] ] := {-1, 1, I, -I };
                      end_if;
                      /* call again recursive to simplify exp( k[1] ) */
                      return( v[1]*mapSets( exp( k[1] ), blockedVars union {v[1]} ) );
                    end_if;
                  end_if;  
                end_if;
                break;

              of "arg" do
                i := op(xpr,1);
                if ( testtype( i, DOM_IDENT ) and contains( Vars minus blockedVars, i ) ) then
                  /* try do determine the sign of the set and evaluate arg if possible */
                  sig := sign(sets[i]);
                  if type(sig)=DOM_SET then
                    if sig intersect {0,1} <> {} then
                      v := {0};
                    else
                      v := {};
                    end_if;
                    if contains(sig, -1) then
                      v := v union {PI}
                    end_if;
                    if nops(v)=1 then
                      return( op(v) );
                    else
                      sets[ i ] := v;
                      return( i );
                    end_if;
                  end_if;
                end_if;

              of "sin" do
                if ( testtype( op(xpr,1), DOM_IDENT ) and
                    contains( Vars minus blockedVars, op(xpr,1) ) )
                  and type(sets[op(xpr,1)] ) <> Dom::Interval
                  then
                  v := sin( sets[op(xpr,1)] );
                  sets[op(xpr,1)] := v;
                  xpr := op(xpr,1);
                end_if;
                break;

              of "ln" do
                i := op(xpr);
                if ( testtype( i, DOM_IDENT ) and
                    contains( Vars minus blockedVars, i ) ) then
                  v := sets[i];
                  if testtype( v, Dom::Interval ) then
                    sig := sign(v);
                    if type(sig)=DOM_SET then
                      if not contains(sig, -1) then
                        /* if the set is a positive interval we can map ln */
                        v := ln(v);
                        sets[i] := v;
                        xpr := op(i);
                      elif not contains(sig, 1) then
                        /* if the set is a negative interval we can map ln to "-set" and have to add I*PI */
                        v := ln(-v);
                        sets[i] := v;
                        xpr := op(i) + I*PI;
                      end_if;
                    end_if;
                  end_if;
                end_if;
                break;

              of "cos" do
                if ( testtype( op(xpr,1), DOM_IDENT ) and
                    contains( Vars minus blockedVars, op(xpr,1) ) )
                  and type(sets[op(xpr,1)] ) <> Dom::Interval
                  then
                  v := cos( sets[op(xpr,1)] );
                  sets[op(xpr,1)] := v;
                  xpr := op(xpr,1);
                end_if;
                break;

            otherwise:
              break;

        end_case;

        i := indets(xpr);
        if not testtype(xpr,DOM_IDENT ) and nops(i)=1 and
          blockedVars intersect i={} and contains( sets, op(i) ) then
          i := op(i);
          if testtype( sets[i], DOM_SET ) then
            v := map( sets[i], X->subs(xpr,i=X,EvalChanges) );
            if map(v, testtype, Type::Union( DOM_RAT, DOM_COMPLEX, DOM_INT, DOM_FLOAT ) )={TRUE} then
              xpr := i;
              sets[ i ] := v;
            end_if;
          end_if;
        end_if;

        return( xpr );
      end_proc;

      xpr := dom::expr(S);
      /* sets is a table containing all sets of the ImageSet S */
      sets := table( (zip( dom::variables(S), dom::sets(S), _equal )));

      assume(_and(op(res, 1) in op(res, 2) $res in sets));

      /* If one of the subsets is an (affine-)linear ImageSet then flatten the parameter into the bording ImageSet.
       * For Example Dom::ImageSet( I*x, x, I*R_ ) = R_
       */
      for i in select( sets,
                      proc(X)
                      begin
                        if not testtype( op(X,2), Dom::ImageSet ) then
                          return( FALSE );
                        end_if;
                      if dom::nvars(op(X,2))<>1 then
                        return( FALSE )
                      end_if;
                      return( TRUE=dom::isLinear( op(X,2) ) );
                      end )
      do
          x := genident();
          xpr := subs( xpr, i[1]=subs(dom::expr(i[2]),
                                      op(dom::variables(i[2],1))=x ) );
          sets[ x ] := op(dom::sets(i[2]),1);
          delete sets[i[1]]
      end_for;

      /* Vars is a set containing all variables which represent a single interval or R_ */
      Vars := ( map( {op(select( sets, X->( testtype( op(X,2), Dom::Interval )
                                           or op(X,2)=R_ ) ))}, op, 1 ) );

      res := mapSets( xpr, {} );


      if type(res) = DOM_IDENT
        or
        Simplify::defaultValuation([res, op(map([op(sets)], op, 2))]) <
        Simplify::defaultValuation([dom::expr(S), op(dom::sets(S))])
        then
        S := Dom::ImageSet( res, [ i[1] $ i in sets ], [ i[2] $ i in sets ] )
      end_if;
      return( S );
    end_proc;

    Simplify:=
    proc(S)
      local i, newvars, sets, l, newexpressions, xpr, v;
    begin
  
      sets:= dom::sets(S);

      // expand finite sets
      l:=contains(map(dom::sets(S), domtype), DOM_SET);
      if l>0 then
        newvars:=dom::variables(S);
        v:= newvars[l];
        xpr:= S::dom::expr(S);
        newexpressions:= map(sets[l], el-> subs(xpr, v=el));
        delete newvars[l];
        delete sets[l];
        return(_union(op(map(newexpressions,
                             el -> simplify(Dom::ImageSet(el, newvars, sets))
                             ))))
      end_if;

      // first simplify the expression
      xpr := hold(Simplify)(dom::expr(S), Steps=50);
      for i from 1 to dom::nvars(S) do
        xpr := hold(_assuming)( xpr, op(dom::variables(S),i) in op(dom::sets(S),i) );
      end_for;
//      S:=extsubsop(S, 1 = eval(xpr) );

      // simplify all sets in the list of sets
      S:= extsubsop(S, 3 = map(dom::sets(S), Simplify));

      S := dom::flattenImageSet( S );

      // simplify { x; x in F} to F

      if testtype( S, Dom::ImageSet ) and dom::nvars(S)=1 and
        op(dom::variables(S),1)=dom::expr(S) then
        return(op(dom::sets(S),1))
      end_if;

      // no further heuristisc at the moment
      S
    end_proc;

    // overload Simplify::complexity
    complexity:=
    proc(iset)
    begin
      200 +
      Simplify::complexity(dom::expr(iset)) +
      4*Simplify::complexity(dom::sets(iset))
    end_proc;

    // overload property::normalGroebner
    normalGroebner:=
    proc(iset)
      local xpr, sets, xprnew, setsnew, vars;
      save MAXEFFORT;
    begin
      if MAXEFFORT < 100 then
        return(iset)
      end_if;
      
      xpr:= dom::expr(iset);
      sets:= dom::sets(iset);
      vars := dom::variables(iset);
      vars := map(vars, X->X=genident("Dom::ImageSet::normalGroebner"));
      xpr := subs(xpr, vars);
      MAXEFFORT:= MAXEFFORT/2;
      if traperror((xprnew:= property::normalGroebner(xpr, args(2..args(0))))) <> 0 then
        // expression cannot be evaluated - set is empty
        return({})
      end_if;  
      setsnew:= property::normalGroebner(sets, args(2..args(0)));
      if xprnew <> xpr or setsnew <> sets then
        dom::new(xprnew, map(vars, op, 2), setsnew)
      else
        iset
      end_if;
    end_proc;
    
    
    _subset :=
    proc(A, B)
      local v, l, r;
      save MAXEFFORT;
    begin
      if domtype(A) <> Dom::ImageSet then
        assert(domtype(B) = Dom::ImageSet);
        if A = Z_ and dom::sets(B) = [Z_] and
          (l:= dom::isisetLinear(B)) <> FALSE then
          // B = {l[1]*k + l[2]; k in Z_}
          // we are asking whether (n-l[2])/l[1] in Z_ for every integer n
          // if l[2]/l[1] is not in Z_, then 0 is not in B
          // otherwise, our condition becomes n/l[1] in Z_, i.e.,
          // 1/l[1] in Z_
          case is(l[2]/l[1] in Z_ and 1/l[1] in Z_)
            of FALSE do
              return(FALSE)
            of TRUE do
              return(TRUE)
              // of UNKNOWN: we don't know anything
          end_case;
        end_if;
        
        return(hold(_subset)(A, B ))
      end_if;

      
      // A is an image set    

      case type( B )
        of "_minus" do
          // case A subset (X minus Y)
          v := Dom::ImageSet::_subset( A, op(B,1) );

          // if A is no subset of X then A is no subset of X minus X
          if v=FALSE then return( FALSE ); end_if;
          if v<>TRUE then break; end_if;

          v := solvelib::isEmpty( A intersect op(B,2) );
          if v in {TRUE,FALSE} then return( v ); end_if;
          break;

        of solvelib::BasicSet do
          // every imageset is a subset of C_
          if B=C_ then
            return(TRUE)
          end_if;
          if B=R_ then
            l := dom::isisetLinear( A );
            if l<>FALSE and _subset( op(dom::sets(A)), R_ )=TRUE then
              if map( l, is, Type::Real )=[TRUE, TRUE] then
                return( TRUE )
              end_if;
            end_if;
          elif B=Z_ then
            if dom::sets(A) = [Z_] and
              (l:= dom::isisetLinear(A)) <> FALSE then
              // A = {l[1]*k + l[2]; k in Z_}
              // A is a subset of Z_ if l[1] and l[2] are integer
              if is(l[1] in Z_ and l[2] in Z_, Goal=TRUE) then
                return(TRUE);
              end_if;
            end_if;
          end_if;
          break;

        of Dom::ImageSet do
          // special case: both imageset are linear in Z_
          if ( dom::nvars(A)=1 and dom::nvars(B)=1 ) then
            MAXEFFORT := MAXEFFORT/2;
            l := Type::Linear( dom::expr(A), dom::variables(A) );
            r := Type::Linear( dom::expr(B), dom::variables(B) );
            if ( l<>FALSE and r<>FALSE ) then
              if dom::sets(A)=[Z_] and dom::sets(B)=[Z_] then
                v:= is( (l[2]-r[2])/r[1], Type::Integer ) and
                    is( l[1]/r[1], Type::Integer);
                if v<>UNKNOWN then
                  return( v );
                end_if;
              end_if;
              if l=r and _subset( dom::sets(A), dom::sets(B) )=TRUE then
                return( TRUE );
              end_if;
            end_if;
            MAXEFFORT := MAXEFFORT*2;
          end_if;

          break;
      end_case;

      hold(_subset)(A, B )
    end_proc;

    float:= S -> dom::new(float(dom::expr(S)),
                          dom::variables(S),
                          float(dom::sets(S)));



    // MacKichan: for compatibility reasons
    setPreferredIdentifiers:= solvelib::setPreferredIdents;



    print:=
    proc(x)
      local i, str, height, base;
      save TEXTWIDTH;
    begin
      if PRETTYPRINT then
        _in:
        TEXTWIDTH := TEXTWIDTH - 2;
        str := strprint(All, expr(x));
        delete TEXTWIDTH;

        if str[1][1] = "\n" then
          str[6] := str[6]-1;
          str[2] := str[2]-1;
        elif str[1][1..2] = "\r\n" then
          str[6] := str[6]-1;
          str[2] := str[2]-1;
        end_if;

        height := str[2];  base := str[6];
        if height = 2 then
          height := 1;
          base   := 1;
        end_if;
        str := "":
        for i from 1 to height do
          if i = base then
            str := str." | \b"
          else
            str := str." | \n"
        end_if
        end_for;

        return({_outputSequence(expr(x), stdlib::Exposed(str),
                                ((if i>1 then
                                    stdlib::Exposed(", ")
                                  else
                                    null()
                                  end_if),
                                 hold(_in)(dom::variables(x)[i],
                                           dom::sets(x)[i]))
                                $ i=1..nops(dom::sets(x)))})
        else
        {_outputSequence(expr(x), stdlib::Exposed(" | "),
                                ((if i>1 then
                                    stdlib::Exposed(", ")
                                  else
                                    null()
                                  end_if),
                                 hold(_in)(dom::variables(x)[i],
                                           dom::sets(x)[i]))
                                $ i=1..nops(dom::sets(x)))}
        end_if
      end_proc;

    expr2text:=
    proc(x)
    begin
      if nops(dom::sets(x)) = 1 then
        hold(Dom::ImageSet)(expr(x), dom::variables(x)[1], dom::sets(x)[1])
      else
        hold(Dom::ImageSet)(expr(x), dom::variables(x), dom::sets(x))
      end_if
    end_proc;

    TeX:=
    proc(x)
      local i:DOM_INT;
    begin
      "\\left\\{\\left. ".generate::TeX(dom::expr(x))." \\right| ".
      _concat(((if i<>1 then ", " else "" end_if).
               generate::TeX(dom::variables(x)[i])."\\in".
               generate::TeX(dom::sets(x)[i]))
              $i=1..dom::nvars(x)). "\\right\\}"
    end_proc;

    Content:=
    proc(Out, x)
      local i, num, bvars, condition, sets;
    begin
      bvars := dom::variables(x);
      sets := dom::sets(x);
      num := nops(sets);
      if num = 1 then
        condition := Out::Ccondition(Out(hold(_in)(bvars[1], sets[1])));
      else
        condition :=
           Out::Ccondition(
                Out::Capply(Out::Cand,
                            Out(hold(_in)(bvars[i], sets[i])) $ i = 1..num));
      end_if;

      Out::Cset(Out::Cbvar(Out(op(bvars))),
                condition,
                Out(expr(x)))
    end_proc;


    hasmsign:=proc(iset:dom)
    begin
      stdlib::hasmsign(iset::dom::expr(iset));
    end_proc;

    hull := proc( T )
      local i;
    begin
      hull( subs( dom::expr(T), dom::variables(T)[i]=hull(dom::sets(T)[i]) $ i=1..nops(dom::variables(T)) ) );
    end_proc;
   // initialization

  begin
    userinfo(20, "Constructing domain Dom::ImageSet");


    /* local methods */



    mapplus:=
    proc(iset, other)
    begin
      iset:= iset::dom::avoidAliasProblem(iset, freeIndets(other));
      iset::dom::map(iset, _plus, other)
    end_proc;

    mapmult:=
    proc(iset, other)
    begin
      iset:= iset::dom::avoidAliasProblem(iset, freeIndets(other));
      iset::dom::map(iset, _mult, other)
    end_proc;

    mapleftpower:=
    proc(iset, other)
      local A, x;
    begin
      x:= genident();
      iset:= iset::dom::avoidAliasProblem(iset, freeIndets(other));
      A := Type::Linear( iset::dom::expr(iset), [op(iset::dom::variables(iset),1)] );
      if ( iset::dom::nvars(iset)=1 and FALSE<>( A ) and testtype( other, Type::PosInt ) ) then
        if ( A[2]=0 ) then
          return( Dom::ImageSet( _power(x*A[1],other), x, op(iset::dom::sets(iset),1)) );
        end_if;
      end_if;
      iset::dom::map(iset, _power, other)
    end_proc;

  end_domain:
