//    
/*++
Dom::Integer -- domain of integer numbers

An integer is represented by a system-integer (DOM_INT). There is
no DOM_EXT to represent an integer. This fact is expressed by the axiom
'Ax::systemRep'.

Methods:-
testtype(x,D)      - returns TRUE if x is of type DOM_INT
_invert(x)         - returns x if x is 1 or -1 and FAIL otherwise
_divide(x,y)       - returns x/y if y divides x and FAIL otherwise
divides(x,y)       - returns TRUE iff x divides y
euclideanDegree(x) - returns the euclidean degree of x, ie., abs(x)
unitNormal(x)      - returns the unit normal part of x, ie., abs(x)
unitNormalRep(x)   - returns the unit normal rep. of x, ie. the
		     list [abs(x), sign(x), -sign(x)]
gcd(x1,x2,...)     - returns the gcd of the xi
gcdex(x,y)         - returns g,v,w such that g = gcd(x,y) = x*v + y*w
lcm(x1,x2,...)     - returns the lcm of the xi
quo(x,y)           - returns the euclidean quotient of x and y, ie., x div y
rem(x,y)           - returns the euclidean reminder of x and y, ie., modp(x,y)
convert(x)         - returns the value of x converted to Dom::Integer
min(x,...)	   - returns the minimal argument
max(x,...)	   - returns the maximal argument
factor(x)	   - returns the factorisation of x
irreducible(x)	   - returns TRUE iff x is prime
random()	   - returns a random number in the range -999..999
random(n)	   - returns a random number between 0 and n-1
random(m..n)	   - returns a random number between m and n
++*/

domain Dom::Integer

local i,j;  // only to avoid warnings 

inherits Dom::Numerical;

category Cat::EuclideanDomain, Cat::FactorialDomain, Cat::DifferentialRing,
    Cat::OrderedSet;

axiom Ax::canonicalRep, Ax::systemRep, Ax::canonicalOrder,
    Ax::canonicalUnitNormal, Ax::closedUnitNormals,
    Ax::efficientOperation("_divide"), Ax::efficientOperation("_mult");

// entries 

    info_str:= "domain of integer numbers";
    
   testtype:=
   proc(i,T) 
   begin
     // since dom is a facade domain, this cannot be overloaded by i
     assert(T = dom);
     bool(domtype(i) = DOM_INT)
   end_proc;

    _invert:= proc()
	local p;
    begin
	p:= proc() begin FAIL end_proc;
	p(1):= 1;
	p(-1):= -1;
	p
    end_proc();

    _divide:= (i,j) -> if modp(i, j) = 0 then (i div j) else FAIL end_if;

    divides:= (i,j) -> bool(modp(j, i) = 0);

    isUnit:= proc()
	local p;
    begin
	p:= i -> FALSE;
	p(1):= TRUE;
	p(-1):= TRUE;
	p
    end_proc();

    associates:= (i,j) -> bool(abs(i) = abs(j));

    euclideanDegree:= abs;

    unitNormal:= abs;

    unitNormalRep:= proc(i) begin
        if i = 0 then
            [ 0, 1, 1 ]
        else
	    [ abs(i), sign(i), sign(i) ]
        end_if
    end_proc;

    gcd:= igcd;

    gcdex:= igcdex;

    lcm:= ilcm;

    quo:= _div;

    rem:= modp;

    factor:= ifactor;

    irreducible:= isprime;

    divide:= (i,j) -> op([ i div j, modp(i, j) ]);

    convert:= i -> if domtype(i) = DOM_INT then i else FAIL end_if;

    convert_to:= proc(e, T) begin
	if domtype(e) <> DOM_INT then FAIL else
	    case T
	    of Dom::Integer do
	    of Dom::Rational do
	    of DOM_INT do
	    of Dom::Numerical do e; break;
	    of DOM_FLOAT do
	    of Dom::Float do float(e); break;
	    otherwise FAIL;
	    end_case
	end_if
    end_proc;
    
    randomGen:= table((-999..999)=random(-999..999));
    
    random:= proc(r) begin
        if args(0) = 0 then
            dom::randomGen[-999..999]()
        else
            if not contains(dom::randomGen, r) then
                sysassign(dom::randomGen[r], random(r))
            end_if;
            dom::randomGen[r]()
        end_if
    end_proc;

    TeXrep := x -> "\\mathbb{Z}";


end_domain:

