/* ======================================= Lodo =======================================

DESCRIPTION:

  This domain represents linear ordinary differential operators over an 
  arbitrary commutative differential ring of charakteristic zero.
  Note: It is highly recommend to use only coefficient rings with unique zero
        representation.

CALL:

 Dom::LinearOrdinaryDifferentialOperator(Var, Dvar, < Ring >) 
 where
   Var    => indeterminate of type DOM_IDENT; 
             * Default: Df
   Dvar   => differential indeterminate of type DOM_IDENT; 
             * Default: x
   Ring   => arbitrary commutative differential ring of the Domains-package;
             * Default: Dom::ExpressionField(normal) 

METHODS: 

 - PARAMETER:
    ++ a,b   : this Domain
    ++ c     : Ring
    ++ f     : DOM_EXPR
    ++ m     : DOM_INT
    ++ v,l,la: DOM_LIST
    ++ e     : any of {DOM_EXPR, DOM_LIST}
    ++ a,b  = differential operators
    ++ c    = element of the coefficient ring 
    ++ f    = expression or any function
    ++ m    = exponent of the symmetric power 
    ++ v    = list of coefficient ring elements
    ++ e    = e is an ordinary linear differential equation (in f) or
    ++        its coefficients vector or a polynomial expression in Var
    ++ l    = list of non negative integers
    ++ la   = list of (irreducible) differential operators

 - new(e)
   new(e, f)
    ++ creates a Domain element.

 - adjoint(a)
    ++ returns the adjoint of a=c_0+c_1*Var+...+c_n*Var^n, which is 
    ++ a^*=c_0-Var*c_1+...+(-1)^n*Var^n*c_n.

 - companionSystem(a)
    ++ returns the companion system matrix associated with a.

 - exponentialZeros(a)
    ++ computes a fundamental set of all exponential zeros 
    ++ (i.e. {y | a(y)=0 and D(y)/y in R}) of a, 
    ++ if all coefficients are rational functions in Dvar. 
    ++ Note: This function works only properly on operators for which
    ++ ode::expsols works properly.

 - evalLODO(a,f)
    ++ applies f on a using the standard derivative with respect to Dvar
    ++ and simplifies the recieved expression.

 - D(a)
    ++ calculates the derivative of a, which is var*a.

 - Dpoly(<l>, a)
    ++ calculates the derivative of a. When optional a list l is given, then
    ++ l=[1,...,1] with length(l)=n computes the n-th derivative of a. If 
    ++ l=[], then a is returned unchanged.

 - factor(a)
    ++ computes a (complete) factorization [a_1,...,a_m] of a=a_1(...(a_m)), 
    ++ if all coefficient are rational functions and deg(a)<4. For operators 
    ++ of higher degree, currently only left and right factors of degree 1 
    ++ will be found.

 - func_call(a, f, < Unsimplified >)
    ++ applies f on a using the standard derivative with respect to Dvar
    ++ and simplifies the recieved expression.
    ++ When the option Unsimplified is given, the recieved expression will not 
    ++ be simplified.

 - leftDivide(a, b)
    ++ computes q, r, such that a=b*q+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.

 - leftExtendedEuclid(a, b)
    ++ computes [[r0, s0, t0], [s1, t1]] such that
    ++ leftGcd(a,b) = r0 = a*s0+b*t0 and rightLcm(a,b) = -a*s1 = b*t1. 

 - leftExtendedGcd(a, b)
    ++ computes [r0, s0, t0] such that leftGcd(a,b) = r0 = a*s0+b*t0.

 - leftGcd(a,b)
    ++ computes the value g of highest degree such that a=g*a0 and b=g*b0
    ++ for some values a0 and b0.

 - leftLcm(a,b)
    ++ computes the value s of lowest degree such that s=a0*a=b0*b
    ++ for some values a0 and b0.

 - leftQuotient(a, b)
    ++ computes q such that a=b*q+r with deg(r)=0 or deg(r) < deg(b).

 - leftRemainder(a, b)
    ++ computes r such that a=b*q+r with deg(r)=0 or deg(r) < deg(b).

 - liouvillianZeros(a, <Transform | Irreducible>)
 - liouvillianZeros(la)
    ++ see 'solve'. 

 - mkLODO(v)
   mkLODO(Ly, y(x))
    ++ returns the corresponding linear differential operator
    ++ v[1]+v[2]*Var+...+v[n]*Var^(n-1) or
    ++ a_n*Var^n+...+a_0, where Ly = a_n*diff(y(x), x$n)+...+a_0*y(x).

 - polynomialZeros(a) 
    ++ computes a fundamental set of all polynomial zeros of a,  
    ++ if all coefficients are rational functions in Dvar. 

 - rationalZeros(a) 
    ++ computes a fundamental set of all rational zeros of a,  
    ++ if all coefficients are rational functions in Dvar. 

 - rightDivide(a, b)
    ++ computes q, r, such that a=q*b+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.

 - rightExtendedEuclid(a, b)
    ++ computes [[r0, s0, t0], [s1, t1]] such that
    ++ rightGcd(a,b) = r0 = s0*a+t0*b and leftLcm(a,b) = -s1*a = t1*b. 

 - rightExtendedGcd(a, b)
    ++ computes [r0, s0, t0] such that rightGcd(a,b) = r0 = s0*a+t0*b.

 - rightGcd(a,b)
    ++ computes the value g of highest degree such that a=a0*g and b=b0*g
    ++ for some values a0 and b0.

 - rightLcm(a,b)
    ++ computes the value s of lowest degree such that s=a*a0=b*b0
    ++ for some values a0 and b0.

 - rightQuotient(a, b)
    ++ computes q such that a=q*b+r with deg(r)=0 or deg(r) < deg(b).

 - rightRemainder(a, b)
    ++ computes r such that a=q*b+r with deg(r)=0 or deg(r) < deg(b).

 - rlfactor(a)
    ++ this is procedure is intended to be a subprocedure of factor. It 
    ++ computes at least one right and one left factor of a, if them exits.

 - solve(a, <Transform | Irreducible>)
 - solve(la)
    ++ computes the liouvillian solutions of a first degree operator a and
    ++ of a second degree operator a with rational function coefficients
    ++ or liouvillian solutions of a (factor) list la of irreducible 
    ++ operators, whereby the last operator of the list corresponds to the  
    ++ right-hand factor (note, maybe not *all* liouvillian solutions of la  
    ++ will be found).
    ++ Higher degree operators are currently only partially treated. 
    ++ The two options only affects by second degree operators:
    ++ When option Transform is given, the unimodular transform is
    ++ executed unconditionally.
    ++ When option Irreducible is given, a is assumed to be irreducible.  
    ++ NOTE: currently this function only works properly for operators
    ++       for which the function ode::expsols works properly.
    ++ Otherwise: no warranty! 

 - symmetricPower(a, m)
    ++ computes the m-th symmetric power of a. This is the lowest degree
    ++ linear differential operator whose solution space consists 
    ++ exactly of all possible m-th powerproducts of solutions of a.

 - unimodular(a, <Transform>)
    ++ transforms operator a in an operator with unimodular Galois group
    ++ (i.e. the Wronskian of the operator is a rational function) and
    ++ returns a table of equation='transformed operator' and
    ++ factorOfTransformation='Wn', i.e. a solution of 'transformed operator'
    ++ multiplied with Wn is a solution of operator a.
    ++ When the option Transform is given, a is transformed unconditionally
    ++ (even if a has yet an unimodular Galois group).
	       
REFERENCES: 
 - M. Bronstein & M. Petkovsek (1993). On Ore Rings, Linear Operators 
   and Factorisation. Bericht Nr. 200  ETH Zuerich 
 - M. Bronstein & M. Petkovsek (1996). An introduction to pseudo-linear
   algebra. Theoretical Computer Science 157 No. 1, 3-33. 
 - Fakler, W. (1997). Algorithms for Solving Linear Ordinary Differential
   Equations. mathPAD 7 No.1, 50-59.

EXAMPLE:
 >> lodo := Dom::LinearOrdinaryDifferentialOperator(Df, x);
 >> L1 := lodo(Df^2+(x+1)*Df+2*x);
 >> L2 := lodo(Df+1);
 >> L1::dom::leftDivide(L1, L2);
 >> factor(L1);

++*/

/*
  DEPENDENCIES: 

    OrePolyCat.mu, DistPoly.mu, UniPoly.mu, OrePoly.mu,
    sympower.mu, LiouvSol.mu, ExpSol.mu, PolySol.mu, tools.mu
*/

domain Dom::LinearOrdinaryDifferentialOperator(var=Df,dvar=x,R=Dom::ExpressionField(normal))
    
local UP, coerce, hasField;
 
inherits  Dom::UnivariateSkewPolynomial(var,id,id,e->diff(e,dvar),0,R);
category  Cat::UnivariateSkewPolynomial(R); 
axiom    (if R::hasProp(Ax::normalRep) or 
            (R::constructor=Dom::ExpressionField and R::hasProp(Ax::systemRep)) 
          then 
            Ax::normalRep 
          end_if),
         (if R::hasProp(Ax::canonicalRep) then 
           Ax::canonicalRep 
         end_if);

  // entries: 

  new:= proc(e,f)
    local i,res;
    begin
      if args(0)=0 then 
        error("Wrong number of arguments") end_if;
      if type(e)=DOM_LIST then
        e:= map(e, expr);
        e:= _plus(e[i]*var^(i-1) $i=1..nops(e));
      elif args(0)=2 then
        e:= ode::isLODE(expr(e),f,Hlode,{},{});
        if nops(e)=4 then 
          e:= expand(subs(e[1],[(diff(f,dvar$i)=var^i) $ i=0..e[4]]));
        end_if;
      else 
        e:= expr(e); 
      end_if;
      res:= poly(e,[var],R);
      if res=FAIL then 
        return(FAIL);
      else 
        new(dom,res); 
      end_if;
   end_proc;     

  leftDivide:= if hasField then
      proc(a,b)
       local q,h,da,db,inv;
       begin
         if map({args()},domtype) <> {dom} then 
           error("Illegal arguments"); 
         end_if;
         q:= dom::poly(dom::zero);
         da:= degree(a);
         db:= degree(b); 
         inv:= R::_invert(lcoeff(b));
         while da >= db and not iszero(a) do
           h:= poly(R::_mult(inv,lcoeff(a)),[var],R)*poly(var^(da-db),[var],R);
           q:= q+h;
           a:= a-dom::ore_mult(b, new(dom,h));
           da:= degree(a);
         end_while;
         return(table(hold(quotient) = new(dom,q), hold(remainder) = a));
       end_proc;
    end_if;

  rightDivide:= if hasField then
      proc(a,b)
        local q,h,da,db,inv;
        begin
          if map({args()},domtype) <> {dom} then 
            error("Illegal arguments");
          end_if;
          q:= dom::poly(dom::zero);
          da:= degree(a);
          db:= degree(b); 
          inv:= R::_invert(lcoeff(b));
          while da >= db and not iszero(a) do
            h:= poly(R::_mult(inv,lcoeff(a)),[var],R)*poly(var^(da-db),[var],R);
            q:= q+h;
            a:= a-dom::ore_mult(new(dom,h), b);
            da:= degree(a);
          end_while;
          return(table(hold(remainder) = a, hold(quotient) = new(dom,q)));
        end_proc;
     end_if;

  Dpoly:= proc() 
    begin
      if args(0) = 1 then 
        return(dom::ore_mult(dom::new(var), args(1)));
      elif args(1)=[] then 
        return(args(2));
      else 
        case op(args(1), 1) 
        of 1 do 
          dom::Dpoly([op(args(1),2..nops(args(1)))],dom::Dpoly(args(2))); 
          break;
        otherwise 
          if op(args(1),1) > 1 then 
            return(dom::zero)
          else 
            error("Illegal argument");
          end_if;
        end_case;
      end_if;
    end_proc;

  D:= dom::Dpoly;
   
  adjoint:= proc(a)
      local r;
    begin
      r:= dom::zero;
      while not iszero(a) do
        r:= r + dom::intmult(dom::ore_mult(lterm(a),coerce(lcoeff(a))),(-1)^degree(a) );
        a:= dom::reductum(a);
      end_while;
      return(r);
    end_proc;

  func_call:= proc(a,f,o=" ") // Option o: Unsimplified 
       local i,avec;
     begin
       f:= expr(context(f));
       if o=Unsimplified then 
         avec:= map(dom::vectorize(a),expr,{},{});
         return(_plus(avec[i]*diff(f,dvar $ i-1) $ i=1..nops(avec)));
       else  
         return(dom::evalLODO(a, f)); 
       end_if;
     end_proc;

  evalLODO:= proc(a,f)
      local eq,n,s;
      save y, Int;
    begin
      delete y;
      eq:= dom::func_call(a, y(dvar), Unsimplified);
      n:= dom::degree(a);
      f:= eval(text2expr(expr2text(f))); 
      eq:= subs(eq, y(dvar)=f,EvalChanges);
      s:= simplify(numer(eq));
      if iszero(s) then 
        return(s);
      else
        return(normal(s/simplify(denom(eq))));
      end_if;
    end_proc;

  mkLODO:= proc(a, f)
      local i,res;
    begin
      case type(a)
      of DOM_LIST do 
        a:= map(a, expr);
        res:= dom::new(_plus(a[i]*var^(i-1) $i=1..nops(a))); 
        break;
      otherwise
        a:= ode::isLODE(expr(a),f,Hlode,{},{});
        if nops(a)=4 then
          res:= dom::new( expand(subs(a[1],[(diff(f,dvar$i)=var^i) $ i=0..a[4]])));
        else 
          res:= FAIL;
        end_if;
      end_case;
      if res=FAIL then 
        error("cannot convert this expression"); 
      else 
        return(res); 
      end_if;
    end_proc;

  symmetricPower:= if hasField then
    proc(a:dom,m:Type::PosInt)
      local i;
      save y;
    begin
      delete y;
      a:= ode::symPower(dom::func_call(a,y(dvar),Unsimplified),y,dvar,dom::degree(a),m,{},{});
      return(dom::new(expand(subs(a,[(diff(y(dvar),dvar$i)=var^i) 
                                  $ i=0..ode::order(a,{y},{},{})]))));
    end_proc;
  end_if;

  solve:= proc(a)
       local o,res,restargs,solveOptions,odeOptions;
       save y;
     begin
       delete y;
       if domtype(a) = DOM_LIST then 
         if map({op(a)},domtype) <> {dom} then 
           error("Expecting a list of linear differential operators");
         end_if;
         return(ode::combineSolutionsOfFactors(
                  map(a,dom::func_call,y(dvar),Unsimplified),y,dvar,
                  _plus(op(map(a, dom::degree))),{},{})
                )
         end_if;  
       case degree(a)
       of 0 do 
         return({0});
       of 1 do 
         return({exp(int(-expr(dom::reductum(a))/expr(lcoeff(a)),dvar))});
       otherwise
         o:= args(2..args(0));                      
         restargs:= {o};
         restargs:= map(restargs, elem -> if elem <> {} then elem end_if);
         solveOptions:= map(restargs, 
                            elem -> if indets(elem) intersect 
                                          {Real, 
                                           MaxDegree,
                                           IgnoreAnalyticConstraints, 
                                           IgnoreSpecialCases,
                                           IgnoreProperties} 
                                       <> {}  
                                    then elem end_if);                     
         // Insert valid options to be used in 'ode' here. 
         // Currently, here in the linear situation no additional
         // options seem to make sense.
         odeOptions:= restargs intersect { Type = ExactFirstOrder, 
                                           Type = ExactSecondOrder,
                                           Type = Homogeneous,
                                           Type = Riccati,
                                           Type = Lagrange,
                                           Type = Bernoulli,
                                           Type = Abel,
                                           Type = Chini,
                                           Type = Clairaut };
         restargs:= restargs minus (solveOptions union odeOptions);
         o:= op(restargs);                   
         res:= ode::liouvillianSolutions(dom::func_call(a,y(dvar),Unsimplified),
                                         y(dvar),o,solveOptions,odeOptions);
         if res = {} then
           return(ode::specfunc(dom::func_call(a,y(dvar),Unsimplified),y,dvar,
                                solveOptions,odeOptions));
         else
           return(res);
         end_if;
       end_case;
     end_proc;

  polynomialZeros:= proc(a)
      save y;
    begin
      delete y;
      if iszero(degree(a)) then
        return({0})
      end_if;
        return(ode::polynomialSolutions(dom::func_call(a,y(dvar),Unsimplified),y(dvar),{},{}));
    end_proc;

  rationalZeros:= proc(a)
      save y;
    begin
      delete y;
      if iszero(degree(a)) then 
        return({0}); 
      end_if;
      return(ode::rationalSolutions(dom::func_call(a,y(dvar),Unsimplified),y(dvar),{},{}));
    end_proc;

  exponentialZeros :=
  proc(a)
    save y;
  begin
    delete y;
    if iszero(degree(a)) then 
      return({0});
    end_if;
    return(ode::exponentialSolutions(dom::func_call(a,y(dvar),Unsimplified),y(dvar),{},{}));
  end_proc;

  // liouvillianZeros := dom::solve;

  unimodular:= (if hasField then
    proc(a:dom, o=" ")
      save y;
      local l;
    begin
      delete y;
      if iszero(degree(a)) then
	      return(table(hold(equation)=a,hold(factorOfTransformation)=R::one));
      end_if;
      l:= ode::mkUnimodular(expr(a(y(dvar),Unsimplified)),y,dvar,degree(a),o,
                            {},{});
      return(table(hold(equation)=dom::mkLODO(l[1],y(dvar)),
                   hold(factorOfTransformation)=R(l[4])));
    end_proc
   end_if);

  companionSystem:= (if hasField then
     proc(a:dom)
       local M,d,v,i,j;
     begin
       d:= dom::degree(a);
       if d=0 then 
         error("Degree must be positive");
       else
         M:= Dom::Matrix(R);
         v:= dom::vectorize(mapcoeffs(dom::monic(a),_negate));
         delete v[d+1];
         return(M(array(1..d,1..d,
                        [[(if i+1=j then 
                             R::one 
                           else 
                             R::zero 
                           end_if) $ j=1..d] $ i=1..d-1,v])));
       end_if;
     end_proc;
   end_if);

  factor:= (if hasField then
    proc(a)
      local i,fl,typ;
    begin
      typ:= "irreducible";
      i:= 1;
      fl:= [a];
      if degree(a)>3 then 
        warning("Only right and left factors of degree 1 will be found");
        typ:= "unknown";
      end_if;
      while i <= nops(fl) do
        if degree(fl[i])>1 then 
          fl[i]:= op(dom::rlfactor(fl[i])) 
        end_if;
        i:= i+1;
      end_while;
      return(Factored::create([R::one,(fl[i],1) $ i=1..nops(fl)],typ,dom));
    end_proc
  end_if);

  factors:= (if hasField then
    proc(a)
      local i,fl;
    begin
      i:= 1;
      fl:= [a];
      if degree(a)>3 then 
        warning("Only right and left factors of degree 1 will be found");
      end_if;
      while i <= nops(fl) do
        if degree(fl[i])>1 then 
          fl[i]:= op(dom::rlfactor(fl[i]));
        end_if;
        i:= i+1;
      end_while;
      return(fl);
    end_proc;
  end_if);

  rlfactor:= (if hasField then
    proc(a)
      local da,faclist,adjlist,n,i;
    begin
      da:= degree(a);
      if (da=0 or da=1) then 
        return([a]);
      end_if;
      adjlist:= [];
      faclist:= dom::pseudoFactor(a,dom::exponentialZeros(a));
      if degree(faclist[1]) > 1 and da > 2 then
        a:= dom::adjoint(faclist[1]);
        adjlist:= dom::pseudoFactor(a,dom::exponentialZeros(a));
        n:= nops(adjlist);
        adjlist:= map([adjlist[n-i] $ i=0..(n-1)],dom::adjoint);
        delete faclist[1];
      end_if;
      return(_concat(adjlist,faclist));
    end_proc;
  end_if);

  pseudoFactor:= (if hasField then
    proc(a,l)     // can be improved! 
      local faclist,i,a2,lasta2,y,rationalizeIfCan;
    begin
      // The following procedure will do some of simplifier's work
      rationalizeIfCan:= proc(f,x)
                           local pow,p;
                         begin
                           f:= Factored::convert_to(factor(simplify(f)),DOM_EXPR);
                           pow:= map(op(rationalize(f),2), 
                                     e -> (e:=op(e,2);
                                           if type(e)="_power" then 
                                             op(e,2); 
                                           else 
                                             null();
                                           end_if));
                           if pow <> {} then
                             p:= sort([op(pow)])[1];
                             return(expand(numer(f)/x^p)/expand(denom(f)/x^p));
                           else 
                             return(f);
	                         end_if
                         end_proc;
      y:= genident("y");
      if iszero(nops(l)) or args(0)=1 then 
        return([a]); 
      end_if;
      l:= [op(map(l,expr))];
      faclist:= [];
      if degree(a) = nops(l) then 
        delete (l[nops(l)]); 
      end_if;
      lasta2:= dom::one;
      for i from 1 to nops(l) do
        // Wronskian is called without R, since then the chances to
        // get simplified coefficients are better 
        a2:= ode::wronskian([op(l,1..i),y(dvar)],dvar,{},{});
        a2:= dom::mkLODO(expr(mapcoeffs(ode::ode2poly(
                      ode::normalize(expr(a2),y,dvar,i,{},{}),y,dvar,i,{},{}),
                      rationalizeIfCan, dvar)),y(dvar));
        faclist:= [dom::rightQuotient(a2,lasta2),op(faclist)];
        lasta2:= a2;
      end_for;
      return([dom::rightQuotient(a,a2),op(faclist)]);
    end_proc;
  end_if);        

// ================================= M A I N =================================
          
begin
  if args(0) > 3 then 
    error("Wrong number of arguments"); 
  end_if;
  if domtype(var) <> DOM_IDENT or domtype(dvar) <> DOM_IDENT then 
    error("ot a variable")
  end_if;
  if args(0) = 3 then
    if R::dom=DOM_DOMAIN then
      if R::hasProp(Dom::BaseDomain) = FAIL then
        R := Dom::ExpressionField(normal)
      elif not(R::hasProp(Cat::CommutativeRing) and
	       R::hasProp(Cat::DifferentialRing) and
               iszero( R::characteristic() ) ) then
        error("Illegal coefficient ring")
      end_if;
    else 
      error("Illegal coefficient ring");
    end_if;
    //The following if-statement would nearly always solve the termination 
    //problem ("unique zero") by polynomial division and yield a correct degree.
    // if R::constructor=Dom::ExpressionField and R::hasProp(Ax::systemRep) then
    //   R:= Dom::ExpressionField(id, iszero@normal)
    // end_if;
  end_if;
  UP:= Dom::UnivariatePolynomial(var, R);
  coerce:= UP::new;
  hasField:= bool(R::hasProp(Cat::Field));
end_domain:

  
