/* ----------------------------------------------------
Dom::Matrix - the domain of (dense or sparse) matrices
--------------------------------------------------------
--------------------------------------------------------
ToDo:
1) see some minor things in the listing of the methods
   further down below
2) **!** The check for floats (as implemented in _invert,
   say) should also be implemented in operation _mult (in
   _plus, too, if the overhead is not too large). In the
   float case, call numeric::mult(A, B) (to be implemented)
   which should call hardware floats.
3) In _invert, the odering of multiplications/divisions
   works correctly for non-commutative rings, too.
   In gaussElim, this ordering must be checked and
   probably still needs to be fixed!
--------------------------------------------------------

Internally, an m x n Matrix A is represented by
extop(A,3) =[p1,..,p.n], where p1,..,p.n are univariate
polynomials (in hold(`#_X_`)) of max degree m.
The polys represent the columns, i.e:
     A[i,j] = coeff(extop(A,3)[j],i).

Internal representation: 4 operands
1st operand: r = row dimension
2nd operand: c = column dimension
3rd operand: a list with c polynomials representing the columns
4th operand: the string "FLAG".

At the moment, the 4th operand "FLAG" is just a dummy. It may
serve for storing information on the contents of the matrix
(e.g., "are there just floats in the matrix?" etc). This could
help to speed up things like passing data to hardware tools,
because no type-checking of the operands would be necessary.  
For this particular purpose, flags such as
  "Unknown"
  "RealFloat"
  "ComplexFloat"
  "RealNumerical"    // convertible to real floats
  "ComplexNumerical" // convertible to non-real floats
  "Symbolic"         // not convertible to floats
would be appropriate. E.g., _plus, _mult etc. could query this
flag at no costs and, in case of floats, could call hardware to 
multiply matrices.
Disadvantage: *each method* that could possibly change the
entries of the matrix would need to update this flag (expensive)
or replace the flag by "Unknown".
-----------------------------------------------------*/

/*--------------------------------------------------------------
Dom::Matrix -- the domains of (sparse) matrices

Dom::Matrix(<R>)

R - (optional) coefficient domain (a Cat::SemiRng)

A domain Dom::Matrix(R) represents all matrices over the
Cat::SemiRng R. If the argument 'R' is missing, the coefficient
domain will be the domain 'Dom::ExpressionField()'.

An element of a domain created by Dom::Matrix (a matrix),
has 4 operands 'r, c, L, "FLAG"'. The first two operands give
the dimension [r,c] of the matrix and the third operand L is a
list of univariate polynomials representing the matrix columns.

Methods:
        _concat          done
        _divide          done
        _index           done
        _invert          done
        _mult            done
        _negate          done
        _plus            done
        _power           done
        assignElements   done by Kai
        col              done by Kai
        concatMatrix     done by Kai
        conjugate        done by Kai
        convert          done
        convert_to       done
                         ToDo: target Dom::DenseMatrix nicht implementiert
                             (?? wirklich ?? Gefahr: automatisches
                                 DenseMatrix*Matrix --> DenseMatrix)
        create           done by Kai
                         ToDo: an coercing-Strategy von 'new' anpassen
        delCol           done by Kai
        delRow           done by Kai
        diff             done by Kai
        equal            done
        evalp            done by Kai
        exp              done
        expr             done
        expand           done by Kai
        expr2text        done by Kai
        factor           done by Kai
        float            done
        gaussElim        done
        has              done by
        identity         done by Kai
        Im               done by walter (21.11.2006)
        internalPolyVar  done by walter (10.12.2004)
        is               done
        iszero           done
        length           done by Kai
        kroneckerProduct done by Walter (1.10.2003)
        map              done by Kai
        mapNonZeroes     done
        matdim           done
        mkSparse         done
        new              done
        nops             done
        norm             done by Kai
        nonZeros         done
        nonZeroes        done
        nonZeroOperands  done
        op               done by Kai
        print            done
        doprint          done
        random           done
        randomDimen      done
        Re               done by walter (21.11.2006)
        row              done by Kai
        set_index        done
        setRow           done by Kai
        setCol           done by Kai
        simplify         done
        stackMatrix      done by Kai
        swapRow          done by Kai
        swapCol          done by Kai
        subs             done
        subsex           done
        subsop           done by Kai
        TeX              done by Kai
        tr               done by Kai
        transpose        done by Kai
        unapply          done by Kai  // deleted by ccr
        zip              done by Kai
-----------------------------------------------------------------------*/

domain Dom::Matrix( R )
    local R2polyCoerce,
          Rcoerce,
          Rid,
          Rnormal,
          Rzero,
          Rone,
          Rcharacteristic,
          polyR
         ;
    inherits Dom::BaseDomain;
    category Cat::Matrix(R);
    axiom if R::hasProp( Ax::canonicalRep ) then Ax::canonicalRep end_if;

// -----------------------------------------
//               entries
// -----------------------------------------
    isSparse:= TRUE;
    isDense := FALSE;
    randomDimen:= [10,10];
    Name:= if dom::constructor = Dom::Matrix and
	      R = Dom::ExpressionField() then
                hold(Dom::Matrix)()
           end;
    internalPolyVar := `#_X_`; // access to the internal poly variable

// -----------------------------------------
//                methods
// -----------------------------------------

/*-------------------------------------------------------------------
  evaluate -- eval of a matrix object

 This method is called by the kernel evaluator for full evaluation
 of a matrix with the current LEVEL
-------------------------------------------------------------------*/
// not called by _index anymore: no efficiency loss (hopefully)

evaluate:= proc(A)
begin
  if extop(A, 1) = 0 or
     extop(A, 2) = 0 then
     A
  else
     extsubsop(A, 3 = map(extop(A, 3), mapcoeffs, eval));
  end_if:
end_proc;
/*-------------------------------------------------------------------
  new  --  construct a new instance of Dom::Matrix(R)
           r = number of rows
           c = number of rows
           new(something)  coerces something to Dom::Matrix(R).
                           'something' may be DOM_ARRAY, Dom::DenseMatrix(..),
                           Dom::SquareMatrix(..), Dom::Matrix(..),
                           Dom::MatrixGroup(..),
                           or some (nested) list of data
           new(r,c)        returns zero Dom::Matrix(R)
           new(r,c, f)     with function f returns A[i,j]:= f(i,j)
           new(r,c,list)   converts list to Dom::Matrix(R) via mkSparse
           new(r,c,list,Diagonal) creates a diagonal (sparse) matrix
           new(r,c,list,Banded) creates a Toeplitz matrix

  Calls:
  case 1 (new(something)): dom::coerce to construct (sparse) matrices from
                           arrays, Dom::DenseMatrix etc
  case 2 (new(r,c))      : new(dom, ..) to construct zero (sparse) matrix
  case 3 (new(r,c,f))    : new(dom, ..) to construct (sparse) matrix with
                           indexing function f
  case 3 (new(r,c,f))    : new(dom, ..) to construct (sparse) matrix with
                           indexing function f
-------------------------------------------------------------------*/
new:= proc(r, c, f)
local i, j, a, eq, polyList, converted, ff, ind;
begin
  case args(0)
  of 1 do
     if testargs() then
       if r::dom::hasProp( Cat::Matrix ) <> TRUE and
          domtype(r) <> DOM_LIST and
          (domtype(r) = DOM_ARRAY and op(r,[0,1]) > 2) and
          (domtype(r) = DOM_HFARRAY and op(r,[0,1]) > 2) 
          then error("invalid argument")
       end_if
     end_if;
     if (r:= dom::coerce( r )) = FAIL then
          error("unable to define matrix over ".expr2text(R))
     else return( r )
     end_if;
  of 2 do
     /* Modification by Nicolas Thiery (4/12/2001)
        to deal with 0xn and nx0 matrices */
     if not testtype( r, Type::NonNegInt ) or not testtype( c, Type::NonNegInt )
        then error("expecting dimension as nonnegative integers")
        else return( new(dom,r,c,[poly(Rzero, [hold(`#_X_`)], polyR) $ c], "FLAG") )
     end_if
  of 3 do
     /* Modification by Nicolas Thiery (4/12/2001)
        to deal with 0xn and nx0 matrices */
     if (not testtype( r, Type::NonNegInt )) or
        (not testtype( c, Type::NonNegInt )) then
        error("expecting dimensions as nonnegative integers")
     end_if;
     converted:= FALSE;
    
     case domtype(f)
     of DOM_ARRAY do
     of DOM_HFARRAY do
        if r*c <> nops(f) then
           error("cannot convert input array to the requested format"):
        end_if;
        if op(f, [0,1]) <> 2 or
           op(f, [0, 2, 1]) <> 1 or
           op(f, [0, 2, 2]) <> r or
           op(f, [0, 3, 1]) <> 1 or
           op(f, [0, 3, 2]) <> c then
           // need to reformat the array
           if domtype(f) = DOM_ARRAY then
              f:= array(1..r, 1..c, [op(f)]);
           else
              f:= hfarray(1..r, 1..c, [op(f)]);
           end_if;
        end_if;
        if (f:= dom::coerce(f)) = FAIL then
            error("unable to define matrix over ".expr2text(R))
        else return(f)
        end_if;
        // f:= [op(f)];
        // Proceed to 'DOM_LIST' below:
     end_case;

     case domtype(f)
     of DOM_TABLE do
        //=======================================================
        // sparse input by a table f = table((i1,j1)=value1, ..)
        //=======================================================
        // convert table f to a list of equations
        // [(i1,j1) = value1, (i2,j2) = value2, ...],
        // then proceed with the 'of DOM_LIST' branch
        if Rcoerce <> id then
          f:= map(f, Rcoerce);
        end_if;
        if has(f, FAIL) then
           error("could not convert input ".expr2text(op(eq, 2)).
                 " to type ".expr2text(R)):
        end_if;
        if R2polyCoerce <> id then
           f:= map(f, R2polyCoerce);
        end_if;
        if has(f, FAIL) then
           error("could not convert input ".expr2text(op(eq, 2)).
                 " to internal representation"):
        end_if;
        converted:= TRUE;
        // proceed to the code in 'of DOM_LIST'
     of DOM_LIST do
        //=======================================================
        // sparse input by a list
        //  f = [(i1,j1)=value1, (i2,j2) = value2, ..]
        //=======================================================
        // The following code is valid both for f = table(...)
        // as well as f = [(i1, j1) = value1, ..]
        // Check whether f = table or f = list of equations.
        // If not, skip this branch and proceed to the 'otherwise' branch
        // where a list is interpreted in a different way
        if nops(f) = 0 then
           // error("expecting a list or table of entries; no data received"):
           return(new(dom,r,c,[poly(Rzero, [hold(`#_X_`)], polyR) $ c],"FLAG"));
        end_if;
        // Check only the first equation.
        // Otherwise, lots of memory is wasted
        if type(op(f, 1)) = "_equal" then
           //----------------------------------------------------------------
           // Do not do the following initialization! It is cheap in the
           // beginning, but when you start filling the polyList, it becomes
           // terribly slow!
           // polyList:= [[[Rzero, 0] $ r] $ c];
           /*----------------------------------------------------------------
           // Instead: use tables first to store the columns. This allows
           // fast sparse indexed writing into the polyList:
           ----------------------------------------------------------------*/
           polyList:= [table() $  c];
           for eq in f do
             if type(eq) <> "_equal" then
                error("expecting a list or a table of entries");
             end_if;
             if r = 1 and nops(op(eq, 1)) = 1 then
                // colum vector via [j1=value1, j2=value2, ..]
                eq:= (1, op(eq, 1)) = op(eq, 2):
             end_if;
             if c = 1 and nops(op(eq, 1)) = 1 then
                // row vector via [i1=value1, i2=value2, ..]
                eq:= (op(eq, 1), 1) = op(eq, 2):
             end_if;
             if nops(op(eq, 1)) <> 2 then
                error("expecting a list of specifications 'i = value' ".
                      "or '(i, j) = value' for the entries");
             end_if;
             i:=  op(eq, [1, 1]):
             j:=  op(eq, [1, 2]):
             if i < 1 or i > r or j < 1 or j > c then
               error("illegal index (".expr2text(i, j).") found in input table/list");
             end_if;
             if not converted then
                if Rcoerce <> id then
                  eq:= op(eq, 1) =  Rcoerce(op(eq, 2));
                end_if;
                if has(eq, FAIL) then
                   error("could not convert input ".expr2text(op(eq, 2)).
                         " to type ".expr2text(R)):
                end_if;
                if R2polyCoerce <> id then
                  eq:= op(eq, 1) =  R2polyCoerce(op(eq, 2));
                end_if;
                if has(eq, FAIL) then
                   error("could not convert input ".expr2text(op(eq, 2)).
                         " to internal representation"):
                end_if;
             end_if;
             // write into the polyList:
             polyList[op(eq, [1, 2])][op(eq, [1, 1])]:=
                          [op(eq, 2), op(eq, [1, 1])]; // [coeff, exponent]
           end_for;
           //----------------------------------------------------------------
           // convert the tables representing the columns into
           // nested lists suitable for conversion to polys
           //  'entry in tabelle' presently produces a problem due to buggy flattening
           // polyList:= map(polyList, tabelle ->
           //            [op(entry, 2) $ entry in tabelle]);
           //
           polyList:= map(polyList, tabelle ->
                         [op(tabelle, [i, 2]) $ i = 1 .. nops(tabelle)]);
           //----------------------------------------------------------------
           polyList:= map(polyList, poly, [hold(`#_X_`)], polyR);
           if has(polyList, FAIL) then
              error("unable to convert input to internal representation");
           end_if;
           return( new(dom,r,c,polyList,"FLAG") )
        else // f is a list but not a list of equations
            a:= dom::mkSparse(r, c, f);
            if a = FAIL
               then error("unable to define matrix over ".expr2text(R))
               else [r,c]:= a[2];
                    return( new(dom,r,c,a[1], "FLAG") )
            end_if
        end_if;
     otherwise //  domtype( f ) <> DOM_TABLE, DOM_LIST 
          //=======================================================
          // consider f as a function of (i,j)
          // do row wise to give rows a chance to become sparse:
          a:= [ 0 $ c ];
          //if Rcoerce <> id then
          //   f:= Rcoerce@f;
          //end_if;
          ff:= f;
          if R2polyCoerce <> id then
               ff:= (i, j) ->  R2polyCoerce(f(i, j));
          end_if;
          for j from 1 to c do
              a[j]:= poly([[ff(i,j),i] $ i=1..r],[hold(`#_X_`)], polyR);
              if a[j] = FAIL
                 then error("unable to convert function value");
              end_if;
          end_for;
          return( new(dom,r,c,a, "FLAG") )
     end_case;
     break;
  of 4 do // create diagonal matrix
     case args(4)
     of hold(Diagonal) do
        if domtype( f ) <> DOM_LIST then
           // regard f as a function
           //if Rcoerce <> id then
           //   f:= Rcoerce@f;
           //end;
           ff:= f;
           if R2polyCoerce <> id then
              ff:= (i, j) ->  R2polyCoerce(f(i, j));
           end;
           a:=[poly([[ff(i,i),i]],[hold(`#_X_`)], polyR) $ i=1..min(r,c),
                      poly(Rzero, [hold(`#_X_`)], polyR) $ i=min(r,c)+1..c];
        else
           // f is a list of diagonal elements. Need to coerce:
           if Rcoerce <> id then
              f:= map(f, Rcoerce);
           end;
           if R2polyCoerce <> id then
              f:= map(f, R2polyCoerce);
           end;
           a:=[poly([[f[i],i]],[hold(`#_X_`)], polyR) $ i=1..min(r,c,nops(f)),
                     poly(Rzero, [hold(`#_X_`)], polyR) $ i=min(r,c,nops(f))+1..c];
        end_if;
        if has( a, FAIL )
           then error("unable to define matrix over ".expr2text(R))
           else return( new(dom,r,c,a, "FLAG") )
        end_if
     of hold(Banded) do
        ind:= 2*stdlib::min(r,c) - 1; // give at most n elements in f
        if domtype( f ) <> DOM_LIST or
           nops( f ) mod 2 = 0 or
           nops( f ) > ind
        then error("expecting a list with a number of elements which ".
                        "is odd and not greater than ".expr2text(ind))
        else
           // f is a list of Toeplitz entries. Need to coerce:
           if Rcoerce <> id then
              f:= map(f, Rcoerce);
           end;
           if R2polyCoerce <> id then
              f:= map(f, R2polyCoerce);
           end;
           ind:= nops(f) div 2;
           a:= [poly([[f[ind+j-i+1],i] $ i=max(1,j-ind)..min(r,j+ind)],
                     [hold(`#_X_`)], polyR) $ j=1..c];
           if has( a,FAIL ) then
                error("unable to define matrix over ".expr2text(R))
           else return( new(dom,r,c,a, "FLAG") )
           end_if
        end_if
     otherwise error("expecting 'Diagonal' or 'Banded' as options")
     end_case
 otherwise error("wrong no of args")
 end_case
end_proc;

/*-------------------------------------------------------------------
   create: fast construction of a (sparse) matrix by a list of
           polynomials. Only for internal use! No checking!
           A list of polynomials does not contain any information
           on the number of rows, so we need to pass them:

   create(r, c, list_of_polynomials);

   Use only if you know *exactly* what you are doing!
-------------------------------------------------------------------*/

create:= proc(r, c, f)
   local i, j, a, converted, eq, polyList, ff, ind;
begin
  case args(0)
  of 1 do
       return( dom::coerce( r ) )
  of 2 do
       return( new(dom,r,c,[poly(Rzero, [hold(`#_X_`)], polyR) $ c], "FLAG") )
  of 3 do
     converted:= FALSE;
     case domtype( f )
     of DOM_TABLE do
        //=======================================================
        // sparse input by a table f = table((i1,j1)=value1, ..)
        //=======================================================
        // convert table f to a list of equations
        // [(i1,j1) = value1, (i2,j2) = value2, ...],
        // then proceed with the 'of DOM_LIST' branch
        if Rcoerce <> id then
          f:= map(f, Rcoerce);
        end_if;
//      if has(f, FAIL) then
//         error("could not convert input ".expr2text(op(eq, 2)).
//               " to type ".expr2text(R)):
//      end_if;
        if R2polyCoerce <> id then
           f:= map(f, R2polyCoerce);
        end_if;
//      if has(f, FAIL) then
//         error("could not convert input ".expr2text(op(eq, 2)).
//               " to internal representation"):
//      end_if;
        converted:= TRUE;
        // proceed to the code in 'of DOM_LIST'
     of DOM_LIST do
        //=======================================================
        // f = a list of polynomials
        //=======================================================
        if nops(f) <> 0 and domtype(f[1]) = DOM_POLY then
           return(new(dom, r, c, f, "FLAG"));
        end_if;
        //=======================================================
        // sparse input by a list
        //  f = [(i1,j1)=value1, (i2,j2) = value2, ..]
        //=======================================================
        // The following code is valid both for f = table(...)
        // as well as f = [(i1, j1) = value1, ..]
        // Check whether f = table or f = list of equations.
        // If not, skip this branch and proceed to the 'otherwise' branch
        // where a list is interpreted in a different way
        if nops(f) = 0 then
           error("expecting a list or table of entries; no data received"):
        end_if;
        // Check only the first equation.
        // Otherwise, lots of memory is wasted
        if type(op(f, 1)) = "_equal" then
           // Do not do the following initialization! It is cheap in the
           // beginning, but when you start filling the polyList, it becomes
           // terribly slow!
           // polyList:= [[[Rzero, 0] $ r] $ c];
           /*----------------------------------------------------------------
           // Instead: use tables first to store the columns. This allows
           // fast sparse indexed writing into the polyList:
           ----------------------------------------------------------------*/
           polyList:= [table() $  c];
             for eq in f do
//               if type(eq) <> "_equal" then
//                  error("expecting a list or a table of entries");
//               end_if;
                 if r = 1 and nops(op(eq, 1)) = 1 then
                    eq:= (1, op(eq, 1)) = op(eq, 2):
                 end_if;
                 if c = 1 and nops(op(eq, 1)) = 1 then
                    eq:= (op(eq, 1), 1) = op(eq, 2):
                 end_if;
//               if nops(op(eq, 1)) <> 2 then
//                  error("expecting a specification '(i, j) = value' for the entries");
//               end_if;
                 i:=  op(eq, [1, 1]):
                 j:=  op(eq, [1, 2]):
//               if i < 1 or i > r or j < 1 or j > c then
//                  error("illegal index (".expr2text(i, j).") found in input table/list");
//               end_if;
                 if not converted then
                    if Rcoerce <> id then
                      eq:= op(eq, 1) =  Rcoerce(op(eq, 2));
                    end_if;
//                  if has(eq, FAIL) then
//                     error("could not convert input ".expr2text(op(eq, 2)).
//                           " to type ".expr2text(R)):
//                  end_if;
                    if R2polyCoerce <> id then
                      eq:= op(eq, 1) =  R2polyCoerce(op(eq, 2));
                    end_if;
//                  if has(eq, FAIL) then
//                     error("could not convert input ".expr2text(op(eq, 2)).
//                           " to internal representation"):
//                  end_if;
                 end_if;
                 polyList[op(eq, [1, 2])][op(eq, [1, 1])]:=
                              [op(eq, 2), op(eq, [1, 1])]; // [coeff, exponent]
             end_for;
             //----------------------------------------------------------------
             // convert the tables representing the columns into
             // nested lists suitable for conversion to polys
             //  'entry in tabelle' presently produces a problem due to buggy flattening
             // polyList:= map(polyList, tabelle ->
             //            [op(entry, 2) $ entry in tabelle]);
             //
             polyList:= map(polyList, tabelle ->
                           [op(tabelle, [i, 2]) $ i = 1 .. nops(tabelle)]);
             //----------------------------------------------------------------
             polyList:= map(polyList, poly, [hold(`#_X_`)], polyR);
//           if has(polyList, FAIL) then
//              error("unable to convert input to internal representation");
//           end_if;
             return( new(dom,r,c,polyList,"FLAG") )
      //else // f is a list, but not a list of equations.
             // Jump to the 'otherwise' branch
              end_if;
     otherwise
        //=======================================================
        if domtype( f ) <> DOM_LIST then
            // consider f as a function of (i,j)
            // do row wise to give rows a chance to become sparse:
            a:= [ 0 $ c ];
            ff:= f;
            if R2polyCoerce <> id then
               ff:= (i, j) ->  R2polyCoerce(f(i, j));
            end_if;
            for j from 1 to c do
                a[j]:= poly([[ff(i,j),i] $ i=1..r],[hold(`#_X_`)], polyR);
//              if a[j] = FAIL
//                 then error("unable to convert function value");
//              end_if;
            end_for;
            return( new(dom,r,c,a, "FLAG") )
        else // f is a list
            a:= dom::mkSparse(r, c, f);
            if a = FAIL
               then error("unable to define matrix over ".expr2text(R))
               else [r,c]:= a[2];
                    return( new(dom,r,c,a[1], "FLAG") )
            end_if
        end_if
     end_case;
  of 4 do // create diagonal matrix
     //=======================================================
     // create(r, c, f, "FLAG"), f = a list of polynomials
     //=======================================================
     if nops(f) <> 0 and domtype(f[1]) = DOM_POLY then
        return(new(dom, r, c, f, "FLAG"));
     end_if;
     //=======================================================
     //
     //=======================================================
     case args(4)
     of hold(Diagonal) do
        if domtype( f ) <> DOM_LIST then
             if R2polyCoerce <> id then
                f:= R2polyCoerce@f;
             end;
             a:=[poly([[f(i,i),i]],[hold(`#_X_`)], polyR) $ i=1..min(r,c),
                 poly(Rzero, [hold(`#_X_`)], polyR) $ i=min(r,c)+1..c];
        else
             if R2polyCoerce <> id then
                f:= map(f, R2polyCoerce);
             end;
             a:=[poly([[f[i],i]],[hold(`#_X_`)], polyR) $ i=1..min(r,c,nops(f)),
                        poly(Rzero, [hold(`#_X_`)], polyR) $ i=min(r,c,nops(f))+1..c];
        end_if;
        return( new(dom,r,c,a, "FLAG") )
     of hold(Banded) do
        ind:= 2*stdlib::min(r,c) - 1; // give at most n elements in f
        if domtype( f ) <> DOM_LIST or
           nops( f ) mod 2 = 0 or
           nops( f ) > ind
        then error("expecting a list with a number of elements which ".
                   "is odd and not greater than ".expr2text(ind));
        else ind:= nops(f) div 2;
             if R2polyCoerce <> id then
                f:= map(f, R2polyCoerce);
             end;
             a:= [poly([[f[ind+j-i+1],i] $ i=max(1,j-ind)..min(r,j+ind)],
                       [hold(`#_X_`)], polyR) $ j=1..c];
             if has( a,FAIL )
                then error("unable to define matrix over ".expr2text(R))
             end_if;
             return( new(dom,r,c,a, "FLAG") )
        end_if;
     otherwise error("expecting 'Diagonal' or 'Banded' as options")
     end_case
   otherwise error("wrong no of arguments")
 end_case
end_proc;


/*-------------------------------------------------------------------
   mkSparse  --  convert array or nested list to a poly list

         array(a..a+r,b..b+c,..)         -> [[poly,...,poly],[r,c]]
         [[a11,a12,..], [a21,a22,..],..] -> [[poly,...,poly],[r,c]]
         [a1, a2, ..] -> [[a1],[a2],..]  -> [[poly],[r,1]]
-------------------------------------------------------------------*/
mkSparse:= proc(a)
local i, j, r, c, __a, t, offsi, offsj, jj, cols, jcols;
begin
case args(0)
of 1 do
    case domtype(a)
    of DOM_ARRAY do
       if has(a,NIL) then
           userinfo(1,"unable to define matrix over ",R);
           return( FAIL )
       end_if;
       if Rcoerce <> id then
          a:= map(a, Rcoerce);
       end_if;
       if R2polyCoerce <> id then
          a:= map(a, R2polyCoerce);
       end_if;
       if op(a,[0,1]) = 1 then
         offsi:= op(a,[0,2,1])-1;
         r:= op(a,[0,2,2]) - offsi;
         a:= [[poly([[a[i+offsi],i]$i=1..r],[hold(`#_X_`)],polyR)],[r,1]];
         if has(a, FAIL) then
            return(FAIL)
         else
            return(a);
         end_if:
       end_if;
       if op(a,[0,1]) = 2 then
         offsi:= op(a,[0,2,1])-1;
         offsj:= op(a,[0,3,1])-1;
         r:= op(a,[0,2,2]) - offsi;
         c:= op(a,[0,3,2]) - offsj;
         t:= [ 0 $ c ];
         for j from 1 to c do
           jj:= j + offsj;
           t[j]:= poly([[a[i+offsi,jj],i] $ i=1..r ], [hold(`#_X_`)], polyR);
           if t[j] = FAIL then return(FAIL) end_if;
         end_for;
         return([t,[r,c]]);
      end_if;
      // now op(a,[0,1])>2. This can't be a matrix
      userinfo(1,"unable to define matrix over ",R);
      return( FAIL );
    of DOM_LIST do
        // a list of raw data to be converted to Dom::Matrix
        r:= nops(a);
       /* Modification by Nicolas Thiery (4/12/2001)
          to deal with 0xn and nx0 matrices
          Warning: there is an ambiguity here: [] can represent a
          vector with no components, or a list of 0 vectors of
          arbitrarily length.
          I decided that [] represents the 0x0 matrix.
        */
        if r = 0 then
           c := 0;
        elif domtype(a[1]) <> DOM_LIST  then //convert [a1,a2,..]->[[a1],[a2],..]
          c := 1;
          a:= map(a, (()->([args(1)])))
        elif map({op(a)},domtype) <> {DOM_LIST} then
             return( FAIL )
        else c:= max(op(map(a,nops)))
        end_if;
        break
    of DOM_HFARRAY do
        if traperror((a:= array(op(a, [0,i]) $ i = 2..nops([op(a, 0)]), [op(a)]))) = 0 then
           return(dom::mkSparse(a))
        else
           return( FAIL )
        end_if;
    otherwise
        return( FAIL )
    end_case;
    break
of 3 do
    r := a; c := args(2);
    /* Modification by Nicolas Thiery (4/12/2001)
       to deal with 0xn and nx0 matrices */
    if not testtype( r,Type::NonNegInt ) or
               not testtype( c,Type::NonNegInt ) then
       userinfo(1,"expecting nonnegative integers as dimension");
       return( FAIL )
    end_if;
    a := args(3);
    /* handle special cases for vectors
       M( 1,2,[1,2] ) -> [1,2]
       M( 2,1,[1,2] ) -> +- -+
                         | 1 |
                         | 2 |
                 i       +- -+
    */
    if r = 1 then
         if nops(a) > c then error("number of columns does not match") end_if;
         if domtype(op(a)) <> DOM_LIST then a:= [a] end_if;
    elif c = 1 then
         if nops(a) > r then
           error("number of rows does not match")
         elif domtype(op(a)) = DOM_LIST then
           error("expecting list of entries")
         end_if
    end_if;
    t := map({op(a)},domtype);
    if t = {DOM_LIST} then
      if max(op(map(a, nops))) > c then
         error("number of columns does not match")
      end_if;
      if nops(a) > r then
        error("number of rows does not match")
      end_if;
    elif contains(t,DOM_LIST) then
         return( FAIL )
    elif nops(a) = r*c then
         // input as a plain list [a11, .. , a.1.c, a21, ..., a.r.c]
         // convert to a nested list
         a:= [[a[(i-1)*c + j] $ j = 1.. c] $ i = 1 .. r];
    elif nops(a) > r then
       error("number of rows does not match")
    end_if;
    break
otherwise
    return( FAIL )
end_case;

// define matrix by a list of lists

   if domtype(a)<> DOM_LIST
      then error("should not arrive here")
   end_if;

   // allow to write matrix([a,b,c])
   // instead of matrix([[a],[b],[c]])
   // if map({op(a)},domtype) <> {DOM_LIST} // way too expensive!!
   // Heuristics: check only the first entry:
   if nops(a) > 0 and domtype(a[1]) <> DOM_LIST 
      then a:= map(a, ()-> [args(1)]);   
   end_if;

   // a should now be of the form a = [[a11,a12,..],[a21,a22,..],.. ]
   if Rcoerce <> id then
     a:= map(a, map, Rcoerce);
   end_if;
   if R2polyCoerce <> id then
      a:= map(a, map, R2polyCoerce);
   end_if;
   if has(a, FAIL) then return(FAIL); end_if;

   // now convert nested list a to sparse poly representation __a
   __a:= [poly(Rzero, [hold(`#_X_`)], polyR) $ c]; // initialize container

   cols:= map(a, nops); // cols[i] = length of a[i] = length of i-th row
    /* Modification by Nicolas Thiery (4/12/2001)
       to deal with 0xn and nx0 matrices */
   if c=0 or min(op(cols)) >= c
   then // all rows have at least c entries (further elements
        // will be ignored)
        // use for loop to give polys a chance to become sparse
        for j from 1 to c do
          __a[j]:= poly([[a[i][j],i] $ i=1..nops(a)], [hold(`#_X_`)], polyR);
          if has(__a[j], FAIL) then return(FAIL) end_if;
        end_for;
   else // some rows need to be filled with zeroes
        jcols:= [i $ i=1.. min(r,nops(a))]; // prototype of list of
                                            // row indices in col j
        for j from 1 to min(c,max(op(cols))) do
           t:= map(jcols, i-> if cols[i]<j then null() else i end_if);
           __a[j]:= poly([[a[i][j],i] $ i in t], [hold(`#_X_`)], polyR);
           if has(__a[j], FAIL) then return(FAIL) end_if
        end_for;
   end_if;
   [__a, [r, c]];
end_proc;

/*--------------------------------------------------------------------
  _concat  --  appends matrices horizontally (see concatMatrix)
--------------------------------------------------------------------*/

_concat:= dom::concatMatrix;

/*-------------------------------------------------------------------
  _index_intern --  indexed access;  implementation for _index and indexval
-------------------------------------------------------------------*/
_index_intern:= proc(x, idx : DOM_LIST, evaluate = TRUE : DOM_BOOL)
local a, i, j, i1, i2, j1, j2, r, c, column;
begin

  [r,c]:= [extop(x,1),extop(x,2)];
  if nops(idx) = 2 then
    // default case: call x[i,j] or x[i1..i2,j1..j2]
    i := idx[1]; j := idx[2];
     if domtype(i) = DOM_INT and domtype(j) = DOM_INT then
        if (i < 1 or i > r) then
           context(hold(error)("first index out of range"))
        end_if;
        if (j < 1 or j > c) then
           context(hold(error)("second index out of range"))
        end_if;
        a:= coeff(extop(x,3)[j],i);
        if evaluate = TRUE then
           a:= eval(a);
        end_if;
        if polyR = Expr or R=polyR then return(a) end_if;
        if R<>polyR then return(Rcoerce(a)) end_if;
     end_if;

     if type(i) = "_range" or type(j) = "_range" then
        // i=i1..i2 and/or j=j1..j2
        // insert code for extraction of submatrix i1..i2,j1..j2
        i1:= op(i,1);
        if type(i)="_range" then i2:= op(i,2) else i2:= i1; end_if;
        j1:= op(j,1);
        if type(j)="_range" then j2:= op(j,2) else j2:= j1; end_if;
        if domtype(i1) <> DOM_INT or
           domtype(i2) <> DOM_INT or
           domtype(j1) <> DOM_INT or
           domtype(j2) <> DOM_INT then
           return(hold(_index)(x, op(idx)));
        end_if;
        if i2>r or j2>c or i1<1 or j1<1 then
           context(hold(error)("index out of range"));
        end_if;
        x:= extop(x,3); // the poly list
        x:= [x[j] $ j=j1..j2]; // reduced poly list
        for j from 1 to nops(x) do
            column:= poly2list(x[j]);
            column:= map(column, proc(a) begin
                             if a[2]>=i1 and a[2]<=i2
                                then [a[1],a[2]-i1+1]
                                else null()
                             end_if
                           end_proc);
            x[j]:= poly(column, [hold(`#_X_`)], polyR);
        end_for;
        return(new(dom, i2-i1+1, j2-j1+1, x, "FLAG"));
     end_if;
    
  elif  nops(idx) = 1 then
    i := idx[1]; j := FAIL;
     //call _index(x,i) == x[i] == x[i,1] or x[1,i]

     /* Modification by Nicolas Thiery (4/12/2001)
        to allow for 0xn and nx0 matrices.
     */
     if r <> 1 and c <> 1 then
        context(hold(error)("missing column index"));
     end_if;
     if r = 1 then
        if domtype(i) = DOM_INT then
           if i<1 or i>c then
              context(hold(error)(" column index out of range"));
           end_if;
           // do not use a recursive call to _index,
           // but duplicate above code for speed
           a:= coeff(extop(x,3)[i],1);
           if evaluate = TRUE then
              a:= eval(a);
           end_if;
           if polyR = Expr or R=polyR then return(a) end_if;
           if R<>polyR then return(Rcoerce(a)) end_if;
        end_if;
        if type(i)="_range" then
             i2:= op(i,2); i1:= op(i,1);
        else i2:= i1:= i;
        end_if;
        if domtype(i1) <> DOM_INT or
           domtype(i2) <> DOM_INT then
           return(hold(_index)(x, op(idx)));
        end_if;
        if i1<1 or c<i2 then
           context(hold(error)("column index out of range"));
        end_if;
        x:= extop(x,3); // the poly list
        x:= [x[i] $ i=i1..i2]; // reduced poly list
        return(new(dom, r, i2-i1+1, x, "FLAG"));
     end_if;
     if c = 1 then
        if domtype(i) = DOM_INT then
           if i<1 or i>r then
              context(hold(error)("row index out of range"));
           end_if;
           // do not use a recursive call to _index,
           // but duplicate above code for speed
           a:= coeff(extop(x,3)[1],i);
           if evaluate = TRUE then
              a:= eval(a);
           end_if;
           if polyR = Expr or R=polyR then return(a) end_if;
           if R<>polyR then return(Rcoerce(a)) end_if;
        end_if;
        if type(i)="_range" then
             i2:= op(i,2); i1:= op(i,1)
        else i2:= i1 := i;
        end_if;
        if domtype(i1) <> DOM_INT or
           domtype(i2) <> DOM_INT then
           return(hold(_index)(x, op(idx)));
        end_if;
        if i1<1 or r<i2 then
           context(hold(error)("column index out of range"));
        end_if;
        x:= extop(x,3); // the poly list
        column:= poly2list(x[1]);
        column:= map(column, proc(a) begin
                       if a[2]>=i1 and a[2]<=i2
                           then [a[1],a[2]-i1+1]
                             else null()
                           end_if
                       end_proc);
        x[1]:= poly(column, [hold(`#_X_`)], polyR);
        return(new(dom, i2-i1+1, 1, x, "FLAG"));
     end_if;
  elif  nops(idx) = 0 then
     context(hold(error)("no indices specified"));
  else
     context(hold(error)("expecting no more than 2 indices"));
  end_if;
  return(hold(_index)(x, op(idx)));
end_proc;
        
_index   := x -> dom::_index_intern(x, [args(2..args(0))], TRUE);
indexval := x -> dom::_index_intern(x, [args(2..args(0))], FALSE);
        
/*-------------------------------------------------------------------
  set_index  --  indexing of matrices
      set_index(x, i, j, y2)  <--> x[i,j]:= y2;
      set_index(x, i, j)      <--> x[i]:= j;
-------------------------------------------------------------------*/
set_index:= 
proc(x,i,j,y2)
  local polyList, column;
begin
  if args(0) = 4 and type(i) = "_range" or type(j) = "_range" then 
    return(x::dom::set_index2(x,i,j,y2))
  end_if;
  if args(0) = 3 then
    if extop(x,1) > 1 and extop(x,2) > 1 then
      warning("expecting row and column index");
      return(x);
    end_if;
  end_if;
  if testargs() then
     // check correct type of entry only on interactive level
     if args(0) = 4 then
          y2:= Rcoerce(y2);
          if y2 = FAIL then
             warning("new entry not compatible ".
                     "with coefficient domain");
             return(x);
          end_if;
     else j:= Rcoerce(j);
          if j = FAIL then
             warning("new entry not compatible ".
                     "with coefficient domain");
             return(x);
          end_if;
     end_if
  end_if;
  if args(0) = 4 then
       y2 := R2polyCoerce(y2):
       if i>extop(x,1) or j>extop(x,2) or i<1 or j<1 then
          warning("index out of range");
          return(x);
       end_if;
       column:= extop(x,3)[j];
       column:= polylib::setcoeff(column, i, y2);
       if column <> FAIL then
            // do not use subsop on the list extop(x, 3). Too slow!
            // return(extsubsop(x, 3=subsop(extop(x,3), j = column)))
            // Instead, use an indexed assignment to change the list:
            polyList:= extop(x, 3);
            polyList[j] := column;
            return(extsubsop(x, 3=polyList));
       else warning("new entry not compatible with coefficient domain");
            return(x);
       end_if;
  elif extop(x,1) = 1 then
       // a single row = list of 1st order polys
       if i>extop(x,2) or i<1 then
          warning("index out of range");
          return(x);
       end_if;
       j := R2polyCoerce(j):
       column:= extop(x,3)[i];
       column:= poly([[j, 1]], [hold(`#_X_`)], op(column,3));
       if column <> FAIL then
            // do not use subsop on the list extop(x, 3). Too slow!
            // return(extsubsop(x, 3=subsop(extop(x,3), i = column)))
            // Instead, use an indexed assignment to change the list:
            polyList:= extop(x, 3);
            polyList[i] := column;
            return(extsubsop(x, 3=polyList));
       else warning("new entry not compatible ".
                    "with coefficient domain");
            return(x)
       end_if;
  else // a single column = [poly]
       if i>extop(x,1) or i<1 then
          warning("index out of range");
          return(x);
       end_if;
       j := R2polyCoerce(j):
       column:= extop(x,3)[1];
       column:= polylib::setcoeff(column, i, j);
       if column <> FAIL then
            return(extsubsop(x, 3=[column]));
       else warning("new entry not compatible ".
                    "with coefficient domain");
            return(x)
       end_if;
  end_if;
end_proc;

/*-------------------------------------------------------------------
  set_index2  --  substituting submatrices via indexing
      set_index(x, i1..i2, j1..j2, y)  <--> x[i1..i2,j1..j2]:= y;

  The matrix y must be of the same type as x or has to be converted
  to a matrix of the same type as x. If this fails, the matrix x 
  is returned. Otherwise, the submatrix x[i1..i2, j1..j2] of x is
  substituted by y.

  NOTE: This procedure can only be called from set_index. And 
        this is only the case, if set_index is called with exactly 
        four argument, at least one of them a range. 
      
-------------------------------------------------------------------*/

set_index2:= proc(x, r_range, c_range, y)
  local rx, cx, ry, cy, j, colx, colsx, coly, X, R;
begin 
  if ((y:= x::dom::convert(y))) = FAIL then 
    warning("new submatrix is incompatible with the original matrix");
    return(x);
  end_if;

  rx:= extop(x,1);
  cx:= extop(x,2);
  ry:= extop(y,1);
  cy:= extop(y,2);

  if type(r_range) = "_range" and type(c_range) = "_range" then 
    if r_range[1] <= r_range[2] and 
       c_range[1] <= c_range[2] and 
       r_range[1] >= 1 and 
       r_range[2] <= rx and 
       c_range[1] >= 1 and 
       c_range[2] <= cx and 
       r_range[2] - r_range[1] + 1 = ry and 
       c_range[2] - c_range[1] + 1 = cy then 
      colsx:= extop(x, 3);
      if r_range = 1 .. rx then
         colsx:= colsx[1..c_range[1]-1].extop(y,3).colsx[c_range[2]+1..cx] ;
      else
         for j from c_range[1] to c_range[2] do 
           colx:= colsx[j];
           [X, R]:= [op(colx, [2, 1]), op(colx, 3)]:
           coly:= extop(y, 3)[j - c_range[1] + 1];
           colx:= select(poly2list(colx), L -> L[2] < r_range[1] or
                                               L[2] > r_range[2] );
           if r_range[1] = 1 then
             colx:= poly(colx, [X], R) + coly;
           else
             colx:= poly(colx, [X], R) + poly(X^(r_range[1]-1), [X], R)*coly;
           end_if;
           colsx[j]:= colx;
         end_for:
      end_if;
      return(extsubsop(x, 3 = colsx));
    else 
      warning("incompatible range");
      return(x);
    end_if;

  elif type(r_range) = "_range" and type(c_range) = DOM_INT then 
    if cy = 1 and 
       r_range[1] <= r_range[2] and 
       r_range[1] >= 1 and 
       r_range[2] <= rx and 
       c_range    >= 1 and 
       c_range    <= cx and 
       r_range[2] - r_range[1] + 1 = ry then 
      colsx:= extop(x, 3);
      if r_range = 1 .. rx then
         colsx[c_range]:= extop(y, 3)[1];
      else
         colx:= colsx[c_range];
         [X, R]:= [op(colx, [2, 1]), op(colx, 3)]:
         coly:= extop(y, 3)[1];
         colx:= select(poly2list(colx), L -> L[2] < r_range[1] or
                                             L[2] > r_range[2] );
         if r_range[1] = 1 then
           colx:= poly(colx, [X], R) + coly;
         else
           colx:= poly(colx, [X], R) + poly(X^(r_range[1]-1), [X], R)*coly;
         end_if;
         colsx[c_range]:= colx;
      end_if;
      return(extsubsop(x, 3 = colsx));
    else 
      warning("incompatible range");
      return(x);
    end_if;

  elif type(c_range) = "_range" and type(r_range) = DOM_INT then 
    if ry = 1 and 
       c_range[1] <= c_range[2] and 
       c_range[1] >= 1 and 
       c_range[2] <= cx and 
       r_range    >= 1 and 
       r_range    <= rx and 
       c_range[2] - c_range[1] + 1 = cy then 
      for j from c_range[1] to c_range[2] do 
        x[r_range,j]:= y[1, j - c_range[1] + 1];
      end_for;
      return(x);
    else 
      warning("incompatible range");
      return(x);
    end_if;
  else 
    warning("expecting ranges or integers as matrix indices");
    return(x);
  end_if;
  return(x);
end_proc;


/*-------------------------------------------------------------------
  doprint  --  print a matrix as a sparse table-like objekt

Changed 8.10.03: printing is not done by an explicit print anymore,
                 but by returning an object, sending the object
                 through the standard output system.

  The standard print method does not display large matrices
  as arrays, because it would create a dense array filled
  with lots of zeroes (a Windows machine is likely to crash
  due to memory overload).
  With doprint, the user can enforce printing of the (sparse) matrix
  entries. The result contains only a list of the nonzero entries:
    doprint(A) ->
        Dom::Matrix(R)(r, c, [(i1, j1) = value, ...])
-------------------------------------------------------------------*/
doprint:= proc(x)
local r, c, t;
begin
    [r, c, t]:= [extop(x, 1),
                 extop(x, 2),
                 [op(x::dom::convert_to(x, DOM_TABLE))]
                ];
    t:= sort(t, (x, y) -> if op(x, [1, 1]) < op(y, [1, 1]) then
                               TRUE;
                          elif op(x, [1, 1]) = op(y, [1, 1]) and
                               op(x, [1, 2]) < op(y, [1, 2]) then
                               TRUE;
                          else FALSE;
                          end);

    subsop(hold(dummy)(r, c, t), 0 = dom);
end_proc;

/*-------------------------------------------------------------------
  print  --  print a matrix as an array
-------------------------------------------------------------------*/
printMaxSize := 500;

setPrintMaxSize:= proc(m)
local old;
begin
  old:= dom::printMaxSize;
  if not ((domtype(m) = DOM_INT and m > 0) or (m = infinity)) then
     error("expecting a positive integer");
  end_if:
  sysassign(dom::printMaxSize, m):
  return(old);
end_proc;

print:=
proc(x)
  local r,c, asciiArt;
begin
   [r,c]:= x::dom::matdim(x);
   if r*c > dom::printMaxSize then
        warning("This matrix is too large for display. ".
                "If you want to see all non-zero entries ".
                "of large matrices, use doprint(..)."):
      return(expr2text(subsop(hold(dummy)(r, c, ["..."]), 0 = dom)));
   end_if;
  
  if PRETTYPRINT then
    asciiArt := () -> stdlib::Exposed(_concat(op(args())));
     map(dom::convert_to(x, DOM_ARRAY), generate::sortSums);
   else // PRETTYPRINT:= FALSE;
     if Pref::mackichan() = TRUE then
       map(dom::convert_to(x, DOM_ARRAY), generate::sortSums);
     else
       x::dom::expr2text(x);
     end_if;
   end_if;
end_proc;

/*-------------------------------------------------------------------
   expr -- returns the matrix as an element of DOM_ARRAY, whereby
           each entry was converted to an expression (using R::expr).
-------------------------------------------------------------------*/
expr:= x -> map(dom::convert_to(x, DOM_ARRAY), R::expr);

/*-------------------------------------------------------------------
   float -- maps float to the elements of the matrix, but only if
    the domain of the elements has a float method.
            The result is FAIL or Dom::Matrix(Ring)
-------------------------------------------------------------------*/
float:= proc(x)
local y, f;
begin
  // if R::float = FAIL then return(FAIL) end_if;
  y:= extop(x,3);
  if polyR = Expr then
     return(new(dom, extop(x, 1), extop(x, 2),
                map(y, mapcoeffs, float), "FLAG"));
  else f:= R::float;
       if f = FAIL then f:= float; end_if;
       if Rcoerce <> id then
          f:= f@Rcoerce;
       end_if;
       if R2polyCoerce <> id then
          f:= R2polyCoerce@f;
       end_if;
       y := map(y, mapcoeffs, f);
       if has(y, FAIL) then
          return(FAIL)
       else
          return(new(dom, extop(x, 1), extop(x, 2), y , "FLAG"));
       end_if;
  end_if;
end_proc;

/*-----------------------------------------------------------------------
   convert -- convert objects to a Dom::Matrix(R)
              domtype(object) may be:
                    Dom::Matrix(R)        -> return immediately
                    DOM_ARRAY                   -> call mkSparse
                    DOM_HFARRAY                 -> call mkSparse
                    DOM_LIST                    -> call mkSparse
                    Dom::DenseMatrix(AnyRing)   -> call mkSparse
                    Dom::SquareMatrix(AnyRing)  -> call mkSparse
                    Dom::MatrixGroup(AnyRing)   -> call mkSparse
                    Dom::Matrix(AnotherRing) ->replace coeff ring
-----------------------------------------------------------------------*/
convert:= proc(x)
local r,c, xpolyR, xRcoerce;
begin
  if args(0) <> 1 then return( FAIL ) end_if;
  case domtype(x)
  of dom do return( x )
  of DOM_ARRAY do
  of DOM_HFARRAY do
  of DOM_LIST do
      x:= dom::mkSparse( x );
      if x = FAIL
      then return( FAIL )
      else [r,c]:= x[2];
            return(new(dom, r, c, x[1], "FLAG"));
      end_if
  otherwise
    case x::dom::constructor
    of dom::constructor do
       // convert from Dom::Matrix(anotherRing) to
       // Dom::Matrix(thisRing), thisRing = polyR;
       // Here extop(x, 3) = [poly(..,anotherRing),..,poly(..,anotherRing)]

       [r,c]:= [ extop(x,1), extop(x,2) ];
       xpolyR:= op(extop(x,3)[1],3); // extract coeff ring from first polynomial
       if xpolyR = polyR then
          // we know that dom::coeffRing <> x::dom::coeffRing
          //--------------------------------------------------------------
          // Problem: if x = Matrix(anotherRing)(..) and if dom::coeffRing
          // does not accept the entries from anotherRing and if
          // polyR(anotherRing) = polyR(dom::coeffRing) then we do not
          // notice on the poly level that no conversion is possible (e.g.,
          // dom::coeffRing = Dom::Integer, anotherRing = Dom::Integer.
          // Both have the same polyR = Expr!)
          //--------------------------------------------------------------
          xRcoerce:= x::dom::coeffRing::coerce;
          x:= extop(x, 3);
          x:= map(x, col -> mapcoeffs(col, R2polyCoerce@Rcoerce@xRcoerce));
       elif x::dom::coeffRing::characteristic <> FAIL and
            x::dom::coeffRing::characteristic <> 0  then
            // Unfortunately, the internal representation IntMod(c)
            // uses _mods, whereas the default representation of
            // Dom::IntegerMod(c) uses _modp. We have to reconvert
            // to the _modp representation before converting to another
            // ring:
            xRcoerce:= x::dom::coeffRing::coerce;
            x:= map(extop(x, 3), poly2list);
            if polyR = Expr then
               x:= map(x, map, c -> [expr(Rcoerce(xRcoerce(c[1]))), c[2]]):
            else
               x:= map(x, map, c -> [R2polyCoerce(Rcoerce(xRcoerce(c[1]))), c[2]]):
            end_if;
            x:= map(x, col -> poly(col, [hold(`#_X_`)], polyR));
       elif dom::coeffRing::characteristic <> FAIL and
            dom::coeffRing::characteristic <> 0  
            and xpolyR = Expr then
                // some irresponsible user might have put domain elements
                // such as Dom::IntegerMod(p) into a Dom::Matrix() with
                // xpolyR = Expr. If the target domain is IntMod(q), the 
                // polys would create FAIL from Dom::IntegerMod(p) objects.
                // We need to a apply expr explicitly to the polys in x,
                // then we may replace the rings in the polys:
                x:= map(x, expr);
                if traperror((
                  x:= subs(extop(x,3),xpolyR=polyR); //replace the coeff ring
                            // inside the polynomials. This does the coercion!
                  )) <> 0 then
                  return(FAIL);
               end_if:
       else
            if traperror((
              x:= subs(extop(x,3),xpolyR=polyR); //replace the coeff ring
                        // inside the polynomials. This does the coercion!
           )) <> 0 then
              return(FAIL);
            end_if;
       end_if;
       if has(x, FAIL) then
            return(FAIL)           // poly coercion failed
       else return(new(dom,r,c,x, "FLAG"));// polys are succesfully changed
       end_if;
    // do Dom::DenseMatrix *after* Dom::Matrix: otherwise, if we set
    // Dom::DenseMatrix:= Dom::Matrix, we would enter this branch and end up
    // in dom::mkSparse, which is not supposed to act on a Cat::Matrix object.
    of Dom::SquareMatrix do
    of Dom::MatrixGroup do
    of Dom::DenseMatrix do
//     x:= extop(x, 3);
//     x:= dom::mkSparse(x);
       x:= dom::mkSparse(expr(x));
       if x = FAIL then return( FAIL ) end_if;
       [r,c]:= x[2];
       return(new(dom, r, c, x[1], "FLAG"));
     otherwise return( FAIL )
     end_case
  end_case
end_proc;

/*-----------------------------------------------------------------------
convert_to  --  convert matrix e to some domain type F

     F = Dom::Matrix(R)  -> return immediately
     F = DOM_ARRAY             -> return dense array
     F = LIST                  -> return dense list of lists
-----------------------------------------------------------------------*/
convert_to:= proc(e,F)
local i, j, t, r, c, B, jthcol, nt, koeffs, exponents,
      epolyR, theotherpolyR, Output, polyList, list, FRcoerce;
begin
  if args(0) <> 2 then return( FAIL ) end_if;
  if domtype(e) = F then return(e) end_if;
  if domtype(e) <> dom then
     return(FAIL);
     /*
     error("(Dom::Matrix(..))::convert_to(e,F) called ".
           "with domtype(e) <> Dom::Matrix(..)");
     */
  end_if;
  [r, c]:= e::dom::matdim(e);
  if domtype(F) <> DOM_DOMAIN // if F = element of some domain, then
  then F:= domtype(F)         // convert to this domain
  end_if;
  case F
  of DOM_ARRAY do
     B:= array(1..r,1..c,[[Rzero $ c] $ r]);
     for j from 1 to c do
         jthcol:= extop(e,3)[j];
         nt:= nterms(jthcol);
         koeffs:= [coeff(jthcol)];
         exponents:= map(poly2list(jthcol),op,2);
         (B[exponents[i],j]:= koeffs[i];) $ i=1..nt;
         if R<>polyR and R<>Dom::ExpressionField() then
            (B[i,j]:= Rcoerce(B[i,j]);) $ i=1..r;
         end_if;
     end_for;
     return(B);
  of DOM_HFARRAY do
     /* Modification by Nicolas Thiery (4/12/2001)
        to deal with 0xn and nx0 matrices */
     if r = 0 or c=0 then
        warning("Cannot convert properly 0xn and nx0 matrices to DOM_HFARRAY");
        if r=0 then r:=1; end_if;
        if c=0 then c:=1; end_if;
        return(hfarray(1..r,1..c,[[Rzero $ c] $ r]));
     end_if;
     B:= hfarray(1..r,1..c,[[float(0) $ c] $ r]);
     for j from 1 to c do
         jthcol:= extop(e,3)[j];
         nt:= nterms(jthcol);
         koeffs:= [coeff(jthcol)];
         exponents:= map(poly2list(jthcol),op,2);
         (B[exponents[i],j]:= koeffs[i];) $ i=1..nt;
         if R<>polyR and R<>Dom::ExpressionField() then
            (B[i,j]:= Rcoerce(B[i,j]);) $ i=1..r;
         end_if;
     end_for;
     return(B);
  of DOM_LIST do
      t:= extop(e,3);
      return( [[coeff(t[j],i) $ j=1..extop(e,2)] $ i=1..extop(e,1)] )
  of DOM_TABLE do
      t:= extop(e,3);
      Output:= table();
      for j from 1 to c do
         polyList:= poly2list(t[j]);
         if Rcoerce <> id then
            polyList:= map(polyList, list -> [Rcoerce(list[1]), list[2]]);
            if has(polyList, FAIL) then
              return(FAIL);
            end_if;
         end_if;
         for list in polyList do
            Output[list[2],j]:= Rcoerce(list[1]);
         end_for;
      end_for;
      return(Output);
  end_case;

  case F::constructor
  of Dom::Matrix do
      // convert from Dom::Matrix(R) to
      // Dom::Matrix(anotherRing).
      // Have a look into the internal representation. If
      // the 'official' rings are different, but the poly rings coincide,
      // only the coeffRing has to be changed but no other coercion
      // is necessary.
      epolyR:= op(extop(e,3)[1],3); // extract from first polynomial
      // create an empty matrix of the other type and extract
      // the other poly ring from the first polynomial
      theotherpolyR:= op(extop(F(1, 1),3)[1],3);
      if theotherpolyR = FAIL then return(FAIL) end_if;
      [r,c]:= [ extop(e,1), extop(e,2) ];
      // now we are sure that we need to coerce:
      if epolyR = theotherpolyR then
         // we know that F::coeffRing <> dom::coeffRing
         //--------------------------------------------------------------
         // Problem: if F::coeffRing does not accept the entries from
         // dom::coeffRing and if polyR(F::coeffRing) = polyR(dom::coeffRing)
         // then we do not notice that no conversion is possible (e.g.,
         // dom::coeffRing = Dom::Integer, F::coeffRing = Dom::Integer.
         // Both have the same polyR = Expr!)
         //--------------------------------------------------------------
         e:= extop(e, 3);
         if F::coeffRing = Dom::ExpressionField() then
              FRcoerce:= Rcoerce;
         else if Rcoerce = id then
                   FRcoerce:= F::coeffRing::coerce;
              else FRcoerce:= F::coeffRing::coerce@Rcoerce;
              end_if;
         end_if;
         if F::coeffRing::characteristic <> FAIL and
            F::coeffRing::characteristic <> 0 then
            // in this case, F uses R2polyCoerce = expr
              e:= map(e, col -> mapcoeffs(col, expr@FRcoerce));
         else // in this case, F uses R2polyCoerce = id
              if FRcoerce <> id then
                 e:= map(e, col -> mapcoeffs(col, FRcoerce));
              end_if;
         end_if;
      else
         if dom::coeffRing::characteristic <> FAIL and
            dom::coeffRing::characteristic <> 0  then
            // Unfortunately, the internal representation IntMod(c)
            // uses _mods, whereas the default representation of
            // Dom::IntegerMod(c) uses _modp. We have to reconvert
            // to the _modp representation before converting to another
            // ring:
            e:= map(extop(e, 3), poly2list);
            e:= map(e, map, c -> [expr(Rcoerce(c[1])), c[2]]):
            if F::coeffRing <> Dom::ExpressionField() then
               FRcoerce:= F::coeffRing::coerce;
               if FRcoerce = FAIL then
                  return(FAIL);
               end_if;
               e:= map(e, map, c -> [FRcoerce(c[1]), c[2]]):
            end_if;
            if F::coeffRing::characteristic <> FAIL and
               F::coeffRing::characteristic <> 0 then
               e:= map(e, map, c -> [expr(c[1]), c[2]]);
            end_if;
            e:= map(e, col -> poly(col, [hold(`#_X_`)], theotherpolyR));
         else
            e:= extop(e, 3);
         end_if;
         if traperror((
            e:= subs(e,epolyR = theotherpolyR); //replace the coeff ring
                      // inside the polynomials. This does the coercion!
            )) <> 0 then
            return(FAIL);
         end_if;
      end_if;
      if has(e, FAIL) then
           return(FAIL) // poly coercion failed
      else return(new(F,r,c,e, "FLAG"));// polys are succesfully changed
      end_if;
  of Dom::SquareMatrix do
      if F::matdim <> e::dom::matdim(e) then return( FAIL ) end_if;
  of Dom::MatrixGroup do
  of Dom::DenseMatrix do
     if traperror((t:= F(expr(e)))) = 0 then
        return(t)
     else
        return(FAIL);
     end_if;
  // return( F(e::dom::convert_to(e, DOM_ARRAY)));
  end_case;
  return( FAIL )
end_proc;

/*-----------------------------------------------------------------------
  matdim  --  return the dimension of a matrix
-----------------------------------------------------------------------*/
matdim:= proc(x) begin [extop(x,1),extop(x,2)] end_proc;

/*-----------------------------------------------------------------------
  _mult  --  multiply matrices

  Multiplies two or more matrices or numbers.
  If an argument is not a matrix then it will be converted to an element
  of R if possible, and scalar multiplication is performed. Otherwise
  the domain of the arument is asked to perform the multiplication
  (which could return FAIL).
-----------------------------------------------------------------------*/

_mult:= if R::hasProp(Ax::systemRep )
        or R::hasProp(Cat::CommutativeRing)
        // in both cases R is a commutative ring, so do not worry about
        // ordering factors in a product. In particular, way may use
        // map(..,_mult,..) to multiply with scalar factors.
   then proc()
      local MatrixArgs, ScalarArgs, i, j, x, y, nargs, dims, mindim, minposition;
   begin
     nargs:= args(0);
     if nargs = 0 then return(1) end_if;
     if nargs = 1 then return(args()); end_if;
     if (i:= contains(map([args()], x -> x::dom::constructor), 
                      Dom::PseudoTensor)) > 0 then
        return((args(i))::dom::_mult(args()));
     end_if;
     // First, get rid of scalar factors. The indices of the scalar arguments
     // are stored in ScalarArgs, indices of matrix arguments are stored in
     // MatrixArgs:
     MatrixArgs:=[];
     ScalarArgs:=[];
     for i from 1 to nargs do
         // consider  Dom::Matrix(R1)(data)*Dom::Matrix(Dom::Matrix(R1))(data).
         // When the _mult method of the first factor is used, we should
         // interpret this as FAIL and use the _mult method of the second
         // factor instead
         if (args(i))::dom::hasProp(Cat::Matrix)=TRUE then
              MatrixArgs:=append(MatrixArgs, i);
         elif type(args(i)) = "_power" and
              (op(args(i), 1))::dom::hasProp(Cat::Matrix)=TRUE then
              // we have f1*f2*..*matrix^n* f.(i+1)* ...,
              // where args(i) = matrix^n is some symbolic matrix
              // power (with symbolic n):
              return(hold(_mult)(  
                                  _mult(args(j)) $ j = 1..i-1,
                                  args(i),
                                  _mult(args(j)) $ j = i+1..args(0)
                    ));
         else ScalarArgs:=append(ScalarArgs, i);
         end_if;
     end_for;

     // collect scalar factors to a single scalar factor
     if nops(ScalarArgs)>0 then
        ScalarArgs:= _mult(args(i) $ i in ScalarArgs);
        if nops(MatrixArgs)=0 then
           return(ScalarArgs);
        end_if;
        nargs:= nops(MatrixArgs);
        if nargs=1 then
           return(dom::_mult1(ScalarArgs,args(MatrixArgs[1])))
        end_if;
     else ScalarArgs:=1;
     end_if;
     // combine matrix factors to 2 factors x and y
     if nargs=2 then x:= args(MatrixArgs[1]);
                     y:= args(MatrixArgs[2]);
     end_if;
     if nargs>2 then
       // Multiplication of more than 2 matrices should include a
       // strategy of minimizing the costs by searching for a good
       // ordering of products of 2 matrices, e.g.,
       // A*B*C = (A*B)*C or A*(B*C) ? Note:
       // cost(AB=A*B)= n1*n2*n3
       // cost(AB*C  )= n1*n3*n4
       // cost(BC=B*C)= n2*n3*n4
       // cost(A*BC  )= n1*n2*n4
       // Consequently:
       // if n1*n3*(n2+n4) <= (n1+n3)*n2*n4
       //    then  implement A*B*C as (A*B)*C
       //    else  implement A*B*C as A*(B*C)
       // end_if;
       // The following code implements a heuristic search strategy:
       // Idea: look for position of matrix with minimal
       // dimension (either rowdim or coldim). Start to
       // multiply this matrix with its neighbours to the
       // left from right to left. This keeps the minimal
       // dimension as the column dimension of all the
       // products in
       //    x := args(1)*(..(.*args(minposition-1)).
       // Then compute
       //    y := ((args(minposition)*..)*..).. * args(nargs)
       // from left to right keeping the minimal dimension
       // as the row dimension of all the products in y.
       // Finally, return x*y.

       // First, search for position of matrix with minimal dimension.
       // Generate list dims = [n1,n2,n3,..,n.k,n.(k+1)], where
       // k = nargs  and
       //     dimension(1st matrix)  = n1 times n2
       //             ...
       //     dimension(last matrix) = n.k times n.(k+1)
       dims:= [(args(MatrixArgs[1]))::dom::matdim(args(MatrixArgs[1]))[1],
               (args(MatrixArgs[i]))::dom::matdim(args(MatrixArgs[i]))[2]
            $ i=1..nargs
              ];
       // search for position of argument with smallest dimensions
       // (either rowdim or coldim)
       mindim:=min(op(dims));
       // Important: prefer to have smallest dimension as far to the
       // right as possible. Note that matrix*column is faster than
       // row*matrix, if the internal matrix representation uses columns
       // as polynomials. For this reason do a downto loop when looking
       // for the position of the minimal dimension. This picks out the
       // rightmost position, if there are several minimal values.
       for minposition from nargs+1 downto 1 do
         if dims[minposition] = mindim then break; end_if;
       end_for;
       // now minposition with 1<= minposition <= nargs+1 is found.
       // Split  args(1)*..*args(nargs) = x * y, where one multiplies
       //     x:= args(1)*..*args(minposition-1)
       // from right to left and
       //     y:= args(minposition)*..*args(nargs)
       // from left to right. Finally, x*y has to be returned.
       if minposition>1 then
          x:= args(MatrixArgs[minposition-1]);
          for i from minposition-2 downto 1 do
             x:= args(MatrixArgs[i])*x;
          end_for;
       end_if;
       if minposition <= nargs then
          y:= args(MatrixArgs[minposition]);
          for i from minposition+1 to nargs do
             y:= y*args(MatrixArgs[i]);
          end_for;
       end_if;
       if minposition = 1 then
          if ScalarArgs = 1
             then return(y);
             else if domtype(y)=dom
                    then return(dom::_mult1(ScalarArgs,y))
                    else // we do not know the domtype of y, so let
                         // the factors take care of themselves:
                         return(ScalarArgs*y)
                  end_if;
          end_if;
       end_if;
       // we know x::dom = dom, so use _mult1 to avoid overhead
       if minposition = nargs+1 then
          if ScalarArgs = 1
             then return(x);
             else return(x::dom::_mult1(ScalarArgs,x))
          end_if;
       end_if;
     end_if; // end of if nargs > 2
     // now we have the scalar factor ScalarArgs and 2 matrices x and y.
     // First, absorb scalar factors into x. Note that x::dom = dom.
     if ScalarArgs<>1 then x:= dom::_mult1(ScalarArgs,x) end_if;
     // Now, we are finally left with the multiplication of 2 matrices
     // x and y, where x::dom = dom.
     dom::_mult2(x,y);
    end_proc;
   else
   //==================================================================
   //        the coefficient ring R is not commutative
   //==================================================================
   proc(A,B)
      local i, j, nargs, k, MatrixArgs, dims, mindim, minposition;
   begin
     nargs:= args(0);
     if nargs = 0 then return(1) end_if;
     if nargs(0) = 1 then return(A) end_if;
     if (i:= contains(map([args()], x -> x::dom::constructor), 
                      Dom::PseudoTensor)) > 0 then
        return((args(i))::dom::_mult(args()));
     end_if;
     if args(0) = 2 then
        if (type(args(1)) = "_power" and
            (op(args(1), 1))::dom::hasProp(Cat::Matrix)=TRUE)
           or
            (type(args(2)) = "_power" and
             (op(args(2), 1))::dom::hasProp(Cat::Matrix)=TRUE)
           then
              // we have matrix^n * factor,
              // where args(1) = matrix^n is some symbolic matrix
              // power (with symbolic n):
              return(hold(_mult)(args(1), args(2)));
        end_if;
     end_if;
     if args(0) > 2 then 
       MatrixArgs:=[0 $ nargs];
       for i from 1 to nargs do
          if (args(i))::dom::hasProp(Cat::Matrix) = TRUE and
             dom::convert(args(i)) <> FAIL then
             MatrixArgs[i]:= i;
          end_if;
          if type(args(i)) = "_power" and
              (op(args(i), 1))::dom::hasProp(Cat::Matrix)=TRUE then
              // we have f1*f2*..*matrix^n* f.(i+1)* ...,
              // where args(i) = matrix^n is some symbolic matrix
              // power (with symbolic n):
              return(hold(_mult)(
                                  _mult(args(j)) $ j = 1..i-1,
                                  args(i),
                                  _mult(args(j)) $ j = i+1..args(0)
                    ));
           end_if;
       end_for;
       MatrixArgs:= map(MatrixArgs, el -> if el = 0 then null() else el end_if);
       // Multiplication of more than 2 matrices should include a
       // strategy of minimizing the costs by searching for a good
       // ordering of products of 2 matrices, e.g.,
       // A*B*C = (A*B)*C or A*(B*C) ? Note:
       // cost(AB=A*B)= n1*n2*n3
       // cost(AB*C  )= n1*n3*n4
       // cost(BC=B*C)= n2*n3*n4
       // cost(A*BC  )= n1*n2*n4
       // Consequently:
       // if n1*n3*(n2+n4) <= (n1+n3)*n2*n4
       //    then  implement A*B*C as (A*B)*C
       //    else  implement A*B*C as A*(B*C)
       // end_if;
       // The following code implements a heuristic search strategy:
       // Idea: look for position of matrix with minimal
       // dimension (either rowdim or coldim). Start to
       // multiply this matrix with its neighbours to the
       // left from right to left. This keeps the minimal
       // dimension as the column dimension of all the
       // products in
       //    x := args(1)*(..(.*args(minposition-1)).
       // Then compute
       //    y := ((args(minposition)*..)*..).. * args(nargs)
       // from left to right keeping the minimal dimension
       // as the row dimension of all the products in y.
       // Finally, return x*y.

       // First, search for position of matrix with minimal dimension.
       // Generate list dims = [n1,n2,n3,..,n.k,n.(k+1)], where
       // k = nargs  and
       //     dimension(1st matrix)  = n1 times n2
       //             ...
       //     dimension(last matrix) = n.k times n.(k+1)

       dims:= [(args(MatrixArgs[1]))::dom::matdim(args(MatrixArgs[1]))[1],
               (args(MatrixArgs[i]))::dom::matdim(args(MatrixArgs[i]))[2]
            $ i=1..nops(MatrixArgs)
              ];
       // search for position of argument with smallest dimensions
       // (either rowdim or coldim)
       mindim:=min(op(dims));
       // Important: prefer to have smallest dimension as far to the
       // right as possible. Note that matrix*column is faster than
       // row*matrix, if the internal matrix representation uses columns
       // as polynomials. For this reason do a downto loop when looking
       // for the position of the minimal dimension. This picks out the
       // rightmost position, if there are several minimal values.
       for minposition from nops(MatrixArgs) + 1 downto 1 do
         if dims[minposition] = mindim then break; end_if;
       end_for;
       if minposition = nops(MatrixArgs) + 1 then
          minposition:= minposition - 1;
       end_if;
       minposition:= MatrixArgs[minposition];

       // now minposition with 1<= minposition <= nargs+1 is found.
       // Split  args(1)*..*args(nargs) = x * y, where one multiplies
       //     x:= args(1)*..*args(minposition-1)
       // from right to left and
       //     y:= args(minposition)*..*args(nargs)
       // from left to right. Finally, x*y has to be returned.
       // end_if; // end of if nargs > 2

       if minposition = 1 then
          A:= args(1);
          for i from 2 to nargs do
             A:= _mult(A, args(i));
          end_for;
          return(A);
       end_if;
       if minposition = nargs + 1 then
          A:= args(nargs);
          for i from nargs-1 downto 1 do
             A:= _mult(args(i), A);
          end_for;
          return(A);
       end_if;
       A:= _mult(args(i) $ i = 1..minposition-1);
       B:= _mult(args(i) $ i = minposition..nargs);
       if A::dom::_mult <> FAIL then
          A:= A::dom::_mult(A, B)
       elif B::dom::_mult <> FAIL then
          A:= B::dom::_mult(A, B)
       else
          A:= _mult(A, B)
       end_if;
       return(A);
     end_if;
     //========================================
     // Here we are in the case: args(0) = 2!!!
     //========================================
     case domtype(A)
     of dom do
        case domtype(B)
        of dom do
           return(A::dom::_mult2(A,B))
        of DOM_INT do
           return(A::dom::_multNC2(A,B))
        of R do
           return(A::dom::_multNC2(A,B))
        otherwise
           if (k:= dom::coerce(B)) <> FAIL then
              return(dom::_mult( A,k ))
           elif (k:= Rcoerce(B)) <> FAIL then
              return(A::dom::_multNC2(A,k)) // extsubsop(x,3=map(extop(x,3),_mult,k)) )
           else
              // Is the domain of y be able to perform the multiplication?
              return( B::dom::_mult(A,B) )
           end_if
        end_case
     of DOM_INT do
        case domtype(B)
        of dom do
           return(B::dom::_multNC1(A,B))
        of DOM_INT do
           return(A*B)
        of R do
           return(B::dom::_mult(A,B))
        end_case
     of R do
        case domtype(B)
        of dom do
           return(B::dom::_multNC1(A,B))
        of DOM_INT do
           return(A::dom::_mult(A,B))
        of R do
           return(B::dom::_mult(A,B))
        end_case
     otherwise
        if (k:= dom::coerce(A)) <> FAIL then
           return(dom::_mult( k,B ))
        elif (k:= Rcoerce(A)) <> FAIL then
           return(B::dom::_multNC1(A,B))
        else
           // unable to perform the multiplication!
           return( FAIL )
        end_if
     end_case;
  end_proc;
end_if;

//--------------------------------------------------------------
// _multNC1 = subroutine called by _mult: multiplication x*y of
// matrix y with scalar factor x  for the non-commutative case
//--------------------------------------------------------------
_multNC1:= proc(x,y)
local c, polyList/*, helpPoly, i*/;
begin
  if domtype(y) <> dom then error("should not happen"); end_if;
  c:= extop(y,2);
  case domtype(x)
  of DOM_INT do // Is x an integer? Note, that R has system representation.
     case x
     of 1 do return( y )
     of 0 do
        c:= extop(y,2);
        y:= extsubsop(y,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     of -1 do return(dom::_negate(y));
     otherwise
        // should be o.k also in non commutative case!
        y:= extsubsop(y,3=map(extop(y,3),mapcoeffs,_mult,x));
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     end_case
  of R do // Is x an element of R ?
     case(x)
     of Rone do return(y)
     of Rzero do
        c:= extop(y,2);
        y:= extsubsop(y,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     otherwise // do not use x::expr, x could be an element of a base domain
       x := R2polyCoerce(x);
       if has(x, FAIL) then
          return(FAIL);
       end_if;
       polyList:= map(extop(y, 3), mapcoeffs, (c, x) -> x*c, x);
       y:= extsubsop(y, 3 = polyList);
       if has(y, FAIL) then return(FAIL) else return(y) end_if;
     end_case;
  otherwise // can x be converted into R ?
     if (x:= Rcoerce(x)) <> FAIL then
       x := R2polyCoerce(x);
       if has(x, FAIL) then
          return(FAIL);
       end_if;
       polyList:= map(extop(y, 3), mapcoeffs, (c, x) -> x*c, x);
       y:= extsubsop(y, 3 = polyList);
       if has(y, FAIL) then return(FAIL) else return(y) end_if;
    else return(FAIL) // unable to perform the multiplication
    end_if;
  end_case;
  return(FAIL);
end_proc;


//--------------------------------------------------------------
// _multNC2 = subroutine called by _mult: multiplication x*y of
// matrix x with scalar factor y  for the non-commutative case
//--------------------------------------------------------------
_multNC2:= proc(x,y)
local c, polyList;
begin
  c:= extop(x,2);
  // if domtype(x) <> dom then error("should not happen"); end_if;
  case domtype(y)
  of DOM_INT do // Is y an integer? Note, that R has system representation.
     case y
     of 1 do return( x )
     of 0 do
        c:= extop(x,2);
        x:= extsubsop(x,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(x, FAIL) then return(FAIL) else return(x) end_if;
     of -1 do return(dom::_negate(x));
     otherwise
       y := R2polyCoerce(y);
       if has(y, FAIL) then
          return(FAIL);
       end_if;
       polyList:= map(extop(x, 3), mapcoeffs, _mult, y);
       x:= extsubsop(x, 3 = polyList);
       if has(x, FAIL) then return(FAIL) else return(x) end_if;
     end_case
  of R do // Is x an element of R ?
     case(y)
     of Rone do return(x)
     of Rzero do
        c:= extop(x,2);
        x:= extsubsop(x,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(x, FAIL) then return(FAIL) else return(x) end_if;
     otherwise // do not use y::expr, y could be an element of a base domain
     if (y:= Rcoerce(y)) <> FAIL then
         // y:= extsubsop(y,3=map(extop(y,3),mapcoeffs,_mult,x) ); // !!!Vorsicht !!!
       y := R2polyCoerce(y);
       if has(y, FAIL) then
          return(FAIL);
       end_if;
       polyList:= map(extop(x, 3), mapcoeffs, _mult, y);
       x:= extsubsop(x, 3 = polyList);
       if has(x, FAIL) then return(FAIL) else return(x) end_if;
     else return(FAIL)
     end_if;
     end_case;
  otherwise // can y be converted into R ?
     if (y:= Rcoerce(y)) <> FAIL then
         y:= R2polyCoerce(y);
         x:= extsubsop(x,3=map(extop(x,3),mapcoeffs,_mult,y) ); // !!!Vorsicht !!!
         if has(x, FAIL) then return(FAIL) else return(x) end_if;
    else return(FAIL) // unable to perform the multiplication
    end_if;
  end_case;
  return(FAIL);
end_proc;

//--------------------------------------------------------------
// _mult1 = subroutine called by _mult: multiplication x*y of
// matrix y with scalar factor x, commutative case
//--------------------------------------------------------------
_mult1:= proc(x,y)
local c;
begin
  // if domtype(y) <> dom then error("should not happen"); end_if;
  case domtype(x)
  of DOM_INT do // Is x an integer? Note, that R has system representation.
     case x
     of 1 do return( y )
     of 0 do
        c:= extop(y,2);
        y:= extsubsop(y,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     of -1 do return(dom::_negate(y));
     otherwise
        x:= R2polyCoerce(x);
        if has(x, FAIL) then
           return(FAIL);
        end_if;
        y:= extsubsop(y,3=map(extop(y,3),mapcoeffs,_mult,x));
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     end_case
  of R do // Is x an element of R ?
     case(x)
     of Rone do return(y)
     of Rzero do
        c:= extop(y,2);
        y:= extsubsop(y,3= [poly(Rzero, [hold(`#_X_`)], polyR) $ c] );
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     otherwise // do not use x::expr, x could be an element of a base domain
        x:= R2polyCoerce(x);
        if has(x, FAIL) then
           return(FAIL);
        end_if;
        y:= extsubsop(y,3=map(extop(y,3),mapcoeffs,_mult,x) ); // !!!Vorsicht !!!
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
     end_case;
  otherwise // can x be converted into R ?
     if (x:= Rcoerce(x)) <> FAIL then
        x:= R2polyCoerce(x);
        if has(x, FAIL) then
           return(FAIL);
        end_if;
        y:= extsubsop(y,3=map(extop(y,3),mapcoeffs,_mult,x) ); // !!!Vorsicht !!!
        if has(y, FAIL) then return(FAIL) else return(y) end_if;
    else return(FAIL) // unable to perform the multiplication
    end_if;
  end_case;
  return(FAIL);
end_proc;

//--------------------------------------------------------------
// _mult2 = subroutine called by _mult: multiplication x*y of
// matrix x with matrix y.
// x is Dom::Matrix(R), y can be of any matrix type and
// will be converted to Dom::Matrix(R).
//--------------------------------------------------------------
_mult2:= proc(A, B)
   local AA, BB, C,m,n,nn,p,j,k,nt,np1,coeffs,exponents,polyListA, polyListB;
    begin // we may assume that A::dom = dom
       //----------------------------------------------------------
       // watch out: if A or B is a matrix over a matrix ring,
       // we may have to interpret one of the factors as a 'scalar'
       //----------------------------------------------------------
       if A::dom::coeffRing::hasProp(Cat::Matrix) and
          (BB:= A::dom::coeffRing::convert(B)) <> FAIL then
          return(A::dom::_mult(A, BB))
       end_if;
       if B::dom::coeffRing::hasProp(Cat::Matrix) and
          (AA:= B::dom::coeffRing::convert(A)) <> FAIL then
          return(B::dom::_mult(AA, B))
       end_if;
       //----------------------------------------------------------
       // end of 'watch out'
       //----------------------------------------------------------
       [m,n] := A::dom::matdim(A);
       [nn,p]:= B::dom::matdim(B);
       if n<>nn then
          // return(FAIL);
          // Andi's Wunsch: Konsistenz zwischen '_plus' und '_mult'.
          //                Wenn die Dimensionen nicht passen, gibt
          //                '_plus' eine Fehlermeldung aus, also auch
          //                '_mult'.
          error ("dimensions do not match")
       end_if;
       if domtype(A) <> domtype(B) then
          C:= dom::coerce(B);
          if C=FAIL then
             return(B::dom::_mult(A, B));
          end_if;
          B:= C;
       end_if;
       /* Ueberschreiben von B durch A*B */
       polyListA:= extop(A, 3):
       polyListB:= extop(B, 3):
       for j from 1 to p do
          nt:= nterms(polyListB[j]);
          if nt <> 0  // nt=0 -> B[j]=zero poly -> A*B[j] = zero poly. Keep it!
          then
               coeffs:= [coeff(polyListB[j])];
               if nt = n // B[j] is dense, no need to figure out exponents via
                         // poly2list (much slower than extracting coeffs via coeff!).
                         // Also, mapcoeffs(p,_mult,a) is faster than multcoeffs(p,a)!
               then np1:= n+1;
                    polyListB[j]:=
                         _plus(mapcoeffs(polyListA[np1-k],_mult,coeffs[k]) $ k=1..nt)
               else exponents:= map(poly2list(polyListB[j]),op,2);
                    polyListB[j]:=
                         _plus(mapcoeffs(polyListA[exponents[k]],_mult,coeffs[k])$k=1..nt);
/* ----------------------
  //  Do not use extop(B, 3), extsubsop(B, 3 = ...)!!
  //  For large and very sparse matrices, this is much slower than
  //  working with lists polyListB etc.
  //      nt:= nterms(extop(B,3)[j]);
  //      if nt <> 0  // nt=0 -> B[j]=zero poly -> A*B[j] = zero poly. Keep it!
  //      then
  //         coeffs:= [coeff(extop(B,3)[j])];
  //         if nt = n // B[j] is dense, no need to figure out exponents via
  //                   // poly2list (much slower than extracting coeffs via coeff!).
  //                   // Also, mapcoeffs(p,_mult,a) is faster than multcoeffs(p,a)!
  //         then np1:= n+1;
  //              B:= extsubsop(B, 3 = subsop(extop(B,3), j =
  //                          _plus(mapcoeffs(extop(A,3)[np1-k],_mult,coeffs[k]) $ k=1..nt)
  //                          ));
  //         else
  //              exponents:= map(poly2list(extop(B,3)[j]),op,2);
  //              B:= extsubsop(B, 3 = subsop(extop(B,3), j =
  //                          _plus(mapcoeffs(extop(A,3)[exponents[k]],_mult,coeffs[k])$k=1..nt)
  //                            ));
---------------- */
             end_if;
          end_if;
       end_for;
       if has(polyListB, FAIL) then
          return(FAIL);
       end_if;
       return(new(dom, m, p, polyListB, "FLAG"));
    end_proc;

/*-----------------------------------------------------------------
   _plus  --  add matrices

     _plus(A)        -> A
     _plus(A+B)      -> A+B
     _plus(A+B+C+..) -> A+B+C+..
-----------------------------------------------------------------*/
    _plus:= proc()
    local r, c, A, B, tmp, i, j, i0, k;
    begin
      if args(0) = 1 then return(args(1)) end_if;
      if (i:= contains(map([args()], x -> x::dom::constructor), 
                       Dom::PseudoTensor)) > 0 then
         return((args(i))::dom::_plus(args()));
      end_if;

      for i from 1 to args(0) do
        if type(args(i)) = "_power" and
             (op(args(i), 1))::dom::hasProp(Cat::Matrix)=TRUE then
             // we have f1*f2*..*matrix^n* f.(i+1)* ...,
             // where args(i) = matrix^n is some symbolic matrix
             // power (with symbolic n):
             return(hold(_plus)(  
                                 _plus(args(j)) $ j = 1..i-1,
                                 args(i),
                                 _plus(args(j)) $ j = i+1..args(0)
                   ));
        elif type(args(i)) = "_mult" and hastype(args(i), "_power") then  
  	  tmp:= op(args(i));					          
          for k from 1 to nops(tmp) do 
	    if type(op(tmp, k)) = "_power" and 
	       (op(op(tmp, k),1))::dom::hasProp(Cat::Matrix)=TRUE then
              return(hold(_plus)(  
                                  _plus(args(j)) $ j = 1..i-1,
                                  args(i),
                                  _plus(args(j)) $ j = i+1..args(0)
                    ));
	    end_if;
	  end_for;
        end_if;
      end_for;

      // search for a term of type dom:
      for i from 1 to args(0) do
        if domtype(args(i)) = dom then
             i0:= i;
             break;
        else next;
        end_if;
      end_for:
      if i > args(0) then
         // Huh? No term of type dom in a sum?
         // Should not happen (because dom::_plus was called)
         return(FAIL);
      end_if;
      A:= args(i0):
      [r, c]:= dom::matdim(A):
      for i from 1 to args(0) do
          if i = i0 then
             next;
          end_if;
          B:= args(i);
          if domtype(B) <> dom and (B:= dom::coerce(B)) = FAIL then
             if (tmp:= R2polyCoerce(Rcoerce(args(i)))) = FAIL then
                  userinfo(1, "operands are not compatible");
                  return(FAIL);
             else // case matrix A + scalar B
//                if r <> c then
//                  return(FAIL);
//                end_if;
                  B:= A::dom::create(r, c, [poly([[tmp, j]], [hold(`#_X_`)], polyR) $ j=1..min(r,c),
                                            poly(0, [hold(`#_X_`)], polyR) $ c - min(r,c)]):
             end_if:
          end_if;
          if dom::matdim(B) <> [r,c] then
            error ("dimensions do not match")
          end_if;
          A:= extsubsop(A, 3= zip(extop(A,3), extop(B,3), _plus));
          if has(A, FAIL) then
             userinfo(1, "operands are not compatible");
             return(FAIL);
          end_if;
      end_for;
      return(A)
    end_proc;

/*-----------------------------------------------------------------------
  _power  --  positive and negative integer matrix powers

  Returns FAIL if
      i is not an integer
      i = 0 and R has no unit
      i<0 and inverse(A) does not exist in the coefficient domain

  Uses Repeated Squaring.
------------------------------------------------------------------------*/
    _power:= proc(A, i)
    local T, t, r, c, polyR, Apower;
    begin
       if args(0) < 2 then
          error("expecting 2 arguments"):
       end_if;

       if domtype(i) <> DOM_INT then
          if domtype(i) = DOM_RAT and denom(i) = 2 then 
            return(_power(linalg::sqrtMatrix(A), numer(i)));
          end_if;
          [r,c]:= A::dom::matdim(A);
          // special case: simplify powers of diagonal matrices
          if r = c then
             if A::dom::nonZeroes(A) > A::dom::nonZeroes(A) then
                // this cannot be a diagonal matrix
                return(hold(_power)(A, i));
             end_if;
             T:= A::dom::convert_to(A, DOM_TABLE);
             for t in T do
                 // t is the equation (k, j) = T[k,j]
                 t:= [op(t, 1)];
                 if t[1] <> t[2] then
                    // this is not a diagonal matrix
                    return(hold(_power)(A, i));
                 end_if;
             end_for;
             return(A::dom::new(r, c, map(T, _power, i)));
          else
             return(hold(_power)(A, i));
          end_if;
       end_if;

       // the special case i = 1 must be treated first, before any
       // error of the type "not a square matrix" can occur
       if i = 1 then return(A): end_if;
       [r,c]:= A::dom::matdim(A);
       if r<>c then error("not a square matrix"); end_if;
       case i
       of 0 do
          if R::one = FAIL then
               userinfo(1,"coefficient domain does not have a unit");
               return( FAIL )
          else polyR :=op(extop(A,3)[1],3); //the poly coeff ring
               return(new(dom,r,r,[poly([[Rone,i]],[hold(`#_X_`)],polyR)$i=1..c], "FLAG"));
          end_if;
       of  1 do return(A);
       of -1 do return(dom::_invert(A));
       otherwise
         if i<0 then A:= dom::_invert(A);
                     if A=FAIL then return(FAIL) end_if;
                     i:= -i;
         end_if;
         /* power ueber Repeated Squaring */
         while TRUE do /* search for first non-trivial bit of i */
           if i=1 then return(A) end_if;
           if i mod 2 = 1 then break; end_if;
           i:= i div 2;
           A:= A*A;
         end_while;
         /* if i = 2^n then A^i was returned above */
         Apower:=A;
         while TRUE do
           Apower:= Apower*Apower;
           i:= i div 2;
           if i mod 2 = 1 then A:= A*Apower; end_if;
           if i=1 then return(A) end_if;
         end_while;
         error("should not arrive here");
      end_case;
    end_proc;


/*---------------------------------------------------------------
  _negate  --  _negate a matrix
---------------------------------------------------------------*/
_negate:= proc(x)
begin
   extsubsop(x, 3=map(extop(x,3), _negate));
end_proc;

/*---------------------------------------------------------------
  _invert  --  compute the inverse of a matrix

  Returns A^(-1) if A is regular, FAIL otherwise.
---------------------------------------------------------------*/
_invert:=

if R = Dom::ExpressionField() then
   linalg::inverse

elif R::hasProp( Cat::IntegralDomain ) and
   not R::hasProp( Cat::Field ) then

   //-------------------------------------------
   // a) build extended scheme  A | identity,
   // b) call gaussElim
   // c) do the backsubstitution
   //-------------------------------------------
   proc(x)
      local L, m, n, i, j, k, R_divide, r, B, diag, charind, list, nonzeroes;
   begin
      m:= extop(x,1);
      n:= extop(x,2);
      nonzeroes:= x::dom::nonZeroes(x);
      if n <> m then
         userinfo(1,"expecting a square matrix");
         return( FAIL )
      end_if;
      // Append the n x n identity matrix to x.
      // Then use fraction free 'gaussElim' to
      // determine the rank (to see, whether the
      // matrix is invertible).
      list:= [poly([[1,i]], [hold(`#_X_`)], polyR) $ i = 1..n];
      x:= extsubsop(x, 3 = _concat(extop(x,3), list));
      x:= extsubsop(x, 2 = 2*n);
      //-----------------------
      // Gaussian Elimination
      //-----------------------
      x:= x::dom::gaussElim(x);
      r:= x[2];
      charind:= x[4];
      x:= extop(x[1],3);
      // r stores the rank of the matrix: if r < n then
      // the input matrix will not be invertible!
      if r < n or {i $ i = 1..n} minus charind <> {} then
         userinfo(1,"matrix is not invertible");
         return( FAIL )
      end_if;
      //===================================================================
      // convert x from polys = columns to polys = rows. Use B as container
      //                      -> the new strategy
      //===================================================================
      // In analogy to transpose:
      // For sparse matrices the version using L:= poly2list(x[j])
      // is faster than the version using coeff(x[i], j).
      // For dense matrices it is the other way round.
      // Regard a matrix as 'reasonably sparse', if at most
      // half its entries are nonzero:
      if nonzeroes < m*n/2 then //sparse case
         // B = container for the rows
         B:= [[[Rzero,0] $ 2*n] $ m];
         for j from 1 to 2*n do
            L:= poly2list(x[j]);
            for list in L do
               // fill the rows
               B[list[2]][j]:= [list[1], 2*n - j];
            end_for;
         end_for;
         // So far, the row B[i] is just a list.
         // Convert the rows to polynomials.
         for i from 1 to m do
           B[i]:= poly(B[i], [hold(`#_X_`)], polyR);
         end_for;
      else //dense case
         B:= [0 $ m]; // initialize container for rows
         // build up row for row to avoid doubling of memory
         // due to doubly nested sequences
         for i from 1 to m do
            B[i] := poly([[coeff(x[j],i), 2*n-j] $j=1..2*n],[hold(`#_X_`)],polyR);
         end_for;
      end_if;
      // Now B = x is a list, but with polys = rows instead of polys = columns
      x:= B;

      R_divide := R::_divide;

      // transform the submatrix x(1..n,1..n) to identity matrix
      // by backsolving

      for k from n downto 1 do
         x[k]:= mapcoeffs(x[k], R_divide, coeff(x[k], 2*n-k));
         if x[k] = FAIL then
            userinfo(1,"computation fails over its coefficient domain");
            return(FAIL);
         end_if;
         for j from k-1 downto 1 do
            diag:= coeff(x[j], 2*n-k);
            x[j]:= x[j] - mapcoeffs(x[k], _mult, diag);
         end_for;
      end_for;

      // Reconversion of polys = rows to polys = cols.
      // Note: We only need to touch the second n x n block
      //       of the matrix storing the inverse matrix!

     nonzeroes:= _plus(nops(poly2list(B[i])) $ i=1..m);
     if nonzeroes < m*n/2 then
       //sparse case
       // B = container for the columns
       B:= [[[Rzero,0] $ m] $ n];
       for i from m downto 1 do
         L:= poly2list(x[i]);
         for list in L do
           // fill the columns
           if list[2] < n then
              // Bug: Hier war die Reihenfolge vertauscht.
              //      B[list[2]+1][i]:= [list[1], i];
              B[n-list[2]][i]:= [list[1], i];
           end_if;
         end_for;
       end_for;
       // So far, the row B[j] is just a list.
       // Convert the columns to polynomials.
       for j from 1 to n do
         B[j]:= poly(B[j], [hold(`#_X_`)], polyR);
       end_for;
     else //dense case
       B:= [0 $ n]; // initialize container for columns
       // build up column for column to avoid doubling of memory
       // due to doubly nested sequences
       for j from 1 to n do
           B[j]:= poly([[coeff(x[i], n-j), i] $i=1..m],[hold(`#_X_`)],polyR);
           if has(B[j],FAIL) then error("should not happen (5)") end_if;
       end_for;
     end_if;
     return(new(dom, n, n, B, "FLAG"));
   end_proc;

 elif R::hasProp( Cat::Field ) then
   //----------------------------------------------
   // a) floatmode and DIGITS < 16:
   //   -->  call numeric::inverse (which uses HardwareFloats )
   // b) not floatmode or DIGITS > 15:
   //   -->  direct implementation of Gauss-Jordan
   //----------------------------------------------
  proc(A)
  local floatmode, types, row, m, n, B, L, list, i, j, nt, minnt,
        usenormal, nonormalTypes, nonzeroes,
        jthcol, pos, piv, pivpos, pivindex, pivfound,
        pivotsize, thissize, rownorm, minlength, thislength,
        term, terms, l;
  begin
     userinfo(1, "using Gaussian elimination (LR decomposition)"):
     [m, n]:= dom::matdim(A);
     if m <> n then error("not a square matrix"); end_if:

     //-------------------------------
     // check for floats
     //-------------------------------
     // The following seems efficient. Some timings (without HardwareFloats)
     // n:=50:
     // a:=/*sparse*/matrix(linalg::hilbert(n)):
     // a[n, n]:= 1.0 * I;time(1/a)
     //      3720 msec (davon 50 msec fuer float check)
     // n:= 200:
     // a:= /*sparse*/matrix(n, n, [-1,2,-1],Banded):
     // a[n, n]:= 2.0*I;
     // time(1/a);
     //      2090 msec (davon 20 msec fuer float check)
     //-------------------------------
     floatmode:= FALSE;
     if R = Dom::Float then
        floatmode:= TRUE;
        if DIGITS <= 15 then
           // all elements of A are floats.
           // Call numeric::inverse, which tries HardwareFloats
           // for DIGITS <= 15;
           B:= numeric::inverse(A, ReturnType = Dom::Matrix());
           if not has(B, FAIL) then
              return(Dom::Matrix(R)(B));
           // else proceed to the MuPAD code below
           end_if;
        end_if;
     elif polyR = Expr then
       //---------------------------------------------------
       // is there a float element in the matrix?
       //---------------------------------------------------
       for i from 1 to n do
         row:= {coeff(extop(A, 3)[i])};
         types:= map(row, domtype):
         if has(types, DOM_FLOAT) then
            floatmode := TRUE;
            break;
         end_if;
         row:= select(row, c -> domtype(c) = DOM_COMPLEX);
         types:= map(row, domtype@op, 2); // domtypes of imaginary parts
         if has(types, DOM_FLOAT) then
            floatmode := TRUE;
            break;
         end_if;
       end_for;
       //---------------------------------------------------
       // There is a float element in the matrix. Try to convert
       // all elements to floats. We need a copy B of A, because
       // we still need the original A if no float conversion is
       // possible.
       //---------------------------------------------------
       if floatmode then
          B:= extop(A, 3);
          for i from 1 to n do
            B[i]:= mapcoeffs(B[i], float):
            types:= map({coeff(B[i])}, domtype):
            if types minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
               floatmode:= FALSE;
               break;
            end_if;
          end_for;
       end_if;
       //---------------------------------------------------
       // if floatmode, all elements of A were converted to floats.
       // Call numeric::inverse, which tries HardwareFloats!
       // If DIGITS > 15, numeric::inverse will not try HardwareFloats.
       // Use the generic sparse code below instead of numeric::inverse
       //---------------------------------------------------
       if floatmode and DIGITS <= 15 then
          B:= numeric::inverse(A, ReturnType = Dom::Matrix());
          if not has(B, FAIL) then
             if R = Dom::ExpressionField() then
                  return(B);
             else return(Dom::Matrix(R)(B));
             end_if;
          end_if;
       end_if;
     end_if;

     //------------------------------------------------------------------
     // start of generic code for inverting a (sparse) matrix over any
     // ring of Cat::Field:
     //------------------------------------------------------------------

     //-----------------------------------------------
     // convert A from polys = columns to polys = rows.
     // Use B as container
     //-----------------------------------------------
     // In analogy to transpose:
     // For sparse matrices the version using L:= poly2list(x[j])
     // is faster than the version using coeff(x[i], j).
     // For dense matrices it is the other way round.
     // Regard a matrix as 'reasonably sparse', if at most
     // half its entries are nonzero:
     if A::dom::nonZeroes(A) < m*n/2
     then //sparse case
       A := extop(A,3);
       // B = container for the rows
       B:= [[[Rzero,0] $ n] $ m];
       for j from 1 to n do
         L:= poly2list(A[j]);
         for list in L do
           // fill the rows
           B[list[2]][j]:= [list[1], n - j];
         end_for;
       end_for;
       // So far, the row B[i] is just a list.
       // Convert the rows to polynomials.
       for i from 1 to m do
         B[i]:= poly(B[i], [hold(`#_X_`)], polyR);
       end_for;
     else //dense case
       A := extop(A,3);
       B:= [0 $ m]; // initialize container for rows
       // build up row for row to avoid doubling of memory
       // due to doubly nested sequences
       for i from 1 to m do
           B[i] := poly([[coeff(A[j],i), n-j] $j=1..n],[hold(`#_X_`)],polyR);
       end_for;
     end_if;
     // Now B = A but with polys = rows instead of polys = columns
     A:= B;

     if has(A, FAIL) then error("should not happen (1)") end_if;
     if R::one = FAIL then error("should not happen (2)") end_if;

     // There is no need to use normal if only numerical
     // types are involved:
     usenormal:= FALSE;
     nonormalTypes:= {DOM_FLOAT, DOM_COMPLEX,
                      DOM_INT, DOM_RAT, DOM_INTERVAL};
     if R::normal <> FAIL and not floatmode then
        for i from 1 to m do
          types:= map({coeff(A[i])}, domtype):
          if types minus nonormalTypes <> {} then
             usenormal:= TRUE;
             break;
          end_if;
        end_for;
     end_if;

     // initialize B as identity matrix, interpretation: polys=rows
     B:= [poly([[Rone,j]], [hold(`#_X_`)], polyR) $ j=1..n];
     if has(B, FAIL) then error("should not happen (3)") end_if;

     piv:= [ 0 $ n]; // initialize container for pivot elements
     for j from 1 to n do
         if usenormal = TRUE then
            for i from j to n do
              A[i]:= mapcoeffs(A[i], Rnormal);
              B[i]:= mapcoeffs(B[i], Rnormal);
            end_for;
         end_if;

         nt:=n-j;
         // j.th col of A from diagonal downwards. Only store the
         // non-trivial terms in jthcol. First generate a dense list.
         // Eliminate trivial entries by converting to a polynomial,
         // then reconvert to list by poly2list. This is faster
         // than sparseList:= select(denseList , x -> not iszero(x[1])):

         userinfo(2, "searching for pivot element in column ".expr2text(j));

         jthcol:= poly([[coeff(A[i],nt), i] $ i=j..n ], [hold(`#_X_`)], polyR);
         jthcol:= poly2list(jthcol);
         if jthcol=[] then
         userinfo(1,"matrix not invertible");
         return(FAIL);
         end_if;
         // start pivot search
         pivpos:= 1;         // position of pivot element in list jthcol
         pivfound:= FALSE;   // pivot element not found yet.
         minnt:= nt+1;       // minimal number of terms in row. The row
                             // with the smallest number of terms is used
                             // as pivot row in order to preserve sparsity
         if floatmode then
            //-------------------------------
            // floating point pivot strategy
            //-------------------------------
            pivotsize:= 0; // size of potential pivot elements in jthrow.
                           // Should be maximal (this is the only
                           // criterion for the floatmode pivot search).
            for pos from 1 to nops(jthcol) do // just search the non-trivial terms
               i:= jthcol[pos][2]: //row index i corresponding to
                                   //pos-th term in jthcol
               // cut diagonal element from i-th row
               A[i]:= lmonomial(A[i],Rem)[2];
               // A[i] is now remainder of row i without j.th element
               // j.th elements are stored in jthcol

               if not pivfound then
                  nt:= nterms(A[i]); //number of non-trivial terms in
                                     // remainder of row i
                  if nt=0 then // ideal pivot element!!
                      pivfound:= TRUE;
                      pivpos:= pos;
                      break;
                  end_if;
                  rownorm:= norm(A[i]);
                  if iszero(rownorm) then
                      // should have been found before. (?)
                      // Still, this is an ideal pivot element
                      pivfound:= TRUE;
                      pivpos:= pos;
                      break;
                   else
                     thissize:= abs(jthcol[pos][1]/rownorm);
                     if thissize > pivotsize then
                        pivpos:= pos;  // new candidate for pivot element
                        pivotsize:= thissize;
                     end_if;
                  end_if;
               end_if;
            end_for;
         else
            //-------------------------------
            // symbolic pivot strategy
            //-------------------------------
            minlength:= 0;      // complexity of potential pivot elements in jthrow.
                                // Should be minimal (this is the secondary
                                // criterion for the pivot search).
            for pos from 1 to nops(jthcol) do // just search the non-trivial terms
               i:= jthcol[pos][2]: //row index i corresponding to
                                   //pos-th term in jthcol
               // cut diagonal element from i-th row
               A[i]:= lmonomial(A[i],Rem)[2];
               // A[i] is now remainder of row i without j.th element
               // j.th elements are stored in jthcol
               if not pivfound then
                  nt:= nterms(A[i]); //number of non-trivial terms in
                                     // remainder of row i
                  if nt=0 then pivfound:= TRUE; end_if; // ideal pivot element!!
                  //pivot criterion nt<minnt is good strategy for non-float
                  //computations.
                 if nt<minnt then // primary pivot criterion
                    pivpos:= pos; minnt:= nt;
                 else // secondary pivot criterion
                      thislength:= length(jthcol[pos][1]);
                      // avoid initialization of minlength by infinity,
                      // this would be too expensive.
                     if minlength=0 then minlength:= thislength; end_if;
                     if nt=minnt and thislength<minlength then
                         pivpos:= pos; minlength:= thislength;
                     end_if;
                 end_if;
               end_if;
            end_for;
         end_if;
         // jth pivot element found, store in table piv
         piv[j]:=   jthcol[pivpos][1];
         pivindex:= jthcol[pivpos][2];
         userinfo(3, "choosing pivot element ".expr2text(Rcoerce(piv[j])).
                     " (row ".expr2text(pivindex).")"):
         // do the elimination
         for term in jthcol do
           i:= term[2];
           if i<>pivindex then
             l:= term[1]/piv[j];
             if l=FAIL then
                error("should not happen (".expr2text(3,j).")");
             end_if;
             case l
             of R::one do
                       A[i]:=A[i]-A[pivindex];
                       B[i]:=B[i]-B[pivindex];
                       break;
             of R::negate(R::one) do
                       A[i]:=A[i]+A[pivindex];
                       B[i]:=B[i]+B[pivindex];
                       break;
             otherwise /* -----------------------------------------------
                       // fast but not correct for non-commutative rings:
                       A[i]:= A[i]-mapcoeffs(A[pivindex],_mult,l);
                       B[i]:= B[i]-mapcoeffs(B[pivindex],_mult,l);
                       ------------------------------------------------*/
                       A[i]:= A[i]-mapcoeffs(A[pivindex], (x, y) -> y*x, l);
                       B[i]:= B[i]-mapcoeffs(B[pivindex], (x, y) -> y*x, l);
             end_case;
           end_if;
         end_for; // elimination done

         if pivindex<>j then // exchange rows
            [A[j],A[pivindex]]:= [A[pivindex],A[j]];
            [B[j],B[pivindex]]:= [B[pivindex],B[j]];
         end_if;
     end_for;
     // ----------- elimination finished ----------------------
     // Now A is in upper triangular form with diagonal elements
     // piv[1],..,piv[n] and strictly upper triangular rows given
     // by the polynomials A[1],..A[n-1], A[n] = poly(0,x) :
     // All piv[i]<>0, otherwise FAIL would have been returned above.
     // Start backsubstitution:
     // ----------------------------------------------------------
     // fast, but not correct for non-commutative rings:
     // B[n]:= mapcoeffs(B[n],_divide,piv[n]);
      //----------------------------------------------------------
     B[n]:= mapcoeffs(B[n], (x, y) -> y*x, 1/piv[n]);
     if B[n]=FAIL then error("should not happen (4)") end_if;
     for i from n-1 downto 1 do
         terms:= poly2list(A[i]);
      //---------------------------------------------------------------------
      // fast, but not correct for non-commutative rings:
      // B[i]:= B[i] -
      //       _plus(mapcoeffs(B[n-term[2]],_mult,term[1]) $ term in terms );
      // B[i]:= mapcoeffs(B[i], _divide, piv[i]);
      //---------------------------------------------------------------------
         B[i]:= B[i] -
             _plus(mapcoeffs(B[n-term[2]], (x, y) -> y*x, term[1]) $ term in terms );
         B[i]:= mapcoeffs(B[i], (x, y) -> y*x, 1/piv[i]);
         if B[i]=FAIL then error("should not happen (4)") end_if;
         if usenormal = TRUE then
              B[i]:= mapcoeffs(B[i], Rnormal);
           end_if;
     end_for;
     // Now B stores the inverse of A with polys = rows.

     //-----------------------------------------
     // Still need to reconvert to polys = cols.
     //-----------------------------------------

     nonzeroes:= _plus(nops(poly2list(B[i])) $ i=1..m);
     if nonzeroes < m*n/2 then //sparse case
        // A = container for the columns
        A:= [[[Rzero,0] $ m] $ n];
        for i from 1 to m do
           L:= poly2list(B[i]);
           for list in L do
              // fill the columns
              A[list[2]][i]:= [list[1], i];
           end_for;
        end_for;
        // So far, the row A[j] is just a list.
        // Convert the columns to polynomials.
        for j from 1 to n do
           A[j]:= poly(A[j], [hold(`#_X_`)], polyR);
        end_for;
     else //dense case
        A:= [0 $ n]; // initialize container for columns
        // build up column for column to avoid doubling of memory
        // due to doubly nested sequences
        for j from 1 to n do
           A[j] := poly([[coeff(B[i], j), i] $i=1..m],[hold(`#_X_`)],polyR);
           if has(A[j],FAIL) then
              error("should not happen (5)")
           end_if;
       end_for;
     end_if;
     return(new(dom, m, n, A, "FLAG"));
  end_proc;

  elif R::hasProp( Cat::Ring ) then

    //------------------------------------------------
    // a special fast routine for modular inverses
    // over Dom::IntegerMod(N). Here, N is not a prime.
    //------------------------------------------------
    if R::constructor = Dom::IntegerMod then
       Dom::Utilities::IntModInverse
    else
       //------------------------------------------------
       // Last resort:  Try to determine the inverse matrix
       // via the formula
       //
       //          det(x)^(-1) * adjoint(x)
       //
       // Beware: this is terribly slow!
       //------------------------------------------------

       // Try to determine the inverse matrix via the formula
       //
       //          det(x)^(-1) * adjoint(x)
       proc(x)
         local d;
       begin
         // STRATEGY:
         // Compute the inverse of the determinant first.
         // If d = FAIL, the matrix is not invertible and
         // there is no need to compute the adjoint matrix.

         d:= linalg::det(x)^(-1):

         if d = FAIL then
           return(FAIL);
         else
           return( d * linalg::adjoint(x) )
         end_if
      end_proc
     end_if;
  else
    FAIL
 end_if;

/*----------------------------------------------------------------------
  _divide  --  compute a/b for a matrix a.
----------------------------------------------------------------------*/
    _divide:= proc(a,b)
    begin
      b:= 1/b;
      if b <> FAIL
        then return( dom::_mult( a,b ) )
        else userinfo(1,"inverse of 2nd matrix cannot be computed");
             return( FAIL )
      end_if
    end_proc;

/*----------------------------------------------------------------------
  iszero  --  test if a matrix is the zero matrix
----------------------------------------------------------------------*/
    iszero:= proc(x)
    begin
       if iszero(extop(x, 1)) or
          iszero(extop(x, 2)) then
          return(TRUE)
       else
          return(bool({op(map(extop(x,3),iszero))} = {TRUE}));
       end_if;
    end_proc;

/*----------------------------------------------------------------------
  testeq  --  overloads the function 'testeq' for matrices
----------------------------------------------------------------------*/
    testeq:= proc(x,y)
      local d;
    begin
        if x::dom <> dom then error("should not happen"); end_if;
        if domtype(x) <> domtype(y) then
              y:= x::dom::coerce(y);
              if y = FAIL then return(FALSE) end_if;
              // do not try x:= y::dom::coerce(x), because
              // y::dom may lead to an error message (e.g., if y = a number)
        end_if;
        if extop(x,1) <> extop(y,1) then return(FALSE) end_if;
        if extop(x,2) <> extop(y,2) then return(FALSE) end_if;
        d:= map({op(x - y)}, elem -> testeq(elem, Rzero));
        if contains(d, FALSE) then return(FALSE) end_if;
        if contains(d, UNKNOWN) then return(UNKNOWN) end_if;
        return(TRUE);
    end_proc;

/*----------------------------------------------------------------------
  equal  --  test if two matrices are equal
----------------------------------------------------------------------*/
    equal:= proc(x,y)
    begin
        if x::dom <> dom then error("should not happen"); end_if;
        if domtype(x) <> domtype(y) then
              y:= x::dom::coerce(y);
              if y = FAIL then return(FALSE) end_if;
              // do not try x:= y::dom::coerce(x), because
              // y::dom may lead to an error message (e.g., if y = a number)
        end_if;
        if extop(x,1) <> extop(y,1) then return(FALSE) end_if;
        if extop(x,2) <> extop(y,2) then return(FALSE) end_if;
        if extop(x,3) =  extop(y,3) then return(TRUE); end_if;
        return(FALSE)
    end_proc;

/*----------------------------------------------------------------------
  gaussElim  --  Gaussian elimination

  Syntax:

  gaussElim(A)
  gaussElim(A, ColumnElimination)

  A  -- a (sparse) matrix

  Synopsis:

  Performs Gaussian elimination on A and returns a list with
   - a row echolon form of A,
   - the rank of A,
   - the determinant of A if it is defined
     (otherwise FAIL will be the third
     operand of the list)
   - a set of indices representing the column vectors
     that span the image of A.

*)With the option 'ColumnElimination', a lower triangular
  echelon form of A is returned that is obtained by eliminating
  columns (instead of rows) in the Gaussian elimination.
  This lower triangular echelon form is the transpose
  of gaussElim(transpose(A))[1].
  For matrices with number_of_rows >> number_of_columns,
  gaussElim(A, ColumnElimination) should be (much) faster
  than gaussElim without ColumnElimination.

*)If R has a method "pivotSize", the pivot element of
  smallest size will be choosen, where pivotSize must return
  a positive real number representing 'the size' of an element.
---------------------------------------------------------------------*/
gaussElim:= if R::hasProp( Cat::Field ) and
               R::hasProp( Ax::efficientOperation("_invert") )
then
      // -----------------------------
      // ordinary Gaussian elimination
      // -----------------------------
      proc(x)
          local columnElimination,floatmode, row, types, rownorm,
                thissize,
                det, m, n, i, j, l, k, t, p, ps, detsign, charind,
                B, nn, kk, term, pivot, oldpivot, sizeOfPivot,
                ithcol, nt, pivRow, tmp, usenormal, nonormalTypes,
                nonzeroes, L, list, count, neqs, zeroTest;
      begin
         //===================================================================
         /* Zero test heuristic for matrices with floatable entries:
           ===================================================================

            For matrices whose components can be converted to floating point
            enrties, we will use a special heuristic a zero test instead of
            only using 'isnonzero'. This is the case of args(2) = `#FloatCheck`.
         */
         //===================================================================
         if args(0) = 2 and args(2) = `#FloatCheck` then
           zeroTest:= x -> if iszero(x) = TRUE then
                             TRUE;
                           elif numeric::isnonzero(x) <> TRUE then
                             TRUE;
                           else
                             FALSE;
                           end_if:
         else
           zeroTest:= iszero;
         end_if:
         //===================================================================
         columnElimination:= has([args()], ColumnElimination);
         nonzeroes:= x::dom::nonZeroes(x);
         m:= extop(x,1); // number of rows
         n:= extop(x,2); // number of columns
         //===================================================================
         // convert x from polys = columns to polys = rows. Use B as container
         //===================================================================
         if not columnElimination then
              B:= extop(x::dom::transpose(x), 3):
         else B:= extop(x, 3):
              [m, n]:= [n, m]:
         end_if;
         //=========================================================
         // convert rows from sum(B.i.j*`#_X_`^j) to sum(B.i.j*`#_X_`^(n-j))
         //=========================================================
         for i from 1 to m do
           // sparse
           B[i]:= poly(map(poly2list(B[i]), x -> [x[1], n-x[2]]), [hold(`#_X_`)], polyR);
           // dense
           //  B[i]:= poly([[coeff(B[i],j), n-j] $j=1..n],[hold(`#_X_`)],polyR);
         end_for;

         //-------------------------------
         // check for floats
         //-------------------------------
         floatmode:= FALSE;
         if R = Dom::Float then
            floatmode:= TRUE;
         elif polyR = Expr then
            //---------------------------------------------------
            // is there a float element in the matrix?
            //---------------------------------------------------
            for i from 1 to m do
              row:= {coeff(B[i])};
              types:= map(row, domtype):
              if has(types, DOM_FLOAT) then
                 floatmode := TRUE;
                 break;
              end_if;
              row:= select(row, c -> domtype(c) = DOM_COMPLEX);
              types:= map(row, domtype@op, 2); // domtypes of imaginary parts
              if has(types, DOM_FLOAT) then
                 floatmode := TRUE;
                 break;
              end_if;
            end_for;
            //---------------------------------------------------
            // There is a float element in the matrix. Try to convert
            // all elements to floats. We need a copy x of B, because
            // we still need the original A if no float conversion is
            // possible.
            //---------------------------------------------------
            if floatmode then
              x:= B;
              for i from 1 to m do
                B[i]:= mapcoeffs(B[i], float):
                types:= map({coeff(B[i])}, domtype):
                if types minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
                   floatmode:= FALSE;
                   B:= x;
                   break;
                end_if;
              end_for;
           end_if;
         end_if;
         //----------------------------------------------
         // end of float check --> floatmode = TRUE/FALSE
         //----------------------------------------------

         usenormal:= FALSE;
         // There is no need to use normal if only numerical
         // types are involved:
         nonormalTypes:= {DOM_FLOAT, DOM_COMPLEX,
                          DOM_INT, DOM_RAT, DOM_INTERVAL};
         if R::normal <> FAIL and not floatmode then
            for i from 1 to m do
              types:= map({coeff(B[i])}, domtype):
              if types minus nonormalTypes <> {} then
                 usenormal:= TRUE;
                 break;
              end_if;
            end_for;
         end_if;

         //----------------------------------------------------
         // Row compression: move all trivial equations to the
         // bottom of the matrix. Then, temporarily, put
         // m = number of non-trivial rows for the elimination,
         //----------------------------------------------------
         count:= 0;
         neqs := m;
         for i from 1 to neqs do
             if iszero(B[i]) then // note that B[i] is a polynomial and
                                  // hence zeroTest cannot be applied to
                                  // it
                m:= m - 1;
             else
                count:= count + 1;
                B[count]:= B[i];
             end_if;
         end_for;
         // fill the bottom of the matrix with trivial rows
         for i from m + 1 to neqs do
             B[i]:= poly(Rzero, [hold(`#_X_`)], polyR);
         end_for;

         //--------------------------------------------------
         // start elimination
         //--------------------------------------------------

         x:= B;
         ps:= R::pivotSize; // if ps = FAIL then no pivot strategy will be used, otherwise
                            // the pivot in respect to < will be choosen
         if has([args()], DiagonalStrategy) then
            // DiagonalStragey is an undocumented Option.
            // Only used by Series::expr2diffeq
            ps:= FAIL;
         end_if;
         userinfo(1,"perform (ordinary) Gaussian elimination");
         j:= 1; // j = the row to be used for the elimination step
         detsign:= 1;  // sign of the determinant
         det:= R::one; // the determinant
         charind:= {}; // the characteristic column indices

         for i from 1 to n do
             // we are in the i-th column and eliminate all elements
             // below the 'pivot' element (j, i)

             if usenormal = TRUE then
                for k from j to m do
                  x[k]:= mapcoeffs(x[k], Rnormal);
                end_for;
             end_if;

             if j = m then
                 // j is 'at the bottom', there is no element below (j, i).
                 // --> no need for elimination. However, we still need
                 // to find characteristic elements to the right of (j, i).
                 // Search for the first non-trivial element (j, k) with
                 // j fixed and k = i, .. , n.
                 k:= i;
                 if usenormal = TRUE then
                    while k <= n do
                       if not zeroTest(Rnormal(coeff(x[j],n-k))) then
                          charind:= charind union {k};
                          break
                       else
                          k:= k + 1
                       end_if
                    end_while;
                 else // usenormal = FALSE -> same code, but without any normal
                    while k <= n do
                       if not zeroTest(coeff(x[j],n-k)) then
                          charind:= charind union {k};
                          break
                       else
                          k:= k + 1
                       end_if
                    end_while;
                 end_if;
                 break; // exit the 'for i from 1 to n' loop
             end_if;

             //-----------------------------------------
             // Convert the i-th column to a sparse list
             //-----------------------------------------

             // consider i.th col of x from element j downwards. Only store the
             // non-trivial terms in ithcol. First generate a dense list.
             // Eliminate trivial entries by converting to a polynomial,
             // then reconvert to list by poly2list. This is faster
             // than sparseList:= select(denseList , x -> not iszero(x[1])):
             nt:=n - i; // exponent in the row polys corresponding to column i
             ithcol:= poly([[coeff(x[k], nt), k] $ k=j..m ], [hold(`#_X_`)], polyR);
             ithcol:= poly2list(ithcol);

             //-----------------------------
             // Search for the pivot element
             //-----------------------------
             userinfo(3,"search for pivot in column ",i);
             p:= 0; // the row index of the pivot element
             oldpivot:= 0:
             sizeOfPivot:= -1: // the size of the pivot element

             nn:= nops(ithcol);
             for kk from 0 to nn - 1 do
                 // Note that the first element in ithcol is the
                 // *last* non-trivial element: run *backward*
                 // through ithcol. This way, we search downwards,
                 // i.e., we guarantee the same pivoting strategy
                 // used by Dom::DenseMatrix:
                 if p <> 0 then
                    // a pivot candidate was found before.
                    // Store it in case the next pivot candidate
                    // is no good.
                    oldpivot:= pivot;
                 end_if;
                 term:= ithcol[nn - kk];
                 pivot:= term[1]; // candidate for the pivot element
                 k:= term[2]: //row index k corresponding to
                              //this term in ithcol
                 // cut first element from the k-th row
                 row:= lmonomial(x[k],Rem)[2];
                 nt:= nterms(row); //number of non-trivial terms in
                                   // remainder of row k
                 if usenormal = TRUE then
                    pivot:= Rnormal(pivot);
                 end_if;
                 if zeroTest(pivot) then
                    pivot:= oldpivot;
                    next;
                 end_if;
                 if nt = 0 then // ideal pivot element
                    p:= k;
                    break;
                 end_if;
                if floatmode then
                   rownorm:= norm(row);
                   if iszero(rownorm) then // ideal pivot element!!
                      p:= k;
                      break;
                   else
                     thissize:= specfunc::abs(pivot/rownorm);
                     if thissize > sizeOfPivot then
                        p:= k;  // new candidate for pivot element
                        sizeOfPivot:= thissize;
                     else
                        pivot:= oldpivot;
                     end_if;
                   end_if;
                else // symbolic pivot strategy
                   if ps = FAIL then
                      p:= k;  // the current pivot element is in row k
                      break;  // break the loop 'for term in ithcol'
                   elif sizeOfPivot = -1 then
                      //initialize with the first inspected candidate
                      sizeOfPivot:= ps(pivot);
                      p:= k;
                   elif (t:= ps(pivot)) < sizeOfPivot then
                      sizeOfPivot:= t;
                      p:= k;
                   else
                      pivot:= oldpivot;
                   end_if
                end_if;
             end_for;

             if p = 0 then
                userinfo(3,"no pivot found");
                det:= R::zero;
                next; // proceed to the next i
                      // without incrementing j
             end_if;

             //----------------------------
             // The pivot element is found:
             // it is the element (p, i)
             //----------------------------

             if m = n and det <> R::zero then
                det:= det * R::coerce(pivot);
                if usenormal = TRUE then
                   det:= Rnormal(det);
                end_if;
             end_if;

             charind:= charind union {i};

             // do the elimination below the element (j, i):

             // cut diagonal element from the row with the pivot element
             pivRow := lmonomial(x[p],Rem)[2];

             for term in ithcol do
                 k:= term[2]; // the index of the row to be eliminated
                 if k <> p then
                    // cut diagonal element from the row with the pivot element
                    // x[k] is now the remainder of row k without i.th element
                    x[k] := lmonomial(x[k],Rem)[2];
                    l:= term[1]/pivot;
                    //if usenormal then
                    //   l:= Rnormal(l);
                    //end_if;
                    case l
                    of  Rone do x[k]:=x[k] - pivRow;
                                break;
                    of -Rone do x[k]:=x[k] + pivRow;
                                break;
                    otherwise x[k]:= x[k]-mapcoeffs(pivRow,_mult,l);
                    end_case;
                    //if usenormal then
                    //   x[k]:= mapcoeffs(x[k], Rnormal);
                    //end_if;
                 else
                    // k = p: we are in the pivot row.
                    // No need to eliminate, but we need
                    // to normalize
                    // if usenormal and normal <> id then
                    //   x[k]:= mapcoeffs(x[k], Rnormal);
                    // end_if;
                 end_if;
             end_for;   // elimination done

             // swap rows: apart from row p, all other rows
             // were truncated and do not contain the i-th
             // element any more
             if p <> j then
                 userinfo(3,"swap rows: ",p, j);
                 [x[j], x[p]] := [x[p], x[j]];
                 detsign:= - detsign
             end_if;

             j:= j + 1  // proceed to next column i, update row j -> j+1
         end_for;
         //------------------
         // End of Elimination
         //------------------

         // 'activate' the trivial rows at the bottom
         // of the matrix by restoring m = number of columns
         m:= neqs;

         //-----------------------------------------
         // Still need to reconvert to polys = cols.
         //-----------------------------------------
         //=========================================================
         // convert rows from sum(x.i.j*`#_X_`^(n-j)) to sum(x.i.j*`#_X_`^j)
         //=========================================================
         for i from 1 to m do
           // sparse
           x[i]:= poly(map(poly2list(x[i]), x -> [x[1], n-x[2]]), [hold(`#_X_`)], polyR);
           // dense
           //  x[i]:= poly([[coeff(x[i],j), n-j] $j=1..n],[hold(`#_X_`)],polyR);
         end_for;

         //=========================================================
         // convert row representation to column representation
         //=========================================================
         if columnElimination then
            tmp:= x;
            [m, n]:= [n, m]:
         else
           nonzeroes:= _plus(nops(poly2list(x[i])) $ i=1..m);
           if nonzeroes < m*n/2 then //sparse case
              // tmp = container for the columns
              tmp:= [[[Rzero,0] $ m] $ n];
              for i from 1 to m do
                 L:= poly2list(x[i]);
                 for list in L do
                    // fill the columns
                    tmp[list[2]][i]:= [list[1], i];
                 end_for;
              end_for;
              // So far, the row tmp[j] is just a list.
              // Convert the columns to polynomials.
              B:= tmp;
              for j from 1 to n do
                 tmp[j]:= poly(B[j], [hold(`#_X_`)], polyR);
              end_for;
           else //dense case
              tmp:= [0 $ n]; // initialize container for columns
              // build up column for column to avoid doubling of memory
              // due to doubly nested sequences
              for j from 1 to n do
                 tmp[j] := poly([[coeff(x[i], j), i] $i=1..m],[hold(`#_X_`)],polyR);
                 if has(tmp[j],FAIL) then
                    error("should not happen (5)")
                 end_if;
              end_for;
           end_if;
         end_if:

         if m = n then
         //=======================================================
         // Nicolas Thiery (4/12/2001)
         // added condition n>0 to deal with 0xn and nx0 matrices.
         //=======================================================
              if det <> R::zero and usenormal = TRUE and n>0 then
                 det:= Rnormal( det * R::coerce(coeff(x[n],n)) );
              end_if;
              if det <> R::zero and usenormal = FALSE and n>0 then
                 det:= det * R::coerce(coeff(x[n],n));
              end_if;
              if detsign = -1 then
                  return([new(dom,m,n,tmp, "FLAG"), nops(charind), -det, charind]);
             else return([new(dom,m,n,tmp, "FLAG"), nops(charind), det, charind]);
             end_if
         else return([new(dom,m,n,tmp, "FLAG"), nops(charind), FAIL, charind]);
         end_if
     end_proc
elif R::hasProp( Cat::IntegralDomain ) then
// -------------------------------------------
// two-step fraction free Gaussian elimination
// -------------------------------------------
proc(x,opt)
  option remember;
  local columnElimination, n, m, i, j, l, f, f0, f1, f2, r, k, p, l0, ps,
        sig, charind, Rdivide, B, tmp, c11, c12, c21, c22, e11, e12,
        nonzeroes, L, list;
  begin
    columnElimination:= has([args()], ColumnElimination);
    Rdivide:= (if R::constructor=Dom::DistributedPolynomial and R::quo <> FAIL then
                  R::quo // we know the division is always exact here,
                         // and R::_divide may return FAIL for Dom::Numerical
               else
                 _divide
               end_if);
    ps:= R::pivotSize; // if ps = FAIL then no pivot strategy will be used, otherwise
                       // the pivot in respect to < will be choosen
    nonzeroes:= x::dom::nonZeroes(x);
    m:= extop(x,1); // number of rows
    n:= extop(x,2); // number of columns
    //===================================================================
    // convert x from polys = columns to polys = rows. Use B as container
    //===================================================================
    if not columnElimination then
         x:= extop(x::dom::transpose(x), 3):
    else x:= extop(x, 3);
         [m, n]:= [n, m];
    end_if:
    //=========================================================
    // convert rows from sum(x.i.j*`#_X_`^j) to sum(x.i.j*`#_X_`^(n-j))
    //=========================================================
    for i from 1 to m do
      // sparse
      x[i]:= poly(map(poly2list(x[i]), x -> [x[1], n-x[2]]), [hold(`#_X_`)], polyR);
      // dense
      //  x[i]:= poly([[coeff(x[i],j), n-j] $j=1..n],[hold(`#_X_`)],polyR);
    end_for;

if R::normal <> FAIL then
    userinfo(1,"perform 2-step fraction free Gaussian elimination");
    sig:= 1;      // the sign of the determinant
    f0:= Rone;  // the determinant
    charind:= {k $ k=1..n}; // the characteristic column indices
    k:= 2;
    r:= 2;
    for k from 2 to 2*(n div 2) step 2 do
      if r > m then break end_if;
        userinfo(3,"search for non-zero determinant");
        l0:= p:= FAIL; // initial value
        for i from r-1 to m do
          f1:= -coeff(x[i],n-k);
          // f2 := x[i,k-1];
          f2:= coeff(x[i],n-k+1);
          for j from i+1 to m do
            // f := _plus( _mult( f2,x[j,k] ), _mult( f1,x[j,k-1] ) );
            f:= f2*coeff(x[j],n-k) + f1*coeff(x[j],n-k+1);
            if not iszero(Rnormal(f)) then
              if ps = FAIL then
                p:= [i,j,f];
                break
              elif l0 = FAIL then
                l0:= ps(f);
                p:= [i,j,f]
              elif (l:=ps(f)) < l0 then
                l0:= l;
                p:= [i,j,f]
              end_if
            end_if;
          end_for;
          if j <= m and ps = FAIL then break end_if
        end_for;
        if p = FAIL then
          userinfo(3,"  non non-zero determinant found in step ",k);
          // initialize single step
          if r = 2 then
            f0 := Rone;
          else
            //    f0:= x[r-2,k-2];
            f0:= coeff(x[r-2],n-k+2);
          end_if;
          break
        end_if;
        if p[1] <> r-1 then
          i:= p[1];
          userinfo(3,"swap rows ",r-1,i);
          // x:= assignElements( x, ( [r-1,o]=x[i,o], [i,o]=x[r-1,o]) $ o=k-1..n );
          tmp:= x;
          x[r-1]:= tmp[i];
          x[i]:= tmp[r-1];
          sig:= -sig
        end_if;
        if p[2] <> r then
          j:= p[2];
          userinfo(3,"swap rows ",j,r);
          // x:= assignElements( x, ( [r,o]=x[j,o], [j,o]=x[r,o] ) $ o=k-1..n );
          tmp:= x;
          x[j]:= tmp[r];
          x[r]:= tmp[j];
          sig:= -sig
        end_if;
        // x[r-1,k-1] = 0 ?
        // if iszero( Rnormal(x[r-1,k-1]) ) then
        if iszero( Rnormal(coeff(x[r-1],n-k+1) )) then
          userinfo(3,"swap rows ",r-1,r);
          // x:= assignElements( x, ( [r-1,o]=x[r,o], [r,o]=x[r-1,o] ) $ o=k-1..n );
          tmp:= x;
          x[r]:= tmp[r-1];
          x[r-1]:= tmp[r];
          p[3]:= -p[3]; // |0 a\\c d| = -ac = -|c d\\0 a|
          sig:= -sig;
        end_if;
        f := Rdivide(p[3],f0);
        //changed by Kai:
        c11:= coeff(x[r-1],n-k);
        c12:= coeff(x[r-1],n-k+1);
        c21:= coeff(x[r],n-k+1);
        c22:= coeff(x[r],n-k);
        //changed by Kai:
        for i from r+1 to m do
          // f1 := Rnormal(Rdivide(
          //        _plus( Rmult( x[r-1,k],x[i,k-1] ),
          //        Rnegate(Rmult( x[r-1,k-1],x[i,k] )) ),
          //      f0 ));
          //changed by Kai:
          e11:= coeff(x[i],n-k+1);
          e12:= coeff(x[i],n-k);
          f1 := Rnormal(
                Rdivide(  c11 * e11 -
                          c12 * e12, f0 ));
          //changed by Kai:
          // f2 := Rnormal(Rdivide(
          //        _plus( Rmult( x[r,k-1],x[i,k] ),
          //          Rnegate( Rmult( x[r,k],x[i,k-1] )) ),
          //      f0 ));
          f2 := Rnormal(
                  Rdivide( c21 * e12   -
                           c22 * e11, f0 ));
          for j from k+1 to n do
            // x[i,j] := Rnormal(Rdivide(
            //     _plus(Rmult(f,x[i,j]),Rmult(f1,x[r,j]), Rmult(f2,x[r-1,j]) ),
            //     f0
            // ));
            x[i]:= x[i] - poly([[coeff(x[i],n-j), n-j]], [hold(`#_X_`)], polyR)
                        + poly([[Rnormal(
                                   Rdivide(  f*coeff(x[i],n-j)   + f1*coeff(x[r],n-j) +
                                            f2*coeff(x[r-1],n-j), f0)),n-j]], [hold(`#_X_`)], polyR);
          end_for;
          // x[i,k] := Rzero; x[i,k-1] := Rzero
          x[i]:= x[i] - poly([[coeff(x[i],n-k), n-k]], [hold(`#_X_`)], polyR);
          x[i]:= x[i] - poly([[coeff(x[i],n-k+1), n-k+1]], [hold(`#_X_`)], polyR);
        end_for;
        for j from k+1 to n do
          // x[r,j] := Rnormal(Rdivide(
          //     _plus( Rmult( x[r-1,k-1], x[r,j] ),
          //     Rnegate(Rmult( x[r-1,j],x[r,k-1] )) ),
          // f0 ));
          x[r]:= x[r] - poly([[coeff(x[r],n-j), n-j]], [hold(`#_X_`)], polyR)
                      + poly([[Rnormal(
                                        Rdivide( c12                 *
                                                 coeff(x[r],n-j)     -
                                                 coeff(x[r-1],n-j)   *
                                                 c21
                                                 , f0 )),n-j]], [hold(`#_X_`)], polyR);
        end_for;
        // x[r,k-1] := Rzero;
        x[r]:= x[r] - poly([[c21, n-k+1]], [hold(`#_X_`)], polyR);
        // x[r,k] := f;
        x[r]:= x[r] - poly([[c22 - f, n-k]], [hold(`#_X_`)], polyR);
        // new divisor
        f0:= f;
        // next step
        r:= r+2;
      end_for;
      r:= r-2;
      if n mod 2 = 1 or r <= n then
        userinfo(2,"perform single-step algorithm");
        f1:= k-1;
        r:= r+1;
        for k from f1 to n do
          if r > m then
            break
          end_if;
          userinfo(2,"searching for pivot element in row/column ",r,k);
          p:= m+1; l0:= FAIL; // initial value
          for i from r to m do
            // if not iszero(Rnormal(x[i,k])) then
            if not iszero(Rnormal(coeff(x[i],n-k))) then
              if ps = FAIL then p := i; break
              elif l0 = FAIL then
                // l0 := ps(x[i,k]);
                l0:= ps(coeff(x[i],n-k));
                p:= i
                //elif (l:=ps(x[i,k])) < l0 then l0 := l; p := i
              elif (l:=ps(coeff(x[i],n-k))) < l0 then
                l0:= l;
                p:= i
              end_if
            end_if
          end_for;
          if p <= m then
            if r <> p then
              userinfo(3,"swap rows ",r,p);
              // x:= assignElements( x,( [r,l]=x[p,l], [p,l]=x[r,l] ) $ l=k..n );
              tmp:= x;
              x[r]:= tmp[p];
              x[p]:= tmp[r];
              sig:= -sig
            end_if;
            for i from r+1 to m do
              f2:= -coeff(x[i],n-k);
              e11:= coeff(x[r],n-k);
              for j from k+1 to n do
              // x[i,j] := Rnormal(Rdivide(
              //    _plus( Rmult( x[r,k],x[i,j] ), Rmult( x[r,j],f2 ) ),f0
              // ))
              x[i]:= x[i] - poly([[coeff(x[i],n-j), n-j]],
                                 [hold(`#_X_`)], polyR)
                          + poly([[Rnormal(
                                           Rdivide(
                                               e11  *
                                               coeff(x[i], n-j) +
                                               coeff(x[r],n-j)  *
                                               f2
                                              ,f0 )),n-j]],
                                                         [hold(`#_X_`)],
                                                         polyR);
              end_for;
              // x[i,k] := Rzero
              x[i]:= x[i] - poly([[coeff(x[i],n-k), n-k]],
                                 [hold(`#_X_`)], polyR);
            end_for;
            // f0 := x[r,k];
            // Bug Fix: Hier stand vorher 'f0:= e11', d.h. der Wert von f0 wurde
            //          nicht korrekt aktualisiert!
            f0:= coeff(x[r],n-k);
            r:= r+1
          else
            charind:= charind minus {k}
          end_if
        end_for;
        charind:= charind minus {l $ l = k..n};
        r:= r-1
      end_if;
      if n = m then
        if r = m then
          if sig = -1 then
            f0:= -f0;
          end_if
        else
          f0:= Rzero
        end_if
      else
        f0:= FAIL
      end_if;

      //-----------------------------------------
      // Still need to reconvert to polys = cols.
      //-----------------------------------------
      //=========================================================
      // convert rows from sum(x.i.j*`#_X_`^(n-j)) to sum(x.i.j*`#_X_`^j)
      //=========================================================

      for i from 1 to m do
        // sparse
        x[i]:= poly(map(poly2list(x[i]), x -> [x[1], n-x[2]]), [hold(`#_X_`)], polyR);
        // dense
        //  x[i]:= poly([[coeff(x[i],j), n-j] $j=1..n],[hold(`#_X_`)],polyR);
      end_for;

      //=========================================================
      // convert row representation to column representation
      //=========================================================
      if columnElimination then
         tmp:= x;
         [m, n]:= [n, m]:
      else
        nonzeroes:= _plus(nops(poly2list(x[i])) $ i=1..m);
        if nonzeroes < m*n/2 then //sparse case
           // tmp = container for the columns
           tmp:= [[[Rzero,0] $ m] $ n];
           for i from 1 to m do
              L:= poly2list(x[i]);
              for list in L do
                 // fill the columns
                 tmp[list[2]][i]:= [list[1], i];
              end_for;
           end_for;
           // So far, the row tmp[j] is just a list.
           // Convert the columns to polynomials.
           B:= tmp;
           for j from 1 to n do
              tmp[j]:= poly(B[j], [hold(`#_X_`)], polyR);
           end_for;
        else //dense case
           tmp:= [0 $ n]; // initialize container for columns
           // build up column for column to avoid doubling of memory
           // due to doubly nested sequences
           for j from 1 to n do
              tmp[j] := poly([[coeff(x[i], j), i] $i=1..m],[hold(`#_X_`)],polyR);
              if has(tmp[j],FAIL) then
                 error("should not happen (5)")
              end_if;
           end_for;
        end_if;
      end_if;
      return([new(dom,m,n,tmp, "FLAG"),r,f0,charind]);
else // Rnormal = FAIL
    userinfo(1,"perform 2-step fraction free Gaussian elimination");
    sig:= 1;      // the sign of the determinant
    f0:= Rone;  // the determinant
    charind:= {k $ k=1..n}; // the characteristic column indices
    k:= 2;
    r:= 2;
    for k from 2 to 2*(n div 2) step 2 do
      if r > m then break end_if;
        userinfo(3,"search for non-zero determinant");
        l0:= p:= FAIL; // initial value
        for i from r-1 to m do
          f1:= -coeff(x[i],n-k);
          // f2 := x[i,k-1];
          f2:= coeff(x[i],n-k+1);
          for j from i+1 to m do
            // f := _plus( Rmult( f2,x[j,k] ), Rmult( f1,x[j,k-1] ) );
            f:= f2*coeff(x[j],n-k) + f1*coeff(x[j],n-k+1);
            if not iszero(f) then
              if ps = FAIL then
                p:= [i,j,f];
                break
              elif l0 = FAIL then
                l0:= ps(f);
                p:= [i,j,f]
              elif (l:=ps(f)) < l0 then
                l0:= l;
                p:= [i,j,f]
              end_if
            end_if;
          end_for;
          if j <= m and ps = FAIL then break end_if
        end_for;
        if p = FAIL then
          userinfo(3,"  non non-zero determinant found in step ",k);
          // initialize single step
          if r = 2 then
            f0 := Rone;
          else
            //    f0:= x[r-2,k-2];
            f0:= coeff(x[r-2],n-k+2);
          end_if;
          break
        end_if;
        if p[1] <> r-1 then
          i:= p[1];
          userinfo(3,"swap rows ",r-1,i);
          // x:= assignElements( x, ( [r-1,o]=x[i,o], [i,o]=x[r-1,o]) $ o=k-1..n );
          tmp:= x;
          x[r-1]:= tmp[i];
          x[i]:= tmp[r-1];
          sig:= -sig
        end_if;
        if p[2] <> r then
          j:= p[2];
          userinfo(3,"swap rows ",j,r);
          // x:= assignElements( x, ( [r,o]=x[j,o], [j,o]=x[r,o] ) $ o=k-1..n );
          tmp:= x;
          x[j]:= tmp[r];
          x[r]:= tmp[j];
          sig:= -sig
        end_if;
        // x[r-1,k-1] = 0 ?
        // if iszero( Rnormal(x[r-1,k-1]) ) then
        if iszero( coeff(x[r-1],n-k+1) ) then
          userinfo(3,"swap rows ",r-1,r);
          // x:= assignElements( x, ( [r-1,o]=x[r,o], [r,o]=x[r-1,o] ) $ o=k-1..n );
          tmp:= x;
          x[r]:= tmp[r-1];
          x[r-1]:= tmp[r];
          p[3]:= -p[3]; // |0 a\\c d| = -ac = -|c d\\0 a|
          sig:= -sig;
        end_if;
        f := Rdivide(p[3],f0);
        // changed by Kai:
        c11:= coeff(x[r-1],n-k);
        c12:= coeff(x[r-1],n-k+1);
        c21:= coeff(x[r],n-k+1);
        c22:= coeff(x[r],n-k);
        //changed by Kai:
        for i from r+1 to m do
          // f1 := Rnormal(Rdivide(
          //        _plus( Rmult( x[r-1,k],x[i,k-1] ),
          //        Rnegate(Rmult( x[r-1,k-1],x[i,k] )) ),
          //      f0 ));
          //changed by Kai:
          e11:= coeff(x[i],n-k+1);
          e12:= coeff(x[i],n-k);
          f1 := Rdivide(  c11 * e11 -
                          c12 * e12, f0 );
          //changed by Kai:
          // f2 := Rnormal(Rdivide(
          //        _plus( Rmult( x[r,k-1],x[i,k] ),
          //          Rnegate( Rmult( x[r,k],x[i,k-1] )) ),
          //      f0 ));
          f2 := Rdivide( c21 * e12   -
                         c22 * e11, f0 );
          for j from k+1 to n do
            // x[i,j] := Rnormal(Rdivide(
            //     _plus(Rmult(f,x[i,j]),Rmult(f1,x[r,j]), Rmult(f2,x[r-1,j]) ),
            //     f0
            // ));
            x[i]:= x[i] - poly([[coeff(x[i],n-j), n-j]], [hold(`#_X_`)], polyR)
                        + poly([[ Rdivide(  f*coeff(x[i],n-j)   + f1*coeff(x[r],n-j) +
                                            f2*coeff(x[r-1],n-j), f0),n-j]], [hold(`#_X_`)], polyR);
          end_for;
          // x[i,k] := Rzero; x[i,k-1] := Rzero
          x[i]:= x[i] - poly([[coeff(x[i],n-k), n-k]], [hold(`#_X_`)], polyR);
          x[i]:= x[i] - poly([[coeff(x[i],n-k+1), n-k+1]], [hold(`#_X_`)], polyR);
        end_for;
        for j from k+1 to n do
          // x[r,j] := Rnormal(Rdivide(
          //     _plus( Rmult( x[r-1,k-1], x[r,j] ),
          //     Rnegate(Rmult( x[r-1,j],x[r,k-1] )) ),
          // f0 ));
          x[r]:= x[r] - poly([[coeff(x[r],n-j), n-j]], [hold(`#_X_`)], polyR)
                      + poly([[ Rdivide( c12                 *
                                         coeff(x[r],n-j)     -
                                         coeff(x[r-1],n-j)   *
                                         c21
                                         , f0 ),n-j]], [hold(`#_X_`)], polyR);
        end_for;
        // x[r,k-1] := Rzero;
        x[r]:= x[r] - poly([[c21, n-k+1]], [hold(`#_X_`)], polyR);
        // x[r,k] := f;
        x[r]:= x[r] - poly([[c22 - f, n-k]], [hold(`#_X_`)], polyR);
        // new divisor
        f0:= f;
        // next step
        r:= r+2;
      end_for;
      r:= r-2;
      if n mod 2 = 1 or r <= n then
        userinfo(2,"perform single-step algorithm");
        f1:= k-1;
        r:= r+1;
        for k from f1 to n do
          if r > m then
            break
          end_if;
          userinfo(2,"searching for pivot element in row/column ",r,k);
          p:= m+1; l0:= FAIL; // initial value
          for i from r to m do
            // if not iszero(Rnormal(x[i,k])) then
            if not iszero(coeff(x[i],n-k)) then
              if ps = FAIL then p := i; break
              elif l0 = FAIL then
                // l0 := ps(x[i,k]);
                l0:= ps(coeff(x[i],n-k));
                p:= i
                //elif (l:=ps(x[i,k])) < l0 then l0 := l; p := i
              elif (l:=ps(coeff(x[i],n-k))) < l0 then
                l0:= l;
                p:= i
              end_if
            end_if
          end_for;
          if p <= m then
            if r <> p then
              userinfo(3,"swap rows ",r,p);
              // x:= assignElements( x,( [r,l]=x[p,l], [p,l]=x[r,l] ) $ l=k..n );
              tmp:= x;
              x[r]:= tmp[p];
              x[p]:= tmp[r];
              sig:= -sig
            end_if;
            for i from r+1 to m do
              f2:= -coeff(x[i],n-k);
              e11:= coeff(x[r],n-k);
              for j from k+1 to n do
              // x[i,j] := Rnormal(Rdivide(
              //    _plus( Rmult( x[r,k],x[i,j] ), Rmult( x[r,j],f2 ) ),f0
              // ))
              x[i]:= x[i] - poly([[coeff(x[i],n-j), n-j]],
                                 [hold(`#_X_`)], polyR)
                          + poly([[ Rdivide( e11  *
                                             coeff(x[i], n-j) +
                                             coeff(x[r],n-j)  *
                                             f2
                                             ,f0 ),n-j]],
                                                         [hold(`#_X_`)],
                                                         polyR);
              end_for;
              // x[i,k] := Rzero
              x[i]:= x[i] - poly([[coeff(x[i],n-k), n-k]],
                                 [hold(`#_X_`)], polyR);
            end_for;
            // f0 := x[r,k];
            // Bug Fix: Hier stand vorher 'f0:= e11', d.h. der Wert von f0 wurde
            //          nicht korrekt aktualisiert!
            f0:= coeff(x[r],n-k);
            r:= r+1
          else
            charind:= charind minus {k}
          end_if
        end_for;
        charind:= charind minus {l $ l = k..n};
        r:= r-1
      end_if;
      if n = m then
        if r = m then
          if sig = -1 then
            f0:= _negate(f0)
          end_if
        else
          f0:= Rzero
        end_if
      else
        f0:= FAIL
      end_if;

      //-----------------------------------------
      // Still need to reconvert to polys = cols.
      //-----------------------------------------
      //=========================================================
      // convert rows from sum(x.i.j*`#_X_`^(n-j)) to sum(x.i.j*`#_X_`^j)
      //=========================================================

      for i from 1 to m do
        // sparse
        x[i]:= poly(map(poly2list(x[i]), x -> [x[1], n-x[2]]), [hold(`#_X_`)], polyR);
        // dense
        //  x[i]:= poly([[coeff(x[i],j), n-j] $j=1..n],[hold(`#_X_`)],polyR);
      end_for;

      //=========================================================
      // convert row representation to column representation
      //=========================================================
      if columnElimination then
         tmp:= x;
         [m, n]:= [n, m]:
      else
        nonzeroes:= _plus(nops(poly2list(x[i])) $ i=1..m);
        if nonzeroes < m*n/2 then //sparse case
           // tmp = container for the columns
           tmp:= [[[Rzero,0] $ m] $ n];
           for i from 1 to m do
              L:= poly2list(x[i]);
              for list in L do
                 // fill the columns
                 tmp[list[2]][i]:= [list[1], i];
              end_for;
           end_for;
           // So far, the row tmp[j] is just a list.
           // Convert the columns to polynomials.
           B:= tmp;
           for j from 1 to n do
            tmp[j]:= poly(B[j], [hold(`#_X_`)], polyR);
           end_for;
        else //dense case
           tmp:= [0 $ n]; // initialize container for columns
           // build up column for column to avoid doubling of memory
           // due to doubly nested sequences
           for j from 1 to n do
              tmp[j] := poly([[coeff(x[i], j), i] $i=1..m],[hold(`#_X_`)],polyR);
              if has(tmp[j],FAIL) then
                 error("should not happen (5)")
              end_if;
           end_for;
        end_if;
      end_if;
      return([new(dom,m,n,tmp, "FLAG"),r,f0,charind]);
    end_if; // of if Rnormal = FAIL
  end_proc;
end_if; // of if R::hasProp( Cat::Field ) and ...
        //    elif R::hasProp( Cat::IntegralDomain )

/*-------------------------------------------------------------
  transpose  --  compute the transpose of a matrix
-------------------------------------------------------------*/

transpose:= proc(x)
local r, c, smallmatrix, s, L, i, j, list;
begin
  r := extop( x,1 );
  c := extop( x,2 );
  // `#_X_`:= op(x[1],[2,1]);

  // For sparse matrices the version using L:= poly2list(x[j])
  // is faster than the version using coeff(x[i], j).
  // For dense matrices it is the other way round.
  // Regard a matrix as 'reasonably sparse', if at most
  // half its entries are nonzero:
  if x::dom::nonZeroes(x) < r*c/2
  then //sparse case
       x := extop(x,3);
       // s = container for the rows
       smallmatrix:= bool(r*c <= 2*10^4) or //heuristics
                     bool(r = 1) or         //tested in
                     bool(c = 1);           //MuPAD 3.0
       if smallmatrix then
            s:= [[[Rzero,0] $ c] $ r];
       else // for very large sparse matrices,
            // memory would blow up in the
            // nested list above, whenn filling
            // in the entries. Use a table (sparse
            // structure) instead.
            s:= table((i = table()) $ i=1..r):
       end_if;
       for j from 1 to c do
         L:= poly2list(x[j]);
         for list in L do
           // fill the rows
           s[list[2]][j]:= [list[1], j];
         end_for;
       end_for;
       // So far, the row s[i] is just a list/table
       // Convert the rows to polynomials.
       if smallmatrix then
            for i from 1 to r do
              s[i]:= poly(s[i], [hold(`#_X_`)], polyR);
            end_for;
       else for i from 1 to r do
              s[i]:= poly(map([op(s[i])], op, 2), [hold(`#_X_`)], polyR);
            end_for;
            // Here, s is a table with s[i] = poly representing the i-th
            // row of the matrix = the i-th column of the transpose.
            // Convert the table of polys to a list of polys
            // (do not use map([op(s)], op, 2), because the operands
            // of a table are not necessarily in the correct order
            s:= [s[i] $ i = 1..r];
       end_if;
       return(new(dom, c, r, s, "FLAG"));
  else //dense case
       x := extop(x,3);
       s:= [0 $ r]; // initialize container for rows
       // build up row for row to avoid doubling of memory
       // due to doubly nested sequences
       for j from 1 to r do
           s[j] := poly([[coeff(x[i],j), i] $i=1..c],[hold(`#_X_`)],polyR);
       end_for;
       // Now, interpret rows as columns:
       return(new(dom, c, r, s, "FLAG"));
  end_if;
end_proc;

/*--------------------------------------------------------------------
  nonZeroes  --  return the number of non-zero elements of a matrix
--------------------------------------------------------------------*/

nonZeroes:= proc(a)
local c, i;
begin
  c := extop( a,2 );
  _plus(nops(poly2list(extop(a, 3)[i])) $ i=1..c);
end_proc;

/*-------------------------------------------------------------------
  nonZeros  --  return the number of non-zero elements of a matrix

  This is just an 'alias' for 'nonZeroes' to make Americans
  as happy as British Customers.
-------------------------------------------------------------------*/

nonZeros:= dom::nonZeroes;

/*-------------------------------------------------------------
    nonZeroOperands -- return a sequence of all operands that
                       are nonzero. The ordering of the elements
                       has no correspondence to the ordering inside
                       the matrix.
Call:  nonZeroOperands(A)

Typical usage: find out what type the operands have. One
       should not use something like map({op(A)}, type),
       because op(A) returns lots of 0s. These are reduced
       by the set, but this may be expensive for large
       matrices that are very sparse.
-------------------------------------------------------------*/
nonZeroOperands:= proc(A)
local r, c, nonzeroes, t, i, j, term, count;
begin
  if args(0) <> 1 then
    error("expecting exactly one argument"):
  end_if;

  r:= extop(A, 1);
  c:= extop(A, 2);

  // extract *all* coefficients
  A:= map(extop(A, 3), poly2list):
  nonzeroes:= _plus(nops(A[i]) $ i=1..c):
  t:= [0 $ nonzeroes];
  count:= 0:
  for j from 1 to c do
    // consider the j-th column A[j]
    for term in A[j] do
     count:= count + 1;
     t[count]:= term[1];
    end_for:
 end_for;
 if Rcoerce <> id then
    t:= map(t, Rcoerce);
 end_if;
 return(op(t));
end_proc;

indets:= proc(A)
begin
  return(indets([A::dom::nonZeroOperands(A)], args(2..args(0))))
end_proc;

freeIndets:= proc(A)
begin
  return(freeIndets([A::dom::nonZeroOperands(A)], args(2..args(0))))
end_proc;

numericindets:= proc(A)
begin
  return(numeric::indets([A::dom::nonZeroOperands(A)], args(2..args(0))))
end_proc;

/*-----------------------------------------------
  stackMatrix  --  concat two matrices vertically
-----------------------------------------------*/
stackMatrix:= proc()
    local a, b, cb, rb, ra, ca, t, j, i, L1, L2, tmp;
    begin
    if args(0) = 0 then return( null() ) end_if;
    if args(0) = 1 then return( args(1) ) end_if;

    ra := extop( args(1),1 );
    ca := extop( args(1),2 );
     a := extop( args(1),3 );
    tmp:= ra;
    // `#_X_`:= op(a[1],[2,1]);
    for i from 2 to args(0) do
        b:= extop(args(i),3);
        rb:= extop(args(i),1);
        cb:= extop(args(i),2);
        t:= [0 $ min(ca,cb)];
        if ca <> cb then
            error("number of columns of all matrices have to be the same");
        end_if;
        for j from 1 to min(ca,cb) do
            L1:= poly2list(a[j]);
            L2:= poly2list(b[j]);
            L2:= map(L2, list -> [list[1],list[2]+tmp] );
            L1:= _concat(L1,L2);
            t[j]:=poly(L1,[hold(`#_X_`)],polyR);
        end_for;
        a:= t;
        tmp:= tmp + rb;
    end_for;
    new( dom,tmp,ca,a, "FLAG" )
end_proc;

/*-------------------------------------------------------------------
  concatMatrix  --  append two matrices horizontally
-------------------------------------------------------------------*/
concatMatrix:= proc(a,b)
    local ra, rb, ca, cb, i, tmp;
    begin
        if args(0) = 0 then
           return( null() );
        end_if;
        if args(0) = 1 then
            return( a )
        elif args(0) > 2 then
            tmp:= args(1);
            for i from 2 to args(0) do
                tmp:= dom::concatMatrix(tmp,args(i));
            end_for;
            return(tmp);
        end_if;
        if a::dom <> dom then
           a:= dom::convert(a);
           if a = FAIL then
              return(FAIL);
           end_if;
        end_if;
        if b::dom <> dom then
           b:= dom::convert(b);
           if b = FAIL then
              return(FAIL);
           end_if;
        end_if;
        ra := extop( a,1 ); ca := extop( a,2 ); a := extop( a,3 );
        rb := extop( b,1 ); cb := extop( b,2 ); b := extop( b,3 );
        /* The following test has been removed by Nicolas Thiery
           (4/12/2001), to let sparse matrices
           grow dynamically with concatMatrix, and so allow to deal with
           sparse matrices with "infinite" number of rows.
           Should this be configurable ?

           if ra <> rb then
              error("number of rows of all matrices have to be the same");
           end_if;
        */
        new( dom,max(ra,rb),ca+cb,_concat(a,b), "FLAG")
end_proc;

/*-------------------------------------------------------------------
  setRow  --  replace a row of a matrix by a new one
-------------------------------------------------------------------*/
setRow:= proc(x,p,row)
local r, c, t, j, L, pp, Rgn;
begin
    if domtype(row) = DOM_LIST then
        row:= map(row, R2polyCoerce@Rcoerce);
        if has(row, FAIL) then return(FAIL) end_if;
        r := extop( x,1 ); c := extop( x,2 ); t := extop( x,3 );
        for j from 1 to c do
            if not iszero(coeff(t[j],p)) then
                L:= poly2list(t[j]);
                L:= map(L, list -> if list[2]=p then [row[j],p] else list end);
                t[j]:=poly(L,[hold(`#_X_`)],polyR);
            else
                L:= poly2list(t[j]);
                L:= append(L,[row[j],p]);
                t[j]:=poly(L,[hold(`#_X_`)],polyR);
            end_if;
        end_for;
    else //in this case row is a vector of type 'Matrix'
        if extop(row,2) <> extop(x,2) then
          error("wrong number of columns of the row vector");
        end_if;
        if x::dom <> row::dom and
           x::dom::constructor <> Dom::SquareMatrix and
           x::dom::constructor <> Dom::MatrixGroup
        then
          if (row:= x::dom::coerce(row)) = FAIL then
            error("cannot convert row vector to a vector of type ".expr2text(x::dom));
          end_if;
        elif x::dom::constructor = Dom::SquareMatrix or
             x::dom::constructor = Dom::SquareMatrix
          then
            Rgn:= x::dom::coeffRing;
            if has(map({op(row)}, elem -> coerce(elem,Rgn)), FAIL) then
              error("cannot convert coefficient ring of the row vector to ".
                    expr2text(x::dom::coeffRing));
            end_if;
        end_if;
        r := extop( x,1 ); c := extop( x,2 ); t := extop( x,3 );
        pp := extop( row,3 );
        for j from 1 to c do
            if not iszero(coeff(t[j],p)) then
                L:= poly2list(t[j]);
                L:= map(L, list -> if list[2]=p then
                                   [coeff(pp[j],1),p] else list end);
                t[j]:=poly(L,[hold(`#_X_`)],polyR);
            else
                L:= poly2list(t[j]);
                L:= append(L,[coeff(pp[j],1),p]);
                t[j]:=poly(L,[hold(`#_X_`)],polyR);
            end_if;
         end_for;
    end_if;
    new(dom,r,c,t, "FLAG");
end_proc;

/*-------------------------------------------------------------------
  setCol  --  replace a column of a matrix by a new one
-------------------------------------------------------------------*/

setCol:= proc(x,p,col)
    local c, r, t, i, Rgn;
    begin
    r := extop( x,1 );
    c := extop( x,2 );
    t := extop( x,3 );
    if domtype(col) = DOM_LIST then
         col:= map(col, R2polyCoerce@Rcoerce);
         if has(col, FAIL) then return(FAIL) end_if;
         t[p]:= poly([[col[i],i] $i=1..r],[hold(`#_X_`)], polyR);
    else //in this case col is a vector of type 'Matrix'
         if extop(col,1) <> r then
           error("wrong number of rows of the column vector");
         end_if;
         if x::dom <> col::dom and
            x::dom::constructor <> Dom::SquareMatrix and
            x::dom::constructor <> Dom::MatrixGroup
         then
           if (col:= x::dom::coerce(col)) = FAIL then
             error("cannot convert column vector to a vector of type ".
                   expr2text(x::dom));
           end_if;
         elif x::dom::constructor = Dom::SquareMatrix or
              x::dom::constructor = Dom::MatrixGroup
         then
           Rgn:= x::dom::coeffRing;
           if has(map({op(col)}, elem -> coerce(elem,Rgn)), FAIL) then
             error("cannot convert coefficient ring of the column vector to ".
                   expr2text(x::dom::coeffRing));
           end_if;
         end_if;
         t[p]:= extop(extop(col,3), 1);
    end_if;
    new(dom, r, c, t, "FLAG");
end_proc;

/*-------------------------------------------------------------------
  row  --  extract a row of a matrix
-------------------------------------------------------------------*/

row:= proc(x,p)
        local c, j, col;
        begin
        if extop(x,1) < p then
            error( "Index out of range ");
        else
        c:= extop(x,2);
        x:= extop(x,3); // the poly list
        x:= [x[j] $ j=1..c]; // reduced poly list
            for j from 1 to nops(x) do
                col:= poly2list(x[j]);
                col:= map(col, proc(a) begin
                        if a[2]>=p and a[2]<=p
                            then [a[1],a[2]-p+1]
                            else null()
                        end_if
                        end_proc);
                    x[j]:= poly(col, [hold(`#_X_`)], polyR);
            end_for;
        end_if;
        new(dom, 1, c, x, "FLAG");
end_proc;

/*-------------------------------------------------------------------
  col  --  extract a column of a matrix
-------------------------------------------------------------------*/

col:= proc(x, p)
        local r;
        begin
        r:= extop(x,1);
        x:= [extop(extop(x,3),p)];
        new(dom, r, 1, x, "FLAG");
end_proc;


/*-------------------------------------------------------------------
  multCol  --  multiply the p-th column of a matrix by a
               scalar factor f
-------------------------------------------------------------------*/

multCol:= proc(x, p, f)
            local r, c;
          begin
            r:= extop(x,1);
	    c:= extop(x,2);
            x:= extop(x,3);
       	    f:= map(f, R2polyCoerce@Rcoerce);
	    if has(f, FAIL) then return(FAIL) end_if;
 	    x[p]:= mapcoeffs(x[p], _mult, f);
 	    if has(x[p], FAIL) then return(FAIL) end_if;
            new(dom, r, c, x, "FLAG");
          end_proc;

/*-------------------------------------------------------------------
  multRow  --  multiply the p-th row of a matrix by a
               scalar factor f
-------------------------------------------------------------------*/

multRow:= proc(x, p, f)
            local r, c, L, j;
          begin
            r:= extop(x,1);
	    c:= extop(x,2);
            x:= extop(x,3);
       	    f:= map(f, R2polyCoerce@Rcoerce);
            if has(f, FAIL) then return(FAIL) end_if;
            for j from 1 to c do
              if not iszero(coeff(x[j],p)) then
                  L:= poly2list(x[j]);
                  L:= map(L, list -> if list[2] = p then [f*list[1],p] else list end);
                  x[j]:=poly(L,[hold(`#_X_`)],polyR);
              end_if;
            end_for;
 	    if has(x, FAIL) then return(FAIL) end_if;
            new(dom, r, c, x, "FLAG");
          end_proc;

/*-------------------------------------------------------------------
  addCol  --  add f times column p to g times column q
-------------------------------------------------------------------*/

addCol:= proc(A, p, q, f, g = 1)
      local r, c, x;
    begin
      r:= extop(A, 1);
      c:= extop(A, 2);
      x:= extop(A, 3);
      f:= R2polyCoerce(Rcoerce(f));
      if has(f, FAIL) then return(FAIL) end_if;
      g:= R2polyCoerce(Rcoerce(g));
      if has(g, FAIL) then return(FAIL) end_if;
      x[q]:= mapcoeffs(x[p], _mult, f) + mapcoeffs(x[q], _mult, g);
      if has(x[p], FAIL) then return(FAIL) end_if;
      new(dom, r, c, x, "FLAG");
    end_proc;

/*-------------------------------------------------------------------
  addRow  --  add g times row q to f times row p
-------------------------------------------------------------------*/

addRow:= proc(A, p, q, f, g = 1)
      local row_q;
    begin
      row_q:= f * A::dom::row(A, p) + g * A::dom::row(A, q);
      A:= A::dom::setRow(A, q, row_q);
    end_proc;

/*-------------------------------------------------------------------
  delRow  --  delete one or more rows of a matrix

  delRow(A, r)
  delRow(A, r1..r2)
  delRow(A, [r1, r2, ..])

  r, r1, r2 --- row indices

-------------------------------------------------------------------*/

delRow:= proc(A, p)
local r, c, t, k, i, j, DEL, L, ind;
begin
    r:= extop( A, 1 );
    c:= extop( A, 2 );
    if r = 1 then return( NIL ); end_if;
    t:= extop( A, 3 );
    if domtype(p) = DOM_INT then
        // delete a single row
        for j from 1 to c do
            L:= poly2list(t[j]);
            L:= map(L, list -> if p = list[2] then null()
                               elif p<list[2] then
                                    [list[1],list[2]-1]
                               else list
                               end_if);
            t[j]:=poly(L, [hold(`#_X_`)], polyR);
        end_for;
        return(new(dom, r-1, c, t, "FLAG"));
    end_if;
    // eliminate several rows in one go.
    // The rows to be deleted are either specified
    // as a range p = r1..r2 or as a list p = [r1, r2, ..]:
    case type(p)
    of "_range" do
       DEL:=[i $ i=extop(p, 1) .. extop(p, 2)];
       break;
    of DOM_LIST do
       // remove duplicates by temporary conversion to
       // a set, then sort the indices from small to large
       DEL:= sort([op({op(p)})]);
       break;
    otherwise
       error("expecting a row index, ".
              "a range of row indices, ".
              "or a list of row indices");
    end_case;
    for j from 1 to c do
       L:= poly2list(t[j]);
       for k from 1 to nops(DEL) do
          ind:= DEL[k]-k+1;
          L:= map(L, list -> if ind = list[2] then null()
                             elif ind<list[2] then
                                  [list[1],list[2]-1]
                             else list
                             end_if);
       end_for;
       t[j]:= poly(L, [hold(`#_X_`)], polyR);
    end_for;
    new(dom, r - nops(DEL), c, t, "FLAG");
end_proc;

/*-------------------------------------------------------------------
  delCol  --  delete a column of a matrix
-------------------------------------------------------------------*/

delCol:= proc(A, p)
    local r, c, l, k, i, j, x;
    begin
    r:= extop( A,1 );
    c:= extop( A,2 );
    if c = 1 then return( NIL ); end_if;
    x:= extop(A, 3);
    if type(p) = "_range" then
         l:=extop(p,1);
         k:=extop(p,2);
         for j from k downto l do
             delete x[j];
         end_for;
    else if domtype(p) = DOM_INT then
           delete x[p];
           l:=p;
           k:=p;
         end_if;
         if domtype(p) = DOM_LIST then
           l:=0;
           k:=nops(p)-1;
           x := eval(subsop(x, i = null() $ i in p));
         end_if;
    end_if;
    new(dom, r, c-k+l-1, x, "FLAG");
end_proc;

/*-------------------------------------------------------------------
  swapRow  --  swap two rows of a matrix
-------------------------------------------------------------------*/
swapRow:= proc(A, p, q)
    local r, c, t, x, j, L, range, i1, i2;
    begin
    if p = q then return( A ); end_if;
    r:= extop( A,1 );
    c:= extop( A,2 );
    t:= extop( A,3 );
    x:= [0 $ c];
    // Bug-Fix: The case of four arguments, where the fourth
    //          argument is an integer or a range of integers
    //          was missing.
    //          The new case is added below!

    //-------------------------------new code----------------------------------
    if args(0) = 4 then
       range:= args(4);
       if type(range) = "_range" then
          i1:= op(range,1);
          i2:= op(range,2);
       else
          i1:= range;
          i2:= i1;
       end_if;
       if domtype(i1) <> DOM_INT or domtype(i2) <> DOM_INT then
          error("expecting an integer or a range of integers as 4. argument");
       end_if;
       for j from i1 to i2 do
          L:= poly2list(t[j]);
          L:= map(L, list -> if list[2] = p then
                                [list[1], q];
                             elif list[2] = q then
                                [list[1], p]
                             else list;
                             end_if
                  );
          t[j]:= poly(L, [hold(`#_X_`)], polyR);
       end_for;
    return(new(dom,r,c,t, "FLAG"));
    end_if;
    //----------------------------end of new code------------------------------

    for j from 1 to c do
        L:= poly2list(t[j]);
        L:= map(L, list -> if   list[2] = p then
                               [list[1], q];
                           elif list[2] = q then
                               [list[1], p]
                           else list;
                           end_if
               );
        x[j]:= poly(L, [hold(`#_X_`)], polyR);
    end_for;
    new(dom,r,c,x, "FLAG");
end_proc;


/*-------------------------------------------------------------------
  swapCol  --  swap two columns of a matrix
-------------------------------------------------------------------*/

swapCol:= proc(A,p,q)
    local r, c, t, range, i1, i2, tmp1, tmp2;
    begin
    if p = q then return( A ); end_if;
    r:= extop( A,1 );
    c:= extop( A,2 );
    t:= extop( A,3 );

    if args(0) = 4 then // swap cols only for the rows given
                        // by the 4th argument (= integer or range)
       range:= args(4);
       if type(range) = "_range" then
          i1:= op(range,1);
          i2:= op(range,2);
       else
          i1:= range;
          i2:= i1;
       end_if;
       if domtype(i1) <> DOM_INT or domtype(i2) <> DOM_INT then
          error("expecting an integer or a range of integers as 4. argument");
       end_if;
       tmp1:= poly2list(t[p]);
       tmp2:= poly2list(t[q]);
       tmp1:= map(tmp1, list -> if list[2] >= i1 and list[2] <= i2 then
                                   list;
                                else
                                   null();
                                end_if);
       tmp1:= poly(tmp1, [hold(`#_X_`)], polyR);
       tmp2:= map(tmp2, list -> if list[2] >= i1 and list[2] <= i2 then
                                   list;
                                else
                                   null();
                                end_if);
       tmp2:= poly(tmp2, [hold(`#_X_`)], polyR);
       t[p]:= t[p] - tmp1 + tmp2;
       t[q]:= t[q] - tmp2 + tmp1;
       return(new(dom, r, c, t, "FLAG"));

    else // args(0) = 3, i.e. the standard case without fourth argument
       [t[p], t[q]]:= [t[q], t[p]];
       return(new(dom, r, c, t, "FLAG"));
    end_if;
end_proc;


/* --------------------------------------------------
  assignElements  --  overloads the function 'assignElements'

Calls:
  assignElements(A, [index1] = value1, [index2] = value2, ...)
  assignElements(A, [[index1], value1], [[index2], value2], ...)

-------------------------------------------------- */

assignElements:= proc(A)
local s, r, c, x, j, k, jk,
  //  subslist,
      polyList,rowindex, value, cnt, eq, Eqs;
begin
   if args(0) = 0 then
      error("expecting at least 1 argument"):
   end_if;

   r:= extop(A,1); c:= extop(A,2); x:= extop(A,3);
   s:= [0 $ c];

   Eqs:= [ 0 $ i=2..args(0)];

   for k from 2 to args(0) do
     //Two different cases can appear: '[i, j] = value' or '[[i,j],value]'
     //These cases are treated first:

     //If the input is of the form '[i, j] = value' then leave it as it is.

     if not type(args(k)) = "_equal" then
        //If it is of the from '[[i,j],value]' then transorm into '[i, j] = value'
        if domtype(args(k)) <> DOM_LIST then
           error("wrong specification, expecting an equation '[i, j] = value' or '[[i,j],value]'");
           else
              Eqs[k-1]:= args(k)[1] = args(k)[2];
        end_if;
     else Eqs[k-1]:= args(k);
     end_if:
   end_for;
   if Rcoerce <> id then
      Eqs:= map(Eqs, x -> (op(x, 1) = Rcoerce(op(x, 2))));
   end_if;
   if R2polyCoerce <> id then
      Eqs:= map(Eqs, x -> (op(x, 1) = R2polyCoerce(op(x, 2))));
   end_if;
   for eq in Eqs do
         jk:= op(eq, [1,2]);
         if s[jk] = 0 then
            s[jk]:= poly2list(x[jk]);
            rowindex:= op(eq,[1,1]); //number of the row where
                                     //we want to replace the index
            value:= op(eq,2);        //the new coefficient, i.e.
                                     //[i,j] = value
            //Here there is a problem: What has to be done,
            //if the coefficient, which has to be replaced,
            //is zero?
            //I use the variable 'cnt' as an indicator to show
            //if the coefficient
            //is zero or not. If cnt is equal to 1,
            //then the coefficient was not equal to
            //zero and already exchanged.
            //If cnt is equal to 0, then the coefficient
            //which has to be replaced was 0 and thereby
            //I use append(s[jk],...) to insert the
            //new coefficient.
            cnt:= 0;
            s[jk]:= map(s[jk], list -> if list[2] = rowindex
                                       then cnt:= 1: [value,list[2]];
                                       else list end_if);
            if cnt = 0 then
                s[jk]:= append(s[jk],[value, rowindex]);
            end_if;
         else
            rowindex:= op(eq,[1,1]); //number of the row where
                                     //we want to replace the index

            value:= op(eq,2);        //the new coefficient, i.e.
                                     //[i,j] = value
            cnt:= 0;
            s[jk]:= map(s[jk], list -> if list[2] = rowindex
                                       then cnt:= 1: [value,list[2]];
                                       else list end_if);
            if cnt = 0 then
               s[jk]:= append(s[jk],[value, rowindex]);
            end_if;
          end_if;
      end_for;
      polyList:= extop(A, 3);
      for j from 1 to c do
         if iszero(s[j]) then next end_if;
         polyList[j]:= poly(s[j], [hold(`#_X_`)], polyR);
      end_for;
      return(extsubsop(A, 3 = polyList));
end_proc;


/*-----------------------------------------------------------------------
  TeX  --  Creates a 'TeX'-Output of matrices
-----------------------------------------------------------------------*/

TeX:= proc(A)
local arr;
begin
   arr:= dom::convert_to(A, DOM_ARRAY);
   return(generate::TeX(arr));
end_proc;

/*-----------------------------------------------------------------------
  map --  Overloads the function 'map' for (sparse)
          matrices. Problems arise, for example, if the user
          maps the function 'exp' onto a Matrix with
          zero coefficients. The following method maps
          functions onto every coefficient. Be aware of losing
          the sparsity in this case!!!

  Here an example:

  >> A:= Dom::Matrix()([[0,0,2],[0,1,0]]);

                       +-         -+
                       |  0, 0, 2  |
                       |           |
                       |  0, 1, 0  |
                       +-         -+

  >> map(A, x -> exp(x) - exp(2));

       +-                                             -+
       |  - exp(2) + 1,   - exp(2) + 1,        0       |
       |                                               |
       |  - exp(2) + 1, exp(1) - exp(2), - exp(2) + 1  |
       +-                                             -+

  >> A:= Dom::Matrix(Dom::IntegerMod(3))([[0, 1, 2, 3]]);

            +-                                  -+
            | 0 mod 3, 1 mod 3, 2 mod 3, 0 mod 3 |
            +-                                  -+

  >> map(A, x -> x^2)   // maps 0 -> 0

            +-                                  -+
            | 0 mod 3, 1 mod 3, 1 mod 3, 0 mod 3 |
            +-                                  -+

  >> map(A, x -> x + 1)  // maps 0 -> 1

            +-                                  -+
            | 1 mod 3, 2 mod 3, 0 mod 3, 1 mod 3 |
            +-                                  -+

  Of course, this function works not quite as fast
  as the function 'mapNonZeroes' below.
---------------------------------------------------------------------*/
// NB: map(matrix, f) is popular programming style of the library.
//     --> map must be ***fast***
//     Even in the dense or semi-dense case, the following version
//     seems to be nearly as fast as 'map' on arrays ;-)
//---------------------------------------------------------------------

map:= proc(x, f)
local r, c, f0, nrows, params, polyList, ind, haszeroes, 
      j, n, i, L, degs,fillingList, RzeroX, ff, g, Args;
begin
  [r, c]:= [extop(x, 1), extop(x, 2)]:
  RzeroX:= x::dom::coeffRing::zero;
  Args := [args(3..args(0))];
  if contains(Args, Unsimplified) <> 0 then
    delete Args[contains(Args, Unsimplified)];
  end_if;
  f0:= RzeroX:
  haszeroes:= bool(x::dom::nonZeroes(x) < r*c):
  if haszeroes and traperror((f0:= f(RzeroX, op(Args)))) <> 0 then
     // there are zero elements in the matrix
       error("cannot evaluate ".expr2text(f)." for matrix components equal to zero");
  end_if;
  nrows:= extop(x,1);
  // forward map to the existing entries. We need to:
  // -- pick out the coefficients of the poly (of type polyR)
  // -- compute f(...)
  // -- reconvert the return value of f to polyR.
  params:= args(3..args(0));
  if params = null() then
    // encapsulating f in another procedure ff is very
    // costly. Avoid this, if no further params are needed.
    if Rcoerce <> id then
         ff:= f@Rcoerce;
    else ff:= f;
    end_if;
  else
    if Rcoerce <> id then
         ff:= proc() begin
                f(Rcoerce(args(1)), params);
              end;
    else ff:= proc() begin
                f(args(1), params);
              end;
    end_if;
  end_if;
  if R2polyCoerce <> id then
     ff:= R2polyCoerce@ff;
  end_if;

  // need an extra iszero(f0) to avoid filling the
  // matrix with empty objects <> Rzero such as
  // float(0) etc (e.g, if f = float; not 0.0 <> Rzero = 0)
  if haszeroes and
     f0 <> RzeroX and
     not iszero(f0) then
       f0:= R2polyCoerce(f0);
       polyList:= extop(x, 3);
       // fill in new elements f(Rzero) in all empty places
       // Beware: f must not have any side effects, i.e.,
       // f is assumed to be such that several calls to f(Rzero)
       // produce the same result (e.g., f = frandom(seed) would not
       // fill in random numbers, but just one value)
       for j from 1 to extop(x, 2) do
         n:= nterms(polyList[j]):
         case n
         of extop(x, 1) do
            // the j-th column is dense:
            // apply f to the existing non-zero entries.
            polyList[j]:= mapcoeffs(polyList[j], ff);
            break; // proceed to next column
         of 0 do
            // the j-th colum was empty
            polyList[j]:= poly([[f0, i] $ i = 1..extop(x, 1)], [hold(`#_X_`)], polyR);
            break; // proceed to next column
         otherwise
            // 0 < n < extop(x, 1), the column is neither empty nor dense
            if degree(polyList[j]) = 0 then
               // the only degree 0 polys should be empty polynomials
               // by the previous case switch
               error("should not happen");
            end_if;
            // The further strategy:
            // Let L:= poly2list(columnPolynomial) = [[c1,e1],[c2,e2],...,[cl,el]].
            // Extract the exponents into the list degs = [e1,e2,...,el].
            // By degs:= {i $ i=1..nrows} minus {op(degs)} one receives a set of the
            // exponents of those terms of columnPolynomial, which are zero and whose
            // degree is less or equal to the degree of columnPolynomial.
	    L:= poly2list(polyList[j]);
            degs:= map(L, list -> list[2]);
            degs:= {i $ i=1..nrows} minus {op(degs)};
	    // Bug -Fix by Kai: L:= map(L, list -> [map(list[1], ff), list[2]]);
	    // does not what it should, since map(q^2, ff) = ff(q)^(ff(2))) for a
	    // matrix component q. So use ff(list[1]).
            L:= map(L, list -> [ff(list[1]), list[2]]);
            // Now create a new list called 'fillingList' by fillingList[i]:= [f0, ind]
            // for all elements ind in the set degs. Put the two lists 'L' and
            // 'fillingList' together and turn it back into a polynomial.
            fillingList:= [0 $ nops(degs)];
            i:= 1;
            for ind in degs do
              fillingList[i]:= [f0, ind];
              i:= i+1;
            end_for;
            L:= _concat(L, fillingList);
            polyList[j]:= poly(L, [hold(`#_X_`)], polyR);
         end_case;
       end_for;
       return(extsubsop(x, 3 = polyList));
  elif f0 = RzeroX then 
       // f0 = f(Rzero) = Rzero. No need to insert new elements
       x:= map(extop(x, 3), row -> if iszero(row) then 
                                      row
                                   else
                                      mapcoeffs(row, ff)
                                   end_if);
       if has(x, FAIL) then return(FAIL); end_if;
       return(new(dom, r, c, x, "FLAG"));
  else // f0 <> RzeroX and iszero(f0)
       // Important for map(Dom::Matrix()(..), factor)
       g:= x -> (x:= ff(x): if iszero(x) then Rzero else x end_if);
       x:= map(extop(x, 3), row -> if iszero(row) then 
                                      row
                                   else
                                      mapcoeffs(row, g)
                                   end_if);
       if has(x, FAIL) then return(FAIL); end_if;
	       return(new(dom, r, c, x, "FLAG"));
  end_if;
end_proc;

// needed for misc::maprec
enableMaprec:= TRUE;

/*-----------------------------------------------------------------------
  mapNonZeroes --  special version of 'map' for matrices
                   This function maps the arguments only on those
                   coefficients of the matrix, which are nonzero.
                   Be aware of problems arising from functions f,
                   for which you get f(Rzero) <>  Rzero.
-----------------------------------------------------------------------*/
    mapNonZeroes:= proc(x, f)
    local r, c, ff, f0, g, params, j;
    begin
       if args(0) < 2 then
          error("expecting at least 2 arguments");
       end_if;

       [r, c]:= [extop(x, 1), extop(x, 2)]:

       // just forward map to the existing entries. We need to:
       // -- pick out the coefficients of the poly (of type polyR)
       // -- compute f(...)
       // -- reconvert the return value of f to polyR.

       params:= args(3..args(0));
       if params = null() then
         // encapsulating f in another procedure ff is very
         // costly. Avoid this, if no further params are needed.
         if Rcoerce <> id then
              ff:= f@Rcoerce;
         else ff:= f;
         end_if;
       else
         if Rcoerce <> id then
            ff:= proc() begin
                    f(Rcoerce(args(1)), params);
                 end;
         else ff:= proc() begin
                    f(args(1), params);
                 end;
         end_if;
       end_if;
       if R2polyCoerce <> id then
          ff:= R2polyCoerce@ff;
       end_if;

       // Beware: mapcoeffs(poly(0, [`#_X_`]), f) produces
       // poly(f(0), [`#_X_`]) !!
       x:= extop(x, 3);
       f0:= ff(Rzero);
       if f0 <> Rzero then
          if iszero(f0) then
             g:= x -> (x:= ff(x): if iszero(x) then Rzero else x end_if);
          else g:= ff;
          end_if;
          for j from 1 to nops(x) do
            if not iszero(x[j]) then
               x[j]:= mapcoeffs(x[j], g);
            end_if;
          end_for;
       else
          // the following is faster than the for loop
          // above by a factor of 2:
          x:= map(x, mapcoeffs, ff);
       end_if;
       if has(x, FAIL) then return(FAIL); end_if;
       return(new(dom, r, c, x, "FLAG"));
    end_proc;

/*--
    simplify  --  simplification of matrix elements
--*/
    simplify:= x -> dom::mapNonZeroes(x, simplify, args(2..args(0)));

/*--
   Simplify  -- dito, but using the new Simplifier
--*/
    Simplify:= x -> dom::mapNonZeroes(x, Simplify, args(2..args(0)));



/*--
    normal  --  normalization of matrix elements
--*/
    normal:= x -> if R::normal = FAIL then
                    x;
                  else
                    dom::mapNonZeroes(x, normal, args(2..args(0)));
                  end_if;

/*--
    _mod  --  mapping the function '_mod' to the matrix components
--*/

    _mod:= proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      A::dom::mapNonZeroes(A, _mod, n)
    end_proc;

/*--
    mods  --  mapping the function 'mods' to the matrix components
--*/

    mods:=
    proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      A::dom::mapNonZeroes(A, mods, n)
    end_proc;

/*--
    modp  --  mapping the function 'modp' to the matrix components
--*/

    modp:=
    proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      A::dom::mapNonZeroes(A, modp, n)
    end_proc;

/*-----------------------------------------------------------------------
  subs  --  'subs' for matrices
-----------------------------------------------------------------------*/
    subs:=
    proc(x)
    begin
      /* Bug-Fix: Vorher stand hier

         extsubsop(x,3= subs(extop(x,3),args(2..args(0))));

         Das funktionierte nicht fr den Fall
         v:= matrix([0,1,2]): subs(v, 0 = 7)
      */
      x::dom::map(x, subs, args(2..args(0)));
    end_proc;

/*-----------------------------------------------------------------------
  subsex  --  'subsex' for matrices
-----------------------------------------------------------------------*/
    subsex:=
    proc(x)
      // local polyList, i;
    begin
      // polyList:= extop(x, 3);
      // for i from 1 to extop(x, 2) do
      //   polyList[i]:= subsex(poly2list(polyList[i]), args(2..args(0)));
      //   polyList[i]:= poly(polyList[i], [hold(`#_X_`)], polyR);
      // end_for;
      // x:= extsubsop(x, 3 = polyList);
      // return(x);

      // Das obige funktionierte nicht fr den Fall
      // v:= matrix([0,1,2]): subsex(v, 0 = 7)

      x::dom::map(x, subsex, args(2..args(0)));

      // Old version of the code - based on substitution within the
      // polynomials:
      // extsubsop(x,3= subsex(extop(x,3),args(2..args(0))));
    end_proc;


/*-----------------------------------------------------------------------
  evalAt - the operator | for matrices
-----------------------------------------------------------------------*/
  evalAt:=
  proc(A)
  begin
    A::dom::map(A, evalAt, args(2..args(0)))
  end_proc;

/*-----------------------------------------------------------------------
  has  --  the function has for matrices

  the kernel function 'has' works as desired on lists of polys.
  However, has(.., `#_X_`) yields TRUE. Deal with this case by
  a warning to the user.
-----------------------------------------------------------------------*/

    has:= proc(x, y)
    local j;
    begin
       if contains({DOM_SET, DOM_LIST}, domtype(y)) then
          for j in y do
             if has(x, j) then 
                return(TRUE);
             end_if;
          end_for:
          return(FALSE);
       end_if;
       if y = hold(`#_X_`) then
          if has(indets(x), y) then
             warning("the special identifier `#_X_` is used for the ".
                     "internal representation of matrices");
             return(TRUE);
          else
             return(FALSE);
          end_if;
       end_if;

/*     // If 'has' is to be a semantical check, we
       // would need coercing:
       if Rcoerce <> id then
          y:= Rcoerce(y):
          if y = FAIL then return(FALSE): end_if;
       end_if:
*/
       if y = R::one then 
          // the element 1 will become poly(.. + 1*#_X_^j + ...)
          // and disappear, so we need a special treatment:
         
          // We do need coercing between the coeff ring 
          // and the internal poly representation (which 
          // is unknown to the user).
          if R2polyCoerce <> id then
             y:= R2polyCoerce(y):
             if y = FAIL then return(FALSE): end_if;
          end_if;
          return(has(map({op(extop(x,3))}, coeff),y));
       end_if;
       if y = R::zero then 
          // Zero entries are not stored explicitly, so we need
          // a special treatment:
          if dom::nonZeroes(x) < extop(x, 1)*extop(x, 2) then
             // there must be zero elements in the matrix
             return(TRUE);
          end_if:
          // check each column for zero elements:
          if R2polyCoerce <> id then
             y:= R2polyCoerce(y):
             if y = FAIL then return(FALSE): end_if;
          end_if;
          for j from 1 to extop(x, 2) do 
            if has( [coeff(extop(x,3)[j])], y ) then
               return(TRUE);
            end_if:
          end_for:
          return(FALSE):
       end_if;
       if domtype(y) = DOM_INT and 
          2 <= y and
          y <= extop(x, 1) then
          // Beware, y could be one of the exponents of the
          // internal components: has(poly(x^2), 2) = TRUE!
          return(has(map({op(extop(x,3))}, coeff),y));
       end_if;
       // All special cases are done. Here is the generic code
       // that just forwards has(.., y) to the internal polys.
       return(has(extop(x,3),y));
    end_proc;

/*-----------------------------------------------------------------------
  length  --  returns the lenght of a matrix

  The length of a matrix is defined by the length of its
  3rd operand, i.e. the length of an array.
-----------------------------------------------------------------------*/

length:= proc(x)
local r, c, l, j, list;
begin
  r:= extop(x,1);
  c:=extop(x,2);
  x:= extop(x,3);
  l:= 1;
  for j from 1 to c do
    x[j]:= poly2list(x[j]);
    for list in x[j] do
      l:= l + length(list[1]);
    end_for;
  end_for;
  return(l);
end_proc;

/*-----------------------------------------------------------------------
  abs        --  compute the absolute values of matrix entries
  conjugate  --  compute the complex conjugate of a matrix
  Re, Im     --  compute the real/imaginary part of a matrix
-----------------------------------------------------------------------*/

abs:= proc(x)
local r, c, t;
begin
  r:=extop(x,1); c:=extop(x,2);
  t:= map(extop(x,3), mapcoeffs, abs);
  if has(t, FAIL) then
     return(FAIL);
  else
     return(new(dom,r,c,t, "FLAG"));
  end_if:
end_proc;

conjugate:= proc(x)
local r, c, t;
begin
  r:=extop(x,1); c:=extop(x,2);
  t:= map(extop(x,3), mapcoeffs, conjugate);
  if has(t, FAIL) then
     return(FAIL);
  else
     return(new(dom,r,c,t, "FLAG"));
  end_if:
end_proc;


Re:= proc(x)
local r, c, t;
begin
  r:=extop(x,1); c:=extop(x,2);
  t:= map(extop(x,3), mapcoeffs, Re);
  if has(t, FAIL) then
     return(FAIL);
  else
     return(new(dom,r,c,t, "FLAG"));
  end_if:
end_proc;

Im:= proc(x)
local r, c, t;
begin
  r:=extop(x,1); c:=extop(x,2);
  t:= map(extop(x,3), mapcoeffs, Im);
  if has(t, FAIL) then
  else
     return(new(dom,r,c,t, "FLAG"));
  end_if:
end_proc;

// ----------- NOTE: --------------------------------------------
// time(mapcoeffs(t[1],conjugate)) <
// time(map(L, list -> [map(list[1],conjugate),list[2]]))
//
// In function conjugate the "List-Method" needs a little more time
// than the "poly-Method" (without having a look at the time
// needed by L:=poly2list(t[1]) and the later conversion from
// list to poly). Therefore it is more clever to use the "poly-Method".
//
// t:=[poly([ [(I-1)^i,i] $ i = 1..10000],[`#_X_`],Dom::Complex)]:
// time(mapcoeffs(t[1],conjugate));
// L:=poly2list(t[1]):
// time(map(L, list -> [map(list[1],conjugate),list[2]]));
// ------------------------------------------------------------------

/*------------------------------------------------------------------
random  --  create random matrices

Calls:
    random()
    random(f)
    random(m, n)
    random(m, n, f)
    random(m, n, nonzeros)
    random(m, n, nonzeros, f)

Details:
 random() --
     returns a random matrix. The dimension is randomly choosen
     in the interval [1..dom::randomDimen[1],1..dom::randomDimen[2]].
     The elements of the random matrix are created by the method
     "random" of R. If such a method does not exist, then FAIL is
     returned.
 random(f) --
     f is interpreted as a random generator and will
     be used to create the coefficients of the matrix.
     The dimension of the matrix will be chosen as above.
 random(m, n) --
     creates an (m x n)-matrix. Its coefficients are chosen
     as in random().
 random(m, n, f) --
     creates an (m x n)-matrix. The coefficients are
     created by using the random generator f.
 random(m, n, nonzeros) --
     nonzeros = a positive integer with 0 <= nonzeros <= m * n.
     Creates an (m x n)-matrix with at most nonzeros non-zero entries.
 random(m, n, nonzeros, f) --
     creates an (m x n)-matrix with at most nonzeros non-zero entries.
     The coefficients of the matrix are created by the random generator f.

Warning: the random generator f:
          i) must allow composition g@f with a function g
             (some domain objects do not have a _fconcat method!)
         ii) must produce values in R (the return values
             of f() are not automatically coerced to R)

Warning: this method is quite different to the random method
         of Dom::DenseMatrix:
          i) Dimensions and random generators can be specified
         ii) (Dom::DenseMatrix(R))::random does not exist if R::random
             does not exist. This is not the case for Dom::Matrix(R):
             the random method always exist. However, it may return
             FAIL, if R::random does not exist and no random generator
             is provided by the user.
------------------------------------------------------------------------*/
random:= proc()
local m, n, mn, j, i, rand, rrand, nonzeros, polyList, reverse,
       randop, randops, /*ops, randpos,*/ col, row, koeff;
begin
  //-----------------------------------------------
  // random()
  //-----------------------------------------------
  if args(0) = 0 then
     m := (random() mod dom::randomDimen[1]) + 1;
     n := (random() mod dom::randomDimen[2]) + 1;
     rand:= R::random;
     if rand = FAIL then return(FAIL) end_if;
     if R2polyCoerce <> id then
        rand:= R2polyCoerce@rand;
     end_if;
     polyList:= [poly([[rand(),i] $i=1..m],[hold(`#_X_`)],polyR) $ j=1..n]:
     if has(polyList, FAIL) then
       error("the random generator does not produce values ".
             "that can be converted to the coefficient ring.");
     end_if;
     return(new(dom, m, n, polyList, "FLAG"));
  end_if;
  //-------------------------------------------------------
  // random(rand), where rand is used as a random generator
  //-------------------------------------------------------
  if args(0) = 1 then
     m := (random() mod dom::randomDimen[1]) + 1;
     n := (random() mod dom::randomDimen[2]) + 1;
     rand:= args(1);
     if R2polyCoerce <> id then
        rand:= R2polyCoerce@rand;
     end_if;
     polyList:= [poly([[rand(),i] $i=1..m],[hold(`#_X_`)],polyR) $ j=1..n]:
     if has(polyList, FAIL) then
       error("the random generator does not produce values ".
             "that can be converted to the coefficient ring.");
     end_if;
     return(new(dom, m, n, polyList, "FLAG"));
  end_if;
  //-------------------------------------------------------
  // For all remaining cases, m, n is specified by the user
  //-------------------------------------------------------
  m:= args(1);
  if domtype(m) <> DOM_INT or m < 0 then
     error("illegal row dimension ".expr2text(m)." (expecting a positive integer)");
  end_if:
  n:= args(2);
  if domtype(n) <> DOM_INT or n < 0 then
     error("illegal column dimension ".expr2text(n)." (expecting a positive integer)");
  end_if:
  //-----------------------------------------------
  // random(m, n)
  //-----------------------------------------------
  if args(0) = 2 then
     rand:= R::random;
     if rand = FAIL then return(FAIL) end_if;
     if R2polyCoerce <> id then
        rand:= R2polyCoerce@rand;
     end_if;
     polyList:= [poly([[rand(),i] $i=1..m],[hold(`#_X_`)],polyR) $ j=1..n]:
     if has(polyList, FAIL) then
       error("the random generator does not produce values ".
             "that can be converted to the coefficient ring.");
     end_if;
     return(new(dom, m, n, polyList, "FLAG"));
  end_if;
  //-----------------------------------------------
  // random(m, n, rand) with a random generator rand
  //-----------------------------------------------
  if args(0) = 3 and domtype(args(3)) <> DOM_INT then
     rand:= args(3);
     if R2polyCoerce <> id then
        rand:= R2polyCoerce@rand;
     end_if;
     polyList:= [poly([[rand(),i] $i=1..m],[hold(`#_X_`)],polyR) $ j=1..n]:
     if has(polyList, FAIL) then
       error("the random generator does not produce values ".
             "that can be converted to the coefficient ring.");
     end_if;
     return(new(dom, m, n, polyList, "FLAG"));
  end_if;

  //------------------------------
  // Implement the following calls:
  //   random(m, n, nonzeros)
  //   random(m, n, nonzeros, rand)
  // with
  //   nonzeros = nonegative integer
  //   rand = a random generator
  //------------------------------

  //--------------
  // test nonzeros
  //--------------
  nonzeros:= args(3);
  if nonzeros < 0 or nonzeros > m*n then
     error("the requested number of non-zero entries ".
           "must be >= 0 and <= ".expr2text(m*n).
           " Requested: ".expr2text(nonzeros));
  end_if;

  //----------------
  // initialize rand
  //----------------
  if args(0) = 3 then
       rand:= R::random;
       if rand = FAIL then return(FAIL) end_if;
  elif args(0) = 4 then
       if domtype(args(3)) <> DOM_INT then
          error("third argument: expecting an integer (the number of random elements)");
       end_if;
       rand:= args(4);
  else error("expecting at most 4 arguments");
  end_if;
  if R2polyCoerce <> id then
     rand:= R2polyCoerce@rand;
  end_if;

  //-------------------------------------------------------------
  // If nonzeros is small, choose nonzero operands that are to be
  // filled in randomly. If nonzeros is large, it is more efficient
  // to choose those operands that should not be filled in.
  // Heuristically, we choose 'large' = 'larger than m*n/2'.
  // This breakeven point was found to be reasonable by playing
  // with (Dom::Matrix(Dom::Integer))::random(100, 100).
  //------------------------------------------------------------
  // Note that 0 <= nonzeros <= m*n
  mn:= m*n;
  if nonzeros <= mn/2 then
       reverse:= FALSE;
  else nonzeros:= mn - nonzeros;
       reverse:= TRUE;
  end;

  //---------------------------------
  // choose nonzero operands that are
  // to be filled in/removed
  //---------------------------------

  // Here, a simple version that is fast and
  // memory efficient.
  // *****************************************
  // * It is important that nonzeros < mn/2. *
  // *****************************************
  // Otherwise, many runs through the while loop
  // would be needed until nonzeros distinct
  // operands from the set {1,2,..,mn} of all
  // operands are chosen:

  // the following random generator r produces
  // uniformly distributed indices in the range 1..mn:

  rrand:= () -> 1 + specfunc::trunc(frandom()*mn);

  // choose random indices until there are exactly
  // nonzero disting indices in randops:
  randops:= {rrand() $ i=1..nonzeros};
  while nops(randops) < nonzeros do
     randops:= randops union {rrand() $ i=1..(nonzeros - nops(randops))};
  end_while;

  /*-------------------------------------------------------
  // Here is a more sophisticated version that
  // produces *uniformly distributed* subsets
  // of length nonzeros from {1, 2, .. , mn}:
  // It is reasonably fast, but a memory waster,
  // because a list of *all* operands is used.

  // Making this **fast** is a bit tricky:
  //   i) create a list ops = [1, .. , m*n] of all operands
  //  ii) choose a random index of this list
  // iii) 'delete' the chosen element from this list
  //      after storing it in in randops
  // Deletion is only done 'virtually', because real deletion
  // is **extremely** expensive for large lists.

  // ops = list of all operands
  // ***************************
  // * ops is a memory waster  *
  // * for nonzeros << m*n ;-( *
  // ***************************
  ops:= [i $ i=1..mn];

  // randops = container for all operands to be filled in
  // or (in reverse mode) to be removed
  randops:= [ 0 $ nonzeros ]:

  vl:= mn; // 'virtual length' of the list ops
  for i from 1 to nonzeros do
    // choose a random operand by choosing a random
    // position randpos with 1 <= randpos <= vl:
    randpos:= specfunc::trunc(frandom()*vl) + 1;
    // We want to remove the chosen operand from the list ops:
    // delete ops[randpos];
    // Well, do not do this, because deletion is **extremely**
    // expensive for large lists. It is much more efficient to
    // leave the original length of ops. Regard it as of
    // 'virtual length' vl by ignoring all elements after position vl.
    // We 'delete' the element at position randpos
    // by moving the 'last element' at position vl of the
    // 'virtual list' to the position randpos and shorten
    // the list 'virtually' be setting vl:= vl - 1.
    // Then, the positions 1 .. vl contain all indices from 1 .. m*n
    // that have not been chosen before. Note that the ordering
    // of the first vl elements in ops is irrelevant, because
    // the position randpos is chosen randomly, anyway.

    randops[i]:= ops[randpos]; // store the chosen operand

    // Finally, 'delete' the chosen operand by replacing it by
    // the 'last element' of the virtual list at position vl:
    ops[randpos]:= ops[vl];
    // the 'virtual length' of the list ops is decreased by 1:
    vl:= vl - 1;
  end_for;
  -------------------------------------------------------------*/

  //--------------------------------------------
  // fill in/delete the randomly chosen operands
  //--------------------------------------------
  // we have chosen all operands that are to be
  // filled in/deleted. They are stored in randops.
  if not reverse then
       // create empty columns
       polyList:= [poly(Rzero, [hold(`#_X_`)], polyR) $ n]:
       // Fill in the randomly chosen operands;
       // randops contains nonzeros *distinct* operands,
       // so the following only produces random entries
       // of the form 0 + koeff, i.e., all entries are
       // generated by rand() (without ever adding 2 such
       // elements)
       for randop in randops do
         // the operand randop corresponds
         // to the following matrix indices
         row := ((randop-1) div n) + 1;
         col := ((randop-1) mod n) + 1;
         // Store this randomly chosen operand in randops:
         koeff:= rand():
         polyList[col]:= polyList[col] +
                         poly([[koeff, row]], [hold(`#_X_`)], polyR);
       end_for;
  else // reverse mode: create a fully random matrix
       // and delete those (few) entries that should
       // not be filled in randomly:
       polyList:= [[[rand(), i] $i=1..m] $ j=1..n];
       // Remove some of the randomly chosen operands;
       for randop in randops do
         // the operand randop corresponds
         // to the following matrix indices
         row := ((randop-1) div n) + 1;
         col := ((randop-1) mod n) + 1;
         polyList[col][row] := [Rzero, row];
       end_for;
       polyList:= map(polyList, poly, [hold(`#_X_`)], polyR);
  end_if;

  if has(polyList, FAIL) then
     error("the random generator does not produce values ".
           "that can be converted to the coefficient ring.");
  end_if;
  return(new(dom, m, n, polyList, "FLAG"));
end_proc;

/*------------------------------------------------------------
  tr  --  compute the trace of a square matrix

  Syntax: tr(A)

  A  --  matrix

  Synopsis:

  A trace of a square matrix A is defined to be the sum of the
  diagonal elements of A.
------------------------------------------------------------*/
tr:= proc(x)
local n, i, R, t;
begin
  n := extop(x,1);
  if n <> extop(x,2) then
    error("expecting a square matrix")
  end_if;
  R:= x::dom::coeffRing;
  x := extop(x,3);
  t:= _plus(coeff(x[i],i) $i=1..n);
  return(R(t));
end_proc;

/*------------------------------------------------------------
  norm()  --  compute the norm of a matrix

  Syntax:

  norm(A)
  norm(A,Frobenius)
  norm(v,k)

  A                    --  matrix
  v                    --  vector
  k                    --  positive integer
  Frobenius, Infinity  --  identifier

  Synopsis:

  norm(A) and norm(A,Infinity) compute the infinity norm of a matrix.
  That is the maximum row sum, where the row sum is the sum of norms
  of each element.
  The infinity norm of a vector is the maximum norm of all
  elements.

  norm(A,1) computes the 1-norm of a matrix which is the maximum sum
  of the norm of the elements of each column.

  norm(A,Frobenius) computes the Frobenius norm of a matrix which is
  the square root of the sum of the squares of the norm of each element.

  norm(v,k) computes the k-norm of a vector for k >= 2.
  Dom norm is defined to be the kth root of the sum of the norm of each
  element raised to the kth power.
  For the 2-norm the function 'linalg::scalarProduct' will be used
  if it is defined. Otherwise the Euclidian scalar product of v will
  be returned, which is defined by

  v[1]*conjugate(v[1]) + v[2]*conjugate(v[2]) + ...

  If the coeffRing of v does not provide a method "conjugate"
  (which must return the complex conjugate of an element) then
  the sum

            v[1]*v[1] + v[2]*v[2] + ...

  will be returned.


  Dom method returns FAIL if the result is not longer an element
  of the coefficient ring of A and v, respectively.

  For all cases except of the 2-norm, the coefficient ring of the
  matrix has to provide the method 'norm', which result for any element
  has to be a number. FAIL will be returned if such a method not exists.

TESTS:

A1::dom::norm(A1,1);
A1::dom::norm(A1);
A1::dom::norm(A1,Infinity);
A1::dom::norm(A1,Frobenius);
v1:= mat1([[1,2,3,4]]):
v1::dom::norm(v1,2);
v2:= mat1([1,2,3,4]):
u1:= mat1([i $ i=1..100]):
u2:= mat1([[i $ i=1..100]]):
u3:= mat1([i/3 $ i=1..500]):
u4:= mat1([[i/3 $ i=1..500]]):
v2::dom::norm(v2,2);
v1::dom::norm(v1,3);
v2::dom::norm(v2,3);
v1::dom::norm(v1,4);
v2::dom::norm(v2,4);
u1::dom::norm(u1,2);
u2::dom::norm(u2,2);
u1::dom::norm(u1,3);
u2::dom::norm(u2,3);
u1::dom::norm(u1,4);
u2::dom::norm(u2,4);
u3::dom::norm(u3,2);
u4::dom::norm(u4,2);
u3::dom::norm(u3,3);
u4::dom::norm(u4,3);
u3::dom::norm(u3,4);
u4::dom::norm(u4,4);
m1:= mat1([7*i/3 $ i=1..5000]):
time(m1::dom::norm(m1,2));
m1::dom::norm(m1,2);
m2:=matrix([7*i/3 $ i=1..5000]):
time(m2::dom::norm(m2,2));
m2::dom::norm(m2,2);
------------------------------------------------------------*/

norm:= proc(x, norm = hold(Infinity))
local i, j, c, r, n, Rnorm, p, m, s, L, list, d, B, v, f;
begin
    if args(0) < 1 or args(0) > 2 then
      error("wrong no of args")
    end_if;
    if norm <> 2 then
        Rnorm := R::norm;
        if Rnorm = FAIL then
            userinfo(1,"the coefficient domain has no method \"norm\"");
            return( FAIL )
        end_if
    end_if;
    r := extop(x,1); c := extop(x,2);
    if domtype(norm) = DOM_FLOAT and iszero(frac(norm)) then
       norm:= round(norm);
    end_if:
    case norm
    of 1 do
        n := Rzero;
        x := extop(x,3);
        for j from 1 to c do
            p := x[j];
            m := nterms(p);
            n:= max(n,
                _plus( Rnorm(nthcoeff(p,i)) $i=1..m)
                );
        end_for;
        break;
    of hold(Infinity) do //Zeilensummennorm
    of hold(Maximum) do
        x := extop(x,3);
        s:= [ Rzero $ r]: // s[i] = i-te Zeilensumme
        for j from 1 to c do
            L:= poly2list(x[j]);
            for list in L do
                s[list[2]]:= s[list[2]] + Rnorm(list[1]);
            end_for:
        end_for:
        n:= max(op(s)):
    break;
    of hold(Frobenius) do //gleiche Vorgehensweise wie bei der
        // n := Rzero;       //Spaltensummennorm.
        x := extop(x,3);
        s:= Rzero;
        for j from 1 to c do
            L:= poly2list(x[j]);
            for list in L do
                s:= s + Rnorm(list[1])^2;
            end_for:
        end_for:
        n := sqrt(s);
        break;

    of 2 do
    of hold(Euclidean) do
    of hold(Spectral) do

        if min(r,c) <> 1 then
          B := linalg::checkForFloats(x);
          if B <> FALSE then
           return( max(numeric::singularvalues(x)) )
          end;

          d := linalg::matdim(x);
          if d[1] > d[2] then
            B := linalg::htranspose( x ) * x;
          else
            B := x * linalg::htranspose( x );
          end;

          v := linalg::eigenvalues( B );
          v:= map(v, abs); // to avoid roundoff trash
          if type(v) <> DOM_SET then
            return(sqrt(max(v)));
          end;

          v := [op(v)];
          if nops(v) = 1 then
             return(sqrt(abs(v[1])))
          end;
          // Note that B is positive semi-definite:
          // All eigenvalues are non-negative real numbers.
          // Converting the symbolic eigenvalues to reals
          // using float yields sometimes complex numbers.
          // Discard imaginary part, which is nearly zero:
          // v := [a_1 + b_1 * I, ..., a_n + b_n * I]
          // f := [a_1, ..., a_n]
          f := float(v);
          f := map(f, x -> if domtype(x) = DOM_COMPLEX then op(x,1)
                           elif domtype(x) = DOM_FLOAT then x
                           else FAIL end);
          if hastype(f, DOM_FAIL) then
             // A contains symbolic elements
            return( sqrt(max( op(map( v, abs ) ) ) ) );
          end;
          m := max(f);
          i := contains(f, m);
          return( sqrt( abs(v[i]) ) );
        end_if;
        x := extop(x,3);
        if R::conjugate <> FAIL then
            if r = 1 then
                n:= _plus( abs(coeff(x[i],1))^2 $ i=1..c );
            else
                L:= poly2list(x[1]);
                n:= Rzero;
                for list in L do
                    n:= n + abs(list[1])^2;
                end_for;
            end_if
        elif r = 1 then
            n:= _plus( coeff(x[i],1)^2 $ i=1..c );
        else // c = 1
            L:= poly2list(x[1]);
            n:= Rzero;
            for list in L do
                n:= n + abs(list[1])^2;
            end_for;
        end_if;
        n:= sqrt(n);
        break
    otherwise
    // compute the k-norm of vector x
    if not testtype( norm,Type::PosInt ) then
        error("invalid matrix norm")
    end_if;
    if r = 1 then
        x := extop(x,3);
        n := _plus( Rnorm(coeff(x[i],1))^norm $ i=1..c )^(1/norm);
    elif c = 1 then
        x := extop(x,3);
        L := poly2list(x[1]);
        n:= Rzero;
        for list in L do
            n:= n + (Rnorm(list[1])^norm);
        end_for;
        n:= n^(1/norm);
    else
        error("invalid matrix norm (expecting a vector)")
    end_if
    end_case;
    return(n);
end_proc;

/*------------------------------------------------------------------
exp  --  compute the exponential of a matrix

Syntax:

        exp(A <,x>)

        A:  square matrix
        x:  indeterminate

Synopsis:

        exp(A,x) computes the matrix exponential defined by

            exp(A, x) = I + A*x + 1/2!*A^2*x^2 + ...

        If the second argument is not given then 1 is set for x.
        When the eigenvalues of A can not be computed then an
        error message will be returned.

        When the sum of the algebraic multiplicities of the
        eigenvalues of A is not equal to the dimension of A
        then FAIL will be returned.

NOTE: Dom method uses the linalg functions
      linalg::jordanForm and linalg::eigenValues.
------------------------------------------------------------------*/

   exp:= if R::hasProp( Cat::Field )
   then
        /*--
            exp  --  compute the exponential of a matrix over a field

            Syntax:
            exp(A <,x>)

            A: square matrix
            x: element of R

            Synopsis:
            exp(A,x) computes the matrix exponential defined by
            exp(A,x) = I + A*x + 1/2!*A^2*x^2 + ...
            If the second argument is not given then 1 is set for x.
            When the eigenvalues of A can not be computed then an
            error message will be returned.

            Background:
            This method first computes the Jordan form of A such that
            A = 1/P*A*P. Then exp(A) = P*exp(J)*1/P, where exp(J) =
            ( exp(J[1]),...,exp(J[k]) ) with the k Jordan blocks J[i].
            Suppose that

                     [ 0, 1, 0 ]
              J[i] = [ 0, 0, 1 ],
                     [ 0, 0, 0 ]

            then we get:

                          [ 1, t, 1/2*t^2]
              exp(J[i]) = [ 0, 1,    t   ].
                          [ 0, 0,    1   ]
        --*/
    proc(A,t=R::one)
    local k, i, j, x, e, n, r, J, P, R_one,
          floatmode, row, types, B, b, Cols;
    begin
      if args(0) < 1 or args(0) > 2 then
         error("wrong no of args")
      end_if;
      n:= extop(A,1);
      if n <> extop(A,2) then
         error("expecting a square matrix")
      elif args(0) = 2 then
         if domtype(t) <> R then
            t:= Rcoerce( t );
            if t = FAIL then
               error("2nd argument is not compatible with the component ring");
            end_if
         end_if
      end_if;
      //---------------------------------------------------
      // does exp make sense mathematcially over this ring?
      //---------------------------------------------------
      if R::exp = FAIL and traperror(exp(R::one)) <> 0 then
         userinfo(1, "exp is not defined over the coefficient ring ".expr2text(R));
         return(FAIL);
      end_if;
     //-------------------------------
     floatmode:= FALSE;
     if R = Dom::Float then
        floatmode:= TRUE;
     elif polyR = Expr then
       //---------------------------------------------------
       // is there a float element in the matrix?
       // If so, try to convert all elements to floats.
       //---------------------------------------------------
       if domtype(t) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;
       if domtype(t) = DOM_COMPLEX and
          domtype(op(t, 2)) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;

       // If t is a float, there is no need
       // to check the matrix for floats
       if not floatmode then
         for i from 1 to n do
           row:= {coeff(extop(A, 3)[i])};
           types:= map(row, domtype):
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
           row:= select(row, c -> domtype(c) = DOM_COMPLEX);
           types:= map(row, domtype@op, 2); // domtypes of imaginary parts
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
         end_for;
       end_if;
       //---------------------------------------------------
       // There is a float element in the matrix. Try to convert
       // all elements to floats. We need a copy B of A, because
       // we still need the original A if no float conversion is
       // possible.
       //---------------------------------------------------
       if domtype(float(t)) <> DOM_FLOAT and
          domtype(float(t)) <> DOM_COMPLEX then
          floatmode := FALSE;
       end_if;

       if floatmode then
          B:= extop(A, 3);
          for i from 1 to n do
            B[i]:= mapcoeffs(B[i], float):
            types:= map({coeff(B[i])}, domtype):
            if types minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
               floatmode:= FALSE;
               break;
            end_if;
          end_for;
       end_if;
     end_if;
     //---------------------------------------------------
     // if floatmode, all elements of A were converted to floats.
     // Call numeric::expMatrix, which tries HardwareFloats!
     //---------------------------------------------------
     if floatmode then
        userinfo(1,"call 'numeric::expMatrix'");
        // Beware: Dom::Matrix(Dom::Float)(..) * I yields FAIL,
        // so we need to convert to a more general ring, first:
        A:= Dom::Matrix()(A);
        return(dom::convert(numeric::expMatrix(t*A)));
     end_if;

    //----------------------------
    //  Start of the symbolic code
    //----------------------------

         k:= linalg::jordanForm( A,hold(All) );
         if k = FAIL then
            userinfo(1,"linalg::jordanForm fails");
            return( FAIL )
         end_if;
         [J,P]:= k;
         R_one:= R::one;
         k:= 1;
         while k <= n do
            e:= J[k,k]; // eigenvalue of A
                        // Compute the exponential of a jordan block:
            x:= Rcoerce( exp(t*e) );
            if x = FAIL then
               userinfo(1,"cannot compute the exponential of a ring element");
               return( FAIL )
            end_if;
            // get the length of the corresponding Jordan chain:
            i:= k;
            while i < n and J[i,i] = e and J[i,i+1] = R_one do i:= i + 1 end_while;
            for r from k to i do
               J[r,r]:= x;
               if r > 1 and not iszero( J[r-1,r] ) then
                  e:= J[r,r] * J[r-1,r];
                  (J[r-j,r] := e * t^j/fact(j) ) $ j=1..r-k
               end_if
            end_for;
            k:= r
          end_while;
          //=========================================
          if n < 7 and max(map({P::dom::nonZeroOperands(P)},length)) > 100 then 
            Cols:= [0 $ n];
            for i from 1 to n do 
              b:= P::dom(n,1);
              b[i]:= 1;
              Cols[i]:= linalg::SSS(P,b)[1];
            end_for;  
            return(P*J*(P::dom::concatMatrix(op(Cols))));
          else   
            return(dom::_mult(P,J,dom::_invert(P)) )
          end_if;          
          //=========================================
      end_proc
   else
      // exp(A) = FAIL, if A is not defined over a 'Cat::Field':
      () -> FAIL
   end_if;

/*----------------------------------------------
sin -- the sine for matrices

defined over ExpressionField and the complex
numbers only


NOTE: The symbolic case is still slow and has
      to be improved!

-----------------------------------------------*/

sin:= proc(A,t=R::one)
   local i, n, floatmode, row, types, B;
   begin
   if R::hasProp( Cat::Field ) then
        /*--
        If there are floating point numbers within the
        matrix we try to compute sin(A) with the help
        'numeric::expMatrix'. The strategy of searching
        for floating point numbers within the matrix A 
        is the same as above in the procedure 'exp'.
        --*/
      if args(0) < 1 or args(0) > 2 then
         error("wrong no of args")
      end_if;
      n:= extop(A,1);
      if n <> extop(A,2) then
         error("expecting a square matrix")
      elif args(0) = 2 then
         if domtype(t) <> R then
            t:= Rcoerce( t );
            if t = FAIL then
               error("2nd argument is not compatible with the component ring");
            end_if
         end_if
      end_if;

      //---------------------------------------------------
      // does sin make sense mathematcially over this ring?
      //---------------------------------------------------
      if R::sin = FAIL and traperror(sin(R::one)) <> 0 then
         userinfo(1, "sin is not defined over the coefficient ring ".expr2text(R));
         return(FAIL);
      end_if;

     //-------------------------------
     floatmode:= FALSE;
     if R = Dom::Float then
        floatmode:= TRUE;
     elif polyR = Expr then
       //---------------------------------------------------
       // is there a float element in the matrix?
       // If so, try to convert all elements to floats.
       //---------------------------------------------------
       if domtype(t) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;
       if domtype(t) = DOM_COMPLEX and
          domtype(op(t, 2)) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;

       // If t is a float, there is no need
       // to check the matrix for floats
       if not floatmode then
         for i from 1 to n do
           row:= {coeff(extop(A, 3)[i])};
           types:= map(row, domtype):
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
           row:= select(row, c -> domtype(c) = DOM_COMPLEX);
           types:= map(row, domtype@op, 2); // domtypes of imaginary parts
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
         end_for;
       end_if;
       //---------------------------------------------------
       // There is a float element in the matrix. Try to convert
       // all elements to floats. We need a copy B of A, because
       // we still need the original A if no float conversion is
       // possible.
       //---------------------------------------------------
       if domtype(float(t)) <> DOM_FLOAT and
          domtype(float(t)) <> DOM_COMPLEX then
          floatmode := FALSE;
       end_if;

       if floatmode then
          B:= extop(A, 3);
          for i from 1 to n do
            B[i]:= mapcoeffs(B[i], float):
            types:= map({coeff(B[i])}, domtype):
            if types minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
               floatmode:= FALSE;
               break;
            end_if;
          end_for;
       end_if;
     end_if;
     //---------------------------------------------------
     // if floatmode, all elements of A were converted to floats.
     // Call numeric::expMatrix, which tries HardwareFloats!
     //---------------------------------------------------
     if floatmode then
        userinfo(1,"call 'numeric::expMatrix'");
        // Beware: Dom::Matrix(Dom::Float)(..) * I yields FAIL,
        // so we need to convert to a more general ring, first:
        A:= Dom::Matrix()(A);
        return(dom::convert(map(
                 (1/2 * I) * (numeric::expMatrix(-I*A) - numeric::expMatrix(I*A)),
                 numeric::complexRound
               )));
     end_if;

     if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then
       return(map((1/2 * I) * (exp(-I*A) - exp(I*A)), rewrite, sincos))
     end_if;
   else
      // exp(A) = FAIL, if A is not defined over a 'Cat::Field':
      return(FAIL);
   end_if;
   end_proc;


/*---------------------------------------------
cos -- the cosine for matrices

defined over ExpressionField and the complex
numbers only


NOTE: The symbolic case is still slow and has
      to be improved!

-----------------------------------------------*/

cos:= proc(A,t=R::one)
   local i, n, floatmode, row, types, B;
   begin
   if R::hasProp( Cat::Field ) then
        /*--
        If there are foating point numbers within the
        matrix we try to compute sin(A) with the help
        'numeric::exp'. The strategy of searching for
        floating point numbers within the matrix A is
        the same as above in the procedure 'exp'.
        --*/
      if args(0) < 1 or args(0) > 2 then
         error("wrong no of args")
      end_if;
      n:= extop(A,1);
      if n <> extop(A,2) then
         error("expecting a square matrix")
      elif args(0) = 2 then
         if domtype(t) <> R then
            t:= Rcoerce( t );
            if t = FAIL then
               error("2nd argument is not compatible with the component ring");
            end_if
         end_if
      end_if;

      //---------------------------------------------------
      // does cos make sense mathematcially over this ring?
      //---------------------------------------------------
      if R::cos = FAIL and traperror(cos(R::one)) <> 0 then
         userinfo(1, "cos is not defined over the coefficient ring ".expr2text(R));
         return(FAIL);
      end_if;

     //-------------------------------
     floatmode:= FALSE;
     if R = Dom::Float then
        floatmode:= TRUE;
     elif polyR = Expr then
       //---------------------------------------------------
       // is there a float element in the matrix?
       // If so, try to convert all elements to floats.
       //---------------------------------------------------
       if domtype(t) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;
       if domtype(t) = DOM_COMPLEX and
          domtype(op(t, 2)) = DOM_FLOAT then
          floatmode := TRUE;
       end_if;

       // If t is a float, there is no need
       // to check the matrix for floats
       if not floatmode then
         for i from 1 to n do
           row:= {coeff(extop(A, 3)[i])};
           types:= map(row, domtype):
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
           row:= select(row, c -> domtype(c) = DOM_COMPLEX);
           types:= map(row, domtype@op, 2); // domtypes of imaginary parts
           if has(types, DOM_FLOAT) then
              floatmode := TRUE;
              break;
           end_if;
         end_for;
       end_if;
       //---------------------------------------------------
       // There is a float element in the matrix. Try to convert
       // all elements to floats. We need a copy B of A, because
       // we still need the original A if no float conversion is
       // possible.
       //---------------------------------------------------
       if domtype(float(t)) <> DOM_FLOAT and
          domtype(float(t)) <> DOM_COMPLEX then
          floatmode := FALSE;
       end_if;

       if floatmode then
          B:= extop(A, 3);
          for i from 1 to n do
            B[i]:= mapcoeffs(B[i], float):
            types:= map({coeff(B[i])}, domtype):
            if types minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
               floatmode:= FALSE;
               break;
            end_if;
          end_for;
       end_if;
     end_if;
     //---------------------------------------------------
     // if floatmode, all elements of A were converted to floats.
     // Call numeric::expMatrix, which tries HardwareFloats!
     //---------------------------------------------------
     if floatmode then
        userinfo(1,"call 'numeric::expMatrix'");
        // Beware: Dom::Matrix(Dom::Float)(..) * I yields FAIL,
        // so we need to convert to a more general ring, first:
        A:= Dom::Matrix()(A);
        return(dom::convert(map(
                 1/2 * (numeric::expMatrix(-I*A) + numeric::expMatrix(I*A)),
                 numeric::complexRound
               )));
     end_if;

     if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then
       return(map(1/2 * (exp(-I*A) + exp(I*A)), rewrite, sincos))
     end_if;
   else
      // exp(A) = FAIL, if A is not defined over a 'Cat::Field':
      return(FAIL);
   end_if;
   end_proc;

/*---------------------------------------------------------------
zip  --  overload the function zip for matrices
---------------------------------------------------------------*/

zip:= proc(A,B,f,d)
local r, c, ff, f0,
      coeffListA, coeffListB, inA, inB,
      rowindicesA, rowindicesB, nthtermA, nthtermB,
      coeffTableNew, i, t,
      polyListA, polyListB,
      column, row, coeffListNew, polyListNew;
begin
   if args(0) <= 2 or args(0) >= 5 then
      error("expecting three or four arguments");
   end_if;
   if domtype(B) <> dom and (B:= dom::coerce(B)) = FAIL then
      userinfo(1, "operands are not compatible");
      return(FAIL);
   end_if;

   // The lists of polynomials:
   polyListA:= extop(A,3);
   polyListB:= extop(B,3);
   // Consistency to Dom::DenseMatrix()! Choose minimum auf
   // row and column dimensions for the dimension of the
   // new matrix.
   // NOTE: The case, that the fourth argument (the default
   //       value) will be needed, can never appear.
   r:= min(extop(A,1), extop(B,1));
   c:= min(extop(A,2), extop(B,2));

   if R = polyR or R = Dom::ExpressionField() then
        ff:= f;
   elif R2polyCoerce = id then
        ff:= (x, y) -> f(Rcoerce(x), Rcoerce(y));
   else ff:= (x, y) -> R2polyCoerce(f(Rcoerce(x), Rcoerce(y)));
   end_if;
   if f <> _divide and traperror((f0:= f(R::zero, R::zero))) = 0 then
      f0:= R2polyCoerce(f0);
   else
      f0:= FAIL;
   end_if;

   // Initialize the list for the polynomials ( = columns of
   // the output matrix)
   polyListNew:= [0 $ c];

   for column from 1 to c do
     coeffListA:= poly2list(polyListA[column]);
     coeffListB:= poly2list(polyListB[column]);
     coeffTableNew:= table():

     // rowindicesA = set of all row indices in col(A, column)
     // rowindicesB = set of all row indices in col(B, column)
     rowindicesA:= {op(map(coeffListA, op, 2))};
     rowindicesB:= {op(map(coeffListB, op, 2))};
     nthtermA:= table():
     for i from 1 to nops(coeffListA) do
       nthtermA[coeffListA[i][2]]:= i;
     end_for;
     nthtermB:= table():
     for i from 1 to nops(coeffListB) do
       nthtermB[coeffListB[i][2]]:= i;
     end_for;

     for row in rowindicesA union rowindicesB do
         // A or B may have larger row dimension than the new matrix
         if row > r then next; end_if;
         inA:= contains(rowindicesA, row) ;
         inB:= contains(rowindicesB, row) ;
         if inA and inB then
            coeffTableNew[row] := [ff(coeffListA[nthtermA[row]][1],
                                      coeffListB[nthtermB[row]][1]), row];
         elif inA then // not inB
            coeffTableNew[row] := [ff(coeffListA[nthtermA[row]][1], Rzero), row];
         elif inB then // not inA
            coeffTableNew[row] := [ff(Rzero, coeffListB[nthtermB[row]][1]), row];
         end_if;
     end_for:

     // Convert table to list;  fill in with f0 = f(Rzero,Rzero)
     // if this value is not trivial
     if iszero(f0) then
          coeffListNew:= map([op(coeffTableNew)], op, 2);
     else coeffListNew:= [[f0, i] $ i=1..r];
          for t in coeffTableNew do
            coeffListNew[op(t,1)]:= op(t,2);
          end_for;
     end_if;
     polyListNew[column]:= poly(coeffListNew, [hold(`#_X_`)], polyR);
   end_for; //for column from 1 to c do
   return(new(dom, r, c, polyListNew, "FLAG"));
end_proc;

/*----------------------------------------------------------------------
        expr2text  --  convert matrix to a string
----------------------------------------------------------------------*/

expr2text:= proc(A)
local L, m, n;
begin
  L:= dom::convert_to(A, DOM_LIST):
  [m, n]:= A::dom::matdim(A):
  if m = 0 and n > 0 then // the dimension cannot be encoded in the form of
                          // an empty list of the call matrix([]), we need
                          // to encode the dimension explicitly
     if R = Dom::ExpressionField() then
          return("matrix(0, ".expr2text(n).", ".expr2text(L).")");
     else return(expr2text(dom)."(0, ".expr2text(n).", ".expr2text(L).")");
     end_if;
  end_if;
  if R = Dom::ExpressionField() then
       return("matrix(".expr2text(L).")");
  else return(expr2text(dom)."(".expr2text(L).")");
  end_if;
end_proc;

/* ---------  use sparse output? ------------
expr2text := proc(x)
   local r, c, t;
   begin
      [r, c, t]:= [expr2text(extop(x, 1)),
                   expr2text(extop(x, 2)),
                   [op(x::dom::convert_to(x, DOM_TABLE))]
                  ];
      t:= sort(t, (x, y) -> if op(x, [1, 1]) < op(y, [1, 1]) then
                                 TRUE;
                            elif op(x, [1, 1]) = op(y, [1, 1]) and
                                 op(x, [1, 2]) < op(y, [1, 2]) then
                                 TRUE;
                            else FALSE;
                            end);

      if R = Dom::ExpressionField() then
           "matrix(".r.",".c.",".expr2text(t).")";
      else expr2text(dom::key)."(".r.",".c.",".expr2text(t).")";
      end_if;
  end_proc;
---------------------------------- */

/*----------------------------------------------------------------------
    op  --  overload the function op to work in the
            same way as the kernel op acts on arrays
----------------------------------------------------------------------*/

op:= proc(x,i)
    local r, c, t, j, k, term, lo, hi;
    begin
    if args(0) < 1 then
       error("expecting at least one argument"):
    end_if;

    r:= extop(x, 1);
    c:= extop(x, 2);
    x:= extop(x, 3);

    if args(0) = 1 then
       // extract *all* coefficients
       t:= [Rzero $ r*c];
       x:= map(x, poly2list);
       for j from 1 to c do
           // consider the j-th column x[j]
           for term in x[j] do
               t[c*(term[2] - 1) + j]:= term[1];
           end_for:
       end_for;
       if (R <> polyR) and
          (R <> Dom::ExpressionField()) then
           t:= map(t, Rcoerce);
       end_if;
       return(op(t));
    end_if;

    if domtype(i) = DOM_INT then
        if i < 0 or i > r*c then
           return(FAIL)
        end_if;
        if i = 0 then
           return((2, 1..r, 1..c));
        end_if;
        k:= i mod c;
        if k = 0 then
            k:= c;
        end_if;
        t:= coeff(x[k], ((i-1) div c) + 1);
        if R = polyR then return(t);
        elif R = Dom::ExpressionField() then return(t);
        else return(Rcoerce(t));
        end_if;
    end_if;
    if type(i) = "_range" then
        lo:= op(i,1);
        hi:= op(i,2);
        /* Modification by Nicolas Thiery (4/12/2001)
           to allow for 0xn and nx0 matrices.
           This also adds range checking for nxm matrices.
           Maybe a partial range should be returned instead of FAIL ?
        */
        if lo<1 or lo>r*c or hi<1 or hi>r*c then
            return(FAIL)
        end_if;
        t:= [ 0 $ hi-lo+1];
        for j from lo to hi do
            k:= j mod c;
            if k = 0 then
                k:= c;
            end_if;
            t[j-lo+1]:= coeff(x[k], ((j-1) div c) + 1);
        end_for;
        if (R <> polyR) and
           (R <> Dom::ExpressionField()) then
           t:= map(t, Rcoerce);
        end_if;
        return(op(t));
    end_if;
    if domtype(i) = DOM_LIST then
        /* Modification by Nicolas Thiery (4/12/2001)
           to allow for 0xn and nx0 matrices.
           This also adds range checking for nxm matrices.
        */
        if i[1] < 0 or i[1] > r*c then return(FAIL) end_if;
        if i[1] = 0 then
           if nops(i) = 1 then
              return((2, 1..r, 1..c))
           else
              return(op([2, 1..r, 1..c], [i[k] $ k = 2..nops(i)]));
           end_if;
        end_if;
        k:= i[1] mod c;
        if k = 0 then k:= c; end_if;
        t:= coeff(x[k], ((i[1]-1) div c) + 1);
        delete i[1];
        if R = polyR then return(op(t,i));
        elif R = Dom::ExpressionField() then return(op(t,i));
        else return(op(Rcoerce(t),i));
        end_if;
    end_if;
    FAIL;
end_proc;

/*---------------------------------------------------------------------
    subsop  --  overload the function op as it operates on arrays
---------------------------------------------------------------------*/

subsop:= proc(x /*, eq1, eq2, .. */ )
local r, c, eq, i, j, k, oldCoeff, newValue;
begin
  r:= extop(x,1);
  c:= extop(x,2);
  x:= extop(x,3);
  for j from 2 to args(0) do
    // process the j-th substitution equation passed to subsop
    eq:= args(j);    // eq:          i      = newValue
                     // oder [i1, i2, ...]  = newValue
    if type(eq) <> "_equal" then
       error("expecting a substitution equation")
    end_if;
    i:= op(eq,1);
    // special case subsop(...,  [i] = newValue).
    // In this case, the list case below crashes,
    // because subsop(..., []) is delegated to the
    // coefficients and the usual subsop does not allow this.
    // Just convert '[i] = newValue' to 'i = newValue'
    if domtype(i) = DOM_LIST and nops(i) = 1 then
       i:= op(i);
    end_if;
    case domtype(i)
    of DOM_INT do // eq:  i = newVAlue
       if i < 1 or i > r*c then return(FAIL) end_if;
       // The i-th operand is found in the folliwing k-th column:
       k:= i mod c;
       if k = 0 then k:= c; end_if;
       newValue:= R2polyCoerce(Rcoerce(op(eq,2)));
       // subtract oldValue
       x[k]:= x[k] + poly([[-coeff(x[k],((i-1) div c)+1),((i-1) div c)+1]],
                          [hold(`#_X_`)], polyR);
       // add newValue
       x[k]:= x[k] + poly([[newValue,((i-1) div c)+1]], [hold(`#_X_`)], polyR);
       break;
     of DOM_LIST do // eq:  [i1, i2, ... ] = newValue
       // the list has at least 2 operands
       if i[1] < 1 or i[1] > r*c then return(FAIL) end_if;
       k:= i[1] mod c;
       if k = 0 then k:= c; end_if;
       oldCoeff:= coeff(x[k],((i[1]-1) div c)+1);
       oldCoeff:= Rcoerce(oldCoeff);
       newValue:= subsop(oldCoeff, [op(i, 2..nops(i))] = op(eq, 2));
       newValue:= R2polyCoerce(newValue);
       // subtract oldValue
       x[k]:= x[k] + poly([[-coeff(x[k],((i[1]-1) div c)+1),((i[1]-1) div c)+1]],
                          [hold(`#_X_`)], polyR);
       // add newValue
       x[k]:= x[k] + poly([[newValue,((i[1]-1) div c)+1]], [hold(`#_X_`)], polyR);
       break;
     otherwise
       error("illegal substitution equation");
     end_case;
  end_for;
  return(new(dom, r, c, x, "FLAG"));
end_proc;


/*---------------------------------------------------------------------
    nops  --  overload the function nops as it operates on arrays
---------------------------------------------------------------------*/

nops:= x -> extop(x, 1)*extop(x, 2);

/*---------------------------------------------------------------------
    expand  --  overload the function expand
---------------------------------------------------------------------*/

expand:= proc(x)
begin
  return(new(dom,extop(x,1),extop(x,2),map(extop(x,3), mapcoeffs, expand, args(2..args(0))), "FLAG"));
end_proc;

/*---------------------------------------------------------------------
    diff --  overload the function diff
---------------------------------------------------------------------*/

diff:= proc(x)
begin
   return(new(dom,extop(x,1),extop(x,2),map(extop(x,3), mapcoeffs, diff, args(2..args(0))), "FLAG"));
end_proc;


/*---------------------------------------------------------------------
    int --  overload the function int  //new in 3.x x \in {?}
---------------------------------------------------------------------*/

int:= proc(x)
begin
   return(new(dom,extop(x,1),extop(x,2),map(extop(x,3), mapcoeffs, int, args(2..args(0))), "FLAG"));
end_proc;

/*---------------------------------------------------------------------
    identity(n) -- return the n x n identity matrix
---------------------------------------------------------------------*/

identity:=
if R::hasProp(Cat::Ring) then
    proc(n)
      local j;
    begin
        if args(0) > 2 or args(0) = 0 then
            error("expecting one argument")
        else
            if not ( testtype( n,DOM_INT ) and n > 0 ) then
                error("expecting a positive integer")
            end_if
        end_if;
        return( new(dom,n,n,[poly([[Rone,j]],[hold(`#_X_`)],polyR) $ j=1..n], "FLAG") );
    end_proc
else return(FAIL);
end_if;

/*-----------------------------------------------------------------------
    factor --  overload the function factor
-----------------------------------------------------------------------*/

factor:= if R::hasProp(Cat::GcdDomain) then
        proc(A)
            local r, c, x, L, j, g;
        begin
            //Strategy: Convert all polynomials to lists,
            //then extract the coefficients, i.e. the
            //first entry of each sublist. Among all
            //these coefficients compute the greatest
            //common divisor.
            r:= extop(A,1);
            c:= extop(A,2);
            x:= extop(A,3);
            L:= [0 $ c]; //The list, which will contain
                         //all coefficients distinct from zero
                         //Note: Sparsity will be kept in this way!
            for j from 1 to c do
                x[j]:= poly2list(x[j]);
                L[j]:= map(x[j], list -> list[1]);
                //Now L[j] contains all coefficients of the
                //polynomial x[j]
            end_for;
            L:= map(L, list -> op(list)); //converting the list of
                                          //L=[[[coeff_1],...,[coeff_n]]...] to
                                          //L=[[coeff_1,...,coeff_n]...]
            //caculate the gcd of all coefficients

            if R = Dom::ExpressionField(normal, iszero@normal) then
               g:= R(gcd(map(op(L), expr)));
            else
               g:= gcd(op(L));
            end_if;

            if g = Rone or g = FAIL then
               return( Factored::create( [1,A,1], "unknown") )
            elif g = Rzero then
               return( Factored::create( [1,
                                new(dom,r,c,[poly(Rzero, [hold(`#_X_`)], polyR) $ c], "FLAG"),
                                1], "unknown") )
            else
                x:= extop(A,3);
                if R = Dom::ExpressionField() then
                   for j from 1 to c do
                       //divide all coefficients by g = gcd(..)
                       x[j]:= mapcoeffs(x[j],Rnormal@_divide,g);
                   end_for;
                else
                   for j from 1 to c do
                      //divide all coefficients by g = gcd(..)
                      x[j]:= mapcoeffs(x[j],_divide,g);
                   end_for;
                end_if;
                A:= new(dom,r,c,x, "FLAG");
                //now return the matrix B = g*A
                return(Factored::create( [g,A,1],"unknown" ))
            end_if
        end_proc:
end_if;

/*-----------------------------------------------------------------------
evalp --  overload the function evalp

Calls:
    evalp(A, x = xValue, y = yValue, ...)

Parameters:
    A       - some Dom::Matrix(SomePolyDomain)(...)
  x, y, ..  - some indeterminate of the elements of SomePolyDomain
 xValue, .. - some elements of the SomePolyDomain::coeffRing
              or objects that can be converted to SomePolyDomain::coeffRing

ReturnValue:
    If *all* entries of A are reduced to some elements of
    the coefficient ring SomePolyDomain::coeffRing, then
    Dom::Matrix(SomePolyDomain::coeffRing)(...) is
    returned.
    Otherwise, Dom::Matrix(SomePolyDomain)(...) is
    returned where all elements are evaluated via evalp.

-----------------------------------------------------------------------*/
evalp:= if R::hasProp(Cat::Polynomial) then
    proc(A)
        local r, c, x, w, L, j, y, list, s;
    begin
        r:= extop(A,1);
        c:= extop(A,2);
        x:= extop(A,3);
        w:= args(2..args(0));
        x:= map(x, mapcoeffs, evalp, w);
        L:= [0 $ c]:
        for j from 1 to c do
            L[j]:= poly2list(x[j]);
        end_for;
        // Check whether the matrix now consists only of components
        // of the coefficient ring of the polynomial domain:

        // L = [ [[p11, 1], [p21, 2], ...],
        //       [[p12, 1], [p22, 2], ...],
        //       ...
        //     ]
        y:= map(L, map, list -> list[1]);
        // y = [ [p11, p21, ...],
        //       [p12, p22, ...],
        //       ...
        //     ]
        y:= map(y, map, degree);
        // y = [ [degree(p11), degree(p21), ...],
        //       [degree(p12), degree(p22), ...],
        //       ...
        //     ]
        y:= map(y, list -> op(list));
        // y = [ degree(p11), degree(p21), ...,
        //       degree(p12), degree(p22), ...,
        //       ...
        //     ]
        y:= map(y, iszero);

        if contains(y, FALSE) <> 0 then
            // Some of the internal polynomials survived (i.e.,
            // have degree > 0). The resulting matrix is again
            // a matrix over the polynomial ring R.
            return(new(dom, r, c, x, "FLAG"))
        else
            // define the matrix over R::coeffRing:
            // convert the encapsulating polynomials
            // from R ( = polynomial ring) to R::coeffRing.
            // We do not want to replicate the strategy
            // R::coeffRing -> polyR here, so we do not use
            // new(Dom::Matrix(R::coeffRing), r, c, [polys])
            // but Dom::Matrix(R::coeffRing)(...), which
            // takes care of the necessary coercions of the input data.
            // Unfortunately, x is a list of column, whereas
            // Dom::Matrix(R::coeffRing)::new accepts
            // only lists of rows as input data.
            // So we have to do a 'transposition' of the nested
            // list x. The following is the equivalent of the code
            // of the transpose method (sparse case):

            // compute s = transpose(L)
            // L = [ [[p11, 1], [p21, 2], ...],
            //       [[p12, 1], [p22, 2], ...],
            //       ...
            //     ]
            s:= [[0 $ c] $ r];  // container for the rows
            for j from 1 to c do
              // L[j] = [[p1.j, 1], [p2.j, 2], .., [p.r.j, r]]
              for list in L[j] do
                // list = [p.i.j, i]
                // fill the i-th row, where i = list[2]
                s[list[2]][j]:= expr(list[1]);
              end_for;
            end_for;
            return(Dom::Matrix(R::coeffRing)(s));
        end_if;
    end_proc
end_if;

/*-------------------------------------------------------------------
kroneckerProduct --  Kronecker product of 2 matrices

Calls:      kroneckerProduct(A, B, ...)
Parameters: A, B - matrices of the same type Dom::Matrix(R)
ReturnValue:  a Dom::Matrix(R) of dimension mA*mB x nA*nB, where
              [mA, nA] = A::dom::matdim(A),
              [mB, nB] = B::dom::matdim(B).
Details:
 * The Kronecker product AB of A and B is the matrix

     [ A[1,1]*B , A[1,2]*B, ...]
     [ A[2,1]*B , A[2,2]*B, ...]
     [   ...    ,   ...   , ...]

   In components:

     AB[mB*(iA - 1) + iB, nB*(jA - 1) + jB] =  A[iA, jA] * B[iB, jB],

   with  iA = 1..mA, iB = 1..mB, jA = 1..nA, jB = 1..nB.

 * This method should be called by a new linalg function
   linalg::kroneckerProduct(A, B) that should convert B
   to A::dom or A to B::dom and then calls
   A::dom::kroneckerProduct(A, B) or
   B::dom::kroneckerProduct(A, B).
-------------------------------------------------------------------*/

kroneckerProduct := proc(A, B)
    local ma, na, mb, nb, a, polyVar, polyR, AB, ja, aa, jb;
  begin
   if args(0) < 1 then
      error("expecting at least one argument"):
   end_if;
   if args(0) = 1 then
      return(A);
   end_if;
   if args(0) > 2 then
      return(dom::kroneckerProduct(A::dom::kroneckerProduct(A, B), args(3..args(0))));
   end_if;
   if A::dom <> B::dom then
      return(FAIL);
   end_if;
   [ma, na]:= A::dom::matdim(A);
   [mb, nb]:= B::dom::matdim(B);
   a:= extop(A, 3):
   B:= extop(B, 3):
   polyVar := extop(a[1], 2); // = [hold(`#_X_`)]
   polyR:= extop(a[1], 3);    // the internal poly ring
   //Initialize container AB for the columns of the product
   AB:= [poly([], polyVar, polyR) $ na*nb];
   for ja from 1 to na do
     if iszero(a[ja]) then
        next;
     end_if;
     aa:= poly2list(a[ja]):
     aa:= map(aa, L -> [L[1], mb*(L[2] - 1)]);
     aa:= poly(aa, polyVar, polyR);
     for jb from 1 to nb do
       if iszero(B[jb]) then
          next;
       end_if;
       assert(nb*(ja - 1) + jb <= na*nb);
       AB[nb*(ja -1) + jb] := aa*B[jb];
     end_for;
   end_for;
   new(A::dom, ma*mb, na*nb, AB, "FLAG");
 end_proc;

// -------------------------------------------------------------------
// (inv)fourier:  overload transform::(inv)fourier
// (inv)laplace:  overload transform::(inv)laplace
// (inv)ztrans:   overload transform::(inv)ztrans
// -------------------------------------------------------------------

fourier := A -> A::dom::mapNonZeroes(A, transform::fourier, args(2..args(0)));
invfourier := A -> A::dom::mapNonZeroes(A, transform::invfourier, args(2..args(0)));
laplace := A -> A::dom::mapNonZeroes(A, transform::laplace, args(2..args(0)));
invlaplace := A -> A::dom::mapNonZeroes(A, transform::invlaplace, args(2..args(0)));
ztrans := A -> A::dom::mapNonZeroes(A, transform::ztrans, args(2..args(0)));
invztrans := A -> A::dom::mapNonZeroes(A, transform::invztrans, args(2..args(0)));

// -------------------------------------------------------------------
// sqrt:  overload 'sqrt'
// -------------------------------------------------------------------

sqrt:= A -> linalg::sqrtMatrix(A,args(2..args(0)));

/*

PROCEDURE:    matrixShape (undocumeted utility method
              created for 'linalg::eigenvalues' computation)

CALLS:        matrixShape(A, <Diagonal>)
              matrixShape(A, <Multiple>)
   
PARAMETERS:   A -- a square matrix of domain Dom::Matrix

RETURN VALUE: matrixShape(A) returns 
                -- "diagonal" if A is a diagonal matrix
                -- "uppertriangular" if A is an upper 
                   triangular matrix
                -- "lowertriangular" if A is a lower 
                   triangular matrix  
                -- 'FAIL' otherwise

              matrixShape(A,Diagonal) returns 
                -- the set containing the diagonal entries 
                   of A

              matrixShape(A,Multiple) returns 
                -- a list of list, i.e. for each diagonal 
                   element d appearing m times on the main 
                   diagonal of the matrix A a list [d,m] is 
                   created. All such lists of two elements 
                   are comprised in a list.
                  
*/

matrixShape:= proc(A)
  local uppertriangular, lowertriangular, L, m, n, j;
begin
  uppertriangular:= TRUE;
  lowertriangular:= TRUE;
  L:= map(extop(A,3), p -> degree(p));
  m:= extop(A,1);
  n:= extop(A,2);
  for j from 1 to n do 
    if L[j] > j then 
      uppertriangular:= FALSE;
      break;
    end_if; 
  end_for;
  L:= map(extop(A,3), p -> poly2list(p));
  for j from 1 to n do 
    if L[j] <> [] and L[j][-1][2] < j then 
      lowertriangular:= FALSE;
      break;
    end_if; 
  end_for;
  if args(0) = 1 then 
    if uppertriangular and lowertriangular then 
      return("diagonal");
    elif uppertriangular then 
      return("uppertriangular") ;
    elif lowertriangular then 
      return("lowertriangular");
    end_if;
  elif (uppertriangular or lowertriangular) and args(0) = 2 then 
    if args(2) = Diagonal then 
      return({A[j,j] $ j = 1..min(m,n)});   
    elif args(2) = Multiple then 
      return([op(Dom::Multiset(A[j,j] $ j = 1..min(m,n)))]);   
    end_if;
  end_if;
  
  return(FAIL);
end_proc;

//---------------------------------
// Free module methods
//---------------------------------

    multcoeffs := dom::_mult; // could be optimized

    mapcoeffs := map;

//---------------------------------------------------
//---------------------------------------------------
/* the body of the domain */
//---------------------------------------------------
//---------------------------------------------------

begin
    if args(0) > 1 then error("wrong no of args") end_if;
    if args(0) = 0 then R := Dom::ExpressionField()
    elif R::hasProp( Cat::Rng ) <> TRUE then
        error("coefficients must be from a Cat::Rng")
    end_if;

    // Default case: the internal polynomials use the
    // same ring as the matrix.
    // Exceptions (for speed):
    //  matrix ring = Dom::ExpressionField() --> poly ring = Expr
    //  matrix ring = Dom::IntegerMod(n)     --> poly ring = IntMod(m)
    // These exceptional cases are handled below

    // defaults
    polyR:= R;

    Rzero := R::zero;
    Rone  := R::one;

    // coefficient domain R is passed to polynomials for internal
    // representation. Replace Dom::ExpressionField()/Dom::IntegerMod
    // by their analogues Expr/IntMod for faster polynomial arithmetic:
    Rcharacteristic:= R::characteristic;
    if Rcharacteristic <> FAIL and
       Rcharacteristic > 0 and
       R = Dom::IntegerMod(Rcharacteristic) then
         polyR:= IntMod(Rcharacteristic);
         Rzero:= 0;
         Rone := 1;
    end_if;
    if R::hasProp(Ax::systemRep) then
         polyR:= Expr;
         Rzero:= R::zero:
         Rone := R::one;
    end_if;

    //----------------------------------------------------------
    // Rcoerce is used in 2 ways:
    // 1) coercing input (but no further coercion to poly, this
    //                    is then done by R2polyCoerce)
    // 2) coercing internal poly representation to output
    // Treat the most important case R = Dom::ExpressionField
    // as a special case for efficiency:
    //----------------------------------------------------------
    Rcoerce:=
      if R = Dom::ExpressionField()
      then id   // no coercion necessary
      else R::coerce
      end_if;
    //----------------------------------------------------------
    // coercing input (already converted by Rcoerce)
    // further to internal poly representation.
    //----------------------------------------------------------
    R2polyCoerce:=
      if polyR = IntMod(Rcharacteristic)
      then expr; // poly(.., IntMod(n)) does not
                 // accept Dom::IntegerMod(n) coeffs ;-(
      else id; // all other cases were converted by Rcoerce.
               // Now it is up to poly to accept and
               // handle these coefficients
      end_if;
    //----------------------------------------------------------
    Rid:= () -> args(1);
    Rnormal:= if R::normal = FAIL then
                 Rid;
              else
                 R::normal
              end_if:
    
         
end_domain:

// end of file
