//    

/*++ ---------------- MultPoly.mu ---------------------

Description:
This domain represents multivariate polynomials in some identifiers over 
commutative coefficient rings.
Most of the functions for DOM_POLY are available.
Note: It is highly recommend to use only coefficient rings with unique zero
      representation.

Call:

 Dom::MultivariatePolynomial(Vars, Ring, Order) 
 where
   Vars    => list of indeterminates of type DOM_INDET; 
              * Default: [x,y,z]
   Ring    => arbitrary commutative ring of the Domains-package;
              * Default: Dom::ExpressionField(normal)
   Order   => set a term order. Currently is possible: 
              LexOrder, DegreeOrder, DegInvLexOrder;
              or any object of type Dom::MonomOrdering          
              * Default: LexOrder

Methods: 

 - used Parameter:
    ++ a,b,ai : this Domain
    ++ v,ais  : DOM_LIST
    ++ vi     : DOM_IDENT
    ++ o      : Dom::MonomOrdering
    ++ v    = variable list
    ++ ais  = list of elements of this Domain
    ++ ai   = homogeneous polynomials of this Domain
    ++ vi   = new variables, which do neither occur in Vars nor in the ring R
    ++ o    = order of Dom::MonomOrdering

 - borderedHessianDet(a, b <, v>)
    ++ returns the determinant of the bordered Hessian matrix of a bordered by
    ++ by b with respect to v. If v is not given, Vars will be used instead.
 - borderedHessianMat(a, b <, v>)
    ++ returns the bordered Hessian matrix of a,
    ++ that is +-                                            -+
    ++         |                                 diff(b,v[1]) |
    ++         |  Matrix( diff(a, v[i], v[j]) )  diff(b,v[2]) |
    ++         |                                      ...     |
    ++         | diff(b,v[1]) diff(b,v[2])  ...        0      |
    ++         +-                                            -+
    ++ If v is not given, Vars will be used instead.
 - hessianDet(a <, v>)
    ++ returns the determinant of the Hessian matrix of a, with respect to v. 
    ++ If v is not given, Vars will be used instead.
 - hessianMat(a <, v>)
    ++ returns the Hessian matrix of a, that is Matrix( diff(a, v[i], v[j]) ). 
    ++ If v is not given, Vars will be used instead.
 - homogeneousComponents(a)
    ++ returns a ordered list of homogeneous components of a. The list is
    ++ descending total degree order.
 - isHomogeneous(a)
    ++ tests, if a is a homogeneous polynomial.
 - jacobianDet(ais <, v>)
    ++ returns the determinant of the Jacobian matrix of a, with respect to v.
    ++ If v is not given, Vars will be used instead.
 - jacobianMat(ais <, v>)
    ++ returns the Jacobian matrix of a, that is Matrix( diff(ais[i], v[j]) ).
    ++ If v is not given, Vars will be used instead.
 - order(a,b,o)
    ++ compares a and b with respect to order o: If a > b then 1
    ++ is returned, if a = b then 0 is returned and if a < b then -1 is
    ++ returned.
 - rewriteHomPoly(a,ais,v)
    ++ computes a polynomial g( op(v) ) over the ring R in the variables v
    ++ such that g( op(ais) )=a, i.e. it returns the homogeneous polynomial
    ++ a expressed in terms of the new variable list v,
    ++ which represents the list of homogeneous polynomials
    ++ ais resepectively. For this the sequence (order) of ais is used in the
    ++ algorithm.
 - rewritePoly(a,[ai=vi] <, "Unsorted">)
    ++ computes a polynomial g(...,vi,...) over the ring R in the variables vi
    ++ such that g(...,ai,...)=a, i.e. it
    ++ returns polynomial a expressed in terms of the new variables vi, which
    ++ by themselves represents the polynomials ai respectively or FAIL if
    ++ this is not possible. This method can be used for representing
    ++ a polynomial with respect to a given polynomial basis.
    ++ When option "Unsorted" is choosen, the list [ai=vi] is not sorted.
    ++ Otherwise, in a precomputation step this list will be sorted in degLex.
    ++ Please note, the order of Vars and ais is important for the algorithm. 
 - sortList(ais,o)
    ++ sorts the polynomials ais with respect to order o 
    ++ in descending order. This sorting method may be not stable.
 - stableSort(ais,o)
    ++ sorts the polynomials ais with with respect to order o
    ++ in descending order. This sorting method is stable.

All Entries:
D, Dpoly, Factor, Rep, SPolynomial, TeX, TeXCoeff, TeXTerm, TeXindet, _div\
ide, _invert, _mult, _negate, _plus, _power, _subtract, allAxioms, allCate\
gories, allEntries, allSuperDomains, associates, borderedHessianDet, borde\
redHessianMat, characteristic, coeff, coeffRing, coerce, content, convert,\
 convert_to, create_dom, decompose, degLex, degRevLex, degree, degreevec, \
diff, dimension, divide, divides, equal, equiv, euclideanDegree, evalp, ex\
pr, factor, func_call, gcd, gcdex, getAxioms, getCategories, getSuperDomai\
n, groebner, ground, has, hasProp, hessianDet, hessianMat, homogeneousComp\
onents, idealGenerator, indets, info, int, intmult, irreducible, isHomogen\
eous, isNeg, isUnit, isone, iszero, jacobianDet, jacobianMat, key, lcm, lc\
oeff, ldegree, lmonomial, lterm, mainvar, makeIntegral, mapcoeffs, monic, \
mult, multcoeffs, new, normalForm, nterms, nthcoeff, nthmonomial, nthterm,\
 numericSolve, one, order, orderedVariableList, ordering, pdioe, pdivide, \
pivotSize, plus, poly, pquo, prem, primpart, print, printMethods, printMon\
omial, printTerm, quo, random, realSolve, reductum, rem, resultant, rewrit\
eHomPoly, rewritePoly, ringmult, sign, solve, sortList, sqrfree, stableSor\
t, subs, subsex, tcoeff, testtype, undefinedEntries, unitNormal, unitNorma\
lRep, variables, whichEntry, zero

See: 
 - Fakler, W. (1994). Algorithmen zur symbolischen Lsung homogener linearer
   Differentialgleichungen. Diplomarbeit. Universitt Karlsruhe.
						      
Example:
>> MP := Dom::MultivariatePolynomial([x,y,z], Dom::Integer);

     Dom::MultivariatePolynomial([x, y, z], Dom::Integer, LexOrder)

>> mp1:= MP(3*x-2);
                                  3 x - 2

>> mp2:= MP(3*x^2-2*x+1);
                                 2
                              3 x  - 2 x + 1

>> mp3:= MP(5*x^3*y^2*z-z^3+5*y^2+1+x);

                          3  2            2    3
                       5 x  y  z + x + 5 y  - z  + 1

>> 5*mp3+2*mp1^2*3*mp2;

          4       3  2          3        2               2      3
     162 x  + 25 x  y  z - 324 x  + 270 x  - 115 x + 25 y  - 5 z  + 29

MP:=Dom::MultivariatePolynomial([x,y],Dom::Rational);
i1:=MP(x*y^(11)-11*x^6*y^6-x^(11)*y);
i2:=-1/121*MP::hessianDet(i1);
i3:=1/20*MP::jacobianDet([i1,i2]);
MP::rewritePoly(i3^2,[i1=I_1,i2=I_2]);
MP::sortList([3*i1,2*i1,i1,i2], Dom::MonomOrdering(DegLex(2)));
MP::stableSort([3*i1,2*i1,i1,i2], Dom::MonomOrdering(DegLex(2)));

MP:=Dom::MultivariatePolynomial([z,x,y],Dom::Rational);
i2:=MP(4*y*z+x^2);
i6:=MP(x*z^5-2*y^3*z^3+x^2*y^2*z^2-x^4*y*z+x*y^5);
i10:=1/16*MP::borderedHessianDet(i2,i6);
i15:=1/40*MP::jacobianDet([i2,i6,i10]);
MP::rewriteHomPoly(i15^2,[i2,i6,i10],[I_2,I_6,I_10]);

++*/

domain Dom::MultivariatePolynomial(Vars=[x,y,z]:Type::ListOf(DOM_IDENT), 
                                   R=Dom::ExpressionField(normal), 
                                   Order=LexOrder)
    local Field,NumVars;
    inherits Dom::DistributedPolynomial(Vars, R, Order);
    category
      if nops(Vars) = 1 then
        Cat::UnivariatePolynomial(R) 
      else
        Cat::Polynomial(R)
      end_if;
    axiom    (if R::hasProp(Ax::normalRep)
                //or (R::constructor=Dom::ExpressionField and
		//    R::hasProp(Ax::systemRep))
                then Ax::normalRep end_if),
             (if R::hasProp(Ax::canonicalRep) then Ax::canonicalRep end_if);

/* entries */

hessianMat :=
  proc(a:dom, vars=Vars :Type::ListOf(DOM_IDENT))
    local da,v,vi,m;
  begin
    if iszero(nops(vars)) then error("Missing variable") end_if;
    m:=[];
    for v in vars do
      da:=diff(a,v);
      m:=append(m, [diff(da,vi) $ vi in vars]);
    end_for;
    (Dom::Matrix(dom))::create(m)
  end_proc;

hessianDet:=
  proc(a:dom, vars=Vars :Type::ListOf(DOM_IDENT))
  begin
      linalg::det(dom::hessianMat(a,vars))
  end_proc;

borderedHessianMat :=
  proc(a,b:dom, vars=Vars :Type::ListOf(DOM_IDENT))
    local da,db,v,vi,m,lastrow;
  begin
    if iszero(nops(vars)) then error("Missing variable") end_if;
    m:=[];
    lastrow:=[];
    for v in vars do
      da:=diff(a,v);
      db:=diff(b,v);
      m:=append(m, [diff(da,vi) $ vi in vars, db]);
      lastrow:=append(lastrow,db)
    end_for;
    m:=append(m, lastrow);
    (Dom::Matrix(dom))::create(m)
  end_proc;

borderedHessianDet:=
  proc(a,b:dom, vars=Vars :Type::ListOf(DOM_IDENT))
  begin
      linalg::det(dom::borderedHessianMat(a,b,vars))
  end_proc;

jacobianMat :=
  proc(la:Type::ListOf(dom), vars=Vars :Type::ListOf(DOM_IDENT))
    local p,vi;
  begin
    if iszero(nops(vars)) then error("Missing variable") end_if;
    (Dom::Matrix(dom))::create([ [diff(p,vi) $ vi in vars] $ p in la ])
  end_proc;

jacobianDet:=
  proc(la:Type::ListOf(dom), vars=Vars :Type::ListOf(DOM_IDENT))
  begin
      linalg::det(dom::jacobianMat(la,vars))
  end_proc;

homogeneousComponents :=
  proc(a:dom)
    local monos, hc, m, d, comp, DOM;
  begin
    if iszero(a) then [a] end_if;
    DOM:=dom;
    monos:= monomials(dom::poly(a),DegreeOrder);
    hc:=[dom::poly(dom::zero)];
    d :=degree(monos[1]);
    comp:=1;
    for m in monos do
      if degree(m)=d then hc[comp]:=hc[comp]+m
      else hc:=append(hc,m);
	comp:=comp+1;
	d:=degree(m)
      end_if
    end_for;
    map(hc, ()->new(DOM,args(1)))
  end_proc;

isHomogeneous :=
  proc(a:dom)
  begin
    bool(nops({map(op(op(dom::poly(a),1)),degree,Vars)})=1)
  end_proc;
    
degLex:=proc(a,b) 
	begin
	  dom::order(a,b,Dom::MonomOrdering(DegLex(NumVars)))
	end_proc;

degRevLex:=proc(a,b) 
	begin
	  dom::order(a,b,Dom::MonomOrdering(DegRevLex(NumVars)))
	end_proc;

order:=
  proc(a:dom,b:dom,o:Dom::MonomOrdering)
    local d1,d2,s;
  begin
    if dom::iszero(a) and dom::iszero(b) then return(0) end_if;
      d1:=dom::degreevec(a,o);
    d2:=dom::degreevec(b,o);
    s:=o(d1,d2);
    if iszero(s) then
      dom::order(a::dom::reductum(a,o),a::dom::reductum(b,o),o)
    else s end_if
  end_proc;

sortList:=
  proc(l:Type::ListOf(dom),o:Dom::MonomOrdering)
    local DOM;
  begin
    DOM:=dom;
    sort(l,(a,b)->bool(DOM::order(a,b,o)=1))
  end_proc;

stableSort:=
  proc(l:Type::ListOf(dom),o:Dom::MonomOrdering)
    local i, DOM;
  begin
    DOM:=dom;
    l:=zip(l,[i$i=1..nops(l)],(a,b)->[a,b]);
    map(sort(l,proc(a,b) local s;
	       begin
		s:=DOM::order(a[1],b[1],o);
		if iszero(s) then
		  if a[2]<b[2] then s:=1 end_if
	     end_if;
	       bool(s=1);
	 end_proc),op,1);
  end_proc;

rewriteHomPoly := if Field then
  proc(a:dom, lp:Type::ListOf(dom), newvars:Type::ListOf(DOM_IDENT))
    local p,DPnv,d,lpPart,newvarsPart,i,k,M_v,A,M_LM,M_P,dv,pos,lca,lc,mp,
          lconst,ga,powerProds,N,b,s,M,dvl,
          linearPosDioSolve,dynamicMult,dynamicPower,dynamicPowerProduct;
  begin
    if testargs() then
      if nops(lp)<>nops(newvars) then
	error("not equal number of polynomials and indeterminates")
      end_if;
      if contains(map(append(lp,a), dom::isHomogeneous),FALSE)<>0 then
	error("all polynomials should be homogeneous")
      end_if;
    end_if;
    // local procedure for solving linear diophantine equations restriced to
    // only positive integers
    linearPosDioSolve :=
    proc(ais:Type::ListOf(Type::PosInt), a:Type::PosInt)
      local allSol;
    begin
      if iszero(nops(ais)) then error("list contains no positive integers")
      end_if;
      if not(iszero(a mod gcd(op(ais)))) then return([]) end_if;
    
      allSol:=proc(ais, a, m)
		local sol;
	      begin
		if iszero(a) then sol:=[ [0$nops(ais)] ]
		else sol:=[] end_if;
		if a>0 then
		  if a>=ais[m] then
		    sol:=map(allSol(ais,a-ais[m],m), s->(s[m]:=s[m]+1; s) )
		  end_if;
		  if m>1 then sol:=_concat(sol,allSol(ais,a,m-1)) end_if;
		end_if;
		sol
	      end_proc;

      allSol(ais,a,nops(ais))
    end_proc;
    
    // local procedures to compute expensive power products more efficient
    dynamicMult:=
    proc(p1,p2)
      option remember;
    begin
      p1*p2
    end_proc;

    dynamicPower:=
    proc(p,n)
      option remember;
    begin
      if iszero(n) then
        poly(R::one,Vars,R)
      elif n=1 then
        p
      else
        dynamicMult(dynamicPower(p,n-1),p)
      end_if
    end_proc;
    
    dynamicPowerProduct:=
    proc(v,pis)
      local i;
    begin
      _mult(dynamicPower(pis[i],v[i]) $ i=1..nops(v))
    end_proc;
      
    // for computing directly on representation:
    a:=dom::poly(a);
    lp:=map(lp,dom::poly);
    d:=degree(a);
    if iszero(d) then return(dom(a)) end_if; // ???? should I return here FAIL
    // take only necessary polynomials of lp and newvars respectively
    lpPart:=[];
    newvarsPart:=[];
    for i from 1 to nops(lp) do
      if degree(lp[i])<=d then
	lpPart:=append(lpPart,lp[i]);
	newvarsPart:=append(newvarsPart,newvars[i])
      end_if
    end_for;
    DPnv:=Dom::MultivariatePolynomial(newvarsPart, R, Order);
    p:=DPnv::poly(DPnv::zero);
    M_v:=linearPosDioSolve(map(lpPart, degree), d);
    if M_v=[] then return(FAIL) end_if;
    // determine all possible degree vectors of leading monomials built up
    // from multiplied polynomials of lpPart with the total degree of a.
    A:=map(lpPart, p->(degreevec(lterm(p))));
    M_LM:=map(M_v, proc(v) local A0,i,k,n;
		   begin
		     n:=nops(A);
		     A0:=[map(A[i],_mult, v[i]) $ i=1..n];
		     [_plus(A0[i][k] $ i=1..n)$ k=1..nops(A[1])]
		   end_proc);
    // compute all possible power products of leading monomials built up
    // from multiplied polynomials of lpPart with the total degree of a
    // represented in newVars.
    M_P:=map(M_v, v->poly([[R::one,v]], newvarsPart, R));
    // representing polynomial a in newvars, where newvars[i]=lp[i]
    while not(iszero(a)) do
      if iszero(degree(a)) then return(FAIL) end_if;
      dv:= degreevec(a);
      pos:=contains(M_LM,dv);
      if iszero(pos) then // strategy failed, use Ansatz instead
	N:=nops(M_v);
        lconst:=[genident("_c_")$i=1..N];
        //powerProds:=map(M_P, mp->poly(eval(subs(op(mp,1),subs_seq)),Vars,R));
	powerProds:=map(M_v, v->dynamicPowerProduct(v,lpPart));
	ga:=poly2list(poly(_plus( lconst[i]*expr(powerProds[i]) $ i=1..N),
			   Vars,polylib::Poly(lconst,R)));
        a:=poly2list(a);
        dvl:=map(a,e->e[2]);
	b:=[R::zero $ nops(ga)];
	A:=[];
	for i from 1 to nops(ga) do
	  A:=append(A,[lcoeff(coeff(ga[i][1],lconst[k],1)) $ k=1..N]);
          pos:=contains(dvl,ga[i][2]);
	  if iszero(pos) then b[i]:=R::zero else b[i]:=a[pos][1] end_if;
	end_for;
	M:=Dom::Matrix(R);
	s:=linalg::matlinsolve(M::create(A),M::create(b),Special);
	if s=[] or s=FAIL then
	  return(FAIL)
	else
	  return(
	    new(DPnv,p+poly(_plus(s[i]*expr(M_P[i]) $i=1..N),newvarsPart,R)));
	end_if
      else  // follow strategy
        //mp:=poly(eval(subs(op(M_P[pos],1),subs_seq)),Vars,R);
	mp:=dynamicPowerProduct(M_v[pos],lpPart);
	lca:= lcoeff(a);
	lc:= lcoeff(mp);
	a:=a-multcoeffs(mp, lca/lc);
	p:=p+multcoeffs(M_P[pos], lca/lc);
     end_if;
    end_while;
    new(DPnv, p)
  end_proc
end_if;
    
rewritePoly := if Field then
  proc(a:dom, leqn:Type::ListOf(Type::Equation(dom,DOM_IDENT)), o=" ")
      // Option: Unsorted 
    local p,lhc,hc,lp,newvars,i,sl,phc,nv,DOM;
  begin
    DOM:=dom;
    lp:=map(leqn,op,1);
    newvars:=map(leqn,op,2);
    nv:=newvars;
    if testargs() then
      if contains(map(lp, dom::isHomogeneous),FALSE)<>0 then
	error("all polynomials on lhs should be homogeneous")
      end_if;
    end_if;
    if iszero(degree(a)) then
      return(new(Dom::MultivariatePolynomial(nv, R, Order), dom::poly(a)))
    end_if;
    if o<>hold(Unsorted) then //cannot use sortList or stableSort since here 
                              //are at once 2 lists to sort
      sl:=zip(zip(lp,newvars,(a,b)->[a,b]), [i$i=1..nops(lp)],
	      (a,b)->append(a,b));
      sl:=sort(sl,proc(a,b) local s;
	      begin
		s:=DOM::degLex(a[1],b[1]);
		if iszero(s) then
		  if a[3]>b[3] then s:=1 end_if
	       end_if;
	       bool(s=1)
	      end_proc);
      lp:=map(sl,op,1);
      newvars:=map(sl,op,2)
    end_if;
    if dom::isHomogeneous(a) then lhc:=[a]
    else lhc:=dom::homogeneousComponents(a)
    end_if;
    p:=poly(R::zero,nv,R);
    for hc in lhc do
      phc:=dom::rewriteHomPoly(hc,lp,newvars);      
      if phc=FAIL then return(FAIL)
      else p:=p+poly(op(phc,1),nv,R) end_if;
    end_for;
    new(Dom::MultivariatePolynomial(nv, R, Order), p)
  end_proc
end_if;

/* body of the domain */

begin
    if args(0) > 3 then error("Wrong no of args") end_if;
    if domtype(Vars) <> DOM_LIST or iszero(nops(Vars))
        then error("Illegal indeterminates") end_if;
    if R::dom=DOM_DOMAIN then
      if R::hasProp(Dom::BaseDomain) = FAIL then
        R := Dom::ExpressionField(normal)
      elif not(R::hasProp(Cat::CommutativeRing)) then
        error("Illegal coefficient ring")
      end_if;
    else
      error("Illegal coefficient ring");
    end_if;
  //The following if-statement would nearly always solve the termination 
  //problem ("unique zero") by polynomial division and yield a correct degree.
    //if R::constructor=Dom::ExpressionField and R::hasProp(Ax::systemRep) then
    //  R:=Dom::ExpressionField(id, iszero@normal)
    //end_if;
    if not contains({LexOrder, DegreeOrder, DegInvLexOrder}, Order)
       and type(Order)<>Dom::MonomOrdering
      then error("Unknown monomial ordering");
    end_if;
    Field   := bool(R::hasProp(Cat::Field));
    NumVars := nops(Vars);
end_domain:


