//    

/*++ ---------------- OrePoly.mu ---------------------

Description:
This is the domain of univariate skew polynomials the so-called
Ore-polynomials. In this implementation the coefficient ring needs to be 
commutative (and to be a differential ring). (The reasons for that are 
efficiency and simplicity. Maybe we will extend it later.). 
A multiplication in an Ore-Domain is defined by x*a = \sigma(a)*x+\delta(a),
where \sigma is an injective ring endomorphism and \delta a \sigma derivation
i.e. it suffice the two conditions \delta(a+b) = \delta(a)+\delta(b) and
\delta(a*b) = \sigma(a)*\delta(b)+\delta(a)*b.
An allowed action of an operator a on an arbitrary element e of an compatible
ring  extension is a map: c*\sigma(e)+\delta(e).
Note: It is highly recommend to use only coefficient rings with unique zero
      representation.

Call:

 Dom::UnivariateSkewPolynomial(var, sigma, inv_sigma, delta, c, Ring) 
 where
   var        => indeterminate of type DOM_IDENT;
                 * Default: A
   sigma      => injective ring endomorphism: Ring -> Ring
                 * Default: id
   inv_sigma  => FAIL or inverse of sigma: Ring -> Ring 
                 * Default: id
   delta      => sigma-Derivation: Ring -> Ring 
                 * Default: e->e::dom::zero
   c          => factor of operator action: element of compatible 
                 extension of Ring
                 * Default: 1
   Ring       => arbitrary commutative ring of the Domains-package;
                 * Default: Dom::ExpressionField(normal)


Methods:
 
 - used Parameter:
    ++ a,b : this Domain
    ++ p   : DOM_POLY
    ++ e   : any of {DOM_EXPR, DOM_LIST}
    ++ c   : DOM_EXPR or domain containing coefficient ring
    ++ n   : DOM_INT
    ++ a,b = a, b are Ore-polynomials of this Category
    ++ n   = n is a nonnegative integer
    ++ e   = e is an arbitrary linear operator in var or
    ++       it's coefficients vector or a polynomial expression in var
    ++ c   = factor of operator action 

 - new(e)
    ++ creates a Domain element.
 - apply(a,e,c)
    ++ returns the operator applied to e by the action c*\sigma(e)+\delta(e).
    ++ The resulting expression is not changed or simplified.
 - func_call(a,e <, Unsimplified>)
    ++ returns the operator applied to e. The resulting expression is expanded
    ++ and simplified and factored unless option Unsimplified is given.
 - leftShift(p)    (INTERNAL-function)
    ++ computes a left shift of the coefficient vector of p, which is the
    ++ same as p*poly(var,[var],Ring). 
 - rightShift(p)   (INTERNAL-function)
    ++ computes a right shift of the coefficient vector of p whereby the 
    ++ ground term is thrown away. This is the same as 
    ++ divide(p,poly(var,[var],Ring),Quo). 
 - ore_mult(a, b)
    ++ computes the multiplication a*b defined by var*a=sigma(a)*var+delta(a).
 - _mult(a,b,...)
    ++ multiplies a,b,... by the non-commutative definition for multiplication
    ++ of Ore-polynomials.
 - _power(a, n)
    ++ computes the n-th power of a by the previous defined operation _mult.
 - leftDivide(a, b)
    ++ computes q, r, such that a=b*q+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.
 - rightDivide(a, b)
    ++ computes q, r, such that a=q*b+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.

All Entries:
Rep, TeX, TeXCoeff, TeXTerm, TeXindet, _invert, _mult, _negate, _plus, _po\
wer, _subtract, allAxioms, allCategories, allEntries, allSuperDomains, app\
ly, characteristic, coeff, coeffRing, coerce, content, convert, convert_to\
, create_dom, degree, degreevec, equal, equiv, euclideanDegree, expr, func\
_call, getAxioms, getCategories, getSuperDomain, ground, has, hasProp, hom\
ogeneousComponents, indets, info, intmult, isHomogeneous, isNeg, isone, is\
zero, key, lcoeff, ldegree, leftDivide, leftExactQuotient, leftExtendedEuc\
lid, leftExtendedGcd, leftGcd, leftLcm, leftQuotient, leftRemainder, lmono\
mial, lterm, mainvar, makeIntegral, mapcoeffs, monic, mult, multcoeffs, ne\
w, nterms, nthcoeff, nthmonomial, nthterm, one, order, orderedVariableList\
, ordering, ore_mult, pivotSize, plus, poly, print, printMethods, printMon\
omial, printTerm, random, reductum, rightDivide, rightExactQuotient, right\
ExtendedEuclid, rightExtendedGcd, rightGcd, rightLcm, rightQuotient, right\
Remainder, ringmult, sign, sortList, stableSort, subs, subsex, tcoeff, tes\
ttype, undefinedEntries, variables, vectorize, whichEntry, zero

See: 
 -  M. Bronstein & M. Petkovsek (1993). On Ore Rings, Linear Operators 
    and Factorisation. Bericht Nr. 200  ETH Zuerich 
 -  M. Bronstein & M. Petkovsek (1996). An introduction to pseudo-linear
    algebra. Theoretical Computer Science 157 No. 1, 3-33.
 -  O. Ore (1933). Theory of non-commutative polynomials. Annals of
    Mathematics 34, 480-508.

Example:

>> ore := Dom::UnivariateSkewPolynomial(P,id,id,e->diff(e,x) ); 
>> p1 := ore(P^2+(x+1)*P+2*x);
>> p2 := ore(P+1);
>> ore::leftDivide(p1, p2);

++*/


/* Dependencies:

 OrePolyCat.mu, DistPoly.mu, MultPoly.mu, UniPoly.mu

*/

alias( rep(x) = extop(x, 1),
       argstest = (if map({args()}, domtype)<>{dom} 
                      then error("illegal arguments") end_if)
):


domain Dom::UnivariateSkewPolynomial( var=A, sigma=id, inv_sigma=id, 
                                      delta= (e->0), c=1,
                                      R=Dom::ExpressionField(normal) )
    local Rs, UP, hasField, isAutomorphism;
    inherits Dom::UnivariatePolynomial(var, R);
    category Cat::UnivariateSkewPolynomial(R);
    axiom    (if R::hasProp(Ax::normalRep) 
                //or (R::constructor=Dom::ExpressionField and
		//    R::hasProp(Ax::systemRep)) 
                then Ax::normalRep end_if),
             (if R::hasProp(Ax::canonicalRep) then Ax::canonicalRep end_if);
 
  // entries: 

  new:= proc(e)
    local i;
    begin
      if args(0) <> 1 then error("wrong no of args") end_if;
      if type(e)=DOM_LIST then
        e := map(e, expr);
        e := _plus(e[i]*var^(i-1) $i=1..nops(e))
        end_if;
      poly(expr(e), [var], R);
      if % = FAIL then return(FAIL)
       else new(dom, %) end_if
    end_proc;

// the following 2 methods are currently not used
/*
  rightShift:= proc(x) local l;
     begin
       l := map(poly2list(x), ()->([args(1)[1],args(1)[2]-1]) );
       if op(l, [nops(l),2])<0 then delete (l[nops(l)])  end_if;
       poly(l, [var], R)
     end_proc;

  leftShift:= proc(x) 
     begin
       poly(map(poly2list(x), ()->([args(1)[1],args(1)[2]+1]) ), [var], R)
     end_proc;
*/

  ore_mult:= proc(a, b)
     local s,l;
     begin
       a := rep(a);
       b := rep(b);
       s := poly(R::zero,[var],R);
       while degree(a)>0 do
         s := s+mapcoeffs(b,R::_mult,coeff(a,0));
         //a := dom::rightShift(a);
         l:=map(poly2list(a), ()->[args(1)[1],args(1)[2]-1] );
         if op(l, [nops(l),2])<0 then delete (l[nops(l)])  end_if;
         a:=poly(l, [var], R);
         //b := dom::leftShift(mapcoeffs(b,sigma))+mapcoeffs(b,delta)
         b:=poly(map(poly2list(mapcoeffs(b,sigma)),
                     ()->([args(1)[1],args(1)[2]+1])),[var],R)
             +mapcoeffs(b,delta)
       end_while;
       new(dom, s+mapcoeffs(b,R::_mult,coeff(a,0)))
     end_proc;

  mult:= proc() local i, m;
     begin
       //if testargs() then
          if map({args()}, domtype) <> {dom} then 
             //error("cannot multiply arguments") end_if;
             return(FAIL)
       end_if; 
       m := args(args(0));
       for i from args(0)-1 downto 1 do  
         m := dom::ore_mult(args(i), m)
       end_for;
       m
     end_proc;

  _mult:= proc() 
      local sel, h, DOM,r,s;
      begin
        //if testargs() then 
        //  if map({args()},domtype) minus 
        //     {dom,op(Rs),Dom::Integer,DOM_INT} <> {}
        //    then  error("cannot multiply arguments") end_if
        //end_if;
        DOM := dom;
        s:=split([args()],()->contains({DOM,op(Rs),Dom::Integer,DOM_INT},
                                       domtype(args(1))));
        //sel := split([args()], testtype, Dom::Integer); 
        sel := split(s[1], testtype, Dom::Integer); 
        if iszero(nops(sel[1])) then
          h:=R::one
        else
          h:=R::_mult(op(sel[1]))
        end_if;
        sel := map(sel[2], e->(if domtype(e)=DOM then e 
                               else new(DOM, poly(expr(e),[var],R)) end) );
        r:=dom::ringmult(dom::mult(op(sel)), h);
        if nops(s[2])=0 then r
        else
          FAIL
        end_if        
      end_proc;

  _power:= proc()  
     begin
       if domtype(args(1))<>dom then 
         error("argument of wrong domain type") end_if;
       args(2);
       if iszero(%) then dom::one
       else
         if domtype(%) = DOM_INT and % > 0 then  
           dom::mult(args(1) $args(2)) 
         else error("NonNegativeInteger as exponent expected") end_if
       end_if
     end_proc;

  leftDivide:= 
    if hasField and isAutomorphism then
      proc(a, b)
       local q, h, da, db, lc;
       begin
         argstest;
         q := dom::poly(dom::zero);
         da := degree(a);
         db := degree(b);
         lc := lcoeff(b); 
         while da >= db and not iszero(a) do
           h:=poly((inv_sigma @@ db)(lcoeff(a)/lc),[var],R)*
               poly(var^(da-db),[var],R);
           q:=q+h;
           a:=a-dom::ore_mult(b, new(dom,h));
           da:=degree(a)
         end_while;
         table(hold(quotient) = new(dom,q), hold(remainder) = a)
       end_proc
     else ()->FAIL
    end_if;

  leftQuotient:=if not isAutomorphism then ()->FAIL end_if;
  leftExactQuotient:=if not isAutomorphism then ()->FAIL end_if;
  leftRemainder:=if not isAutomorphism then ()->FAIL end_if;
  leftGcd:=if not isAutomorphism then ()->FAIL end_if;
  leftExtendedEuclid:=if not isAutomorphism then ()->FAIL end_if;
  leftExtendedGcd:=if not isAutomorphism then ()->FAIL end_if;
  rightLcm:=if not isAutomorphism then ()->FAIL end_if;

  rightDivide:= 
    if hasField then
      proc(a, b)
        local q, h, da, db, lc, inv;
        begin
          argstest;
          q := dom::poly(dom::zero);
          da := degree(a);
          db := degree(b);
          lc := lcoeff(b);
          while da >= db and not iszero(a) do
            inv:=R::_invert((sigma @@ (da-db))(lc));
            h:=poly(R::_mult(inv,lcoeff(a)),[var],R)*poly(var^(da-db),[var],R);
            q:=q+h;
            a:=a-dom::ore_mult(new(dom,h), b);
            da:=degree(a)
          end_while;
          table(hold(remainder) = a, hold(quotient) = new(dom,q))
        end_proc
     end_if;

  apply:=
    proc(a,f,cfac=c)
      local avec,i,theta_c;
    begin
      avec:=map(dom::vectorize(a),expr);
      theta_c:=t->cfac*eval(sigma(t))+eval(delta(t));
      f:=eval(f);
      _plus(avec[i]*eval((theta_c@@(i-1))(f))
            $ i=1..nops(avec));
     // _plus(avec[i]*eval(cfac*(sigma@@(i-1))(f)+(delta@@(i-1))(f))
     //       $ i=1..nops(avec));
    end_proc;

  func_call :=
    proc(a, f, o=" ") // Option o: Unsimplified 
      local r,f0;
    begin
      f0:=context(f); // to get rid of scoping problems
      r:=dom::apply(a,f0);
      if o=Unsimplified then r
      else
        if testtype(r,Type::Numeric) then
          r
        else
          factor(simplify(expand(r)))
        end_if;
      end_if;
    end_proc;
/*
 -------------------------

   Forbidden Functions 
            
 -------------------------
*/
  
  _divide:=()->FAIL; D:=()->FAIL; Dpoly:=()->FAIL; SPolynomial:=()->FAIL;
  borderedHessianDet:=()->FAIL; borderedHessianMat:=()->FAIL; 
  decompose:=()->FAIL; degLex:=()->FAIL; degRevLex:=()->FAIL; 
  diff:=()->FAIL; dimension:=()->FAIL; divex:=()->FAIL;
  divide:=()->FAIL; divides:=()->FAIL; evalp:=()->FAIL; factor:=()->FAIL;
  gcd:=()->FAIL; gcdex:=()->FAIL; groebner:=()->FAIL; hessianDet:=()->FAIL;
  hessianMat:=()->FAIL; idealGenerator:=()->FAIL;
  int:=()->FAIL; irreducible:=()->FAIL; jacobianDet:=()->FAIL; 
  jacobianMat:=()->FAIL; lcm:=()->FAIL; normalForm:=()->FAIL; 
  numericSolve:=()->FAIL; pdioe:=()->FAIL; pdivide:=()->FAIL; pquo:=()->FAIL; 
  prem:=()->FAIL; quo:=()->FAIL; realSolve:=()->FAIL; rewriteHomPoly:=()->FAIL;
  rewritePoly:=()->FAIL; rem:=()->FAIL; resultant:=()->FAIL; sqrfree:=()->FAIL;
  solve:=()->FAIL; solve_sys:=()->FAIL; 
/*
 -------------------------

    Output-Procedures 
            
 -------------------------
*/

  TeXCoeff:= proc(x)
     local str, ex;
     begin
       str := R::TeX(x);
       ex  := R::expr(x);
       case type(ex)
         of "_plus" do str := " + {\\left( ".str." \\right)}"; break;
         of "_mult" do if op(ex, nops(ex)) <> -1 then str := " + {".str."}"
                        else str := " - {".str[2..-1]."}"
                       end_if; break;
         otherwise
           if x = R::one then str := " + " 
             elif x = -R::one then str := " - "
               elif sign(ex) <> -1 then str := " + {".str."}"
               else str := " - {".str[2..-1]."}"     
           end_if
       end_case;
       str
     end_proc;

begin
  if args(0) > 6 then error("wrong no of args") end_if;
  if domtype(var) <> DOM_IDENT then error("not a variable") end_if;
  if {domtype(sigma), domtype(delta)} minus {DOM_PROC, DOM_FUNC_ENV} <> {} 
    then error("function expected") end_if;
  if {domtype(inv_sigma)} minus {DOM_PROC, DOM_FUNC_ENV} <> {} 
    then isAutomorphism := FALSE else isAutomorphism := TRUE end_if; 
  if args(0) = 6 then
    if R::dom=DOM_DOMAIN then
      if R::hasProp(Dom::BaseDomain) = FAIL then
        R := Dom::ExpressionField(normal)
      elif not(R::hasProp(Cat::CommutativeRing)) then
        error("Illegal coefficient ring")
      end_if;
    else
      error("Illegal coefficient ring");
    end_if;
  //The following if-statement would nearly always solve the termination 
  //problem ("unique zero") by polynomial division and yield a correct degree.
    //if R::constructor=Dom::ExpressionField and R::hasProp(Ax::systemRep) then
    //  R:=Dom::ExpressionField(id, iszero@normal)
    //end_if;
  end_if;
  if R::hasProp(Dom::Rational) then
    Rs := {R, DOM_RAT} else Rs := {R} end_if;
  UP := Dom::UnivariatePolynomial(var, R);
  hasField := bool(R::hasProp(Cat::Field));
end_domain:

unalias( rep, argstest ):

