/* ----------------------------------------------------
Dom::PseudoTensor - the domain of symbols that represent tensors
--------------------------------------------------------
Constructor Call: Dom::PseudoTensor(<R>)

                  R - (optional) coefficient domain (a Cat::Ring)

If desired, define the shortcut

    ptensor:= Dom::PseudoTensor();
-----------------------------------------------------------------------*/

domain Dom::PseudoTensor( R )
 // local I_am_local;
    inherits Dom::BaseDomain;
    category Cat::Matrix(R); 
    axiom if R::hasProp( Ax::canonicalRep ) then Ax::canonicalRep end_if;

// -----------------------------------------
//               entries
// -----------------------------------------
    Name:= if dom::constructor = Dom::PseudoTensor and
	      R = Dom::ExpressionField() then
                hold(Dom::PseudoTensor)()
           end;

// -----------------------------------------
//                methods
// -----------------------------------------

isformal:= x -> bool(domtype(extop(x, 3)) <> Dom::Matrix(R));

//--------------------------------------------------
// utility for flattening matrices
// flatten(<r, c>, M) reformats a matrix with vector
// entries (rows or columns) to an r x c matrix with 
// plain entries
// If r, c are not specified, a heuristics is used
// to guess the format.
//--------------------------------------------------
flatten:= proc(r, c, M) 
   local rr, cc, rrr, ccc, i, j, R, rrrr, cccc, decide;
   begin
     if args(0) = 1 then
        M:= r;
     end_if;

     if M::dom::hasProp(Cat::Matrix) <> TRUE then
        error("expecting a matrix");
     end_if;

     [rr, cc]:= M::dom::matdim(M):
     R:= M::dom;

     decide:= FALSE;

     for i from 1 to rr do
       for j from 1 to cc do
         if (M[i,j])::dom::hasProp(Cat::Matrix) = TRUE then
 	   decide:= TRUE;
           [rrrr, cccc]:= (M[i,j])::dom::matdim(M[i,j]);
	   break;
          end_if;   
       end_for;
     end_for;

     if decide then 
       for i from 1 to rr do
         for j from 1 to cc do 
           if M[i,j] = 0 then 
	     M[i,j]:= Dom::Matrix()(rrrr, cccc, [[0 $ cccc] $ rrrr]);
           end_if;
         end_for;
       end_for;
     end_if;

     if (M[1,1])::dom::hasProp(Cat::Matrix) <> TRUE then
       if type(M[1,1]) <> "_index" and rr = 1 and cc = 1 then 
	 return(M[1,1])
       else  
         return(M); // nothing to flatten
       end_if;
     end_if;

     if rr > 1 and cc > 1 then
        return(M);
     end_if;

     [rrr, ccc]:= (M[1,1])::dom::matdim(M[1,1]);

     // New strategy: Single column of columns should not be flatted. 
     //               Single row of rows should not be flatted.
     if cc = 1 and ccc = 1 then 
       if rr > 1 then 
         return(M);
       end_if;
     elif rr = 1 and rrr = 1 then 
       if cc > 1 then 
         return(M);
       end_if;
     end_if;

     // Heuristics:
     // If M is misformated as a column (it might be meant
     // as a row), we swap the interpretation. In both
     // cases, the i-th entry is accessed as M[i].
     if (rr = 1 and ccc <> 1 and rrr = 1) or
        (cc = 1 and rrr <> 1 and ccc = 1) then
        [rr, cc]:= [cc, rr];
     end_if;
     if (rrr = 1 and cc <> 1 and rr = 1) or
        (ccc = 1 and rr <> 1 and cc = 1) then
        // If M[1,1] is misformated as a column (it might be meant
        // as a row), we swap the interpretation. In both
        // cases, the i-th entry of M[1,1] is accessed as M[1,1][i].
	[rrr, ccc]:= [ccc, rrr];
     end_if;

     if args(0) = 1 then
        // no target dimensions r, c were specified,
        // we have to deduce them
        r:= max(rr, rrr); 
        c:= max(cc, ccc); 
     elif (r <> max(rr, rrr) and r = max(cc, ccc)) or
          (c <> max(cc, ccc) and c = max(rr, rrr)) then
          [rr, cc] := [cc, rr]; 
          [rrr,ccc]:= [ccc,rrr]; 
     end_if;
     if r <> max(rr, rrr) or c <> max(cc, ccc) then
        return(M); // error("unexpected format (matrix entries inside pseudo tensor)"):
     end_if;
     
     if   r = rrr and c = cc then 
          return(M::dom::new(r, c, [[M[j][i] $ j=1..c] $ i=1..r]));
     elif r = rr and c = ccc then 
          return(M::dom::new(r, c, [[M[i][j] $ j=1..c] $ i=1..r]));
     elif r = rr and c = cc then 
          return(M::dom::new(r, c, [[M[j][i] $ j=1..c] $ i=1..r]));
     elif r = rrr and c = ccc then 
          return(M::dom::new(r, c, [[M[i][j] $ j=1..c] $ i=1..r]));
     else // this cannot happen
          error("unexpected format (matrix entries inside pseudo tensor)");
     end_if;
     return(M);
end_proc;

/*-------------------------------------------------------------------
  new  --  construct a new instance of Dom::PseudoTensor(R)

  Call:  

         (Dom::PseudoTensor(R))::new(M, furtherargs) 
      or (Dom::PseudoTensor(R))::new(r, c, M, furtherargs) 
      or (Dom::PseudoTensor(R))::new(m, n, Name, furtherargs) 

      or Dom::PseudoTensor(R)(M, furtherargs) 
      or Dom::PseudoTensor(R)(r, c, M, furtherargs) 
      or Dom::PseudoTensor(R)(m, n, Name, furtherargs) 

      or ptensor(M, furtherargs) 
      or ptensor(r, c, M, furtherargs) 
      or ptensor(m, n, Name, furtherargs) 

      if ptensor:= Dom::PseudoTensor() is defined.

  Parameters:  

      r = number of rows (a positive integer)
      c = number of rows (a positive integer)
      M = a matrix, an array, a plain list, a list of lists etc 
          (anything that can be converted to a Dom::Matrix(R))
      furtherargs = arbitrary further arguments that are stored
                    in the internal representation and can be
                    accessed via extop(.., 4), extop(.., 5) etc.

      With this input, a matrix is created via Dom::Matrix(R)(r, c, M) 
      and stored as the 3-rd operand of the pseudo tensor.

      Alternatively:

      m = number of rows (an integer or a symbol)
      n = number of rows (an integer or a symbol)
      Name = a string (the output symbol seen on the screen)
      furtherargs = arbitrary further arguments that are stored
                    in the internal representation and can be
                    accessed via extop(.., 4), extop(.., 5) etc.


      With this input, the Name is stored as the 3-rd operand 
      instead of a concrete matrix. In this case, symbolic 
      dimensions m, n may be specified. This is a 'formal' tensor.

  Internal representation:
           new(dom, 
               r,  // row dimension
               c,  // column dimension
               matrix/string/_mult expression/_plus expression, 
               furtherargs)
-------------------------------------------------------------------*/
new:= proc(r, c, M, furtherargs)
local tmp;
begin
  if args(0) = 0 then
     error("expecting at least 1 argument"):
  end_if;
  // --------------------------------------------------
  // handle the call Dom::PseudoTensor(M, furtherargs).
  // Just check whether the first argument r can be
  // converted to a matrix of type Dom::Matrix(R):
  // --------------------------------------------------
  if traperror((tmp:= Dom::Matrix(R)(r))) = 0 then
     furtherargs:= args(2..args(0));
     if tmp::dom::matdim(tmp) <> [1,1] then  
       M:= dom::flatten(tmp);
       [r, c]:= M::dom::matdim(M);
     else
       M:= tmp;
       r:= 1; 
       c:= 1;
     end_if;
  elif r::dom = dom then
     return(r)
  elif args(0) < 3 then
     // we need r, c, M
     error("expecting at least 3 arguments");
  else
     furtherargs:= args(4..args(0));
  end_if;
  // --------------------------------------------------
  // From here, we know that at least 3 arguments
  // r, c, M were passed by the user
  // --------------------------------------------------

  // --------------------------------------------------
  // handle formal pseudo tensors with symbolic r, c
  // --------------------------------------------------
  if domtype(r) <> DOM_INT or
     domtype(c) <> DOM_INT then
     return(new(dom, r, c, M, furtherargs));
  end_if;

  // --------------------------------------------------
  // handle formal pseudo tensors with integers r, c
  // but without matrix entries. We need a formal name
  // M as a string:
  // --------------------------------------------------
  if domtype(M) = DOM_STRING then
     return(new(dom, r, c, M, furtherargs));
  end_if;

  if domtype(r) = DOM_INT and r < 1 then
     error("expecting a positive row dimension, got: ".expr2text(r));
  end_if;

  if domtype(c) = DOM_INT and c < 1 then
     error("expecting a positive column dimension, got: ".expr2text(c));
  end_if;
  
  if traperror((tmp:= Dom::Matrix(R)(M))) = 0 then
     M:= dom::flatten(r, c, tmp);
     return(new(dom, r, c, M, furtherargs));
  elif M::dom = dom then
     return(M)
  end_if;
  return(new(dom, r, c, M, furtherargs));
end_proc;

//-------------------------------------------
// indexed access:  x[i, j] extracts elements
//-------------------------------------------
_index:= proc(x, indices) 
         begin
           if not x::dom::isformal(x) then
              extop(args(1), 3)[args(2..args(0))];
           else
              hold(_index)(args());
           end_if;
         end_proc;

//--------------------------------------------
//--------------------------------------------
expr2text:= x -> return(expr2text(dom)."(".expr2text(extop(x)).")");

//--------------------------------------------
//--------------------------------------------
print:= x -> extop(x, 3);

//--------------------------------------------
//--------------------------------------------
Content := (Out, data) -> Out(dom::print(data));

//--------------------------------------------
// convert_to(e, F): convert an element of type
// Dom::PseudoTensor(R) to the data type F
//--------------------------------------------
convert_to:= proc(e, F)
  local f;
  begin 
  f:= extop(e, 3):  // string or an element of Dom::Matrix(R)
  if f::dom::hasProp(Cat::Matrix) = TRUE then
     return(f::dom::convert_to(f, F));
  elif e::dom::matdim(e) = [1, 1] then
     // Beware: 1 x 1 matrices are 'flattened' to
     // scalars. Turn it back into a matrix:
     return(F(1, 1, [f]));
  else
     return(FAIL);
  end_if;
end_proc;

//--------------------------------------------
// convert(e): convert an element to type
// Dom::PseudoTensor(R) 
//--------------------------------------------
convert:= proc(e)
  begin 
  if e::dom::hasProp(Cat::Matrix) <> TRUE then
     return(FAIL);
  else
     return(new(dom, op(e::dom::matdim(e)), e::dom::convert(e)));
  end_if;
end_proc;

//--------------------------------------------
// iszero
//--------------------------------------------
iszero:= (x) -> bool( (extop(x, 3))::dom::hasProp(Cat::Matrix) = TRUE  
                      and iszero(extop(x, 3)) );

//--------------------------------------------
// matdim: the matrix dimension
//--------------------------------------------
matdim:= (x) -> [extop(x,1),extop(x,2)];

//--------------------------------------------
// _plus: addition of matrices and matrix symbols
// Beware:  symbol1 + symbol2 + .. + matrix1 + matrix2 + ... 
//          is handled by this routine.
//          However,  2*symbol + 3*symbol 
//          is not handled by this routine and does
//          **not** yield 5*symbol
//--------------------------------------------
_plus:= proc()
   local Args, result, i, j, k, r, c, rr, cc,
         formals, others, dummy, x, y, z, zz,
         mult2list, xfound, cx, cy, rx, ry;
   begin
      if args(0) = 0 then return(R::zero); end_if;
      if args(0) = 1 then return(args(1)); end_if;
      //------------------------------------
      // Replace explicit tensors by their
      // internal matrix:
      //------------------------------------
      Args:= map([args()], x -> if x::dom = dom and not 
                                   x::dom::isformal(x) 
                                then extop(x, 3) 
                                else x; end_if);

      Args:= map(Args, x -> if x::dom = dom and
                               type(extop(x, 3)) = "_plus" 
                               then op(extop(x, 3))
                               else x; end_if);
      //----------------------------------------------
      // The only remaining pseudo tensors in Args are
      // **formal** ones. If there is no formal tensor
      // left, just do the sum of the scalars
      // and matrices and do return:
      //----------------------------------------------
      if not has(map(Args, domtype), dom) then
         result:= _plus(op(Args));
         if result::dom::hasProp(Cat::Matrix) = TRUE then
            return(dom::new(op(result::dom::matdim(result)), result));
         else
            error("unexpected case in _plus");
         end_if;
      end_if;

      // --------------
      // remove zero terms
      // --------------
      Args:= map(Args, x -> if iszero(x) then null() else x end_if);

      //------------------------------------
      // split the terms into scalars and matrices
      // versus formal tensors.
      //------------------------------------
      [formals, others, dummy]:= split(Args, x -> (domtype(x) = dom));

      others:= _plus(op(others));
      if iszero(others) then
          result:= formals;
      else
          result:= [others].formals;
      end_if;

      // -------------------------------------------------
      // Now, result = [matrix or scalar, formal1, formal2, ...]
      // --------------------------------------------
      // Check for correct dimensions and
      // throw an error if there is a mismatch
      //--------------------------------------
      [r, c]:= [0, 0];
      for i from 1 to nops(result) do
          if (result[i])::dom::hasProp(Cat::Matrix) = TRUE then
            [rr, cc]:= (result[i])::dom::matdim(result[i]);
            if [r, c] = [0, 0] then // first matrix or tensor found
               [r, c]:= [rr, cc];   // set the dimension of the sum
               next;
            else
               if (domtype(r) = DOM_INT and 
                   domtype(rr) = DOM_INT and
                   r <> rr) or (
                   domtype(c) = DOM_INT and 
                   domtype(cc) = DOM_INT and
                   c <> cc) then
                  error("cannot add matrices/tensors: dimensions do not match");
               end_if;
            end_if;
          end_if;
      end_for:

      assert(r <> 0 and c <> 0);

      //-----------------------------------------------
      // combine formal pseudo tensors with different factors
      //  c1*formal + c2*formal + .. -->  (c1 + c2)*formal + ..
      //  formal*M1 + formal*M2 + .. -->  formal*(M1 + M2) + ..
      //  etc.
      //-----------------------------------------------
      mult2list:= (x) ->
            // The first operand of a _mult expression with
            // formal pseudo tensors is the coefficient
            if (op(x, 1))::dom <> dom then
               x:= [op(x, 1..nops(x))];
            else
               x:= [R::one, op(x, 1..nops(x))];
            end_if;
      i:= 0:
      while i < nops(result) do
         //combine multiples of the i-th operand of the sum
         i:= i + 1;
         x:= result[i];
         if x::dom::isformal(x) <> TRUE then
            next;
         end_if;
         [rx, cx] := [extop(x, 1..2)];
         if domtype(extop(x, 3)) <> DOM_STRING then
            x:= extop(x, 3);
         end_if;
         if type(x) = "_mult" then
           x:= mult2list(x);
         else
           x:= [R::one, x];
         end_if;
         xfound:= FALSE;
         for j from i+1 to nops(result) do
            y:= result[j];
            if y::dom::isformal(y) <> TRUE then
               next;
            end_if;
            [ry, cy] := [extop(y, 1..2)];
            if domtype(extop(y, 3)) <> DOM_STRING then
              y := extop(y, 3);
            end_if;
            if type(y) = "_mult" then
               y:= mult2list(y);
            else
               y:= [R::one, y]:
            end_if;
            if [rx, cx] <> [ry, cy] then
               next;
            end_if;
            if nops(x) > nops(y) then
               y:= [op(y), (R::one $ nops(x) - nops(y))]
            end_if;
            if nops(x) < nops(y) then
               x:= [op(x), (R::one $ nops(y) - nops(x))]
            end_if;
            z:= zip(x, y, bool@_equal);
            zz:= select(z, bool@_equal, FALSE);
            if nops(zz) > 1 then next; end_if;
            if nops(zz) = 0 then
               xfound:= TRUE;
               x[1]:= x[1] + y[1];
               result[j]:= NIL;
            else //only one factor in x and y differs. 
                 // Add these factors
               k:= contains(z, FALSE);
               if (x[k])::dom::isformal(x[k]) = TRUE or
                  (y[k])::dom::isformal(y[k]) = TRUE then
                  next;
               end_if;
               xfound:= TRUE;
               x[k]:= x[k] + y[k];
               result[j]:= NIL;
            end_if;
         end_for:
         if xfound then
            x:= select(x, _not@_equal, R::one);
            // get rid of factors R::one
            if nops(select(x, iszero)) > 0 and
               domtype(rx) = DOM_INT and
               domtype(cx) = DOM_INT then
                 result[i]:= new(dom, rx, cx, Dom::Matrix(R)(rx, cx));
            elif nops(x) > 1 then
                 result[i]:= new(dom, rx, cx, hold(_mult)(op(x)));
            elif (result[1])::dom = dom then
                 return(result[1]);
            else result[i]:= new(dom, rx, cx, x[1]);
            end_if;
            result:= subs(result, NIL = null());
         end_if;
      end_while;

      //-----------------------------------------------
      // return a formal pseudo tensor
      //-----------------------------------------------
      if nops(result) = 1 then 
        if (result[1])::dom = dom then
          return(result[1]);
        else
          return(dom::new(r, c, op(result) /*, furtherargs??*/));
        end_if;
      else
        return(dom::new(r, c, hold(_plus)(op(result)) /*, furtherargs??*/));
      end_if;
   end_proc;

//----------------------------------------------
//----------------------------------------------
_negate:= x -> (-1)*x;
   
//----------------------------------------------
// _mult: multiplication of scalars and matrices
//        by tensor symbols
//----------------------------------------------
_mult:= proc()
   local Args, tmp, result, start, finish, 
         i, r, c, dim1, dim2, scalars, m1, m2;
   begin
      if args(0) = 0 then return(R::one); end_if;
      if args(0) = 1 then return(args(1)); end_if;
      //------------------------------------
      // Replace explicit tensors by their
      // internal matrix:
      //------------------------------------
      Args:= map([args()], x -> if x::dom = dom and not 
                                   x::dom::isformal(x) 
                                then extop(x, 3) 
                                elif x::dom = dom and  
                                   x::dom::isformal(x) and 
				   x::dom::matdim(x)[1] = 1 and 
				   x::dom::matdim(x)[2] = 1 
				then 
                                   extop(x,3);
				else 
				   x; end_if);

      Args:= map(Args, x -> if x::dom = dom and
                               type(extop(x, 3)) = "_mult" 
                               then op(extop(x, 3))
                               else x; end_if);
      //----------------------------------------------
      // The only remaining pseudo tensors in Args are
      // **formal** ones. If there is no formal tensor
      // left, just do the multiplication of the scalars
      // and matrices and do return:
      //----------------------------------------------
      if not has(map(Args, domtype), dom) then
         if traperror((result:= _mult(op(Args)))) <> 0 then
	   result:= Args[1];
	   for i from 2 to nops(Args) do
 	     if result::dom::hasProp(Cat::Matrix) = TRUE and  
	        (Args[i])::dom::hasProp(Cat::Matrix) = TRUE then 
               m1:= result::dom::matdim(result);
	       m2:= (Args[i])::dom::matdim(Args[i]);
               if m1[1] = m2[1] and m1[2] = 1 and m2[2] = 1 then 
                 result:= result * matrix([Args[i]]);
	       elif m1[1] = 1 and m2[1] = 1 and m1[2] = m2[2] then 
	         result:= matrix([result]) * Args[i];						    
	       else 
	         result:= result * Args[i]; 
	       end_if;
	     else
	       result:= result * Args[i];
	     end_if;
	   end_for;
	 end_if;
         if result::dom::hasProp(Cat::Matrix) = TRUE then
	    m1:= result::dom::matdim(result);
	    if m1[1] = 1 and m1[2] = 1 then
 	       result:= dom::new(op(result::dom::matdim(result)), result);
	       result:= result::dom::flatten(result);
	       return(result);
	    else 
               return(dom::new(op(result::dom::matdim(result)), result));
	    end_if;
         else
            error("unexpected case in _mult");
         end_if;
      end_if;
      //-----------------------------------------------
      // Do the multiplication of matrices and scalars:
      // Multiply several matrices between formal tensor symbols:
      // formal1 * mat1 * mat2 * formal2 * mat3 * ..
      // --> result = [formal1 , (mat1 * mat2) , formal2 , (mat3 * ..)]
      //-----------------------------------------------
      tmp:= Args[1]:
      if domtype(tmp) = dom then
         result:= [Args[1]];
         start:= 2;
      else
         result:= [];
         start:= 1;
      end_if;
      repeat
        finish:= start;
        if finish > nops(Args) then break; end_if;
        while domtype(Args[finish]) <> dom do
          finish:= finish + 1;  // investigate the next term
          if finish > nops(Args) then break; end_if;
        end_while;
        if finish > start then
           result := result.[_mult(Args[i] $ i = start .. finish - 1)]:
        end_if;
        if finish <= nops(Args) then
           result:= result.[Args[finish]];
        end_if;
        start:= finish + 1;
      until finish > nops(Args) end_repeat;

      // ----------------------------------------------
      // Now, result = 
      // [formal, matrix or scalar, formal, matrix or scalar, ...]
      //--------------------------------------
      // check for correct dimensions of the factors
      // and throw an error if there is a mismatch.
      // At the same time, search for the dimension 
      // of the product (store in r, c)
      //--------------------------------------
      r:= 0: // the row dimension of the product
      c:= 0: // the col dimension of the product
      for i from 1 to nops(result) - 1 do
          dim1:= (result[ i ])::dom::matdim(result[ i ]);
          dim2:= (result[i+1])::dom::matdim(result[i+1]);

          // store the first row dimension that is found in r
          // store the last col dimension that is found in c
          if dim1 <> FAIL then 
             c:= dim1[2];  // in case dim2 does not provide c
             if r = 0 then r:= dim1[1]: end_if;
          end_if;
          if dim2 <> FAIL then 
             c:= dim2[2]; 
             if r = 0 then
                r:= dim2[1]; // in case dim1 does not provide r
             end_if;
          end_if;
          if dim1 = FAIL or dim2 = FAIL then next end_if;

          if domtype(dim1[2]) = DOM_INT and
             domtype(dim2[1]) = DOM_INT and
             dim1[2] <> dim2[1] then
             error("cannot multiply matrices/tensors: ".
                   "dimensions do not match");
          end_if;
      end_for:

      assert(r <> 0 and c <> 0);

      //--------------------
      // collect the scalars
      //--------------------
      scalars:= R::one;
      for i from 1 to nops(result) do
          if (result[i])::dom::hasProp(Cat::Matrix) <> TRUE then
             scalars:= scalars*result[i];
             result[i]:= NIL;
          end_if;
      end_for:
      result:= subs(result, NIL = null());

      //-----------------------------------------------
      // stick the scalars into the first matrix object
      // or tensor that is found among the factors of the
      // product
      //-----------------------------------------------
      if scalars <> R::one then
        for i from 1 to nops(result) do
           if (result[i])::dom::hasProp(Cat::Matrix) = TRUE then
              if (result[i])::dom <> dom then
                  result[i]:= scalars*result[i];
                  scalars:= R::one;
                  break;
              end_if;
           end_if;
        end_for:
      end_if:

      if scalars <> R::one then
         result:= [scalars, op(result)];
      end_if;

      //-----------------------------------------------
      // Check for 0 scalars and matrices
      //-----------------------------------------------
      if has(map(result, iszero), TRUE) then
         if domtype(r) = DOM_INT and
            domtype(c) = DOM_INT then
            return(dom::new(r, c, [[R::zero $ c] $ r]));
         end_if;
      end_if;

     //-----------------------------------------------
     // return a formal pseudo tensor
     //-----------------------------------------------
     if nops(result) = 1 then
       return(dom::new(r, c, op(result) /*, furtherargs??*/));
     else
       return(dom::new(r, c, hold(_mult)(op(result)) /*, furtherargs??*/));
     end_if;
   end_proc;


//-------------------------------------
// modify initDomain to allow automatic
// generation of methods using the 
// methods in Dom::Matrix
//-------------------------------------
initDomain:= proc()
             option escape;
             local matrixmethods, old;
             begin
               old := dom::make_slot;
               // the following methods of Dom::Matrix(R)
               // can be used with Dom::PseudoTensor(R).
               // These methods are forwarded to the
               // internal matrix:
               matrixmethods:= {
                // "new",    // there is a special overloading of new
                // "_index", // there is a special overloading of _index
                // "iszero", // there is a special overloading of iszero
                // "_mult",  // there is a special overloading of _mult
                // "_plus",  // there is a special overloading of _plus
                   "float",
                   "set_index",
                   "doprint",
                   "op",
                   "expr",
		   //"subs",
 /* Dom::Matrix has the following further methods:
    Just uncomment a method and it is forwarded automatically
    to the internal matrix, if the pseudo tensor has an internal
    matrix representation:
                   "create",
                   "mkSparse",
                   "printMaxSize",
                   "setPrintMaxSize",
                   "_multNC1",
                   "_multNC2",
                   "_mult1",
                   "_mult2",
                   "_power",
                   "_invert",
                   "_divide",
                   "testeq",
                   "equal",
                   "gaussElim",
                   "transpose",
                   "nonZeroes",
                   "nonZeros", // = dom::nonZeroes;
                   "nonZeroOperands",
                   "indets",
                   "stackMatrix",
                   "concatMatrix",
                   "setRow",
                   "setCol",
                   "row",
                   "col",
                   "multCol",
                   "multRow",
                   "addCol",
                   "addRow",
                   "delRow",
                   "delCol",
                   "swapRow",
                   "swapCol",
                   "assignElements",
                   "TeX",
                   "map",
                   "mapNonZeroes",
                   "simplify",
                   "Simplify",
                   "normal",
                   "_mod",
                   "mods",
                   "modp",
                   "subsex",
                   "evalAt",
                   "has",
                   "length",
                   "conjugate",
                   "random",
                   "tr",
                   "norm",
                   "exp",
                   "sin",
                   "cos",
                   "zip",
                   "subsop",
                   "nops",
                   "expand",
                   "diff",
                   "int",
                   "identity",
                   "is",
                   "factor",
                   "evalp",
                   "kroneckerProduct",
                   "fourier,
                   "invfourier,
                   "laplace,
                   "invlaplace,
                   "multcoeffs,
                   "mapcoeffs",
               */
                   null()
               };

               dom::make_slot := 
               proc(d, entry)
               option escape;
               begin
                 // forward the method to the internal matrix
                 // if extop(object, 3) can be converted to a
                 // matrix:
                 if contains(matrixmethods, entry) then
                   proc()
                   local x, y, furtherargs;
                   begin
                       x:= args(1):
                       if extnops(x) > 3 then
                          furtherargs:= extop(x, 4..extnops(x));
                       else
                          furtherargs:= null();
                       end_if;
                       x:= x::dom::convert_to(x, Dom::Matrix(R));
                       if x = FAIL then
                          return(FAIL);
                       end_if;
                       y:= slot(Dom::Matrix(R), entry)(x, args(2..args(0)));
                       if y::dom::hasProp(Cat::Matrix) = TRUE then
                          return(dom::new(op(y::dom::matdim(y)), y, furtherargs));
                       else
                          return(y);
                       end_if;
                   end_proc:
                 else
                   old(args());
                 end_if;
               end_proc;
             end_proc;

map:= proc(p, f)
      local A;
      begin
        A:= extop(p, 3);
        if domtype(A) = DOM_STRING then
           return(hold(map)(args()));
        end_if;
        if f = stdlib::diff then
          return(
            p::dom::new(map(A, args(2..args(0))))
          );
        else
          return(
            p::dom::new(extop(p, 1), extop(p, 2), map(A, args(2..args(0))))
          );
        end_if;
      end_proc;

subs:= proc(p) 
       begin
         if not p::dom::isformal(p) then 
 	   p::dom::new(extop(p, 1), extop(p, 2), map(extop(p, 3), subs, args(2..args(0))))
	 else
	   return(p)
	 end_if;
       end_proc;

diff:= proc(p, var)
         local m,n, i,res;
       begin
          if p::dom::constructor = Dom::PseudoTensor then
             [m, n]:= p::dom::matdim(p);
             if domtype(var) = DOM_LIST then
	       // return(map(p, stdlib::diff, args(2..args(0))));
               res:= Dom::PseudoTensor()(1,nops(var), [[diff(p, args(3..args(0)), var[i]) 
	                                    $ i = 1..nops(var)]]);
	       if m = 1 or n = 1 then 
	         return(res::dom::flatten(res));
	       else 
	         return(res);
	       end_if;
	     else
               return(map(p, diff, args(2..args(0))));
	     end_if;
          end_if;
          FAIL;
       end_proc;

eval:= proc(p)
       begin
         return(
           p::dom::new(extop(p, 1), extop(p, 2), map(extop(p, 3), eval))
         );
       end_proc;

//---------------------------------------------------
// the body of the domain 
//---------------------------------------------------
begin
  if args(0) > 1 then error("wrong number of arguments") end_if;
  if args(0) = 0 then R := Dom::ExpressionField()
  elif R::hasProp( Cat::Ring ) <> TRUE then
    error("the coefficients must be from a Cat::Rng")
  end_if;
end_domain:
// end of file 
