/*
    Dom::Real  --  the field of real numbers

    The elements of Real are numbers of type DOM_INT, DOM_RAT, 
    DOM_FLOAT or DOM_EXPR. The test whether an expression is real is 
    performed by calling 'rectform'.

    The domain has category Cat::Field, which is very optimistic 
    (indeed false, because for example bool(1.0 = float(3) / float(3))
    FALSE. The category is assumed only because of pragmatism.
    Note that elements may not have an unic representation, for example
    bool( 0 = sin(2)^2 + cos(2)^2 - 1 ) returns FALSE.
*/

domain Dom::Real
  local selectreals;
  inherits Dom::Complex;
  category Cat::OrderedSet, Cat::DifferentialRing, Cat::Field;
  axiom Ax::systemRep, Ax::canonicalOrder,
        Ax::efficientOperation("_divide"), Ax::efficientOperation("_mult"),
        Ax::efficientOperation("_invert");

// ----------------
// --- Methods: ---
// ----------------

convert:= proc(x)
    local tmp;
begin
    if args(0)<>1 then return(FAIL) end_if;

    case domtype(x)
    of DOM_INT     do
    of DOM_RAT     do
    of DOM_FLOAT   do
        return( x )
    of DOM_COMPLEX do
        return( FAIL )
    of DOM_IDENT do
        if contains( Type::ConstantIdents,x ) then
            return( x )
        else  
            return( FAIL )
        end_if
    of DOM_EXPR do
        if testtype(x, Type::Arithmetical)
           and indets(x) minus Type::ConstantIdents = {}
        then
            if iszero(Im(x)) then
                return( Re(x) )
            else
               tmp:= rectform( x );
               if iszero(op(tmp,2)) and op(tmp,3) = 0 and not has(tmp,{hold(Re),hold(Im)}) then
                   return( x )
               end_if
            end_if
        end_if
    otherwise
        return( FAIL )
    end_case
end_proc;

convert_to:= proc(x,T)
begin
    if domtype(x) = T then
        return( x )
    elif T::dom <> DOM_DOMAIN then
        T:= T::dom
    end_if;

    case T
    of DOM_FLOAT do
    of Dom::Float do
        return( float(x) )
    of DOM_COMPLEX do
        return( FAIL )
    of dom do
    of Dom::Complex do
    of Dom::Numerical  do
    of Dom::ArithmeticalExpression do
        return( x )
    of Dom::Integer do
        if domtype(x) = DOM_INT then return( x ) else return( FAIL ) end_if
    of Dom::Rational do
        if domtype(x) = DOM_INT or domtype(x) = DOM_RAT then return( x ) else return( FAIL ) end_if
    otherwise
        if T::constructor = Dom::ExpressionField then
            return(T::coerce( x ))
        else
            return( FAIL )
        end_if
    end_case
end_proc;

_power:= proc(z,n)
begin
    if contains( {DOM_INT,DOM_FLOAT,DOM_RAT}, domtype(n) ) then
        return( z^n )
    elif (n:= dom::coerce(n)) = FAIL then
        error("invalid exponent")
    else
        return( z^n )
    end_if
end_proc;

Im:= 0;

Re:= id;

conjugate:= id;

random:= proc()
    option escape;
    local randgen;
begin
    if args(0) = 0 then 
        return( random() )
    else 
        randgen:= random(args());
        return( () -> randgen() )
    end_if
end_proc;

_less:= _less;

_leequal:= _leequal;

domsolve:=
proc(eqs, vars, options: DOM_TABLE)
begin
  // convert polys to expressions
  eqs:= misc::maprec(eqs, {DOM_POLY}=expr);
  if nops(vars)>1 or contains({DOM_SET, DOM_LIST}, type(eqs)) then
    options[Domain]:= hold(Expr);
    options[VectorFormat]:= TRUE;
    solve(eqs, vars, options);
    selectreals(%, vars, options);
    if %=FAIL then
      hold(solve)(eqs, vars, solvelib::nonDefaultOptions(options))
    else
      %
    end_if
  elif options[VectorFormat] and type(vars) = DOM_LIST and
    not contains({DOM_SET, DOM_LIST}, type(eqs)) then
    options[Domain]:= dom;
    return(solvelib::cartesianPower(solve(eqs, op(vars), options), 1))
  elif contains({DOM_SET, DOM_LIST}, type(vars)) then
    // only one variable - remove brackets or braces, respectively
    hold(_in)(op(vars), solve(eqs, op(vars), options))
  else
    // vars and eqs are identifiers
    options[Domain]:= hold(Expr);
    save vars;
    assume(vars, Type::Real, _and);
    solve(eqs, vars, options) 
  end_if
end_proc;

TeXrep := x -> "\\mathbb{R}";

// ---------------------------
// --- Body of the domain: ---
// ---------------------------
begin

  // local method selectreals to extract the real solutions
  // out of a system
  // solutions must be of type "_in" 
  // may fail in some cases !
  
    selectreals:=
    proc(solutions, vars, options)
      local
      checkRhs:DOM_PROC;
    begin
      
      map(vars, _save);
      map(vars, assume, Type::Real);

// local method checkRhs
      // returns a set (piecewise defined or not)
      // equal to {l} or {} depending whether or not
      // is a list of real solutions
      checkRhs:=
      proc(l: matrix)
        local i, appended, cond;
      begin
        cond:= [];
        appended:= FALSE;

        /*
        for i from 1 to nops(l) do
          if type(l[i]) = "_equal" then
            case is(Im(rhs(l[i])) = 0)
              of FALSE do
              // not a real solution
                return({})
              of UNKNOWN do
                if (indets(rhs(l[i])) intersect {op(vars)})
                  minus Type::ConstantIdents = {} then
                  cond:= cond. [rhs(l[i])];
                else
                // append to list
                  appended:= TRUE;
                  l:= append(l, Im(rectform(rhs(l[i])))=0 )
                end_if;
            end_case;
          end_if;
        end_for;
          */

        /*
        if appended then
          l:= solve(l, vars, Domain=Dom::Real)
        else
*/
          l:= {l};
  //      end_if;
        
        cond:= _and(op(map(cond, _in, R_)));
        piecewise([cond, l], [not cond, {}])
      end_proc;

      case type(solutions)
        of "solve" do
          return(FAIL)
        of piecewise do
          piecewise::extmap(solutions, selectreals, vars, options);
          if has(%, FAIL) then
            return(FAIL)
          else
            return(%)
          end_if
        of DOM_SET do
          map(solutions, checkRhs);
          return(_union(op(%)))
      end_case;

      // default:
      solvelib::cartesianPower(R_, nops(vars)) intersect solutions
   
    end_proc;



    // initialization part of domain
  
    if args(0) <> 0 then
        error("no arguments expected")
    end_if
end_domain:

