/*
    Dom::SquareMatrix -- the domains of square matrices

    Dom::SquareMatrix(n,[R])

    n - dimension (positive integer)
    R - (optional) the coefficient domain (a Cat::Rng)

    A domain Dom::SquareMatrix(n,R) created by Dom::SquareMatrix 
    represents the Cat::Rng of n times n matrices with coefficients 
    from a Cat::Rng R.

    If 'R' is missing then the coefficient domain will be the
    domain 'Dom::ExpressionField()'.

    An element of Dom::SquareMatrix(n,R) has the same representation as
    an element of the domain Dom::Matrix(R). Dom ensures that all 
    methods for Dom::Matrix(R) may be used by Dom::SquareMatrix(n,R).

    Note that the results of the methods "delRow", "delCol", 
    "stackMatrix", "concatMatrix", "row" and "col" will be
    elements of the super domain Dom::/*Dense*/Matrix(R).
 
    The method "create" (inherited from Dom::Matrix(R)) must be used
    with caution because dom method does not tests its arguments
    (thus it is possible to create matrices which are non-square,
    for example).
*/

domain Dom::SquareMatrix(n, R)

    local Rzero, Rone, MatrixR;

    inherits  Dom::/*Dense*/Matrix(R);
    category  Cat::SquareMatrix(R);
    axiom     if R::hasProp( Ax::canonicalRep ) then Ax::canonicalRep end_if;


/*-- Entries --*/
    one:= if Rone <> FAIL then
          extsubsop(MatrixR::identity(n), 0 = dom);
          end_if;
    zero:= extsubsop(MatrixR(n, n), 0 = dom);
    randomDimen:= [n,n];
    Name:= if dom::constructor = Dom::SquareMatrix and
	      R = Dom::ExpressionField() then
                hold(Dom::SquareMatrix)(n) 
           end;

/*-- Methods --*/
    new:= proc(f)
        local r,c,x;
    begin
        case args(0)
        of 0 do
            return( dom::zero )
        of 1 do
            if contains( {DOM_EXEC, DOM_PROC}, domtype(f) ) then
                return( extsubsop( MatrixR(n,n,f),0=dom ) )
            else
                x:= dom::convert(f);
                if x = FAIL then
                    error("invalid argument")
                else
                    return( x )
                end_if
            end_if
        of 2 do
            r:= args(1); c:= args(2);
            if testtype( r,Type::PosInt ) and testtype( c,Type::PosInt ) then
                if r = n and c = n then return( dom::zero )
                else error("invalid dimension")
                end_if
            elif contains( {hold(Diagonal),hold(Banded)},c ) then
                return( extsubsop( MatrixR(n,n,r,c),0=dom ) )
            else
                error("invalid arguments")
            end_if
        of 3 do
        of 4 do
            x:= extsubsop( MatrixR(args()),0=dom );
            if [extop(x,1),extop(x,2)] <> [n,n] then
                error("invalid dimension")
            else
                return( x )
            end_if
        otherwise
            error("wrong no of arguments")
        end_case
    end_proc;

    convert:= proc(e)
        local t;
    begin
        if domtype(e) = dom then return( e ) end_if;
        t := MatrixR::convert( e );
        if t = FAIL then return( FAIL ) end_if;
        if t::dom::matdim(t) <> [n,n] then
            return( FAIL )
        else
            extsubsop( t,0=dom )
        end_if
    end_proc;


   _index:= (x, i, j) -> MatrixR::_index(extsubsop(x, 0 = MatrixR), args(2..args(0)));
/*------------------------------------------------
    _index:= proc(x,i,j)
        local a, r1, r2, b1, b2, e1, e2;
    begin
        if args(0) = 3 then
            if domtype(i) = DOM_INT and domtype(j) = DOM_INT then
                return( context(_index(extop(x,3),i,j)) )
            elif type(i) = "_range" and type(j) = "_range" then
                // return submatrix of x of type Dom::/*Dense*/Matrix(R):
                x:= extop(x,3);
                b1:= op(i,1); e1:= op(i,2);
                b2:= op(j,1); e2:= op(j,2);
                a:= array(1..e1-b1+1, 1..e2-b2+1,
                    ( (r1,r2)=x[b1+r1-1, b2+r2-1] $ r1=1..e1-b1+1 ) $ r2=1..e2-b2+1
                );
                return(new( MatrixR, e1-b1+1, e2-b2+1, a ))
            else
                return( hold(_index)(args()) )
            end_if
        elif domtype(i) = DOM_INT then
            if extop(x,1) = 1 then
                return( context(_index(extop(x,3),1,i)) )
            else
                return( context(_index(extop(x,3),i,1)) )
            end_if
        else
            return( hold(_index)(args()) )
        end_if
    end_proc;
------------------------------------------------*/

    matdim:= x -> [n,n];

    delRow:= proc() 
    begin
        return( MatrixR::delRow(args()) )
    end_proc;

    delCol:= proc()
    begin
        return( MatrixR::delCol(args()) )
    end_proc;

    stackMatrix:= proc()
    begin
        if args(0) = 1 then
            return( args(1) )
        else
            return( MatrixR::stackMatrix(args()) )
        end_if
    end_proc;

    concatMatrix:= proc()
    begin
        if args(0) = 1 then
            return( args(1) )
        else
            return( MatrixR::concatMatrix(args()) )
        end_if
    end_proc;

    identity:= if Rone <> FAIL then
    proc(d)
    begin
        if args(0) <> 1 or not testtype(d,Type::PosInt) then
            error("expecting a positive integer")
        elif d = n then
            return(dom::one)
        else
            MatrixR::identity(d)
        end_if
    end_proc
    end_if;

    row:= proc()
    begin
        MatrixR::row(args())
    end_proc;

    col:= proc()
    begin
        MatrixR::col(args())
    end_proc;

    random:= () -> MatrixR::random(n, n, args());
/*
    random:= if R::random <> FAIL then
        proc()
            local i, j;
        begin
            new( dom,n,n,array( 1..n,1..n,
               [[R::random() $ j=1..n] $ i=1..n]))
           end_proc
    end_if;
*/

/*--
    evalp -- overload 'evalp' for matrices over polynomial coefficient domains
--*/
    evalp:= if R::hasProp(Cat::Polynomial) then
    proc(A)
        local x, y;
    begin
        x:= map(extop(A,3),eval@evalp,args(2..args(0)));
        // Check whether the matrix now consists only of components 
        // of the coefficient ring of the polynomial domain:
        y:= map({op(x)},testtype,R::coeffRing);
        if contains(y,FALSE) then
            // No, hence define the matrix over R:
            return(new( dom,extop(A,1..2),x ))
        else
            // Yes, hence define the matrix over R::coeffRing:
            return(new( Dom::SquareMatrix(n,R::coeffRing),extop(A,1..2),x ))
        end_if
    end_proc
    end_if;

//---------------------------------
// expr2text
//---------------------------------
expr2text:= proc(A)
local L;
begin
  L:= dom::convert_to(A, DOM_LIST):
  expr2text(dom)."(".expr2text(L).")";
end_proc;




//---------------------------------
// the body of the domain:
//---------------------------------
begin
    if args(0) < 1 or args(0) > 2 then error("wrong no of args") end_if;
    if not testtype( n,Type::PosInt ) then
        error("dimension must be a positive integer")
    elif args(0) = 1 then 
        R:= Dom::ExpressionField()
    elif R::dom <> DOM_DOMAIN or R::hasProp( Cat::Rng ) <> TRUE then
        error("expecting a coefficient domain of category 'Cat::Rng'")
    end_if;

    // initialization of local variables:
    Rone:= R::one;
    Rzero:= R::zero;
    MatrixR:= Dom::/*Dense*/Matrix(R)
end_domain:

