/*--------------------------------------
export::stl(filename, xyz, u, v, 
           <Mesh = mesh>, 
          < OutputBox = outputbox,
           <Scaling = scaling> >,
           <Append>,
            Text|Ascii|Bin|Binary|Raw)

   write the triangulation of a surface given by a 
   parametrization [x(u, v), y(u,v), z(u,v)] with 
       u = u_min .. u_max, v = v_min .. v_max 
   to the file called filename. The STL data are 
   scaled such that the surface fits exactly into 
   the output box x0..x1, y0..y1, z0..z1, where 
   x0,..,z1 are specified by the last parameter 
   OutputBox = [x0..x1,y0..y1,z0..z1])

Example: 

export::stl("sphere.stl", 
            [(u,v) -> cos(u)*sin(v), 
             (u, v)-> sin(u)*sin(v), 
             (u, v)-> cos(v)
            ], 
            u = 0..2*PI, 
            v = 0..PI, 
          < mesh >,   // default: Mesh = [25, 25] >,
          < format >  // default: Bin
          < <OutputBox = outputbox>, 
            <Scaling = scaling>  // default: Scaling = Unconstrained
          >
           );

Parameters:
   filename - string (the name of the STL file)
              or an integer (a file descriptor)
   xyz      - a list of expressions 
                  [x(u,v),y(u,v),z(u,v)] 
              or a list 
                  [(u, v) -> x(u, v), 
                   (u, v) -> y(u, v),
                   (u, v) -> z(u, v)]
              of procedures defining a surface
   u        - an equation uname = a .. b
   v        - an equation vname = a .. b
   mesh     - an equation Mesh = [n1, n2]
   outputbox - a list [x0 .. x1, y0 .. y1, z0 .. z1]
               with x0, .. , z1 representing the physical 
               dimensions of the output box.
   scaling  - Constrained or Unconstrained (default)
   format   - either Bin, Binary, Raw (default) or Text, Ascii
  
----------------------------------------- */

export::stl:= proc()
option escape;
local count;
begin
  count:= 0: // utility for counting the stl objects in a file

proc(filename, xyz, u=-1..1, v=-1..1
                   // Mesh = [25, 25]), 
                   // OutputBox = [0..200,0..200,0..300],
                   // Scaling = Unconstrained
                   // Bin
                   // <Append>
                   )
local arg, append, tmp, i,
      format, byteorder,
      mesh, outputbox, scaling,
      res, origFloatFormat,
      plotObjects, argStart;
save solid, vertex, endsolid;
save DIGITS;
begin
  if args(0) < 2 then
     error("expecting at least 2 arguments"):
  end_if;
  if type(xyz)="plotObject" then
    plotObjects := [xyz];
    argStart := 3;
    while (args(0)>=argStart and type(args(argStart))="plotObject") do
      plotObjects := plotObjects.[args(argStart)];
      argStart := argStart+1;
    end_while;
  else
    plotObjects := null();
    argStart := 5;
    if args(0) < 4 then
      error("expecting at least 4 arguments"):
    end_if;
    if domtype(filename) <> DOM_STRING and
      domtype(filename) <> DOM_INT then
      error("1st argument: expecting a file name as a string");
    end_if;
    if domtype(xyz) <> DOM_LIST or nops(xyz) <> 3 then
      error("2nd argument: expecting a parametrization of a surface ".
            "as a list with 3 entries"):
    end_if;
    if type(u) <> "_equal" or type(op(u, 2)) <> "_range" then
      error("3rd argument: expecting an equation u = u0 .. u1"):
    end_if;
    if type(v) <> "_equal" or type(op(v, 2)) <> "_range" then
      error("4th argument: expecting an equation v = v0 .. v1"):
    end_if;
  end_if;
 
  // set defaults
  mesh:= [25, 25]:
  outputbox:= NIL: // use output coordinates given 
                   // by the parametrization
  scaling:= Unconstrained;
  format:= Binary;
  append:= FALSE;
  byteorder:= LittleEndian;

  // overwrite defaults if requested explicitly by the user
  for arg in [args(argStart..args(0))] do
     if arg = Binary then format:= Binary; next; end_if;
     if arg = Bin    then format:= Binary; next; end_if;
     if arg = Raw    then format:= Binary; next; end_if;
     if arg = Text   then format:= Text; next; end_if;
     if arg = Ascii  then format:= Text; next; end_if;
     if arg = Append then append:= TRUE; next; end_if;
     if type(arg) <> "_equal" then
        error("Unexpected argument ".expr2text(arg).
              ". Expecting options such as Binary, Text, ".
              "Mesh = [n1, n2], ".
              "OutputBox = [x0 .. x1, y0 .. y1, z0 .. z1], ".
              "Scaling = Constrained, or ".
              "Scaling = Unconstrained")
     end_if;
     if op(arg, 1) = Mesh then mesh:= op(arg, 2): next; end_if;
     if op(arg, 1) = OutputBox then outputbox:= op(arg, 2); next; end_if;
     if op(arg, 1) = Scaling then scaling:= op(arg, 2); next; end_if;
     error("Unexpected argument ".expr2text(arg));
  end_for:

  if domtype(mesh) <> DOM_LIST or 
     nops(mesh) <> 2 or
     domtype(mesh[1]) <> DOM_INT or
     domtype(mesh[2]) <> DOM_INT or
     mesh[1] < 2 or
     mesh[2] < 2 then
     error("Expecting an equation Mesh = [n1, n2] ".
           "with integers n1 > 1, n2 > 1"):
  end_if;
  if outputbox <> NIL and
     (domtype(outputbox) <> DOM_LIST or
      nops(outputbox) <> 3 or
      type(outputbox[1]) <> "_range" or
      type(outputbox[2]) <> "_range" or
      type(outputbox[3]) <> "_range") then
       error("Expecting a specification ".
             "OutputBox = [x0 .. x1, y0 .. y1, z0 .. z1] ".
             "with numbers x0, .. , z1 representing the ".
             "physical dimensions of the output");
  end_if;
  if not contains({Constrained, Unconstrained}, scaling) then
       error("Expecting a specification ".
             "Scaling = Constrained or Scaling = Unconstrained, ".
             "got: Scaling = ".expr2text(scaling));
  end_if;
  if domtype(filename) <> DOM_STRING then
    if append then
      error("the option 'Append' is only available if the ".
            "output file is specified by a character string");
    elif outputbox <> NIL then
      error("the option 'OutputBox' is only available if the ".
            "output file is specified by a character string");
    end_if;
  end_if;

  plot::STL::initialize();
  // the output box:
  if outputbox <> NIL then
     plot::STL::setScaling(float(outputbox), scaling);
  end_if;

  origFloatFormat:= Pref::floatFormat();
  if plotObjects=null() then
    plotObjects := [plot::Surface(xyz, u, v, UMesh=mesh[1], VMesh=mesh[2])];
  end_if;
  if type(filename)=DOM_STRING then
    plot::STL::openOutputFile(filename, format, append);
  else
    plot::STL::setOutputFile(filename, format, append);
  end_if;
  res := plot::STL(i) $ i in plotObjects;
  plot::STL::flushOutputFile();
  tmp := plot::STL::closeOutputFile();
  Pref::floatFormat(origFloatFormat);
  return();
end_proc:
end_proc():
