/*--
FILE: CF.mu
DESC: general code generator

// see generate::C(...)
// see generate::fortran(...)
// see generate::MATLAB(...)
// see generate::Simscape(...)

     domain_or_funcenv::CF(e, p, t, opts)
        e       : expression
        p       : priority of expression
        t       : expression type ("double" or "int")
        opts : option table

     The return value is a charcater string that contains the code generated 
     according to the specifications given in 'opts'.

--*/


/*--
generate::CF -- return CF-formatted string for expression
generate::CF(e, opts)
  e - expression, equation or list of equations

NOTE: This function must not be called recursively!!! 
      Use corresponding subroutines or slots instead.
--*/
generate::CF:= proc(argsUneval, userOpts, opts) 
   option hold;
   local  optsErr, argsEval, argsT, isEvaluated;
begin
   if args(0) <> 3 then 
      error("wrong number of arguments");
   end_if;

   // Evaluate defaults in the context of the calling function
   // and check the values.
   opts:= context(opts);
   if domtype(opts) <> DOM_TABLE then
      error("table of defaults expected as third argument");
   end_if;
   if (optsErr:= generate::CFcheckDefaults(opts)) <> "" then
      (()->error(optsErr))();
   end_if;

   // Evaluate user options in the context of the calling function
   // and interpret and check the specified values.
   userOpts:= context(userOpts);
   if domtype(userOpts) <> DOM_LIST then
      error("list of user options expected as second argument");
   end_if;
   // Merge global user options and lokal user options and check them
   userOpts:= generate::CFcheckUserOptions(generate::CFdefaults.userOpts, opts);
   
   // Merge verified user options to defaults and/or overwrite defaults
   opts:= table(opts, userOpts);
     
   // Initialize name generators for scalar variables, array variables and functions
   argsT:= [context(argsUneval)];  // prevent generators from creating these variables
                                   // use a list to avoid assignments of procedures!!!
   if generate::CFnewScalarVariable = FAIL or not opts["ContinueEnumeration"] then
      sysassign(generate::CFnewScalarVariable, 
                prog::genidentgenerator(opts["NameOfScalar"], 
                                        "",
                                        opts["NameIndexFirst"],
                                        opts["NameIndexDigits"],
                                        argsT));
   end_if;
   if generate::CFnewArrayVariable = FAIL or not opts["ContinueEnumeration"] then
      sysassign(generate::CFnewArrayVariable, 
                prog::genidentgenerator(opts["NameOfArray"],  
                                        "",
                                        opts["NameIndexFirst"],
                                        opts["NameIndexDigits"],
                                        argsT));
   end_if;
   if generate::CFnewFunctionName = FAIL or not opts["ContinueEnumeration"] then
      sysassign(generate::CFnewFunctionName, 
                prog::genidentgenerator(opts["NameOfFunction"],  
                                        "",
                                        opts["NameIndexFirst"],
                                        opts["NameIndexDigits"],
                                        argsT));
   end_if;
   if generate::CFnewDerivativeName = FAIL or not opts["ContinueEnumeration"] then
      sysassign(generate::CFnewDerivativeName,
                prog::genidentgenerator((if opts["DerivativeVectorNotation"] then
                                            opts["NameOfDerivative"].opts["IndexLeft"];
                                         else
                                            opts["NameOfDerivative"];
                                         end_if), 
                                        (if opts["DerivativeVectorNotation"] then
                                            opts["IndexRight"];
                                         else
                                            "";
                                         end_if),
                                        opts["NameIndexFirstDerivative"],
                                        opts["NameIndexDigits"],
                                        argsT));
   end_if;

   isEvaluated:= FALSE:

   // We want to operate on lists in order to make mapping easy and also to
   // avoid any side effects on matrices and procedures when assigning them
   // to variables, e.g. the automatic change of the procedure name!!!
   if type(argsUneval) = "last" then
      // Special case: incoming args must be evaluated immediately in order
      // to resolve last operator, e.g. %, %1,...
      argsUneval:= [context(argsUneval)];
      isEvaluated:= TRUE:
   else
      argsUneval:= [argsUneval];
   end_if;
    
   // "Mask" objects of types listed in generate::MatrixTypes in order to  
   // make their names still available after the input is fully evaluated.
   // IMPORTANT: this must be done before the arguments are evaluated.
   argsUneval:= generate::CFdoMaskMatrixes(argsUneval);

   // "Mask" piecewise objects in order to make their names still available
   // after the input is fully evaluated.
   // IMPORTANT: this must be done *after* masking matrices since they 
   //            may hold piecewise objects. Furthermore, the function
   //            CFdoMaskPiecewise may evaluate matrices on the fly.
   // IMPORTANT: this must be done before the arguments are evaluated.
   argsUneval:= generate::CFdoMaskPiecewise(argsUneval);

   // Evaluate the input expression in the context of the calling function!
   if isEvaluated then
      argsEval:= argsUneval;
   else
      argsEval:= context(argsUneval);
   end_if;

   // Convert 1x1 matrices to scalar expressions, if needed.
   // Needs to be done before (auto)assignments are handled.
   // Assumption: matrices are masked and already evaluated.
   if opts["Convert1x1MatrixToScalar"] = TRUE then
      argsEval:= misc::maprec(argsEval, {generate::Matrix} = (()->(
                    // Is object generate::Matrix(<name>,<matrix>)
                    // Get matrix dimensions: op(<matrix>,0)
                    // e.g.: 2, 1..3, 1..3 => [1..3, 1..3] => {1..3, 1..3}
                    if {op(op(op(args(),2),0)[2..-1])} = {1..1} then 
                       // Is one-dimensional matrix: convert to scalar expression
                       op(op(args(),2),1);
                    else
                       // do not touch this operand!
                       args();
                    end_if;
                 )));
   end_if;
   
   // Change 'a = ...' to 'a := ...' 
   // This is the default for generate::{C|FORTRAN|MATLAB} since ages.
   if opts["SubsLeftEqualByAssign"] = TRUE then
      argsEval:= generate::CFdoSubsLeftEqualByAssign(argsEval);
   end_if;

   // Unwrap ODEs to sequences of expressions and equations
   // IMPORTANT: handle 'generate::MatrixTypes' first!
   // IMPORTANT: handle 'piecewise' first!
   // IMPORTANT: handle 'SubsLeftEqualByAssign' first!
   if opts["UnwrapOde"] = TRUE then
      argsEval:= generate::CFunwrapOdes(argsEval);
   end_if;

   // Change '...' to 't0 := ...'
   // This is the default for generate::{C|FORTRAN|MATLAB} since ages.
   // IMPORTANT: handle 'generate::MatrixTypes' first!
   // IMPORTANT: handle 'piecewise' first!
   // IMPORTANT: handle 'ode' first!
   // IMPORTANT: handle 'SubsLeftEqualByAssign' first!
   if opts["AutoAddAssign"] = TRUE then
      argsEval:= generate::CFdoAutoAddAssign(argsEval);
   end_if;

   if opts["ReduceOrderOfDerivaties"] = TRUE then
      argsEval:= generate::CFreduceOrderOfDerivaties(argsEval, opts, TRUE);
   end_if;

   // Lists and expression sequences are interpreted as statements sequences.
   // This is the default for generate::{C|FORTRAN|MATLAB} since ages. Embed
   // other objects in a statement sequence (with one statement) to simplify
   // code formatting. 
   argsT:= type(op(argsEval));
   if argsT = DOM_LIST or argsT = "_exprseq" or argsT = "_stmtseq" then
      argsEval:= hold(_stmtseq)(op(op(argsEval)));
   else
      argsEval:= hold(_stmtseq)(op(argsEval));
   end_if;

   // Convert *all* remaining integers to floating point numbers. This is
   // needed for (very) old FORTRAN 77 compilers which have problems with
   // mixed floats and integers (1-BOAH01).
   if opts["IntToFloat"] = TRUE then
      argsEval:= misc::maprec(argsEval, {DOM_INT}=float)
   end_if;

   // generate code for all objects
   argsT:= generate::CFany(argsEval, output::Priority::Noop, opts["ReturnType"], opts);

   // post processing: format the source code nicely and as needed
   generate::CFformatting(argsT, opts);
end_proc:

//-- --------------------------------------------------------------------------
// This feature should stay undocumented. This is for internal use only!!!
// generate::CFdefaults: Options that are to be added to each CF call.
generate::CFdefaults:= []:

/*--
generate::CFsetDefault -- set additional default options for CF calls
generate::CFsetDefault(opts)
   opts : valid CF options options
--*/
generate::CFsetDefault:= proc(opts)
  local theOpts, state;
begin
   state:= generate::CFdefaults;
   // Return current state
   if args(0) = 0 then
      return(state);
   end_if;
   
   // Ensure that 'generate::Mdefaults' is loaded from 'matlab.mu'
   // It holds all vaild internal option names.
   generate::MATLAB(); 

   if args(0) = 1 and domtype(opts) = DOM_LIST then
      // Set: check and convert to internal options
      if opts = [] then
        theOpts:= [];
      else
        theOpts:= generate::CFcheckUserOptions(opts, generate::Mdefaults);
      end_if;
   else
      // Add: check and convert to internal options 
      theOpts:= state.generate::CFcheckUserOptions([args()], generate::Mdefaults);
   end_if;
   
   // Convert "..."=value into #...=value
   theOpts:= map(theOpts, o->(if domtype(lhs(o)) = DOM_STRING then `#`.lhs(o)=rhs(o) else o end));
   
   // Assign and return previous state
   sysassign(generate::CFdefaults, theOpts);
   return(state);
end_proc:
//-- --------------------------------------------------------------------------


/*-- --------------------------------------------------------------------------
     CF formatting slots of domains -------------------------------------------
     --------------------------------------------------------------------------

     CF formatting slots of domains have the following calling syntax:
     
     domain::CF(e, p, t)
        e    : expression
        p    : priority of expression
        t    : expression type ("double" or "int")
        opts : option table

     The return value is a charcater string that contains the code generated 
     according to the specifications given in 'opts'.
--*/

/*--
DOM_INT::CF -- returns a CF formatted string for integer numbers
--*/
DOM_INT::CF:= proc(e, p, t, opts)
   save DIGITS; 
begin
   DIGITS:= opts["Digits"];
   if opts["keepInteger"]=TRUE or t = "int" or opts["IntAsIntWhereObvious"] = TRUE then
     _concat(
       if e<0 and p > output::Priority::Mult then "(" else "" end_if, // needed? just to be sure!
       expr2text(e),
       if e<0 and p > output::Priority::Mult then ")" else "" end_if  // needed? just to be sure!
     );
   else 
     DOM_FLOAT::CF(float(e), p, t, opts) 
   end_if
end_proc:

/*--
DOM_RAT::CF -- returns a CF formatted string for rational numbers
--*/
DOM_RAT::CF:= proc(e, p, t, opts) 
   save DIGITS;
   local s, neg, pp; 
begin
   DIGITS:= opts["Digits"];
   neg := bool(e < 0);
   pp := p;
   if neg then
     e := -e;
     pp := output::Priority::Noop;
   end_if;
   if t = "int" or opts["IntAsIntWhereObvious"] = TRUE 
      and (opts["Language"] = "MATLAB" or opts["Language"] = "Simscape") then 
      // not allowed in C && FORTRAN, since int / int 
      // rounds towards zero in both languages.
      s:= generate::CFop("/", output::Priority::Mult, pp, t, [op(e,1), op(e,2)], opts);
   else
      s:= generate::CFop("/", output::Priority::Mult, pp, t, [float(op(e,1)), float(op(e,2))], opts);
   end_if;
   if neg then
     s := _concat(
       if p >= output::Priority::Mult then "(-" else "-" end_if,  // to be on the safe side
       s,
       if p >= output::Priority::Mult then ")"  else ""  end_if); // to be on the safe side
   end_if;
   s;
end_proc:

/*--
DOM_FLOAT::CF -- returns a CF formatted string for floating point numbers
--*/
DOM_FLOAT::CF:= proc(e, p, t, opts) 
   save  DIGITS;
   local saveFormat, ep;
begin
   if t = "int" and opts["TypeChecking"] then 
      error("not an integer: '".e."'");
   end_if;
   if e = RD_NAN then
      if opts["NaN"] = FAIL then
         return(expr2text(e));
      else
         return(opts["NaN"]);
      end_if;
   end_if;
   if e = RD_INF then
      if opts["Inf"] = FAIL then
         return(expr2text(e));
      else
         return(opts["Inf"]);
      end_if;
   end_if;
   if e = RD_NINF then
      if opts["Inf"] = FAIL then
         return(expr2text(e));
      else
         return("-".opts["Inf"]);
      end_if;
   end_if;
   if e = 0 or iszero(e) then
      if opts["Language"] = "FORTRAN" then 
         return("0.0D0");
      else
         return("0.0");
      end_if;
   end_if;

   DIGITS:= opts["Digits"];
   // use normalized scientific notation x.yEz
   saveFormat:= Pref::floatFormat(opts["DoubleFormat"]);
   e:= expr2text(float(e)); // ensure that it's a floating point number
   Pref::floatFormat(saveFormat);

   // remove trailing 0's
   e:= text2list(e, [".", "e"], Cyclic);
   ep:= op(e,3);
   while ep[-1] = "0" do
      if length(ep) = 1 then 
         break;
      end_if;
      ep:= ep[1..-2];  // only if Pref::trailingZeroes(TRUE)
   end_while;

   if opts["Language"] = "FORTRAN" then 
      // [mantissa,exponent] => [mantissa,exponent,"D","0"]
      if nops(e) = 3 then
         e:= e.["D", "0"]; // currently not needed, but keeps code robust for the future
      end_if;
   else
      // [mantissa,exponent,"e","0"] => [mantissa,exponent]
      if nops(e) >= 3 and e[-2..-1] = ["e", "0"] then
         e:= e[1..-3];
      end_if;
      
   end_if;

   _concat(
     // use parenthesis only in terms with a higher priority than products have
     if e[1][1]="-" and p > output::Priority::Mult then "(" else "" end_if, // to be on the safe side
     op(subs(subsop(e, 3=ep),"e"=opts["DoubleEnotationChar"])),
     if e[1][1]="-" and p > output::Priority::Mult then ")" else "" end_if  // to be on the safe side
   );
end_proc:

/*--
DOM_COMPLEX::CF -- returns a CF formatted string for complex numbers
--*/
DOM_COMPLEX::CF:= proc(e, p, t, opts)
   save DIGITS;
   local plus;
begin
   DIGITS:= opts["Digits"];
   if opts["ComplexI"] = FAIL then
      opts["ComplexPrefix"].
      "(".DOM_FLOAT::CF(float(op(e,1)), output::Priority::Noop, t, opts).",".
          DOM_FLOAT::CF(float(op(e,2)), output::Priority::Noop, t, opts).")";
   else
      // complex numbers are written as expressions
      _concat(
         (if p > output::Priority::Plus and not iszero(op(e,1)) and not iszero(op(e,2)) then "(" else "" end_if),
         (if   op(e,1) =  0 or iszero(op(e,1)) then
             plus:= "";
          else
             plus:= "+";
             DOM_FLOAT::CF(float(op(e,1)), output::Priority::Plus, t, opts);  // here we are in a plus context
          end_if),
         (if   op(e,2) =  1 or iszero(op(e,2) -1) then
             if opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then plus."1" else plus end_if;
          elif op(e,2) = -1 or iszero(op(e,2) +1) then
             if opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then  "-"."1" else  "-" end_if;
          else
             (if op(e,2) < 0 then "" else plus end_if).
             DOM_FLOAT::CF(float(op(e,2)), output::Priority::Plus, t, opts).  // here we are in a plus context
             (if contains({"MATLAB", "Simscape"}, opts["Language"]) and not contains({RD_INF, RD_NINF}, op(e,2)) then "" /* write 1i, 1.0i, 1/2i */ else "*" end_if); 
          end_if),
         opts["ComplexI"], 
         (if p > output::Priority::Plus and not iszero(op(e,1)) and not iszero(op(e,2)) then ")" else "" end_if)  // parentheses around a complex number?!
      );
   end_if;
end_proc:

/*--
Factored::CF -- returns a CF formatted string for Factored
--*/
Factored::CF:= proc(e, p, t, opts)
begin
   // convert to plain expression
   generate::CFany(expr(e), p, t, opts);
end_proc:

/*--
rectform::CF -- returns a CF formatted string for rectform
--*/
rectform::CF:= proc(e, p, t, opts)
begin
   // convert to plain expression
   generate::CFany(expr(e), p, t, opts);
end_proc:

/*--
Series::Puiseux::CF -- returns a CF formatted string for Series::Puiseux
--*/
Series::Puiseux::CF:= proc(e, p, t, opts)
begin
   // convert to plain expression: removing ...+O(...)
   generate::CFany(expr(e), p, t, opts);
end_proc:

/*--
unit::CF -- returns a CF formatted string for unit
--*/
unit::CF:= proc(e, p, t, opts)
   local theUnit, newName;
begin
   // Replace units with identifiers, e.g., 'unit::s' => "UNIT_S"
   theUnit:= e;
   newName:= stringlib::upper(stringlib::subs(expr2text(theUnit),"::"="_"));
   warning("Units are not supported. Replacing unit '".expr2text(theUnit)."' with '".newName."'.");
   return(newName);
end_proc:

/*--
stdlib::Infinity::CF -- returns a CF formatted string for infinity
--*/
stdlib::Infinity::CF:= proc(e, p, t, opts)
   local signum;
begin
   // Special case: op(e) might be an expression like with variables and I.
   // => generate p(e) * inifinity
   if e <> hold(infinity) then // Avoid 'recursive definition'
      if not testtype(op(e), Type::Numeric) or has(op(e), I) then
         return(generate::CFany(hold(_mult)(op(e), infinity), p, t, opts));
      end_if;
   end_if;

   // Special case: numerical expression with RD_NAN
   // => generate RD_NAN
   if has(op(e), RD_NAN) then
      return(generate::CFany(RD_NAN, p, t, opts));
   end_if;
   
   // Handles expressions with math constants like RD_NAN,...
   if is(op(e) < 0) = TRUE then
      signum:= "-";
   else
      signum:= "";
   end_if;
   if opts["Inf"] = FAIL then
      signum."infinity";
   else
      signum.opts["Inf"];
   end_if;
end_proc:

/*--
stdlib::Undefined::CF -- returns a CF formatted string for undefined
--*/
stdlib::Undefined::CF:= proc(e, p, t, opts) 
begin
   if opts["NaN"] = FAIL then
      "undefined";
   else
      opts["NaN"];
   end_if;
end_proc:

/*--
ode::CF -- returns a CF formatted string for an ode object
--*/
ode::CF:= proc(e, p, t, opts)
begin
   generate::CFunverified("ode", opts);
   generate::CFcall("ode", t, [op(e)], opts);
end_proc:

/*--
piecewise::CF -- returns a CF formatted string for a piecewise object
--*/
piecewise::CF:= proc(e, p, t, opts, v=#PiecewiseWithNoName)
   local o, i;
begin
   // special case for assignments 'v:= piecewise()'
   if type(e) = "_assign" then
      v:= op(e,1);                          // name of variables to assign to, ignore old 'v'
      e:= op(e,2);                          // piecewise object to generate code for
   elif v = #PiecewiseWithNoName and opts["AutoAddAssignPiecewise"] then
      v:= generate::CFnewScalarVariable();  // generate new scalar variable: for keeping code robust
   end_if;
   
   if type(e) = "function" then             // unevaluated piecewise may lead to invalid code!
      return(generate::CFany(hold(_assign)(v,eval(e)), p, t, opts));
   end_if;

   // Step-by-step substitute the false-branch ('#') of an if-statement with
   // corresponding if-statements for each cond=expr branch of the piecewise
   // object.   The value of the last false-branch is either specified by an 
   // otherwise branch or is set to undefined. The default order of branches
   // is used, assuming that 'Otherwise' is always last. => 'see CF.tst'
   o:= #;
   for i from 1 to nops(e) do
      if piecewise::condition(e,i) = Otherwise then
         o:= subs(val(o), # = (if v = #PiecewiseWithNoName then                 piecewise::expression(e,i)
                                                           else hold(_assign)(v,piecewise::expression(e,i))
                               end_if));
      else
         o:= subs(val(o), # = hold(_if)(piecewise::condition(e,i),
                              (if v = #PiecewiseWithNoName then                 piecewise::expression(e,i)
                                                           else hold(_assign)(v,piecewise::expression(e,i))
                               end_if), #));
      end_if;
   end_for;
   // if no 'Otherwise'
   o:= subs(val(o), # = (if v = #PiecewiseWithNoName then                 undefined 
                                                     else hold(_assign)(v,undefined)
                         end_if));    
   
   // generate code for the newly created nested if-statement
   generate::CFany(o, p, t, opts);
end_proc:

/*--
DOM_BOOL::CF -- returns a CF formatted string for boolean values
--*/
DOM_BOOL::CF:= proc(e, p, t, opts)
begin
   if e = UNKNOWN then
      if opts["Unknown"] = FAIL then
         (()->error("Boolean value UNKNOWN is not supported in '".opts["Language"]."'"))();
      else
         return(opts["Unknown"]);
      end_if;
   end_if;
    
   if e then 
      if opts["True"] = FAIL then
         "TRUE";
      else
         opts["True"];
      end_if;
   else 
      if opts["False"] = FAIL then
         "FALSE";
      else
         opts["False"];
      end_if;
   end_if;
end_proc:

/*--
DOM_POLY::CF -- returns a CF formatted string for polynomials
--*/
DOM_POLY::CF:= proc(e, p, t, opts)
begin
   generate::CFany(expr(e), p, t, opts);
end_proc:

/*--
DOM_STRING::CF -- returns a CF formatted string for character strings
--*/
DOM_STRING::CF:= proc(e, p, t, opts)
begin
   if   opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      "'".e."'";
   elif opts["Language"] = "FORTRAN" then
      "'".e."'";
   else  // "C"
      expr2text(e);
   end_if;
end_proc:

/*--
DOM_IDENT::CF -- returns a CF formatted string for identifiers
DOM_IDENT::CF(e, p, t, opts)
   e : expression
   p : priority of expression
   t : expression type ("double" or "int")
--*/
DOM_IDENT::CF:= proc(e, p, t, opts)
   local et;
begin
   et:= expr2text(e);

   // Special case: Other languages often do not allow variable
   // names with characters other than (_A-Za-z0-9). Currently,
   // the user is responsible for that. However, we look for ``
   // and identifiers and throw an error.
   if et = "``" or stringlib::contains(et, " ") then
     (()->error("invalid identifier '".et."'"))();
   end_if;
   
   // Special case: e.g. needed for language Simscape 
   // The name of the 'time variable' must be exchanged, e.g.: 't' -> 'time'
   if et = opts["NameOfTimeVariable"] then
      return(opts["RenameTimeVariableTo"]);
   end_if;
   
   // Handle special math constants
   if domtype(opts["MathConstants"]) = DOM_TABLE and contains(opts["MathConstants"], e) then
      if opts["TypeChecking"] and opts["MathConstants"]["type"] <> t then 
         error("unexpected type when using object '".expr2text(e)."'");
      end_if;
      return(generate::CFany(opts["MathConstants"][e]["value"], output::Priority::Noop, t, opts));
    end_if;
    
   // Handle 'undefined' when it was holded!
   if e = undefined or e = hold(undefined) then
      return(stdlib::Undefined::CF(e, p, t, opts));
   end_if;

   // Handle 'infinity' when it was holded!
   if e = infinity or e = hold(infinity) then
      return(stdlib::Infinity::CF(e, p, t, opts));
   end_if;

   // All other identifiers
   // Handle special identifiers: `&mu;` => `mu`
   stringlib::subs(et, "`"="", "&"="", ";"="");
end_proc:

/*--
DOM_FUNC_ENV::CF -- returns a CF formatted string for function environments
--*/
DOM_FUNC_ENV::CF:= proc(e, p, t, opts)
begin
   prog::getname(e);  // returns the name of it
end_proc:

/*--
DOM_VAR::CF -- returns a CF formatted string for local variables
--*/
DOM_VAR::CF:= proc(e, p, t, opts)
begin
   expr2text(e);  // local variable in procedures
end_proc:

/*--
DOM_NULL::CF -- returns a CF formatted string for the null object
--*/
DOM_NULL::CF:= proc(e, p, t, opts)
begin
   "";  // empty string
end_proc:

/*--
DOM_SET::CF -- returns a CF formatted string for a finite set
--*/
DOM_SET::CF:= proc(e, p, t, opts)
begin
   if e = {} then
      e:= undefined; // will be converted to NaN
      if opts["Warnings"] then
         warning("Empty finite sets (DOM_SET) are not supported. ".
                 "Using '".expr2text(e)."' instead."
         );
      end_if;
   else
      e:= e[1]; // first set element, use '_index' to have the right order!!! 
      if opts["Warnings"] then
         warning("Finite sets (DOM_SET) are not supported. ".
                 "Using element '".expr2text(e)."' instead."
         );
      end_if;
   end_if;
   generate::CFany(e, p, t, opts);  
end_proc:

/*--
solvelib::BasicSet::CF -- returns a CF formatted string for a solvelib::BasicSet object
--*/
solvelib::BasicSet::CF:= proc(e, p, t, opts)
begin
   // Is either Z_, Q_, R_ or C_.
   e:= 1;  // choose randomly: 1 is in all of these sets
   if opts["Warnings"] then
      warning("Number ranges Z_, Q_, R_ and C_ are not supported. ".
              "Using element '".expr2text(e)."' instead." 
      );
   end_if;
   generate::CFany(e, p, t, opts);
end_proc:

/*--
DOM_INTERVAL::CF -- returns a CF formatted string for a DOM_INTERVAL object
--*/
DOM_INTERVAL::CF:= proc(e, p, t, opts)
begin
   if type(lhs(e)) = DOM_FLOAT and type(lhs(e)) = DOM_FLOAT then
      // Handle real case
      // DOM_INTERVALs are closed intervals
      e:= Dom::Interval([lhs(e)], [rhs(e)]);
   else
      // Handle complex case
      e:= undefined; // will be converted to NaN
      if opts["Warnings"] then
         warning("Intervals (DOM_INTERVAL) are not supported. ".
                 "Cannot extract an element. ".
                 "Using '".expr2text(e)."' instead."
         );
      end_if;
   end_if;
   generate::CFany(e, p, t, opts);
end_proc:

/*--
Dom::Interval::CF -- returns a CF formatted string for a Dom::Interval
--*/
Dom::Interval::CF:= proc(e, p, t, opts)
begin
   e:= solvelib::getElement(e); // get any element of this interval
   if e = FAIL then
      e:= undefined; // will be converted to NaN
      if opts["Warnings"] then
         warning("Intervals (DOM_INTERVAL, Dom::Interval) are not supported. ".
                 "Cannot extract an element. ".
                 "Using '".expr2text(e)."' instead."
         );
      end_if;
   else
      if opts["Warnings"] then
         warning("Intervals (DOM_INTERVAL, Dom::Interval) are not supported. ".
                 "Using element '".expr2text(e)."' instead."
         );
      end_if;
   end_if;
   generate::CFany(e, p, t, opts);
end_proc:

/*--
Dom::ImageSet::CF -- returns a CF formatted string for a Dom::ImageSet object
--*/
Dom::ImageSet::CF:= proc(e, p, t, opts)
   local elem;
 begin
   elem:= solvelib::getElement(e); // choose randomly: get any element of this image set
   if elem = FAIL then
      e:= undefined; // will be converted to NaN
      if opts["Warnings"] then
         warning("Image sets (Dom::ImageSet) are not supported. ".
                 "Cannot extract an element. ".
                 "Using '".expr2text(e)."' instead."
         );
      end_if;
   else
      e:= elem;
      if opts["Warnings"] then
         warning("Image sets (Dom::ImageSet) are not supported. ".
                 "Using element '".expr2text(e)."' instead."
         );
      end_if;
   end_if;
   generate::CFany(e, p, t, opts);
end_proc:

/*--
DOM_EXPR::CF -- returns a CF formatted string for expressions
--*/
DOM_EXPR::CF:= proc(e, p, t, opts) 
   local f, var; 
begin
   f:= op(e,0);
   // SPECIAL CASE: 'D::CF' must be called with the whole expression e='D(x)(t)'.
   if type(f) = "D" then
      return(D::CF(e,p,t,opts));
   end_if;
    
   if domtype(f) = DOM_IDENT then
      f:  // ggf. loadproc auflsen
      f:= eval(%);
      if domtype(f) = DOM_FUNC_ENV then
         if f::CF = FAIL then
            generate::CFunverified(op(e,0), opts);       
            if domtype(op(f,2)) = DOM_EXEC then
               generate::CFcall(op(f,[2,4]), t, [op(e)], opts);
            elif domtype(op(f,1)) = DOM_EXEC then
               generate::CFcall(op(f,[1,3]), t, [op(e)], opts); // keeps code robust
            else
               generate::CFcall(generate::CFany(op(e,0), output::Priority::Fconcat, t, opts), t, [op(e)], opts);
            end_if
         else
            f::CF(e, p, t, opts);
         end_if
      elif domtype(f) = DOM_PROC then
         // assumption: procedure f is or will be translated as well
         generate::CFunverified(op(e,0), opts);
         generate::CFcall(generate::CFany(op(e,0), output::Priority::Fconcat, t, opts), t, [op(e)], opts);
      else
         // Special case needed for language Simscape: convert 'x(t)' to 'x' for time variable 't'
         if opts["Language"] = "Simscape" then
            var:= op(e);
            if nops(var) <> 1 then
               // No conversion in this case!!!
               // Should this throw an error in the future?
               // error("Only one time variable '".opts["NameOfTimeVariable"]."' expected in function call '".expr2text(e)."'");
            elif "".var <> opts["NameOfTimeVariable"] then
               // No conversion in this case!!!
               // Should this throw an error in the future?
               // error("Time variable '".opts["NameOfTimeVariable"]."' expected in function call '".expr2text(e)."'");
            else
               return(DOM_IDENT::CF(op(e,0),p,t,opts));
            end_if;
            if "".(op(e,0)) = opts["NameOfTimeVariable"] then
              // t(): Do not rename 't' in this context. Seems not make sense.
              f:= opts["NameOfTimeVariable"];
              generate::CFunverified(f, opts);
              return(generate::CFcall(f, t, [op(e)], opts));
            end_if;
         end_if;
         generate::CFunverified(op(e,0), opts);
         generate::CFcall(generate::CFany(op(e,0), output::Priority::Fconcat, t, opts), t, [op(e)], opts);
      end_if
   else
      generate::CFcall(generate::CFany(f, output::Priority::Fconcat, t, opts), t, [op(e)], opts); // keeps code robust
   end_if
end_proc:

/*--
DOM_ARRAY::CF -- returns a CF formatted string for arrays
--*/
DOM_ARRAY::CF:= proc(e, p, t, opts, v=MatrixWithNoName)
   local s, le, ri, S, i, d, cfg, l, dim, carry, nextRow;
begin
   cfg:= op(e,0);    // config of array, e.g.: 2, 1..3, 1..3
   s:= [cfg[2..-1]]; // ranges of array, e.g.:   [1..3, 1..3]
   l:= map(s,i->(i[2]-i[1]+1)); // sizes of dims, e.g.:[3, 3]

   // Special case: e.g. matrix([]) or matrix([[]]) => then l contains a '0'
   // Note: l cannot contain negative values because array() and matrix() do not allow this.
   if contains(l,0) > 0 then
      (()->error("zero-dimensional arrays and matrices are not supported"))();
   end_if;
    
   if contains(map(s,domtype),DOM_LIST) <> 0 then 
      error("cannot handle block structure of arrays"); // currently not supported, but keeps code robust
   elif contains({op(e)},NIL) then
      error("array contains undefined entries");        // currently not supported, but keeps code robust
   end_if;

   // lists of lower and upper bound of the dimensions
   le:= map(s,op,1); 
   ri:= map(s,op,2);   

   // Accept ranges starting with 1 only
   if {op(le)} <> {1} then
      if opts["Warnings"] then
         warning("array index out of range 1..n: ".expr2text(v).expr2text(s));
      end_if;
   end_if;

   dim:= nops(le); // = nops(ri);
   s:= le;         // first index
   S:= "";         // empty output string
   
   if opts["GenerateMatrixStyle"] = "MATLAB" or opts["Language"] = "Simscape" then
      if opts["GenSparseMatrix"] = TRUE or dim > 2 then
         S:= S.generate::CFany(v, output::Priority::Noop, t, opts)." = zeros("._concat(expr2text(i)."," $ i in l)[1..-2].")".opts["StatementDelimiter"]."\n";
      else
         S:= S.generate::CFany(v, output::Priority::Noop, t, opts)." = [\n";
      end_if;
   end_if;
   
   if s[dim] = ri[dim] then
      nextRow:= TRUE; 
   else
      nextRow:= FALSE; 
   end_if;
   if opts["GenSparseMatrix"] = FALSE or not iszero(_index(e,op(s))) then
      // s => zip(s,le,_subtract) wrde Normierung der Indizes auf den 
      // Bereich 0..n-1 bewirken.  Besser scheint die Annahme zu sein
      // MuPAD: 1..n => C: 0..n-1 => jeden Index um 1 dekrementieren.
      // Siehe hierzu '_index::CF'.
      if (opts["GenerateMatrixStyle"] = "MATLAB" or opts["Language"] = "Simscape") and
         opts["GenSparseMatrix"] = FALSE and dim <= 2 then
         S:= S.generate::CFany(_index(e,op(s)),output::Priority::Noop,t, opts);
         if nextRow then S:=S."\n" else S:=S." "; end_if;
      else
         S:= S.generate::CFany(_index(v,op(s)),output::Priority::Noop,t, opts)." = ".generate::CFany(_index(e,op(s)),output::Priority::Noop,t, opts).opts["StatementDelimiter"];
      end_if;
   end_if;
   if s[dim] = ri[dim] then
      nextRow:= TRUE; 
   end_if;

   // next index
   for i from 2 to nops([op(e)]) do
      nextRow:= FALSE;
      carry:= TRUE;
      d:= dim;
      while carry do
         s[d]:= s[d] + 1;
         if s[d] > ri[d] then 
            s[d]:= le[d]; 
            d:= d - 1;
         else 
            carry:= FALSE;
         end_if; 
      end_while;
      if s[dim] = ri[dim] then
         nextRow:= TRUE; 
      end_if;

      if opts["GenSparseMatrix"] = FALSE or not iszero(_index(e,op(s))) then
         // s => zip(s,le,_subtract) wrde Normierung der Indizes auf den 
         // Bereich 0..n-1 bewirken.  Besser scheint die Annahme sein zu
         // MuPAD: 1..n => C: 0..n-1 => jeden Index um 1 dekrementieren.
         // Siehe hierzu '_index::CF'.
         if (opts["GenerateMatrixStyle"] = "MATLAB" or opts["Language"] = "Simscape") and
            opts["GenSparseMatrix"] = FALSE and dim <= 2 then
            S:= S.generate::CFany(_index(e,op(s)),output::Priority::Noop,t, opts);
            if nextRow then S:=S."\n" else S:=S." "; end_if;
         else
            S:= S. (if S = "" then "" else "\n" end_if)
              . generate::CFany(_index(v,op(s)),output::Priority::Noop,t, opts)." = ".generate::CFany(_index(e,op(s)),output::Priority::Noop,t, opts).opts["StatementDelimiter"];
         end_if;
      end_if;
   end_for; 
   
   if opts["GenerateMatrixStyle"] = "MATLAB" or opts["Language"] = "Simscape" then
      if opts["GenSparseMatrix"] = FALSE and dim <= 2 then
         S:= S."]";
      end_if;
   end_if;

   if length(S) > 1 and S[-1] = opts["StatementDelimiter"] then
      S:= S[1..-2];
   end_if;
   S;
end_proc:


// ================================================================================================
// CODE GENERATION: DOM_EXPR TYPE-SLOTS
// ================================================================================================

// _index =========================================================================================

/*--
_index::CF -- returns a CF formatted string for funcenv '_index'
--*/
_index::CF := proc(e, p, t, opts)
   local i;
begin
   if select([op(e,2..nops(e))], o->_lazy_and(testtype(o,Type::Numeric), o < 1)) <> [] then
      if opts["Warnings"] then
         warning("array index out of range 1..n: ".expr2text(e));
      end_if;
   end_if;
   _concat(
      generate::CFany(op(e,1), output::Priority::Index, t, opts), 
      (if opts["IndexStyle"] = "C" then
          // A[?,?] => A[?][?]
          op(map([op(e,2..nops(e))], o -> opts["IndexLeft"].generate::CFany(o-1, output::Priority::Noop, "int", opts).opts["IndexRight"]));
       else 
          // opts["IndexStyle"] = "Fortran" or
          // opts["IndexStyle"] = "MATLAB"
          // A[?,?] => A(?,?)
          opts["IndexLeft"],
          op(zip(map([op(e,2..nops(e))], o -> generate::CFany(o, output::Priority::Noop, "int", opts)),
                 ["," $ i=2..nops(e)-1, ""], _concat)),
          opts["IndexRight"];
       end_if)
   );
end_proc:

// basic arithmetic ===============================================================================

/*--
_plus::CF -- returns a CF formatted string for funcenv '_plus'
--*/
_plus::CF := proc(e, p, t, opts) 
   local  i; 
begin
   _concat(
      (if p >= output::Priority::Plus then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Plus, t, opts),
      (if generate::CFisminus(op(e,i), opts) then "-".generate::CFany(-op(e,i), output::Priority::Plus, t, opts);
       else                                       "+".generate::CFany( op(e,i), output::Priority::Plus, t, opts);
       end_if) $ i=2..nops(e),
      (if p >= output::Priority::Plus then ")" else "" end_if)
   );
end_proc:

/*--
_subtract::CF -- returns a CF formatted string for funcenv '_subtract'
--*/
_subtract::CF := proc(e, p, t, opts)
   local subprio;
begin
   // MuPAD has no priority level for _subtract. However, to avoid unnecessary
   // parenthesis in examples like a-(b), it would be very helpful to have it.
   // Assumption: No other priority level exists between output::Priority::Plus and output::Priority::Mult.
   // Definition: subprio is here defined as the mean of output::Priority::Plus and output::Priority::Mult.
   subprio:= stats::mean([output::Priority::Mult, output::Priority::Plus]);
   _concat( 
      (if p >= output::Priority::Plus then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Plus, t, opts),
      (if generate::CFisminus(op(e,2), opts) then "+".generate::CFany(-op(e,2), output::Priority::Plus, t, opts);
       else                                       "-".generate::CFany( op(e,2), subprio,  t, opts);
       end_if),
      (if p >= output::Priority::Plus then ")" else "" end_if)
   );
end_proc:

/*--
_negate::CF -- returns a CF formatted string for funcenv '_negate'
--*/
_negate::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= output::Priority::Power then "(" else "" end_if),
      generate::CFany(-op(e,1), output::Priority::Plus, t, opts),
      (if p >= output::Priority::Power then ")" else "" end_if)
   );
end_proc:

/*--
_mult::CF -- returns a CF formatted string for funcenv '_mult'
--*/
_mult::CF := proc(e, p, t, opts) 
   local n, d; 
begin
   // Mehr Spezialflle bercksichtigen: x*-1/2,
   if testtype(op(e,nops(e)), Type::Numeric) and op(e,nops(e)) = -1 then
      return(generate::CFunop("-", output::Priority::Plus, p, t, subsop(e, nops(e)=null()), opts));
   end_if;
   n:= select(e, not generate::CFisdiv, opts);
   d:= select(e,     generate::CFisdiv, opts);
   if d = 1 then
      generate::CFop("*", output::Priority::Mult, p, t, [op(e)], opts);
   else
      generate::CFop("/", output::Priority::Mult, p, t, [n, 1/d], opts);
   end_if
end_proc:

/*--
_divide::CF -- returns a CF formatted string for funcenv '_divide'
--*/
_divide::CF := proc(e, p, t, opts)
begin
   if domtype(op(e,1)) = DOM_INT and domtype(op(e,2)) = DOM_INT then
      // create rational number, handle special cases in DOM_RAT::CF
      return(DOM_RAT::CF(op(e,1)/op(e,2), p, t, opts));
   end_if;
   _concat(
      (if p >= output::Priority::Mult then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Mult, t, opts),
      "/", 
      generate::CFany(op(e,2), output::Priority::Mult, t, opts),
      (if p >= output::Priority::Mult then ")" else "" end_if)
   );
end_proc:

/*--
_invert::CF -- returns a CF formatted string for funcenv '_invert'
--*/
_invert::CF := proc(e, p, t, opts)
begin
   if nops(e) <> 1 then 
      error("illegal number of operands for '_invert'");
   end_if;
   generate::CFany(1/op(e), p, t, opts); // use CFany!
end_proc:

/*--
_power::CF -- returns a CF formatted string for funcenv '_power'
--*/
_power::CF := proc(e, p, t, opts)
begin
   if t = "int" and opts["TypeChecking"] then 
      error("unexpected return type in call of function '_power'") 
   end_if;
   if opts["PowerInvertNegIntRatExp"] and (domtype(op(e,2)) = DOM_INT or domtype(op(e,2)) = DOM_RAT) and op(e,2) < 0 then
      // See g600002.
      (if p > output::Priority::Mult then "(" else "" end_if).
      (if t = "int" then "1" else DOM_FLOAT::CF(1.0, 0, "double", opts) end_if).
      "/".
      generate::CFany(op(e,1)^-op(e,2), output::Priority::Mult, t, opts).
      (if p > output::Priority::Mult then ")" else "" end_if);
   elif op(e,2) = 1/2 then
      generate::CFcall("sqrt", "double", [op(e,1)], opts, {1});
   elif opts["PowerKeepIntExp"] and domtype(op(e,2)) = DOM_INT then
      if op(e,2) > 0 and op(e,2) <= opts["PowerThreshold"] and
         (domtype(op(e,1)) = DOM_IDENT or (type(op(e,1)) = "_index" and 
          not hastype(op(op(e,1),2..nops(op(e,1))),DOM_IDENT))) then
         // write as product
         generate::CFop("*", output::Priority::Mult, p, t, [op(e,1) $ op(e,2)], opts);
      else
         if opts["PowerOperator"] = FAIL then
            // Currently only true for C
            generate::CFcall("pow", "double", [op(e)], opts, {1,2});
         else
            // write exponent as (positive) integer
            generate::CFany(op(e,1), output::Priority::Power, t, opts).
            opts["PowerOperator"].
            (if op(e,2) < 0 then "(" else "" end_if).
            DOM_INT::CF(op(e,2),output::Priority::Noop,"int",opts).
            (if op(e,2) < 0 then ")" else "" end_if);
         end_if;
      end_if;
   else
      if opts["PowerOperator"] = FAIL then
         // Currently only true for C
         generate::CFcall("pow", "double", [op(e)], opts, {1,2});
      else
         if generate::CFisminus(op(e,2), opts) then
            generate::CFany(op(e,1), output::Priority::Power, "double", opts).opts["PowerOperator"]."(".generate::CFany(op(e,2),output::Priority::Noop,t, opts).")";
         else
            generate::CFop(opts["PowerOperator"], output::Priority::Power, p, "double", [op(e)], opts);
         end_if;
      end_if;
   end_if     
end_proc:

// log ============================================================================================

/*--
log::CF -- returns a CF formatted string for funcenv 'log'
--*/
log::CF := proc(e, p, t, opts)
begin
   if opts["Log2"] <> FAIL and op(e,1) = 2 then 
      // log(2,...)
      generate::CFcall(opts["Log2"], "double", [op(e,2)], opts, {1});
   elif opts["Log10"] <> FAIL and op(e,1) = 10 then 
      // log(10,...)
      generate::CFcall(opts["Log10"], "double", [op(e,2)], opts, {1});
   else
      generate::CFany(rewrite(e,ln), p, t, opts);
   end_if;
end_proc:

// abs  / sign ====================================================================================

/*--
abs::CF -- returns a CF formatted string for funcenv 'abs'
--*/
abs::CF := proc(e, p, t, opts)
begin
   if t = "int" then
      generate::CFcall("abs", "int", [op(e)], opts)
   elif opts["Language"] = "C" then
      generate::CFcall("fabs", "double", [op(e)], opts, {1});
   else
      generate::CFcall("abs", "double", [op(e)], opts, {1});
   end_if
end_proc:

/*--
sign::CF -- returns a CF formatted string for funcenv 'sign'
--*/
sign::CF := proc(e, p, t, opts)
begin
   if opts["Language"] <> "MATLAB" and opts["Language"] <> "Simscape" then 
      "(".generate::CFany(op(e)/abs(op(e)), p, t, opts).")";
   else
      generate::CFcall("sign", "double", [op(e)], opts, {1});
   end_if;
end_proc:

// div  / mod =====================================================================================

/*--
_div::CF -- returns a CF formatted string for funcenv '_div'
--*/
_div::CF := proc(e, p, t, opts)
begin
   if opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      "fix(".generate::CFop("/", output::Priority::Mult, p, "int", [op(e)], opts).")";
    else
      generate::CFop("/", output::Priority::Mult, p, "int", [op(e)], opts);
   end_if;
end_proc:

/*--
_mod::CF -- returns a CF formatted string for funcenv '_mod'
--*/
_mod::CF := proc(e, p, t, opts)
begin
   if   opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      generate::CFcall("mod", "int", [op(e)], opts);
   elif opts["Language"] = "FORTRAN" then
      generate::CFcall("mod", "int", [op(e)], opts);
   else
      generate::CFop(" % ", output::Priority::Fconcat, output::Priority::Fconcat, "int", [op(e)], opts);
   end_if;
end_proc:

// airy ===========================================================================================

/*--
airyAi::CF -- returns a CF formatted string for funcenv 'airyAi'
--*/
airyAi::CF := proc(e, p, t, opts)
begin
   if t = "int" and opts["TypeChecking"] then 
      error("unexpected return type in call of function 'airyAi'");
   end_if;
   if nops(op(e)) <> 2 then 
      error("illegal number of operands for 'airyAi'");
   end_if;
   if opts["Language"] <> "MATLAB" and opts["Language"] <> "Simscape" then 
      generate::CFunverified("airyAi", opts);
      generate::CFcall("airyAi", "double", [op(e)], opts, {1}, {2});
   else
      // assuming that op(e,2) can only be 0 or 1
      generate::CFcall("airy", "double", [op(e)], opts, {1}, {2}, {1=2});
   end_if;
end_proc:


/*--
airyBi::CF -- returns a CF formatted string for funcenv 'airyBi'
--*/
airyBi::CF := proc(e, p, t, opts)
begin
   if t = "int" and opts["TypeChecking"] then 
      error("unexpected return type in call of function 'airyAi'");
   end_if;
   if nops(op(e)) <> 2 then 
      error("illegal number of operands for 'airyBi'");
   end_if;
   if opts["Language"] <> "MATLAB" and opts["Language"] <> "Simscape"then 
      generate::CFunverified("airyBi", opts);
      generate::CFcall("airyBi", "double", [op(e)], opts, {1}, {2});
   else
      // assuming that op(e,2) can only be 0 or 1
      generate::CFcall("airy", "double", [op(e,1), op(e,2)+2], opts, {1}, {2}, {1=2});
   end_if;
end_proc:

// min / max ======================================================================================

/*--
min::CF -- returns a CF formatted string for funcenv 'min'
--*/
min::CF:= proc(e, p, t, opts)
   local l, i;
begin
   if type(op(e)) = "_exprseq" then
      l:= [op(e)];     // sequence to list
   elif type(op(e)) = DOM_SET then
      l:= [op(op(e))]; // set to list
   elif type(op(e)) = DOM_LIST then
      l:= op(e);
   else
      return(generate::CFany(op(e), output::Priority::Noop, t, opts));
   end_if;
   if nops(l) = 1 then
      return(generate::CFany(op(l), output::Priority::Noop, t, opts));
   end_if;
   if opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      "min([".generate::CFany(l[1],output::Priority::Exprseq,t,opts).((",".generate::CFany(l[i],output::Priority::Exprseq,t,opts))$i=2..nops(l))."])";
   else
      "min(".generate::CFany(l[1],output::Priority::Exprseq,t,opts).",".min::CF(hold(min)(l[2..-1]),1,"double",opts).")";
   end_if;
end_proc:

/*--
max::CF -- returns a CF formatted string for funcenv 'max'
--*/
max::CF:= proc(e, p, t, opts)
   local l, i;
begin
   if type(op(e)) = "_exprseq" then
      l:= [op(e)];     // sequence to list
   elif type(op(e)) = DOM_SET then
      l:= [op(op(e))]; // set to list
   elif type(op(e)) = DOM_LIST then
      l:= op(e);
   else
      return(generate::CFany(op(e), output::Priority::Noop, t, opts));
   end_if;
   if nops(l) = 1 then
      return(generate::CFany(op(l), output::Priority::Noop, t, opts));
   end_if;
   if opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      "max([".generate::CFany(l[1],output::Priority::Exprseq,t, opts).((",".generate::CFany(l[i],output::Priority::Exprseq,t, opts))$i=2..nops(l))."])";
   else
      "max(".generate::CFany(l[1],output::Priority::Exprseq,t, opts).",".max::CF(hold(max)(l[2..-1]),1,"double", opts).")";
   end_if;
end_proc:

// relational operators ===========================================================================

/*--
_equal::CF -- returns a CF formatted string for funcenv '_equal'
--*/
_equal::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Relation, t, opts),
      (  if opts["Language"] = "C" then 
          " == "
       elif opts["Language"] = "FORTRAN" then 
          if domtype(op(e,1)) = DOM_BOOL or domtype(op(e,2)) = DOM_BOOL then 
             " .EQV. "
           else
             " .EQ. "
          end_if;
       elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
          " == "
       end_if),
      generate::CFany(op(e,2), output::Priority::Relation, t, opts),
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then ")" else "" end_if)
   );
end_proc:

/*--
_unequal::CF -- returns a CF formatted string for funcenv '_unequal'
--*/
_unequal::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Relation, t, opts),
      (  if opts["Language"] = "C" then 
          " != "
       elif opts["Language"] = "FORTRAN" then 
          if domtype(op(e,1)) = DOM_BOOL or domtype(op(e,2)) = DOM_BOOL then 
             " .NEQV. "
           else
             " .NE. "
          end_if;
       elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
          " ~= " 
       end_if),
      generate::CFany(op(e,2), output::Priority::Relation, t, opts),
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then ")" else "" end_if)
   );
end_proc:

/*--
_leequal::CF -- returns a CF formatted string for funcenv '_leequal'
--*/
_leequal::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Relation, t, opts),
      (  if opts["Language"] = "C" then 
          " <= "
       elif opts["Language"] = "FORTRAN" then 
          " .LE. "
       elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
          " <= " 
       end_if),
      generate::CFany(op(e,2), output::Priority::Relation, t, opts),
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then ")" else "" end_if)
   );
end_proc:

/*--
_less::CF -- returns a CF formatted string for funcenv '_less'
--*/
_less::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then "(" else "" end_if),
      generate::CFany(op(e,1), output::Priority::Relation, t, opts),
      (  if opts["Language"] = "C" then 
          " < "
       elif opts["Language"] = "FORTRAN" then 
          " .LT. "
       elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
          " < " 
       end_if),
      generate::CFany(op(e,2), output::Priority::Relation, t, opts),
      (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Relation */ then ")" else "" end_if)
   );
end_proc:

/*--
_in::CF -- returns a CF formatted string for funcenv '_in'
--*/
_in::CF := proc(e, p, t, opts)
   local expanded;
begin
   expanded:= expand(e);
   if expanded <> e then
      return(generate::CFany(expanded, p, t, opts));
   end_if;
   error("not yet supported: '".e."'");
end_proc:

// boolean logical operators ======================================================================

/*--
_and::CF -- returns a CF formatted string for funcenv '_and'
--*/
_and::CF := proc(e, p, t, opts)
   local S, i, oldp;
begin
   if nops(e) = 0 then
      return(generate::CFany(TRUE, output::Priority::And, t, opts));
   end_if;

   S:= generate::CFany(op(e,1), output::Priority::And, t, opts);
   if nops(e) = 1 then
      return(_concat(
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::And */ then "(" else "" end_if),
         S,
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::And */ then ")" else "" end_if)
      ));
   end_if;
    
   oldp:= p; 
   p:= output::Priority::And; // left associative, use parenthesis
   for i from 2 to nops(e) do
      if i = nops(e) then
         p:= oldp;  // use original parent priority for last element 
      end_if;
      S:= _concat(
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::And */ then "(" else "" end_if),
         S,
         (  if opts["Language"] = "C" then 
             " && "
         elif opts["Language"] = "FORTRAN" then 
             " .AND. "
         elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
             " && " 
         end_if),
         generate::CFany(op(e,i), output::Priority::And, t, opts),
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::And */ then ")" else "" end_if)
      );
   end_for;
   S;
end_proc:

/*--
_or::CF -- returns a CF formatted string for funcenv '_or'
--*/
_or::CF := proc(e, p, t, opts) 
   local S, i, oldp;
begin
   if nops(e) = 0 then
      return(generate::CFany(FALSE, output::Priority::Or, t, opts));
   end_if;

   S:= generate::CFany(op(e,1), output::Priority::Or, t, opts);
   if nops(e) = 1 then
      return(_concat(
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Or */ then "(" else "" end_if),
         S,
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Or */ then ")" else "" end_if)
      ));
   end_if;
    
   oldp:= p; 
   p:= output::Priority::Or; // left associative, use parenthesis
   for i from 2 to nops(e) do
      if i = nops(e) then
         p:= oldp;  // use original parent priority for last element 
      end_if;
      S:= _concat(
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Or */ then "(" else "" end_if),
         S,
         (  if opts["Language"] = "C" then 
             " || "
         elif opts["Language"] = "FORTRAN" then 
             " .OR. "
         elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
             " || " 
         end_if),
         generate::CFany(op(e,i), output::Priority::Or, t, opts),
         (if p >= generate::CFoutPrioBoolOpMin /* output::Priority::Or */ then ")" else "" end_if)
      );
   end_for;
   S;
end_proc:

/*--
_lazy_and::CF -- returns a CF formatted string for funcenv '_lazy_and'
--*/
_lazy_and::CF := _and::CF: 

/*--
_lazy_or::CF -- returns a CF formatted string for funcenv '_lazy_or'
--*/
_lazy_or::CF := _or::CF: 

/*--
_not::CF -- returns a CF formatted string for funcenv '_not'
--*/
_not::CF := proc(e, p, t, opts) 
begin
   _concat(
      (if p >= output::Priority::Not then "(" else "" end_if),
      (  if opts["Language"] = "C" then 
          "!"
       elif opts["Language"] = "FORTRAN" then 
          ".NOT."
       elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then 
          "~" 
       end_if),
      generate::CFany(op(e,1), output::Priority::Not, t, opts),
      (if p >= output::Priority::Not then ")" else "" end_if)
   );
end_proc:

// programming ====================================================================================

/*--
_stmtseq::CF -- returns a CF formatted string for domtype '_stmtseq'
--*/
_stmtseq::CF:= proc(e, p, t, opts)
   local o, it;
begin
   // handle a single object like a statement sequence with one element
   if type(e) <> "_stmtseq" then 
      e:= hold(_stmtseq)(e);
   end_if;

   // the statement sequence might be empty
   if nops(e) = 0 then
      return("");
   end_if;
   
   // generate code for all operands
   it:= _concat(opts["IndentWidth"] $ opts["IndentLevel"]);
   _concat(
      //if p >= output::Priority::Stmtseq then "(" else "" end_if,
      op(map(e, () -> (
         o:= generate::CFany(args(), output::Priority::Noop, t, opts);
         // Ignore empty expressions (e.g. spare matrix with 0-entries) 
         if o = "" then 
            o 
         else
            _concat(it, o, (if o[-1] = "}" or o[-1] = "\n" then "" else opts["StatementDelimiter"] end_if), "\n");
         end_if;
      )))
      //if p >= output::Priority::Stmtseq then ")" else "" end_if
   );
end_proc:

/*--
_assign::CF -- returns a CF formatted string for funcenv '_assign'
--*/
generate::_assign__CF:= proc(e, p, t, opts)
   local l, r, v;
begin
   l:= op(e,1);
   r:= op(e,2);
   
   // special case for _stmtseq: return value of last statement
   if type(r) = "_stmtseq" then
      // If last statement in statement sequence is an assignment
      // then get left-hand-side (typically a variable) and store
      // it (say 'temp') for later use.
      // If last statement in statement sequence is no assignment
      // then (check its type), create a new variable and rewrite
      // the statement into an assignment 'temp:=...'.  Store the
      // variable for later use.
      // Create statement sequence using 'output::Priority::Noop'
      // Assign 'temp' to 'l'.
      if type(op(r,nops(r))) = "_assign" then
         v:= op((op(r, nops(r))),1);         // get and store lhs
      else
         v:= generate::CFnewScalarVariable();
         r:= subsop(r, nops(r) = hold(_assign)(v, op(r,nops(r))));
      end_if;
      r:= generate::CFany(r, output::Priority::Noop, t, opts);
      return(r.generate::_assign__CF(l=v, p, t, opts));
   end_if;

   // special case for matrices: let 'generate::Matrix::CF' do it
   if type(r) = generate::Matrix then
      return(generate::Matrix::CF(e, output::Priority::Noop, t, opts));
   end_if;
   
   // special case for piecewise: let 'generate::Piecewise::CF' do it
   if type(r) = generate::Piecewise then
      return(generate::Piecewise::CF(e, output::Priority::Noop, t, opts));
   end_if;

   // special case for piecewise: let 'piecewise::CF' do it
   if type(r) = piecewise or type(r) = hold(piecewise) or (type(r) = "function" and op(r,0) = hold(piecewise)) then
      return(piecewise::CF(e, output::Priority::Noop, t, opts));
   end_if;

   l:= generate::CFany(l, output::Priority::Noop, t, opts);

   if type(r) = DOM_PROC then
      // call code generator function for DOM_PROC with nam = l
      return(l." = ".DOM_PROC::CF(r, output::Priority::Stmtseq, t, opts, l));
   end_if;
   
   // handle 'a:= b:= c' for languages that do not support this feature
   if contains({"FORTRAN", "MATLAB", "Simscape"}, opts["Language"]) and type(op(e,2)) = "_assign" then
      l:= generate::CFany(r, output::Priority::Noop, t, opts).opts["StatementDelimiter"]."\n".l;
      r:= generate::CFany(op(e,[2,1]), output::Priority::Relation, t, opts);
   else
      r:= generate::CFany(r, output::Priority::Relation, t, opts);
   end_if;

   // standard assignment
   _concat(
      if p >= output::Priority::Stmtseq then "(" else "" end_if,
      l." = ".r,
      if p >= output::Priority::Stmtseq then ")" else "" end_if
   );
end_proc:

/*--
_if::CF -- returns a CF formatted string for funcenv '_if'
--*/
_if::CF:= proc(e, p, t, opts)
   local co, bt, bf, it;
begin
   it:= _concat(opts["IndentWidth"] $ opts["IndentLevel"]);
   
   co:= generate::CFany(op(e,1), output::Priority::Noop, t, opts);
   bt:= _stmtseq::CF(op(e,2), output::Priority::Noop, t, table(opts, "IndentLevel" = opts["IndentLevel"]+1));
   bf:= _stmtseq::CF(op(e,3), output::Priority::Noop, t, table(opts, "IndentLevel" = opts["IndentLevel"]+1));
   
   if opts["Language"] = "C" then
      "if (".co.") {\n"
         .bt
      .it."} else {\n"
         .bf
      .it."}";
   elif opts["Language"] = "FORTRAN" then
      "if (".co.") then\n"
         .bt
      .it."else\n"
         .bf
      .it."endif";
   elif opts["Language"] = "MATLAB" or opts["Language"] = "Simscape" then
      "if (".co."),\n"
         .bt
      .it."else\n"
         .bf
      .it."end";
   end_if;
end_proc:

/*--
return::CF -- returns a CF formatted string for funcenv 'return'
--*/
return::CF:= proc(e, p, t, opts)
begin
   if nops(e) = 0 then
      if opts["Language"] = "C" then
         "return()";
      else
         "return";
      end_if;
   else
      if opts["Language"] = "C" then
         "return(".generate::CFany(op(e), output::Priority::Noop, t, opts).")";
      else
         error("not yet implemented");
      end_if;
   end_if;
end_proc:

/*--
DOM_PROC::CF -- returns a CF formatted string for procedures
--*/
DOM_PROC::CF:= proc(e, p, t, opts, nam=#unknown)
   local S, vars, stmts;
begin
   if opts["Language"] <> "C" or opts["SupportProcedures"] <> TRUE then
      if nam = #unknown then
         nam:= prog::getname(e);
      end_if;
      if nam = "FUN" then
        error("Anonymous procedure is not allowed in this context");
      end_if;
      warning("Translating MuPAD procedures ('".nam."') is not yet implemented");
      return(nam);
   end_if;
   S:= "";
   
   //S:= "#include <math.h>\n\n";

   //S:= S."extern double z".opts["StatementDelimiter"] // global variables?

   // Get or create name of the procedure
   if op(e,6) = NIL then
      nam:= generate::CFnewFunctionName();
   else
      nam:= op(e,6);
   end_if;
    
   // C function header including parameters
   S:= S.t." ".nam."(";
   if op(e,1) <> NIL then
      S:= S._concat(
         t." ".expr2text(op(e,[1,1])), 
         op(zip([", " $ nops(op(e,1))-1], map([op(e,[1,2..nops(op(e,1))])],o -> t." ".expr2text(o)),id)));
   end_if;
   S:= S.")\n";

   S:= S._concat(opts["IndentWidth"] $ opts["IndentLevel"])."{\n";
   // local variables
   if op(e,2) <> NIL then
      S:= S._concat(opts["IndentWidth"] $ opts["IndentLevel"]+1)._concat(
         op(map([op(e,[2,1..nops(op(e,2))])], o -> t." ".expr2text(o).";\n")));
      S:= S."\n\n";
   end_if;

   // names of parameters and local variables: DOM_VAR(0,2..n)
   vars:= ["this", op(e,2), op(e,1)];

   // substitute DOM_VAR(0, i) by its name
   stmts:= misc::maprec(op(e,4), {DOM_VAR} = (e -> (
              if op(e,1) <> 0 then 
                 error("illegal reference to local variable from outer scope");
              end_if;
              op(vars, op(e,2));
           )));
           
   // create procedure body
   S:= S.generate::CFany(stmts, output::Priority::Noop, t, table(opts, "IndentLevel" = opts["IndentLevel"]+1));
   S:= S._concat(opts["IndentWidth"] $ opts["IndentLevel"])."}";
   S;
end_proc:

// ================================================================================================
// CODE GENERATION: DOM_EXPR: SPECIAL FUNCTIONS (SEE 'generate::CFspecfun') 
// ================================================================================================

fact::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "int",    {1},   None,   table(                                      "MATLAB"="factorial", "Simscape"="factorial"), table(                        ), {},    {1},   {}):  
binomial::CF  := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "int",    {2},   None,   table(                                      "MATLAB"="nchoosek",  "Simscape"="nchoosek" ), table(                        ), {},    {1,2}, {}):  

sqrt::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="sqrt",    "FORTRAN"="sqrt",      "MATLAB"="sqrt",      "Simscape"="sqrt"     ), table(                        ), {1},   {},    {}):  

exp::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="exp",     "FORTRAN"="exp",       "MATLAB"="exp",       "Simscape"="exp"      ), table(                        ), {1},   {},    {}):  
ln::CF        := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="log",     "FORTRAN"="log",       "MATLAB"="log",       "Simscape"="log"      ), table(                        ), {1},   {},    {}):  

sin::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="sin",     "FORTRAN"="sin",       "MATLAB"="sin",       "Simscape"="sin"      ), table(                        ), {1},   {},    {}):  
cos::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="cos",     "FORTRAN"="cos",       "MATLAB"="cos",       "Simscape"="cos"      ), table(                        ), {1},   {},    {}):  
tan::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="tan",     "FORTRAN"="tan",       "MATLAB"="tan",       "Simscape"="tan"      ), table(                        ), {1},   {},    {}):  
csc::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   sin,    table(                                      "MATLAB"="csc",       "Simscape"="csc"      ), table(                        ), {1},   {},    {}):  
sec::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   cos,    table(                                      "MATLAB"="sec",       "Simscape"="sec"      ), table(                        ), {1},   {},    {}):  
cot::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   tan,    table(                                      "MATLAB"="cot",       "Simscape"="cot"      ), table(                        ), {1},   {},    {}):  

arcsin::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="asin",    "FORTRAN"="asin",      "MATLAB"="asin",      "Simscape"="asin"     ), table(                        ), {1},   {},    {}):  
arccos::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="acos",    "FORTRAN"="acos",      "MATLAB"="acos",      "Simscape"="acos"     ), table(                        ), {1},   {},    {}):  
arctan::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="atan",    "FORTRAN"="atan",      "MATLAB"="atan",      "Simscape"="atan"     ), table(                        ), {1},   {},    {}):  
arccsc::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   arcsin, table(                                      "MATLAB"="acsc",      "Simscape"="acsc"     ), table(                        ), {1},   {},    {}):  
arcsec::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   arccos, table(                                      "MATLAB"="asec",      "Simscape"="asec"     ), table(                        ), {1},   {},    {}):  
arccot::CF    := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   arctan, table(                                      "MATLAB"="acot",      "Simscape"="acot"     ), table(                        ), {1},   {},    {}):  

sinh::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="sinh",    "FORTRAN"="sinh",      "MATLAB"="sinh",      "Simscape"="sinh"     ), table(                        ), {1},   {},    {}):  
cosh::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="cosh",    "FORTRAN"="cosh",      "MATLAB"="cosh",      "Simscape"="cosh"     ), table(                        ), {1},   {},    {}):  
tanh::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table("C"="tanh",    "FORTRAN"="tanh",      "MATLAB"="tanh",      "Simscape"="tanh"     ), table(                        ), {1},   {},    {}):  
csch::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   sinh,   table(                                      "MATLAB"="csch",      "Simscape"="csch"     ), table(                        ), {1},   {},    {}):  
sech::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   cosh,   table(                                      "MATLAB"="sech",      "Simscape"="sech"     ), table(                        ), {1},   {},    {}):  
coth::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   tanh,   table(                                      "MATLAB"="coth",      "Simscape"="coth"     ), table(                        ), {1},   {},    {}):  

arcsinh::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="asinh",     "Simscape"="asinh"    ), table(                        ), {1},   {},    {}):  
arccosh::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="acosh",     "Simscape"="acosh"    ), table(                        ), {1},   {},    {}):  
arctanh::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="atanh",     "Simscape"="atanh"    ), table(                        ), {1},   {},    {}):
arccsch::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="acsch",     "Simscape"="acsch"    ), table(                        ), {1},   {},    {}):  
arcsech::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="asech",     "Simscape"="asech"    ), table(                        ), {1},   {},    {}):  
arccoth::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   ln,     table(                                      "MATLAB"="acoth",     "Simscape"="acoth"    ), table(                        ), {1},   {},    {}):  

arg::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1,2}, ln,     table(                                                                                  ), table(                        ), {1},   {},    {}):  
Re::CF        := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(                                      "MATLAB"="real",      "Simscape"="real"     ), table(                        ), {1},   {},    {}):  
Im::CF        := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(                                      "MATLAB"="imag",      "Simscape"="imag"     ), table(                        ), {1},   {},    {}):  
conjugate::CF := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(               "FORTRAN"="conjg",     "MATLAB"="conj",      "Simscape"="conj"     ), table("FORTRAN"="FORTRAN_77"  ), {1},   {},    {}):  

besselI::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="besseli",   "Simscape"="besseli"  ), table(                        ), {2},   {1},   {}):  
besselJ::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="besselj",   "Simscape"="besselj"  ), table(                        ), {2},   {1},   {}):  
besselK::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="besselk",   "Simscape"="besselk"  ), table(                        ), {2},   {1},   {}):  
besselY::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="bessely",   "Simscape"="bessely"  ), table(                        ), {2},   {1},   {}):  

beta::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="beta",      "Simscape"="beta"     ), table(                        ), {1,2}, {2},   {}):  
erf::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(                                      "MATLAB"="erf",       "Simscape"="erf"      ), table(                        ), {1},   {},    {}):
erfc::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(                                      "MATLAB"="erfc",      "Simscape"="erfc"     ), table(                        ), {1},   {},    {}):
psi::CF       := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1,2}, None,   table(                                      "MATLAB"="psi",       "Simscape"="psi"      ), table("FORTRAN"="FORTRAN_2008"), {1},   {2},   {1=2}):  
gamma::CF     := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(               "FORTRAN"="gamma",     "MATLAB"="gamma",     "Simscape"="gamma"    ), table("FORTRAN"="FORTRAN_2008"), {1},   {},    {}):
lngamma::CF   := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {1},   None,   table(               "FORTRAN"="log_gamma", "MATLAB"="gammaln",   "Simscape"="gammaln"  ), table("FORTRAN"="FORTRAN_2008"), {1},   {},    {}):

lambertW::CF  := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "double", {2},   None,   table(                                      "MATLAB"="lambertw",  "Simscape"="lambertw" ), table(                        ), {2},   {1},   {}):  

ceil::CF      := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "int",    {1},   None,   table("C"="ceil",    "FORTRAN"="ceiling",   "MATLAB"="ceil",     "Simscape"="ceil"      ), table("FORTRAN"="FORTRAN_95"  ), {1},   {},    {}):
floor::CF     := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "int",    {1},   None,   table("C"="floor",   "FORTRAN"="floor",     "MATLAB"="floor",    "Simscape"="floor"     ), table("FORTRAN"="FORTRAN_95"  ), {1},   {},    {}):
trunc::CF     := (e,p,t,opts) -> generate::CFspecfun(e,p,t,opts, "int",    {1},   None,   table("C"="trunc",   "FORTRAN"="aint",      "MATLAB"="fix",      "Simscape"="fix"       ), table("FORTRAN"="FORTRAN_77"  ), {1},   {},    {}):


// ================================================================================================
// CODE GENERATION: other math functions 
// ================================================================================================

// summation ======================================================================================

/*--
sum::CF -- returns a CF formatted string for 'sum'
--*/
sum::CF:= proc(e, p, t, opts)
   local o2;
begin
   if nops(e) <> 2 then
      error("invalid number of argument in call of 'sum'");
   end_if;
   generate::CFunverified("sum", opts);
   o2:= op(e,2);
   if type(o2) = DOM_IDENT then // sum(a[i],i)
      return(generate::CFcall("sum", t, [op(e,1), o2], opts));
   end_if;
   if type(o2) = "_equal" then // sum(a[i],i=...)
      if type(op(o2,2)) = "_range" then // sum(a[i],i=1..n)
         return(generate::CFcall("sum", t, [op(e,1), op(o2,1), op(op(o2,2))], opts));
      end_if;
      return(generate::CFcall("sum", t, [op(e,1), op(o2)], opts));
   end_if;
   // unknown format of function call, should never be reached!!! 
   return(generate::CFcall("sum", t, [op(e)], opts));
   // error("not yet implemented");
end_proc:
 
// derivatives ====================================================================================

/*--
D::CF -- returns a CF formatted string for funcenv 'D' (see 'diff')
--*/
D::CF:= proc(e, p, t, opts)
   local fun, var, tvar;
begin
   assert(opts["NameOfTimeVariable"] <> "");
   
   // The input 'e' is expected to be an expression like x'(t).
   // Derivatives of functions and partial derivatives D([n1,...], f) cannot be handled. 
   // This is currently no use case and results in an error or some generic code (maybe garbage).
   if type(e) = "D" then
      if type(op(e,1)) = DOM_LIST then
         (()->error("Partial derivatives are not supported '".e."'"))();
      end_if;
      if opts["Language"] = "Simscape" then
         (()->error("Time derivative with respect to '".opts["NameOfTimeVariable"]."' expected in '".e."'"))();
      end_if;
      return("D(".generate::CFany(op(e), p, t, opts).")");
   end_if;

   // The input 'e' is expected to be an expression like x'(t) = D(x)(t).
   // Partial derivatives D([n1,...], f)(t) are not supported.
   // This is currently no use case and results in an error or some generic code (maybe garbage).
   if type(op(e,[0,1])) = DOM_LIST then
      // D([n1,...], f)(t) => error
      (()->error("Partial derivatives are not supported '".e."'"))();
   end_if;
   
   fun:= op(e,[0,1]);  // 'D(x)(t)' => 'D(x)' => 'x'
   var:= [op(e)];      // 'D(x)(t)' => ['t']
   
   if nops(var) = 0 then // f'() = D(f)()
      (()->error("Missing differentiation variable in '".expr2text(e)."'"))();
   end_if;
    
   // Code generation for Simscape is special in this case 
   if opts["Language"] = "Simscape" then
      if type(fun) = DOM_IDENT and var = [hold(``).opts["NameOfTimeVariable"]] then
         // fun'(t) with "t" is time variable
         return("".fun.".der");
      end_if;
      tvar:= hold(``).opts["NameOfTimeVariable"];
      if {op(var)} <> {tvar} then
         (()->error("Time derivative with respect to '".opts["NameOfTimeVariable"]."' expected in '".e."'"))();
      end_if;
       
      // Any other derivative, e.g., fun''(t) or fun'(t,t) or fun'(a,b) or ...
      return("der(".generate::CFany(fun(op(var)), p, t, opts).")");
   else 
      if nops({op(var)}) > 1 then 
         (()->error("Ambiguous differentiation variable in '".expr2text(e)."'"))();
      end_if;
   end_if;
    
   // The general case of all other languages
   generate::CFunverified("D", opts);
   generate::CFcall(expr2text(op(e,0)), t, [op(e)], opts);
end_proc:

/*--
diff::CF -- returns a CF formatted string for funcenv 'diff' (see 'D')
--*/
diff::CF:= proc(e, p, t, opts)
   local ex, fun, fvar, dvar;
begin
   ex  := op(e,1);             // 'diff(fun(x),var)' => 'fun(x)'
   fun := op(ex,0);            // 'diff(fun(x),var)' => 'fun(x)' => 'fun'
   fvar:= [op(ex)];            // 'diff(fun(x),var)' => 'fun(x)' => ['x']
   dvar:= [op(e,2..nops(e))];  // 'diff(fun(x),var)' => ['var']

   assert(opts["NameOfTimeVariable"] <> "");
   if nops(dvar) = 0 then
     // diff(f(x)) => f(x)
     return(generate::CFany(ex, p, t, opts));
   end_if;
   if opts["Language"] = "Simscape" then
      if {op(dvar)} <> {hold(``).opts["NameOfTimeVariable"]} then
         (()->error("Time derivative with respect to '".opts["NameOfTimeVariable"]."' expected in '".e."'"))();
      end_if;
      // diff(x,var) => fun.der
      if type(ex) <> "function" or (type(ex) = "function" and type(fun) = "D") then
         // 'ex' is any non-functional expression or object
         // Assuming 'ex' is an expression in 't'
         if type(ex) = DOM_IDENT then
            // Needed for reducing higher order derivatives to first order derivatives
            return("".ex.".der");
         else
            // Better return an error?
            return("der(".generate::CFany(ex, p, t, opts).")");
         end_if;
      elif nops(dvar) = 1 and fvar = [hold(``).opts["NameOfTimeVariable"]] and type(op(ex,0)) = DOM_IDENT then
         // Better return an error?
         return("".op(ex,0).".der");
      elif nops(dvar) > 1 or nops(fvar) > 1 then
         return("der(".generate::CFany(hold(diff)(ex,op(dvar[1..-2])), p, t, opts).")");
      elif "".fvar[1] <> opts["NameOfTimeVariable"] or "".dvar[1] <> opts["NameOfTimeVariable"] then
         // No conversion in this case!!!
         // Should it throw an error in the future?
         // error("Time variable '".opts["NameOfTimeVariable"]."' expected in derivation '".expr2text(e)."'");
      else
         return("".fun.".der"); // keeps code robust
      end_if; 
   end_if;

   // The general case of all other languages 
   generate::CFunverified("diff", opts);
   generate::CFcall("diff", t, [op(e)], opts);
end_proc:


// ================================================================================================
// HELPER FUNCTIONS: CODE GENERATION
// ================================================================================================

/*--
generate::CFany -- returns a CF formatted string for any MuPAD object
   This function expects the standard CF parameter list. It generates 
   code for an arbitrary MuPAD object by calling the domain specific
   method.
--*/
generate::CFany:= proc(e, p, t, opts)
   local te, dt;
begin
   if args(0) < 4 then
      error("unexpected object of type DOM_NULL");
   end_if;
   if args(0) > 4 then
      error("unexpected object of type \"_exprseq\"");
   end_if;
   /*
   if args(0) > 4 then
     return(op(map([args(1..args(0)-3)], generate::CFany, args(args(0)-2), args(args(0)-1), args(args(0)))));
   end_if;
   */
   if slot((te:= domtype(e)), "CF") = FAIL then
      te:= type(e);
      dt:= domtype(te);
      if not (dt = DOM_FUNC_ENV or dt = DOM_DOMAIN) or slot(te, "CF") = FAIL then
         error("unexpected object of type '".expr2text(te)."'");
      end_if;
   end_if; 
   te::CF(e, p, t, opts);
end_proc:

/*--
generate::CFunop -- returns a CF formatted string for unary operators
generate::CFunop(sym, p, pe, t, ol, opts)
   sym  : operator symbol, charcater string
   p    : priority of this expression
   pe   : priority of enclosing expression
   t    : type of expression ("double" or "int")
   ol   : operand list
   opts : defaults and options
--*/
generate::CFunop:= proc(sym, p, pe, t, ol, opts)
begin
   _concat(
      (if p < pe then "(" else "" end_if),
      sym, generate::CFany(ol, p, t, opts),
      (if p < pe then ")" else "" end_if)
   )
end_proc:

/*--
generate::CFop -- returns a CF formatted string for n-ary operators
generate::CFop(sym, p, pe, t, ol, opts)
   sym  : operator symbol, charcater string
   p    : priority of this expression
   pe   : priority of enclosing expression
   t    : type of expression ("double" or "int")
   ol   : operand list
   opts : defaults and options
--*/
generate::CFop:= proc(sym, p, pe, t, ol, opts)
begin
   _concat(
      (if p <= pe then "(" else "" end_if),
      op(zip(map(ol, generate::CFany, p, t, opts), [sym $ (nops(ol)-1)], _concat, "")),
      (if p <= pe then ")" else "" end_if)
   )
end_proc:

/*--
generate::CFcall -- returns a CF formatted string for function calls
generate::CFcall(fun, t, ol, opts, toFloat, toKeepInt, toSwap)
   fun         : function name as charcater string
   t           : expression type ("double" or "int")
   ol          : operand list
   opts        : defaults and options
   toFloat     : set of indices of operands that are to be converted to floats
   toKeepInt   : set of indices of operands that are to keep as integer
   toSwap      : set of indices i=j of operands that are to be swapped
--*/
generate::CFcall:= proc(fun, t, ol, opts, toFloat={}, toKeepInt={}, toSwap={})
   local o, i;
begin
   // convert DOM_INT and DOM_RAT to floating point numbers if index is in toFloat
   if toFloat <> {} then
      ol:= [(o:= op(ol, i);
             if contains(toFloat, i) then
                if   domtype(o) = DOM_INT then float(o);
                elif domtype(o) = DOM_RAT then hold(_divide)(float(numer(o)), float(denom(o)));
                else o; end_if;
             else o; end_if) $ i = 1..nops(ol)];
   end_if; 

   // generate the code for the operands and keep operands as integer if necessary 
   ol:= [(o:= op(ol, i);
          if domtype(o) = DOM_INT and contains(toKeepInt, i) then 
             DOM_INT::CF(o, 1, t, table(opts, "keepInteger" = TRUE));
          else 
             generate::CFany(o, 1, t, opts);
          end) $ i = 1..nops(ol)];
        
   // Swap operands if necessary
   if toSwap <> {} then
      // Apply operand swapping only to operand indices that actually exist in the operand list 
      // Note that the actual number of operands may vary for different calls.
      toSwap:= select(toSwap, o -> _lazy_and(lhs(o)>0, rhs(o)>0, lhs(o)<=nops(ol), rhs(o)<=nops(ol))); 
      ol:= subsop(ol, op(map(toSwap, o -> (lhs(o)=op(ol,rhs(o)), rhs(o)=op(ol,lhs(o))))));
   end_if;
    
   // create function call 
   _concat(fun, "(",
      (if ol = [] then
          ""; 
       else 
          op(zip(ol, ["," $ (nops(ol)-1)], _concat, ""));
       end_if), ")");
end_proc:

/*--
generate::CFspecfun -- returns a CF formatted string for a calls of special functions
generate::CFspecfun(e, p, t, opts, ret, params, rewriteTo, fnam, cver, toFloat, toKeepInt, toSwap, obj)
   e           : expression
   p           : priority of expression
   t           : expression type ("double" or "int")
   opts        : defaults and options
   ret         : type of return value
   params      : set of number of arguments that are allowed for this function call
   rewriteTo   : rewrite target, e.g. ln, sin, tan, ...
   fnam        : table of language = target_function_name
   cver        : table of language = target_special_compiler_version_required
   toFloat     : set of indices of operands that are to be converted to floats
   toKeepInt   : set of indices of operands that are to keep as integer
   toSwap      : set of indices i=j of operands that are to be swapped
   obj         : "function" or "object"
--*/
generate::CFspecfun:= proc(e, p, t, opts, ret, params, rewriteTo=None, fnam=table(), cver=table(), toFloat={}, toKeepInt={}, toSwap={}, obj="function")
   local rewResult;
begin
   // check if return type is valid
   if opts["TypeChecking"] and t <> Any and (t = "int" and ret = "double") then 
      error("unexpected return type in call of function '".prog::getname(op(e,0))."'");
   end_if;
   if contains(fnam, opts["Language"]) then 
      // direct code generation for this function is supported for given language
      fnam:= fnam[opts["Language"]];
      if contains(cver, opts["Language"]) then 
         // However, compiling this code requires a special compiler version
         // Note: code generation might be stopped by 'generate::CFrequires'
         generate::CFrequires(cver[opts["Language"]], fnam, opts, obj);
      end_if;
   elif rewriteTo <> None then
      // direct code generation for this function is not supported for the given language,
      // but rewrite target is specified. Try to rewrite. 
      rewResult:= eval(rewrite(e,rewriteTo));
      if rewResult <> e then
         return(generate::CFany(rewResult, p, t, opts));
      end_if;
      fnam:= prog::getname(op(e,0));            // should never be reached!!!
      warning("Rewriting '".fnam."' to '".expr2text(rewriteTo)."' failed.");
      generate::CFunverified(fnam, opts, obj);  // should never be reached!!!
   else
      // neither direct code generation nor rewriting is supported for this 
      // function for the given language. Try to generate *unverified* code.
      // Note: code generation might be stopped by 'generate::CFunverified'
      fnam:= prog::getname(op(e,0));
      generate::CFunverified(fnam, opts, obj);
   end_if;
   // check number of operand: for all target languages! 
   // TODO: move this test up? not yet!
   if params <> {} and not contains(params,nops(e)) then 
      error("invalid number of operands in call of function '".fnam."'");
   end_if;
   // generate a function call while (optional) keeping some operands as intergers.
   generate::CFcall(fnam, t, [op(e)], opts, toFloat, toKeepInt, toSwap);
end_proc:


// ================================================================================================
// HELPER FUNCTIONS: REWRITING EXPRESSIONS [COMPATIBILITY MODES]
// ================================================================================================

/*--
Handle the following types as matrices: "Mask" them in the input with an internal 
domain element that stores the *value* and the *name* of the matrix (the variable 
that holds this matrix. With this the name of the matrix is still available after
evaluating the input and can be used during the code generation where needed. 
--*/
generate::MatrixTypes:= {DOM_ARRAY,     hold(array),   array,   "array", 
                         DOM_HFARRAY,   hold(hfarray), hfarray, "hfarray", 
                         Dom::Matrix(), hold(matrix),  matrix,  "matrix",
                         null()}:
generate::Matrix:= newDomain("generate::Matrix"): 
             // => new(generate::Matrix, "name", value)
generate::Matrix::CF:= proc(e, p, t, opts)
begin
   // warning("Generating code for matrix '".op(e,1)."'");
   // special case for assignments 'v:= piecewise()'
   if type(e) = "_assign" then
      e:= subsop(op(e,2), 1=op(e,1)); // rename matrix locally: use name given as lhs
   end_if;
   DOM_ARRAY::CF(op(e,2), p, t, opts, op(e,1)); 
end_proc:
/*
// generate::Simscape(matrix([[x'(t),2],[x''(t),4]]))
generate::Matrix::maprec:= proc()
begin
   warning("in generate::Matrix::maprec");
   args();
end_proc:
*/

/*--
generate::CFdoMaskMatrixes -- substitute matrices and identifier that hold a matrix by a
                              new(generate::Matrix, name, value) object. This masking is
                              needed due to the fact that the name of a matrix might get
                              lost when the input expression is evaluated.
generate::CFdoMaskMatrixes()
   args() : expression(s)
--*/
generate::CFdoMaskMatrixes:= proc()
   local t;
begin
   // NOTE: args() is not assigned to a local variable to avoid any side effect,
   // e.g. like the implicit change of the procedure name when assigning it!!!!!

   if domtype(args()) = DOM_LIST then
      return(map(args(), generate::CFdoMaskMatrixes));
   end_if;
   t:= type(args());

   // Do not look for matrices in holded expressions
   if t = "hold" then
      return(args());
   end_if;

   // detect matrices and function calls that are known to generate matrices
   if contains(generate::MatrixTypes, t) or (t = "function" and contains(generate::MatrixTypes, op(args(),0))) then
      // Evaluate the content of 'args()' and mask it using 'generate::Matrix'
      new(generate::Matrix, generate::CFnewArrayVariable(), eval(args()));

   // identifier: recursive descend on content
   elif t = DOM_IDENT then
      // Read the content of this variable (level 1 evaluation in procs)
      eval(args());
      // avoid recursion and evaluation of piecewise (which cannot hold a matrix)!
      if % <> args() and type(%) <> hold(piecewise) and type(%) <> piecewise and (type(%) <> "function" or op(args(),0) <> hold(piecewise)) then  
         generate::CFdoMaskMatrixes(%);
         if type(%) = generate::Matrix then
            subsop(%, 1=args()); // set name of matrix to top level variable
          else 
            %;
          end_if;
      else 
         args();
      end_if;

   // expression: recursive descend on operands
   elif domtype(args()) = DOM_EXPR and t <> "_index" then
      op(args(),0)(op(map([op(args())], generate::CFdoMaskMatrixes)));

   // any other object
   else
      args();
   end_if;
end_proc:

generate::Piecewise:= newDomain("generate::Piecewise"): 
             // => new(generate::Piecewise, "name", value)
generate::Piecewise::CF:= proc(e, p, t, opts)
begin
   // warning("Generating code for piecewise '".op(e,1)."'");
   // special case for assignments 'v:= piecewise()'
   if type(e) = "_assign" then
      e:= subsop(op(e,2), 1=op(e,1)); // rename piecewise locally: use name given as lhs
      piecewise::CF(op(e,2), p, t, opts, op(e,1));
   else
      // is of type 'generate::Piecewise'
      piecewise::CF(op(e,2), p, t, opts, 
         // 'op(e,1)' may contain the name of the identifier that evaluated to piecewise.
         // Undo autobinding, if it is not wanted. Also see 'generate::CFdoMaskPiecewise'
         (if opts["AutoBindToVarPiecewise"] = TRUE then op(e,1) 
                                                   else #PiecewiseWithNoName 
          end)
      ); 
   end_if;
end_proc:

/*--
generate::CFdoMaskPiecewise -- substitute piecewise and identifiers that hold a piecewise 
                               by a new(generate::Piecewise, name, value) object. This 
                               masking is needed due to the fact that the name of the 
                               piecewise might get lost when the input expression is 
                               evaluated.
generate::CFdoMaskPiecewise()
   args() : expression(s)
--*/
generate::CFdoMaskPiecewise:= proc()
   local t;
begin
   // NOTE: args() is not assigned to a local variable to avoid any side effect,
   // e.g. like the implicit change of the procedure name when assigning it!!!!!

   if domtype(args()) = DOM_LIST then
      return(map(args(), generate::CFdoMaskPiecewise));
   end_if;
   t:= type(args());

   // Do not look for matrices in holded expressions
   if t = "hold" then
      if type(op(args())) = "function" and op(op(args()),0) = hold(piecewise) then
        return(generate::CFdoMaskPiecewise(eval(args())));
      else
        return(args());
      end_if;
   end_if;

   // detect matrices and function calls that are known to generate matrices
   if t = hold(piecewise) or t = piecewise or (t = "function" and op(args(),0) = hold(piecewise)) then
      // Evaluate the content of 'args()' and mask it using 'generate::Piecewise'
      eval(args());
      if type(%) = hold(piecewise) or type(%) = piecewise then
         new(generate::Piecewise, #PiecewiseWithNoName, %);
      else
         args();
      end_if;

   // identifier: recursive descend on content
   elif t = DOM_IDENT then
      // Read the content of this variable (level 1 evaluation in procs)
      eval(args());
      if % <> args() then  // avoid recursion
         generate::CFdoMaskPiecewise(%);
         if type(%) = generate::Piecewise then
            subsop(%, 1=args()); // set name of matrix to top level variable
          else 
            %; // can this happen? just to be safe. keeps code robust.
          end_if;
      else 
         args();
      end_if;

   // expression: recursive descend on operands
   elif domtype(args()) = DOM_EXPR and t <> "_index" then
      op(args(),0)(op(map([op(args())], generate::CFdoMaskPiecewise)));
      
   // any other object
   else
      args();
   end_if;
 end_proc:

/*--
generate::CFdoSubsLeftEqualByAssign -- rewrites 'a = ...' to 'a := ...' where 'a' must 
                                       be of type DOM_INT or "_index". Otherwise this 
                                       function throws an error.
generate::CFdoSubsLeftEqualByAssign()
   args() : expression(s)
--*/
generate::CFdoSubsLeftEqualByAssign:= proc()
begin   
   // NOTE: args() is not assigned to a local variable to avoid any side effect,
   // e.g. like the implicit change of the procedure name when assigning it!!!!!
   
   if domtype(args()) = DOM_LIST then
      return(map(args(), generate::CFdoSubsLeftEqualByAssign));
   end_if;
    
   if type(args()) <> "_equal" then 
      return(args()) 
    end_if;
    
   if not contains({DOM_IDENT, "_index"}, type(op(args(),1))) then  /* Special case not needed: or type(op(args(),1)) = "_equal" */
      // is '(a=b)=c' or something like 'a+b=c' => both is invalid
      error("invalid left hand side in assignment");
   end_if;
    
   // is 'a = ...' or 'a[i] = ...'  
   return(hold(_assign)(op(args())));
end_proc:
 
/*--
generate::CFdoAutoAddAssign -- rewrites '...' to 't0 := ...' where '...' is any object  
                               excluding assignments. 
generate::CFdoAutoAddAssign()
   args() : expression(s)
--*/
generate::CFdoAutoAddAssign:= proc()
begin
   // NOTE: args() is not assigned to a local variable to avoid any side effect,
   // e.g. like the implicit change of the procedure name when assigning it!!!!!
   
   if domtype(args()) = DOM_LIST then
      return(map(args(), generate::CFdoAutoAddAssign));
   end_if;
    
   // Some objects have to be treated differently...
   case type(args())
     // Nothing to do on assignments
     of "_assign" do
        return(args());
        
     // Objects of type 'generate::Matrix' are treated differently.
     // Ignore them here. See also 'generate::CFdoMaskMatrixes'.
     of generate::Matrix do
        return(args());
        
     // Objects of type 'generate::Piecewise' are treated differently.
     // Ignore them here. See also 'generate::CFdoMaskPiecewise'.
     of generate::Piecewise do
        return(args());

     // Objects of type 'DOM_PROC' are treated differently. 
     // See 'DOM_PROC:CF'.
     of DOM_PROC do
        return(args());
        
     // Ignore any kind of programming statements
     of "return"     do
     of "_stmtseq"   do
     of "_if"        do
        return(args());   
     
     // Default: assign the object to a scalar variable
     otherwise
        hold(_assign)(generate::CFnewScalarVariable(), args());
   end_case;
end_proc:

/*--
generate::CFunwrapOde -- returns a CF formatted string for an ode
   args() : ODE

   EXAMPLE: ode(mx''(t)+b*x'(t)+k*x(t), x(t))
            [diff(mx(t), t, t) + b*diff(x(t), t) + k*x(t) = 0] 
--*/
generate::CFunwrapOde:= proc() 
begin
   //op(args(),2);  // function variable, e.g. 'x'
   //op(args(),3);  // time variable,     e.g. 't'

   if type(op(args(),1)) = DOM_SET then
      // system of ODEs
      [ op(op(args(),4)),                    // initial conditions as sequence, might be empty
        op(map(op(args(),1), ex -> ex = 0))  // sequence of ODEs
      ];
   else
      // single ODE
      [ op(op(args(),4)),                    // initial conditions as sequence, might be empty
        op(args(),1) = 0                     // ODE
      ];
   end_if;
end_proc:

/*--
generate::CFunwrapOdes -- Unwraps ODEs in a given object.
generate::CFunwrapOdes()
   args() : expression(s)
--*/
generate::CFunwrapOdes:= proc()
   local t;
begin
   // NOTE: args() is not assigned to a local variable to avoid any side effect,
   // e.g. like the implicit change of the procedure name when assigning it!!!!!

   if domtype(args()) = DOM_LIST then
      return(map(args(), generate::CFunwrapOdes));
   end_if;
   t:= type(args());

   // detect matrices and function calls that are known to generate matrices
   if t = ode or t = hold(ode) then
      op(generate::CFunwrapOde(args()));

   // identifier: recursive descend on content
   elif t = DOM_IDENT then
      // Read the content of this variable (level 1 evaluation in procs)
      eval(args());
      if type(%) = ode or type(%) = hold(ode) then  
         op(generate::CFunwrapOde(%)); // can this happen? keeps code robust.
      else 
         args();
      end_if;

   // expression: recursive descend on operands
   elif domtype(args()) = DOM_EXPR and t <> "_index" then
      op(args(),0)(op(map([op(args())], generate::CFunwrapOdes)));

   // any other object
   else
      args();
   end_if;
end_proc:

/*--
generate::CFreduceOrderOfDerivaties -- Reduces the order of derivatives by introducing new variables.
                                       This is needed for generating Simscape code.
generate::CFreduceOrderOfDerivaties(exl, opts)
   exl     : list of expression or list of list of expression
   opts    : defaults and options
   flatten : for single expressions return FALSE => a list of list, TRUE => a sequence.

   returns: [variablesList, expressionList, derivativesList]
            or
            variables, expressions

   e.g.:    generate::CFreduceOrderOfDerivaties(m*x''(t)+b*x'(t)+k*x(t), generate::Sdefaults)
            [[x3 = D(x)(t), x4 = D(x3)(t)], [b*D(x)(t) + k*x(t) + m*x4], [D(x), (D@@2)(x)]]
            or
            generate::CFreduceOrderOfDerivaties(m*x''(t)+b*x'(t)+k*x(t), generate::Sdefaults, TRUE)
            x7 = D(x)(t)), x8 = D(x7)(t), b*D(x)(t) + k*x(t) + m*x8

   EXAMPLE: >> generate::Simscape(m*x''(t)+b*x'(t)+k*x(t) = 0):
            >> fprint(Unquoted, 0, %);
                 x1 = x.der;
                 x2 = x1.der;
                 m*x2+k*x+b*x.der == 0.0;
--*/
generate::CFreduceOrderOfDerivaties:= proc(exl, opts, flatten = FALSE)
   local derList, varList, isConst, var, tvar, fvar, dcnt, i, fvar1, fvar2, cder, pos;
begin
   if args(0) < 2 then
      error("unexpected object of type DOM_NULL");
   end_if;
    
   // Recursive call for lists
   if domtype(exl) = DOM_LIST then
      return(map(exl, generate::CFreduceOrderOfDerivaties, opts, flatten));
   end_if;
   
   // Fill derList and varlist (the i-th elements of these lists corresponds to each other)
   // derList[ex]: list of derivatives diff(ex,t,t), ...
   // varList[ex]: list of new variables and their first order derivatives
   derList:= table();
   varList:= table();
   isConst:= table();
   
   // Rewrite x'(t) to diff(x(t),t) and do the same for higher order derivatives
   misc::maprec(exl, {DOM_EXPR} = proc()
    begin
       // Handle expressions of type x'(t) or x''(t) with 'x' is a function in 't'.
       if type(op(args(),0)) = "D" then
          fvar:= op(args(),0);                            
          var:= [op(args())];
          
          if nops(var) = 0 then // f'() = D(f)()
             (()->error("Missing differentiation variable in '".expr2text(exl)."'"))();
          end_if;

          // Determine the order of the derivative and the function name, e.g. 'x' in x''(t)
          dcnt:= 0;
          repeat
             dcnt:= dcnt + 1;
             fvar:= op(fvar);
          until type(fvar) <> "D" end_repeat;

          assert(opts["NameOfTimeVariable"] <> "");
          tvar:= hold(``).opts["NameOfTimeVariable"];
          if {op(var)} <> {tvar} then
             isConst[fvar(op(var))]:= TRUE;
             // Reduce time derivative w.r.t. time variable only when it is applied to "t"
             // For other values x'(a) <> diff(x(a), t);
             // x'(t), x'(t,t) is allowed, but not x'(a,t) or x'(x,b) or ...
             if opts["Warnings"] and type(op(op(args(),0))) = "D" then
                // Warn only if horder is higher then one.
                warning("Reducing the order of time derivatives of time variable '".opts["NameOfTimeVariable"]."' only.");
              //warning("Reducing the order of derivatives w.r.t. time variable '".opts["NameOfTimeVariable"]."'.");
             end_if;
             return(args());
          end_if;
                      
          // fvar is expected to be an identifier
          if type(fvar) = DOM_IDENT then
             // Rewrite derivative from prime notation into diff notation 
                hold(diff)(fvar(op(var)), tvar$dcnt);
          else
             args();
          end_if;
       else
          args();
       end_if;
    end_proc,
    Unsimplified // Do not simplify to avoid any side effects on code generation!!!
   );
   
   // Now exl contains derivaties of type diff(..., ...) only
   // Avoid any assignnment. This may have side effects on DOM_PROCs!!!
   misc::maprec(%, {"diff"} = proc()
    local der;
    begin
       if type(op(args(),1)) = "diff" then
          // Return evaluated input in order to flatten nested diffs
          der:= eval(args()); 
          // Assumption: Still a diff because every nested diff comes from  
          // rewriting diff(x'(t),t) to diff(diff(x(t),t),t). Other nested
          // diffs should have been flatted before.
          assert(type(der) = "diff");
       else
          // Return input unchanged
          der:= args();
       end_if;
       
       // Expressions of type diff(x(t), t), diff(x(t), t, t), ...
       fvar:= op(der,1);
       tvar:= [op(der,2..nops(der))];
       
       // Special case: diff(any) = any ==> do not further analyse this case.
       if nops(tvar) = 0 then 
          return(args());
       end_if;
       
       // Is this a time derivative?
       assert(opts["NameOfTimeVariable"] <> "");
       if {op(tvar)} <> {hold(``).opts["NameOfTimeVariable"]} then
          // Simscape: Can handle time derivatives (w.r.t. time variable) only
          if opts["Language"] = "Simscape" then
             (()->error("Time derivative with respect to '".opts["NameOfTimeVariable"]."' expected in '".der."'"))();
          end_if;
          // Other target languages: Ignore this derivative 
          return(der);
       end_if;
        
       // Get order of derivative and time variable and initialize lists
       dcnt:= nops(tvar);
       tvar:= tvar[1];
       if type(derList[fvar]) <> DOM_LIST then
          derList[fvar]:= [];
          varList[fvar]:= [];
       end_if;
       
       // Introduce new variables and equations for high order derivatives:
       // next_variable = first_order_derivative_of_previous_variable
       fvar1:= fvar;
       for i from 1 to dcnt  do
          cder:= hold(diff)(fvar, tvar$i);
          // position of 'cder' in derList: corresponds to varList
          if (pos:= contains(derList[fvar], cder)) > 0 then
             // Get corresponding name of variable from varList
             fvar1:= lhs(varList[fvar][pos]);
          else
             // Introduce new variable and equation   
             fvar2:= generate::CFnewDerivativeName();
             derList[fvar]:= derList[fvar].[cder];
             varList[fvar]:= varList[fvar].[fvar2 = hold(diff)(fvar1,tvar)];
             fvar1:= fvar2;
          end_if;
       end_for;
       der;
    end_proc,
    Unsimplified // Do not simplify to avoid any side effects on code generation!!!
   );
   
   // Now all new variables and equations are created, if any derivatives were found.
   // print(derList);
   // print(varList);
   // print(isConst);
   if derList = table() then
      return(exl);
   else
      %;  // Avoid any assignnment. This may have side effects on DOM_PROCs!!!
   end_if;
   
   // Reduce / substitute derivatives with help of derList and varList
   // Avoid any assignnment. This may have side effects on DOM_PROCs!!!
   misc::maprec(%, {"diff"} = proc()
    begin
       // Expressions of type diff(x(t), t), diff(x(t), t, t), ...
       fvar:= op(args(),1);
       tvar:= [op(args(),2..nops(args()))];
       
       // Special case: diff(any) = any ==> do not further analyse this case.
       if nops(tvar) = 0 then
          return(args());
       end_if;
       
       // Assumption: derivative exists in both lists, lists have same length
       pos:= contains(derList[fvar], args());
       assert(pos > 0 and pos <= nops(derList[fvar]) and nops(derList[fvar]) = nops(varList[fvar]));
               
       // Return first order derivative or the corresponding new variable
       if pos = nops(derList[fvar]) then
         // Use x.der notation for the highest order derivatives
         rhs(varList[fvar][pos]);
       else
         // Use new variable for lower order derivatives
         lhs(varList[fvar][pos]);
       end_if
    end_proc,
    Unsimplified // Do not simplify to avoid any side effects on code generation!!!
   );
   
   if flatten then
      // Return a sequence of expressions for code generation
      // 1. sequence of new equations without highest order
      // 2. expression where higher order derivatives are substituted
      if opts["Language"] = "Simscape" then
         // op(map(map(rhs(varList), _index, 1..-2), op)), %;
         op(map(map(map(rhs(varList), _index, 1..-2), op), (o)->o[2]=o[1])), %;
      else
         op(map(map(map(rhs(varList), _index, 1..-2), op), subsop, 0=hold(_assign))), %;
      end;
   else
      [map(rhs(varList), op), [%], map(rhs(derList), op)]; // return a list of lists for further post-processing
   end_if;
end_proc:


// ================================================================================================
// HELPER FUNCTIONS: CODE FORMATTING
// ================================================================================================

/*--
generate::CFformatting -- post processor for code formatting
generate::CFformatting(s, opts, n, alreadyIndented)
   s    : source code line, a character string
   opts : defaults and options
   n    : line number, counted when splitting lines
   alreadyIndented : This code is already indented.
--*/
generate::CFformatting:= proc(s, opts, n=1, alreadyIndented=FALSE)
   local i, l, k, prefix;
begin
   // Stop immediately.
   if s = "" then
      return("");
   end_if;
    
   // Determine type of indent
   if n=1 then
      i:= opts["IndentInitial"];
      if alreadyIndented then
         i:= "";
      end_if;
   else 
      i:= opts["IndentInitialContinue"];
   end_if;
   
   // Separate lines from each other
   l:= select(text2list(s, ["\n"]), e->bool(e <> "\n"));

   // Map formatting routine on each line
   if nops(l) > 1 then
      return(_concat(op(map(l, generate::CFformatting, opts, 1, alreadyIndented))));
   end_if;
   
   s:= l[1]; // nops(l) = 1 and s does not contain any '\n'!
   
   // Special formatting rules for FORTRAN
   if opts["Language"] = "FORTRAN" then
      k:= length(opts["IndentInitialContinue"]);
      // Do not indent comment line or continued lines
      if (length(s)>=2 and s[1..2]="C ") or 
         (length(s)>=k and s[1..k]=opts["IndentInitialContinue"]) then
         i:= "";
      end_if;

      // No further line wrapping is necessary
      if length(s) <= 66 then 
         return(i.s."\n");
      end_if;
       
      prefix:= ""; // default
      // Use comment prefix when splitting a comment line into several lines
      if length(s)>1 and s[1..2]="C " then
         prefix:= "C ";
      end_if;
      return(i.s[1..66]."\n".generate::CFformatting(prefix.s[67..-1], opts, n+1, alreadyIndented));
   end_if;
    
   // Default formatting for all languages other than FORTRAN
   i.s."\n";      
end_proc:


// ================================================================================================
// HELPER VALUES: 
// ================================================================================================
generate::CFoutPrioBoolOpMin:= min(
   output::Priority::Equiv,
   output::Priority::Implies,
   output::Priority::Or,
   output::Priority::Xor,
   output::Priority::And
):


// ================================================================================================
// HELPER FUNCTIONS: EXPRESSION ANALYSIS
// ================================================================================================

/*--
generate::CFisdiv -- test if expression is 1/x
generate::CFisdiv(e, opts)
   e   : expression
   opts: defaults and options
--*/
generate::CFisdiv:= proc(e, opts)
begin
   _lazy_and( type(e) = "_power", op(e,2) = -1 );
end_proc:

/*--
generate::CFisminus -- test if expression is -x
generate::CFisminus(e, opts)
   e   : expression
   opts: defaults and options
--*/
generate::CFisminus:= proc(e, opts)
  local lastop;
begin
   // normal form: if last operand is a numeric value
   // then it holds the sign of the whole product.
   if type(e) = "_mult" then
      lastop:= op(e,nops(e));
      if testtype(lastop,Type::Numeric) then
         if domtype(lastop) = DOM_COMPLEX then
            if sign(lastop) = -I or sign(lastop) = -1.0*I then
               return(TRUE);
            end_if;
            return(FALSE);
         else
            if contains({RD_NAN}, eval(lastop)) then
               return(FALSE);
            else
               return(bool(lastop < 0));
            end_if;
         end_if;
      end_if;
   elif testtype(e,Type::Numeric) then
      if domtype(e) = DOM_COMPLEX then
         if sign(e) = -I or sign(e) = -1.0*I then
            return(TRUE);
         end_if;
         return(FALSE);
      else
         if contains({RD_NAN}, eval(e)) then
            return(FALSE);
         else
            return(bool(e < 0))
         end_if;
      end_if;
   end_if;
   FALSE;
end_proc:

// ================================================================================================
// HELPER FUNCTIONS: USER NOTIFICATIONS
// ================================================================================================

/*--
generate::CFunverified -- 'not supported' warning or error message
generate::CFunverified(f, opts, nam)
   f    : function name of 0th operand of a function call.
   opts : defaults and options
   nam  : character string "function", "object", "constant" or...
--*/
generate::CFunverified := proc(f, opts, nam="function")
  local N, n, s, notifier;
begin
   if domtype(f) <> DOM_STRING then
      f:= prog::getname(f);
   end_if;
   N:= stringlib::upper(nam[1]).stringlib::lower(nam[2..-1]);
   n:= stringlib::lower(nam[1]).stringlib::lower(nam[2..-1]);
   s:= N." '".f."' is not verified to be a valid ".opts["Language"]." ".n.".";
   
   // Note: depending on the 'notifier' the code generation might be stopped!
   notifier:= opts["OnUnverifiedName"];
   if notifier = "warning" then
      if opts["Warnings"] then
         (()->warning(s))();
      end_if;
   elif notifier = "error" then
      (()->error(s))();
   end_if;
end_proc:

/*--
generate::CFrequires -- 'requires xxx compiler support' warning or error message
generate::CFrequires(ver, f, opts, nam)
   ver  : a version character string, e.g. "FORTRAN77", "FORTRAN90" or "FORTRAN2008"
   f    : function name of 0th operand of a function call.
   opts : defaults and options
   nam  : character string "function", "object", "constant" or...
--*/
generate::CFrequires := proc(ver, f, opts, nam="function")
  local N, n, s, notifier;
begin
   if domtype(ver) <> DOM_STRING then
      error("compiler version character string expected as first argument");
   end_if;
   if domtype(f) <> DOM_STRING then
      f:= prog::getname(f);
   end_if;
   N:= stringlib::upper(nam[1]).stringlib::lower(nam[2..-1]);
   n:= stringlib::lower(nam[1]).stringlib::lower(nam[2..-1]);
   s:= N." '".f."' requires a ".stringlib::subs(ver,"_"=" ")." compiler.";

   // Note: depending on the 'notifier' the code generation might be stopped!
   notifier:= opts["OnRequires".ver];
   if notifier = "warning" then
      if opts["Warnings"] then
         (()->warning(s))();
      end_if;
   elif notifier = "error" then
      (()->error(s))();
   end_if;
end_proc:

// ================================================================================================
// HELPER FUNCTIONS: DEFAULTS/OPTIONS CHECKER
// ================================================================================================

/*--
generate::CFcheckDefaults -- check contents of opts
generate::CFcheckDefaults()
--*/
generate::CFcheckDefaults := proc(defaults)
begin
   if domtype(defaults) <> DOM_TABLE then
      "table of defaults and options expected as last argument";
   elif domtype(defaults["Language"]) <> DOM_STRING or
       (defaults["Language"] <> "C"         and
        defaults["Language"] <> "FORTRAN"   and
        defaults["Language"] <> "MATLAB"    and
        defaults["Language"] <> "Simscape") then
      "default value \"Language\" must be \"C\" or \"FORTRAN\" or \"MATLAB\" or \"Simscape\"";
   elif domtype(defaults["NameOfTimeVariable"]) <> DOM_STRING then
      "default value \"NameOfTimeVariable\" must be a character string";
   elif domtype(defaults["RenameTimeVariableTo"]) <> DOM_STRING then
      "default value \"RenameTimeVariableTo\" must be a character string";
   elif defaults["Warnings"] <> TRUE and 
      defaults["Warnings"] <> FALSE then
      "default value \"Warnings\" must be TRUE or FALSE";
   elif defaults["SubsLeftEqualByAssign"] <> TRUE and
      defaults["SubsLeftEqualByAssign"] <> FALSE  then
      "default value \"SubsLeftEqualByAssign\" must be TRUE or FALSE";
   elif defaults["AutoAddAssign"] <> TRUE and
      defaults["AutoAddAssign"] <> FALSE  then
      "default value \"AutoAddAssign\" must be TRUE or FALSE";
   elif defaults["AutoAddAssignPiecewise"] <> TRUE and
      defaults["AutoAddAssignPiecewise"] <> FALSE  then
      "default value \"AutoAddAssignPiecewise\" must be TRUE or FALSE";
   elif defaults["AutoBindToVarPiecewise"] <> TRUE and
      defaults["AutoBindToVarPiecewise"] <> FALSE  then
      "default value \"AutoBindToVarPiecewise\" must be TRUE or FALSE";
   elif defaults["ReduceOrderOfDerivaties"] <> TRUE and
      defaults["ReduceOrderOfDerivaties"] <> FALSE  then
      "default value \"ReduceOrderOfDerivaties\" must be TRUE or FALSE";
   elif domtype(defaults["IndentInitial"]) <> DOM_STRING then
      "default value \"IndentInitial\" must be a character string";
   elif domtype(defaults["IndentInitialContinue"]) <> DOM_STRING then
      "default value \"IndentInitialContinue\" must be a character string";
   elif domtype(defaults["IndentWidth"]) <> DOM_STRING then
      "default value \"IndentWidth\" must be a character string";
   elif domtype(defaults["IndentLevel"]) <> DOM_INT or 
      defaults["IndentLevel"] < 0  then
      "default value \"IndentLevel\" must be a non-negative integer";
   elif domtype(defaults["StatementDelimiter"]) <> DOM_STRING then
      "default value \"StatementDelimiter\" must be a character string";
   elif domtype(defaults["Digits"]) <> DOM_INT or 
      defaults["Digits"] < 1 then
      "default value \"Digits\" must be a positive integer";
   elif defaults["IntAsIntWhereObvious"] <> TRUE and 
      defaults["IntAsIntWhereObvious"] <> FALSE then
      "default value \"IntAsIntWhereObvious\" must be TRUE or FALSE";
   elif defaults["IntToFloat"] <> TRUE and 
      defaults["IntToFloat"] <> FALSE then
      "default value \"IntToFloat\" must be TRUE or FALSE";
   elif domtype(defaults["DoubleFormat"]) <> DOM_STRING then
      "default value \"DoubleFormat\" must be a character string";
   elif domtype(defaults["DoubleEnotationChar"]) <> DOM_STRING then
      "default value \"DoubleEnotationChar\" must be a character string";
   elif domtype(defaults["False"]) <> DOM_STRING and
      defaults["False"] <> FAIL then
      "default value \"False\" must be a character string or FAIL";
   elif domtype(defaults["True"]) <> DOM_STRING and
      defaults["True"] <> FAIL then
      "default value \"True\" must be a character string or FAIL";
   elif domtype(defaults["Unknown"]) <> DOM_STRING and
      defaults["Unknown"] <> FAIL then
      "default value \"Unknown\" must be a character string or FAIL";
   elif domtype(defaults["ComplexI"]) <> DOM_STRING and
      defaults["ComplexI"] <> FAIL then
      "default value \"ComplexI\" must be a character string or FAIL";
   elif domtype(defaults["ComplexPrefix"]) <> DOM_STRING then
      "default value \"ComplexPrefix\" must be a character string";
   elif domtype(defaults["PowerOperator"]) <> DOM_STRING and
      defaults["PowerOperator"] <> FAIL then
      "default value \"PowerOperator\" must be a character string or FAIL";
   elif domtype(defaults["PowerThreshold"]) <> DOM_INT or 
      defaults["PowerThreshold"] < 1 then
      "default value \"PowerThreshold\" must be a positive integer";
   elif defaults["PowerInvertNegIntRatExp"] <> TRUE and
      defaults["PowerInvertNegIntRatExp"] <> FALSE  then
      "default value \"PowerInvertNegIntRatExp\" must be TRUE or FALSE";
   elif defaults["PowerKeepIntExp"] <> TRUE and
      defaults["PowerKeepIntExp"] <> FALSE  then
      "default value \"PowerKeepIntExp\" must be TRUE or FALSE";
   elif domtype(defaults["Log2"]) <> DOM_STRING and
      defaults["Log2"] <> FAIL then
      "default value \"Log2\" must be a character string or FAIL";
   elif domtype(defaults["Log10"]) <> DOM_STRING and
      defaults["Log10"] <> FAIL then
      "default value \"Log10\" must be a character string or FAIL";
   elif domtype(defaults["ReturnType"]) <> DOM_STRING then
      "default value \"ReturnType\" must be a character string";
   elif defaults["TypeChecking"] <> TRUE and 
      defaults["TypeChecking"] <> FALSE then
      "default value \"TypeChecking\" must be TRUE or FALSE";
   elif defaults["OnUnverifiedName"] <> "warning" and 
      defaults["OnUnverifiedName"] <> "error" and
      defaults["OnUnverifiedName"] <> FAIL then
      "default value \"OnUnverifiedName\" must be \"warning\", \"error\" or FAIL";
   elif defaults["OnRequiresFORTRAN_77"] <> "warning" and 
      defaults["OnRequiresFORTRAN_77"] <> "error" and
      defaults["OnRequiresFORTRAN_77"] <> FAIL then
      "default value \"OnRequiresFORTRAN_77\" must be \"warning\", \"error\" or FAIL";
   elif defaults["OnRequiresFORTRAN_90"] <> "warning" and 
      defaults["OnRequiresFORTRAN_90"] <> "error" and
      defaults["OnRequiresFORTRAN_90"] <> FAIL then
      "default value \"OnRequiresFORTRAN_90\" must be \"warning\", \"error\" or FAIL";
   elif defaults["OnRequiresFORTRAN_95"] <> "warning" and 
      defaults["OnRequiresFORTRAN_95"] <> "error" and
      defaults["OnRequiresFORTRAN_95"] <> FAIL then
      "default value \"OnRequiresFORTRAN_95\" must be \"warning\", \"error\" or FAIL";
   elif defaults["OnRequiresFORTRAN_2003"] <> "warning" and 
      defaults["OnRequiresFORTRAN_2003"] <> "error" and
      defaults["OnRequiresFORTRAN_2003"] <> FAIL then
      "default value \"OnRequiresFORTRAN_2003\" must be \"warning\", \"error\" or FAIL";
   elif defaults["OnRequiresFORTRAN_2008"] <> "warning" and 
      defaults["OnRequiresFORTRAN_2008"] <> "error" and
      defaults["OnRequiresFORTRAN_2008"] <> FAIL then
      "default value \"OnRequiresFORTRAN_2008\" must be \"warning\", \"error\" or FAIL";
   elif domtype(defaults["NameOfScalar"]) <> DOM_STRING then
      "default value \"NameOfScalar\" must be a character string";
   elif domtype(defaults["NameOfArray"]) <> DOM_STRING then
      "default value \"NameOfArray\" must be a character string";
   elif domtype(defaults["NameOfFunction"]) <> DOM_STRING then
      "default value \"NameOfFunction\" must be a character string";
   elif domtype(defaults["NameOfDerivative"]) <> DOM_STRING then
      "default value \"NameOfDerivative\" must be a character string";
   elif defaults["DerivativeVectorNotation"] <> TRUE and 
      defaults["DerivativeVectorNotation"] <> FALSE then
      "default value \"DerivativeVectorNotation\" must be TRUE or FALSE";
   elif domtype(defaults["NameIndexFirst"]) <> DOM_INT or 
      defaults["NameIndexFirst"] < 0 then
      "default value \"NameIndexFirst\" must be a non-negative integer";
   elif domtype(defaults["NameIndexFirstDerivative"]) <> DOM_INT or 
      defaults["NameIndexFirstDerivative"] < 0 then
      "default value \"NameIndexFirstDerivative\" must be a non-negative integer";
   elif domtype(defaults["NameIndexDigits"]) <> DOM_INT or 
      defaults["NameIndexDigits"] < 0 then
      "default value \"NameIndexDigits\" must be a non-negative integer";
   elif defaults["GenSparseMatrix"] <> TRUE and 
      defaults["GenSparseMatrix"] <> FALSE then
      "default value \"GenSparseMatrix\" must be TRUE or FALSE";
   elif defaults["Convert1x1MatrixToScalar"] <> TRUE and 
      defaults["Convert1x1MatrixToScalar"] <> FALSE then
      "default value \"Convert1x1MatrixToScalar\" must be TRUE or FALSE";
   elif domtype(defaults["GenerateMatrixStyle"]) <> DOM_STRING then
      "default value \"GenerateMatrixStyle\" must be a character string";
   elif domtype(defaults["IndexLeft"]) <> DOM_STRING then
      "default value \"IndexLeft\" must be a character string";
   elif domtype(defaults["IndexRight"]) <> DOM_STRING then
      "default value \"IndexRight\" must be a character string";
   elif domtype(defaults["IndexStyle"]) <> DOM_STRING then
      "default value \"IndexStyle\" must be a character string";
   elif defaults["ContinueEnumeration"] <> TRUE and 
      defaults["ContinueEnumeration"] <> FALSE then
      "default value \"ContinueEnumeration\" must be TRUE or FALSE";
   elif defaults["SupportProcedures"] <> TRUE and 
      defaults["SupportProcedures"] <> FALSE then
      "default value \"SupportProcedures\" must be TRUE or FALSE";
   elif domtype(defaults["MathConstants"]) <> DOM_TABLE then
      "default value \"MathConstants\" must be a table";
   else
      "";
   end_if;
end_proc:


/*--
generate::CFcheckUserOptions -- check for valid user option and interpret them
generate::CFcheckUserOptions()
   usrOpts: list of user options
   opts   : defaults and options
--*/
generate::CFcheckUserOptions:= proc(usrOpts, opts)
  local badOpts, intOpts, tmpOpts, opt, notOptionAlreadyHandled;
begin
  // Options should be given as equations like OptionName=OptionsValue.
  // Otherwise convert them to OptionName=TRUE.
  usrOpts:= map(usrOpts, O -> (if type(O) <> "_equal" then O = TRUE else O end));

  // The left-hand-side of an option must be an identifier.
  badOpts:= select(usrOpts, O -> bool(type(lhs(O)) <> DOM_IDENT));
  if badOpts <> [] then
    if rhs(badOpts[1]) = TRUE then
      (()->error("Invalid option: '".expr2text(lhs(badOpts[1]))."' or '".expr2text(badOpts[1])."'"))();
    else
      (()->error("Invalid option: '".expr2text(badOpts[1])."'"))();
    end_if;
  end_if;

  // Separate internal options (undocumented, starting with '#') from
  // the official options (documented).
  [intOpts, usrOpts, tmpOpts]:= split(usrOpts, O -> bool(expr2text(lhs(O))[1] = "#"));

  // Handle internal (undocumented) options: Convert, e.g. `#Language` 
  // to "Language", check if table 'opts' holds a default value for it.
  intOpts:= map(intOpts, O -> (expr2text(lhs(O))[2..-1] = rhs(O))); // convert
  badOpts:= select(intOpts, O -> bool(lhs(O)="Language" or not contains(opts, lhs(O))));
  if badOpts <> [] then
    if rhs(badOpts[1]) = TRUE then
      (()->error("Invalid option: '".expr2text(`#`.lhs(badOpts[1]))."' or '".expr2text(`#`.lhs(badOpts[1])=rhs(badOpts[1]))."'"))();
    else
      (()->error("Invalid option: '".expr2text(`#`.lhs(badOpts[1])=rhs(badOpts[1]))."'"))();
    end_if;
  end_if;
  // Check the specified values. Do it here to get nicer error messages.
  badOpts:= generate::CFcheckDefaults(table(opts, intOpts));
  if badOpts <> "" then
    (()->error(stringlib::subs(badOpts, "default value"="value of option")))();
  end_if;

  // Handle official (documented) options, depending on the target language
  tmpOpts:= [];
  badOpts:= [];
  for opt in usrOpts do
    notOptionAlreadyHandled:= TRUE;

    case lhs(opt)
      of NoWarning do:
         if rhs(opt) <> TRUE and rhs(opt) <> FALSE then
           badOpts:= [opt];
           break;
         end_if;
         tmpOpts:= tmpOpts.["Warnings" = not rhs(opt)];
         notOptionAlreadyHandled:= FALSE;
         break;
    end_case;

    if notOptionAlreadyHandled then
      case opts["Language"]
        // ----------------------------------------------------------------------
        of "C" do:
           case lhs(opt)
             otherwise:
                badOpts:= [opt];
           end_case;
           break;

        // ----------------------------------------------------------------------
        of "FORTRAN" do:
           case lhs(opt)
             otherwise:
                badOpts:= [opt];
           end_case;
           break;

        // ----------------------------------------------------------------------
        of "MATLAB" do:
        of "Simscape" do:
           case lhs(opt)
             otherwise:
                badOpts:= [opt];
           end_case;
           break;
      end_case;
    end_if;

    // Throw error on invalid options
    if badOpts <> [] then
      if rhs(badOpts[1]) = TRUE then
        (()->error("Invalid option: '".expr2text(lhs(badOpts[1]))."' or '".expr2text(lhs(badOpts[1])=rhs(badOpts[1]))."'"))();
      else
        (()->error("Invalid option: '".expr2text(lhs(badOpts[1])=rhs(badOpts[1]))."'"))();
      end_if;
    end_if;
  end_for;
  return(tmpOpts.intOpts);
end_proc:


// end of file 
