/*--
generate::TeX.mu -- TeX-output of expressions

--*/
alias(OP = output::Priority):

  // store a flag whether we are inside TeXmacs
  proc()
    local texmacs;
    option escape;
  begin
    texmacs := FALSE;
    generate::ifTeXmacs :=
    proc(yes, no)
    begin
      if args(0) = 1 then
        texmacs := args(1);
      else
        if texmacs then yes else no end_if;
      end_if;
    end_proc:
  end_proc():

  //----------------------------------------------------
  // Call generate::whitespaceBeforeFractions(TRUE) to
  // activate an additional whitespace "-\\," between
  // a minus sign and a fraction (with small bitmaps of 
  // TeXed formulas, the minus sign seems to be a part
  // of the 'Bruchstrich'; add this whitespace for a
  // nicer picture).
  //----------------------------------------------------
  // store a flag whether we are using whitespaceBeforeFractions
  proc()
    local whitespacebeforefractions;
    option escape;
  begin
    whitespacebeforefractions := FALSE;
    generate::whitespaceBeforeFractions :=
    proc(yes, no)
    begin
      if args(0) = 1 then
        if args(1) = TRUE or args(1) = FALSE then
           whitespacebeforefractions := args(1);
        else
           error("expecting TRUE or FALSE"):
        end_if;
      else
        if whitespacebeforefractions then yes else no end_if;
      end_if;
    end_proc:
  end_proc():
  
  
/*++
generate::TeX -- return TeX-formatted string for expression

generate::TeX(e,...)

e - expression
++*/

generate::TeX:= proc(e) 
  local i;
  save PRETTYPRINT;
begin 
    PRETTYPRINT := FALSE;
    case args(0)
    of 0 do
        ""; break;
    of 1 do
        generate::tex(e, OP::Noop); break;
    otherwise
        _concat(generate::tex(e, OP::Exprseq),
                (generate::ifTeXmacs(",", ",\\; "),
                 generate::tex(args(i), OP::Exprseq)) $ i=2..args(0));
    end_case 
end_proc:
  
  
/*++
generate::TeXdef -- define TeX-string for expression

generate::TeXdef(e, s)

e - expression
s - TeX-string
++*/

generate::TeXdef:= (e, s)->(
    if args(0) <> 2 then error("wrong no of args") end_if;
    if domtype(s) <> DOM_STRING then error("no TeX string") end_if;
    generate::TeXdefs[e]:= s
);

// ------------------------ internas ----------------------- 

// table of user-defined TeX-strings 

generate::TeXdefs:=
table(
      hold(PI)="\\pi", I="\\mathrm{i}", hold(EULER)="\\gamma",
      op(map([
//              hold(Delta), hold(Phi), hold(Gamma), hold(Lambda),
//              hold(Pi), hold(Theta), hold(Sigma), hold(Upsilon),
//              hold(Omega), hold(Xi), hold(Psi), hold(alpha),
//              hold(beta), hold(chi), hold(delta), hold(epsilon),
//              hold(varepsilon), hold(phi), hold(varphi), hold(gamma),
//              hold(eta), hold(iota), hold(kappa), hold(lambda), hold(mu),
//              hold(nu), hold(pi), hold(varpi), hold(theta), hold(vartheta),
//              hold(rho), hold(varrho), hold(sigma), hold(varsigma),
//              hold(tau), hold(upsilon), hold(omega), hold(xi), hold(psi),
//              hold(zeta), hold(aleph), hold(wp),
              hold(partial), hold(Im), hold(Re),
              
              hold(arcsin), hold(arccos), hold(arctan), hold(arg),
              hold(sin), hold(sinh), hold(cos), hold(cosh), hold(tan),
              hold(tanh), hold(cot), hold(coth), hold(csc),
              hold(sec), hold(ln)
             ], (x -> (x = "\\".x)))),
         op(map([
                 hold(Ci), hold(Ei), hold(Si), hold(arccosh), hold(arccot),
                 hold(arccoth), hold(arccsc), hold(arccsch), hold(arcsec),
                 hold(arcsech), hold(arcsinh), hold(arctanh), hold(csch),
                 hold(sech), hold(dilog), hold(erf), hold(erfc), hold(frac),
                 hold(heaviside), hold(trunc), hold(bernoulli), hold(isprime)
                 ], (x->(x = "\\mathop{\\mathrm{".x."}}\\nolimits"))))
    ):
  
// decorations
generate::TeXaccents :=
  table("circ"     = "widehat",
        "tilde"    = "widetilde",
        "OverBar"  = "overline",
        "UnderBar" = "underline",
        "rarr"     = "vec",
        "check"    = "check",
        "acute"    = "acute",
        "grave"    = "grave",
        "breve"    = "breve",
        "dot"      = "dot",
        "die"      = "ddot",
        "Dot"      = "ddot",
        "tdot"     = "dddot",
        "Bbb"      = "mathbb",
        "bold"     = "mathbf",
        "cal"      = "mathcal",
        "frak"     = "mathfrak"):

// special identifiers
generate::TeXidents :=
  table("alpha"      = "\\alpha{}",
        "beta"       = "\\beta{}",
        "gamma"      = "\\gamma{}",
        "delta"      = "\\delta{}",
        "epsilon"    = "\\epsilon{}",
        "epsiv"      = "\\varepsilon{}",
        "zeta"       = "\\zeta{}",
        "eta"        = "\\eta{}",
        "theta"      = "\\theta{}",
        "thetav"     = "\\vartheta{}",
        "iota"       = "\\iota{}",
        "kappa"      = "\\kappa{}",
        "lambda"     = "\\lambda{}",
        "mu"         = "\\mu{}",
        "nu"         = "\\nu{}",
        "xi"         = "\\xi{}",
        "omicron"    = "o",
        "pi"         = "\\pi{}",
        "piv"        = "\\varpi{}",
        "rho"        = "\\rho{}",
        "rhov"       = "\\varrho{}",
        "sigma"      = "\\sigma{}",
        "sigmav"     = "\\varsigma{}",
        "tau"        = "\\tau{}",
        "upsilon"    = "\\upsilon{}",
        "phi"        = "\\phi{}",
        "phiv"       = "\\varphi{}",
        "chi"        = "\\chi{}",
        "psi"        = "\\psi{}",
        "omega"      = "\\omega{}",
        
        "Gamma"      = "\\Gamma{}",
        "Delta"      = "\\Delta{}",
        "Theta"      = "\\Theta{}",
        "Lambda"     = "\\Lambda{}",
        "Xi"         = "\\Xi{}",
        "Pi"         = "\\Pi{}",
        "Sigma"      = "\\Sigma{}",
        "Upsilon"    = "\\Upsilon{}",
        "Phi"        = "\\Phi{}",
        "Psi"        = "\\Psi{}",
        "Omega"      = "\\Omega{}",

   "Aopf"       = "\\mathbb{A}",
   "Bopf"       = "\\mathbb{B}",
   "Copf"       = "\\mathbb{C}",
   "Dopf"       = "\\mathbb{D}",
   "Eopf"       = "\\mathbb{E}",
   "Fopf"       = "\\mathbb{F}",
   "Gopf"       = "\\mathbb{G}",
   "Hopf"       = "\\mathbb{H}",
   "Iopf"       = "\\mathbb{I}",
   "Jopf"       = "\\mathbb{J}",
   "Kopf"       = "\\mathbb{K}",
   "Lopf"       = "\\mathbb{L}",
   "Mopf"       = "\\mathbb{M}",
   "Nopf"       = "\\mathbb{N}",
   "Oopf"       = "\\mathbb{O}",
   "Popf"       = "\\mathbb{P}",
   "Qopf"       = "\\mathbb{Q}",
   "Ropf"       = "\\mathbb{R}",
   "Sopf"       = "\\mathbb{S}",
   "Topf"       = "\\mathbb{T}",
   "Uopf"       = "\\mathbb{U}",
   "Vopf"       = "\\mathbb{V}",
   "Wopf"       = "\\mathbb{W}",
   "Xopf"       = "\\mathbb{X}",
   "Yopf"       = "\\mathbb{Y}",
   "Zopf"       = "\\mathbb{Z}",
        
   "bullet"     = "\\bullet{}",
   "xvee"       = "\\bigvee{}",
   "supe"       = "\\supseteq{}",
   "times"      = "\\times{}",
   "UnderBar"   = "_",
   "RightCeiling" = "\\rceil{}",
   "Equal"      = "=",
   "in"         = "\\in{}",
   "xodot"      = "\\bigodot{}",
   "prime"      = "\\prime{}",
   "darr"       = "\\downarrow{}",
   "bprime"     = "`",
   "colon"      = ":",
   "NIL"        = "\\text{Nil}",
   "nsub"       = "\\not\\subset{}",
   "forall"     = "\\forall{}",
   "ApplyFunction" = "",
   "lsquo"      = "\\grq{}",
   "bottom"     = "\\perp{}",
   "geq"        = "\\geq{}",
   "Sqrt"       = "\\surd{}",
   "plusmn"     = "\\pm{}",
   "hearts"     = "\\heartsuit{}",
   "rdquor"     = "\\text{''}",
   "equiv"      = "\\equiv{}",
   "lcub"       = "\\{",
   "infin"      = "\\infty{}",
   "Sum"        = "\\sum{}",
   "uArr"       = "\\Uparrow{}",
   "Therefore"  = "\\therefore{}",
   "CapitalDifferentialD" = "\\mathrm{D}",
   "isin"       = "\\in{}",
   "DD"      = "\\mathrm{D}",
   "Intersection" = "\\bigcap{}",
   "rang"       = "\\rangle{}",
   "CloseCurlyQuote" = "\\text{\\rq}",
   "notin"      = "\\not\\in{}",
   "lceil"      = "\\lceil{}",
   "And"        = "\\bigwedge{}",
   "lsqb"       = "{}[",
   "oplus"      = "\\oplus{}",
   "DoubleLeftArrow" = "\\Leftarrow{}",
   "varpi"      = "\\varpi{}",
   "rfloor"     = "\\rfloor{}",
   "Exists"     = "\\exists{}",
   "SmallCircle" = "\\circ{}",
   "rbrace"     = "\\}",
   "lozenge"    = "\\diamond{}",
   "bigsqcup"   = "\\bigsqcup{}",
   "angle"      = "\\angle{}",
   "rpar"       = ")",
   "DoubleDownArrow" = "\\Downarrow{}",
   "rbrack"     = "{}]",
   "varsigma"   = "\\varsigma{}",
   "divide"     = "/",
   "lsquor"     = "\\glq{}",
   "subset"     = "\\subset{}",
   "quest"      = "?",
   "percnt"     = "\\%{}",
   "Backslash"  = "\\backslash{}",
   "deg"        = "^{\\circ}",
   "sharp"      = "\\#{}",
   "ouml"       = "\\text{\\\"o}",
   "there4"     = "\\therefore{}",
   "raquo"      = "\\text{\\frqq}",
   "SubsetEqual" = "\\subseteq{}",
   "iff"        = "\\Leftrightarrow{}",
   "diam"       = "\\diamond{}",
   "OpenCurlyDoubleQuote" = "``",
   "yen"        = "\\text{\\textyen{}}",
   "supset"     = "\\supset{}",
   "Space"      = "\\text{\\space{}}",
   "isinv"      = "\\in{}",
   "nabla"      = "\\nabla{}",
   "semi"       = ";",
   "true"       = "\\text{true}",
   "UpTee"      = "\\perp{}",
   "sdot"       = "\\cdot{}",
   "rArr"       = "\\Rightarrow{}",
   "approx"     = "\\approx{}",
   "cap"        = "\\cap{}",
   "uarr"       = "\\uparrow{}",
   "div"        = "/",
   "cent"       = "\\text{\\textcent}",
   "straightphi" = "\\phi{}",     // <---------------- gibts nicht  
   "ldquo"      = "``",
   "varphi"     = "\\varphi{}",
   "DoubleRightArrow" = "\\Rightarrow{}",
   "CenterDot"  = "\\cdot{}",
   "int"        = "\\int{}",
   "cir"        = "\\circ{}",
   "Dot"        = "\\ddot{}",
   "Implies"    = "\\Rightarrow{}",
   "not"        = "\\neg{}",
   "bsol"       = "\\backslash{}",
   "vnsub"      = "\\not\\subset{}",
   "Im"         = "\\Im{}",
   "biguplus"   = "\\biguplus{}",
   "verbar"     = "|",
   "rangle"     = "\\rangle{}",
   "radic"      = "\\surd{}",
   "vee"        = "\\vee{}",
   "cup"        = "\\cup{}",
   "NULL"       = "\\text{Null}",
   "LeftRightArrow" = "\\leftrightarrow{}",
   "vert"       = "|",
   "lowbar"     = "_",
   "leftarrow"  = "\\leftarrow{}",
   "perp"       = "\\perp{}",
   "ldquor"     = "\\text{\\glqq}",
   "spades"     = "\\spadesuit{}",
   "xsqcup"     = "\\bigsqcup{}",
   "rightarrow" = "\\rightarrow{}",
   "ForAll"     = "\\forall{}",
   "Auml"       = "\\text{\\\"A}",
   "DownArrow"  = "\\downarrow{}",
   "nsubset"    = "\\not\\subset{}",
   "amp"        = "\\&{}",
   "subseteq"   = "\\subseteq{}",
   "cong"       = "\\cong{}",
   "loz"        = "\\diamond{}",
   "dollar"     = "\\${}",
   "NotANumber" = "\\text{NaN}",
   "lang"       = "\\langle{}",
   "bigwedge"   = "\\bigwedge{}",
   "heartsuit"  = "\\heartsuit{}",
   "Leftrightarrow" = "\\Leftrightarrow{}",
   "iexcl"      = "\\text{\\textexclamdown}",
   "excl"       = "!",
   "vprop"      = "\\propto{}",
   "Downarrow"  = "\\Downarrow{}",
   "supseteq"   = "\\supseteq{}",
   "emptyv"     = "\\emptyset{}",
   "lt"         = "<",
   "uml"        = "\\ddot{}",
   "Uuml"       = "\\text{\\\"U}",
   "xcap"       = "\\bigcap{}",
   "lfloor"     = "\\lfloor{}",
   "ohm"        = "\\Omega{}",
   "sup1"       = "^{1}",
   "lbrace"     = "\\{",
   "varepsilon" = "\\varepsilon{}",
   "lpar"       = "(",
   "lbrack"     = "{}[",
   "pm"         = "\\pm{}",
   "Union"      = "\\bigcup{}",
   "TildeTilde" = "\\approx{}",
   "Upsi"       = "\\Upsilon{}",
   "ContourIntegral" = "\\oint{}",
   "minus"      = "-",
   "Congruent"  = "\\equiv{}",
   "xcup"       = "\\bigcup{}",
   "xoplus"     = "\\bigoplus{}",
   "ExponentialE" = "e",
   "laquo"      = "\\text{\\flqq}",
   "hyphen"     = "\\text{-}",
   "SupersetEqual" = "\\supseteq{}",
   "NotEqual"   = "\\not=",
   "LeftAngleBracket" = "\\langle{}",
   "Wedge"      = "\\bigwedge{}",
   "Dagger"     = "\\ddagger{}",
   "sim"        = "\\sim{}",
   "PartialD"   = "\\partial{}",
   "ne"         = "\\not=",
   "lArr"       = "\\Leftarrow{}",
   "RightAngleBracket" = "\\rangle{}",
   "exist"      = "\\exists{}",
   "diamond"    = "\\diamond{}",
   "reg"        = "\\text{\\textregistered}",
   "horbar"     = "\\text{---}",
   "Hat"        = "\\hat{}",
   "DoubleDot"  = "\\ddot{}",
   "InvisibleComma" = "",
   "sub"        = "\\subset{}",
   "varnothing" = "\\emptyset{}",
   "InvisibleTimes" = "",
   "NonBreakingSpace" = "~",
   "sum"        = "\\sum{}",
   "therefore"  = "\\therefore{}",
   "NotSubset"  = "\\not\\subset{}",
   "sup"        = "\\supset{}",
   "apos"       = "'",
   "Colon"      = "::",
   "period"     = ".",
   "sup3"       = "^{3}",
   "brvbar"     = "\\brokenvert{}",
   "UNKNOWN"    = "\\text{Unknown}",
   "le"         = "\\leq{}",
   "langle"     = "\\langle{}",
   "false"      = "\\text{false}",
   "OpenCurlyQuote" = "\\lq{}",
   "varpropto"  = "\\propto{}",
   "part"       = "\\partial{}",
   "setmn"      = "\\backslash{}",
   "CircleDot"  = "\\bigodot{}",
   "and"        = "\\wedge{}",
   "FAIL"       = "\\text{Fail}",
   "ang"        = "\\angle{}",
   "equals"     = "=",
   "ThickSpace" = "\\text{\\space{}}",
   "bigvee"     = "\\bigvee{}",
   "copy"       = "\\copyright{}",
   "quot"       = "'",
   "DoubleLeftRightArrow" = "\\Leftrightarrow{}",
   "hellip"     = "\\dots{}",
   "gt"         = ">",
   "larr"       = "\\leftarrow{}",
   "backprime"  = "`",
   "DifferentialD" = "d",
   "Integral"   = "\\int{}",
   "szlig"      = "\\text{\\ss}",
   "image"      = "\\Im{}",
   "oint"       = "\\oint{}",
   "Subset"     = "\\subset{}",
   "epsi"       = "\\epsilon{}",
   "NotElement" = "\\not\\in{}",
   "spadesuit"  = "\\spadesuit{}",
   "hArr"       = "\\Leftrightarrow{}",
   "ImaginaryI" = "i",
   "otimes"     = "\\otimes{}",
   "Ouml"       = "\\text{\\\"O}",
   "Coproduct"  = "\\coprod{}",
   "trade"      = "\\text{\\texttrademark}",
   "compfn"     = "\\circ{}",
   "CloseCurlyDoubleQuote" = "\\grqq{}",
   "MediumSpace" = "\\text{\\space{}}",
   "Del"        = "\\bigtriangledown{}",
   "iquest"     = "\\text{\\textquestiondown}",
   "dd"         = "d",
   "micro"      = "\\mu{}",
   "blank"      = "\\text{\\space{}}",
   "vartheta"   = "\\vartheta{}",
   "ldsh"       = "\\swarrow{}",     // <-  gibts nicht , sollte Return sein
   "CirclePlus" = "\\bigoplus{}",
   "xotime"     = "\\bigotimes{}",
   "conint"     = "\\oint{}",
   "ii"         = "i",
   "bigodot"    = "\\bigodot{}",
   "Element"    = "\\in{}",
   "centerdot"  = "\\cdot{}",
   "rsquo"      = "\\grq{}",
   "rcub"       = "\\}",
   "lowast"     = "\\ast{}",
   "num"        = "\\#{}",
   "para"       = "\\P{}",
   "bull"       = "\\bullet{}",
   "frac12"     = "\\frac{1}{2}",
   "frac14"     = "\\frac{1}{4}",
   "frac34"     = "\\frac{3}{4}",
   "UpArrow"    = "\\uparrow{}",
   "leq"        = "\\leq{}",
   "LeftFloor"  = "\\lfloor{}",
   "bigcap"     = "\\bigcap{}",
   "prod"       = "\\prod{}",
   "CircleTimes" = "\\bigotimes{}",
   "Verbar"     = "\\|",
   "LeftArrow"  = "\\leftarrow{}",
   "Re"         = "\\Re{}",
   "ge"         = "\\geq{}",
   "bigcup"     = "\\bigcup{}",
   "rceil"      = "\\rceil{}",
   "plus"       = "+",
   "Vee"        = "\\bigvee{}",
   "RightFloor" = "\\rfloor{}",
   "harr"       = "\\leftrightarrow{}",
   "bot"        = "\\perp{}",
   "auml"       = "\\text{\\\"a}",
   "permil"     = "\\permil{}",
   "rsqb"       = "{}]",
   "RightArrow" = "\\rightarrow{}",
   "coprod"     = "\\coprod{}",
   "TildeFullEqual" = "\\cong{}",
   "Vert"       = "\\|",
   "pound"      = "\\pounds{}",
   "Leftarrow"  = "\\Leftarrow{}",
   "UnionPlus"  = "\\biguplus{}",
   "wp"         = "\\wp{}",
   "leftrightarrow" = "\\leftrightarrow{}",
   "GreaterEqual" = "\\geq{}",
   "Uparrow"    = "\\Uparrow{}",
   "middot"     = "\\cdot{}",
   "rsquor"     = "\\grq{}",
   "PlusMinus"  = "\\pm{}",
   "Rightarrow" = "\\Rightarrow{}",
   "dArr"       = "\\Downarrow{}",
   "bigoplus"   = "\\bigoplus{}",
   "amalg"      = "\\coprod{}",
   "comma"      = ",",
   "uuml"       = "\\text{\\\"u}",
   "xwedge"     = "\\bigwedge{}",
   "Tab"        = "\\qquad{}",
   "ast"        = "\\ast{}",
   "aleph"      = "\\aleph{}",
   "ap"         = "\\approx{}",
   "ee"         = "e",
   "upsi"       = "\\upsilon{}",
   "real"       = "\\Re{}",
   "setminus"   = "\\backslash{}",
   "rdquo"      = "\\grqq{}",
   "DoubleUpArrow" = "\\Uparrow{}",
   "Tilde"      = "\\tilde{}",
   "weierp"     = "\\wp{}",
   "diamondsuit" = "\\diamondsuit{}",
   "wedge"      = "\\wedge{}",
   "dagger"     = "\\dagger{}",
   "cdot"       = "\\cdot{}",
   "bigotimes"  = "\\bigotimes{}",
   "Superset"   = "\\supset{}",
   "SquareUnion" = "\\bigsqcup{}",
   "sube"       = "\\subseteq{}",
   "xuplus"     = "\\biguplus{}",
   "LeftCeiling" = "\\lceil{}",
   "sup2"       = "^{2}",
   "commat"     = "@",     // AT-Symbol
   "half"       = "\\frac{1}{2}",

   "D"          = "\\mathrm{D}",
        "E"          = "\\mathrm{e}",
        "I"          = "\\mathrm{i}",
        "hbar"       = "\\hbar{}"):

/*--
generate::tex -- return TeX-formatted string for expression

generate::tex(e,p)

e - expression
p - priority of enclosing expression
--*/

generate::tex:=
proc(e, p)
  local s;
begin
  if contains(generate::TeXdefs, e) then
    generate::TeXdefs[e]
  else
    slot(domtype(e), "TeX")(args());
    if % = FAIL then
      s := DOM_IDENT::TeX(args());
      if s[1] = "`" then
      s := s[2..-2];
      end_if;
      s
    else
      %
    end_if
  end_if;
end_proc:

/*--
generate::TeXquote -- quote TeX-string

generate::TeXquote(s,ep,sp)

s - TeX string
ep - priority of enclosing expression
sp - priority of this expression
--*/

generate::TeXquote:= (s,ep,sp)->
    if ep<sp then s else "\\left(".s."\\right)" end_if;

/*--
generate::TeXseq -- create TeX string for sequence

generate::TeXseq(lb,rb,e,...)

lb, rb - strings with left and right brackets
e      - expression
--*/

generate::TeXseq:= proc(lb,rb,e) 
  local i;
begin 
  _concat(
    lb,
    (case args(0)
     of 2 do ""; break;
     of 3 do generate::tex(e, 0); break;
     otherwise
        generate::tex(e, OP::Exprseq),
        ((",", generate::tex(args(i), OP::Exprseq)) $ i=4..args(0));
     end_case),
    rb) 
end_proc:

/*--
generate::TeXoperator -- TeX formatting of operator

generate::TeXoperator(sym, ep, tp, oP,...)

sym - operator symbol
ep  - priority of enclosing expression
tp  - priority of this expression
oP  - operands
--*/

generate::TeXoperator:= proc(sym, ep, tp, oP) 
  local i;
begin 
    case args(0)
    of 3 do ""; break;
    of 4 do generate::tex(oP, ep); break;
    otherwise
        generate::TeXquote(
            _concat(generate::tex(oP, tp),
                  (sym, generate::tex(args(i), tp))
                     $ i=5..args(0)),
                         ep, tp)
    end_case
end_proc:

/*--
generate::TeXfun -- TeX formatting function calls

generate::TeXfun(o,p,e,...)

o - TeX string of operator
p - priority of enclosing expression
e - operand expressions
--*/

generate::TeXfun:= proc(o,p,e) 
  local i;
begin
  // the \! is necessary because $f\left(x\right)$ is
  // equivalent to $f\mathinner{\left(x\right)}, which breaks
  // the spacing by exactly one thin space, unone by \!.
  o.generate::TeXseq("\\!\\left(", "\\right)", args(i) $ i=3..args(0))
end_proc ;

/* uncomment the following procedure if you prefer f(x) to be
  formatted as 'f x' instead of 'f(x)'... */
/*
generate::TeXfun:= proc(o,p,e) local s, i; begin
    case args(0)
    of 2 do
        o."\\left(\\right)";
        break;
    of 3 do
        s:= generate::tex(e, OP::Fconcat);
        if substring(s,1,7) = "\\left(" then
            o.s
        else
            generate::TeXquote(o." ".s, p, OP::Fconcat)
        end_if;
        break;
    otherwise
        o.generate::TeXseq("\\left(", "\\right)", args(i) $ i=3..args(0))
    end_case
end_proc:
*/

// ------------- TeX formatting of domain elements ------------- 

DOM_INT::TeX:= x -> if x < 0 and args(2) > OP::Plus then
                      "\\left(- ".expr2text(-x)."\\right)";
                    else
                      expr2text(x)
                    end:
DOM_RAT::TeX:=
  (x, prio) -> if sign(x) = 1 then generate::TeXquote("\\frac{".numer(x)."}{".denom(x)."}",
                                   prio, OP::Mult)
               else generate::TeXquote("-".
                                       generate::whitespaceBeforeFractions("\\,"," ").
                                       "\\frac{".numer(-x)."}{".denom(x)."}",
                                   prio, OP::Mult)
               end_if;
DOM_BOOL::TeX:= x -> "\\text{".expr2text(x)."}";
DOM_FAIL::TeX:= DOM_BOOL::TeX;
DOM_NIL::TeX:= DOM_BOOL::TeX;
DOM_NULL::TeX:= "";
DOM_SET::TeX:=  x -> if x = {} then "\\emptyset"
                   else
                     generate::TeXseq("\\left\\{", "\\right\\}",
                                      op(DOM_SET::sort(x)));
                   end_if:

/* Fuer eine ggf. gekuerzte Ausgabe das hier aktivieren:
DOM_SET::TeX :=
proc(x)
  local l;
begin
  l := DOM_SET::sort(x);
  if nops(x) < 10 then
    generate::TeXseq("\\left\\{", "\\right\\}", op(l))
  else
    generate::TeXseq("\\left\\{", ",...", op(l, 1..3)) .
    generate::TeXseq(",", "\\right\\}", op(l, nops(l)-2..nops(l)));
  end_if
end:
*/

DOM_LIST::TeX:= x -> generate::TeXseq("\\left[", "\\right]", op(x));
DOM_POLY::TeX:=
proc(p : DOM_POLY)
  local x, R, e, L, t, S, s, sg, i;
begin
  x := op(p,2);
  R := op(p,3); 
      // we don't need the expression, we use the list-representation instead.
  L := poly2list(p);
  S := "";
  for t in L do
    if bool(sign(op(t,1))=-1) = TRUE then
      s := "-";
      sg := -1;
    else
      s := _if(S="","","+");
      sg := 1;
    end_if;
    e := op(t,2);
    if domtype(e) <> DOM_LIST then e := [e] end_if;
    
    if expr(sg*op(t,1)) <> 1 then
      sg := sg*op(t,1);
      s := s.generate::tex(expr(sg), OP::Mult);
    elif {op(e)} = {0} then
      s := s."1";
    end_if;
    
    for i from 1 to nops(x) do
      if e[i] <> 0 then
      if s <> "" then
        s := s.generate::ifTeXmacs("\\"."*", "\\,");
      end_if;
      s := s.generate::TeX(op(x,i)^e[i]);
      end_if;
    end_for;
    S := S.s;
  end_for;
  if S = "" then S := "0"; end_if;
  S := "\\text{poly}\\left(".S.", ".generate::tex(x, 0);
  if op(R, 0) = hold(IntMod) then
    S := S.", \\mathrm{IntMod}(".expr2text(op(R)).")";
  elif R <> hold(Expr) then
    S := S.", ".generate::tex(R, OP::Exprseq);
  end_if;
//      if (op(R,0)) = IntMod then
//        S := S."\\in".generate::TeX(Z_)."_{".expr2text(op(R,1))."}".expr2texx);
//      elif R = Expr then
//        S := S."\\in\\mathbb{E}".expr2text(x);
//      elif R::TeXrep <> FAIL then
//        S := S."\\in".R::TeXrep().expr2text(x);
//      else
//        S := S.",".expr2text(x).",".generate::TeX(R);
//      end_if;
  return(S."\\right)");
end_proc:

DOM_COMPLEX::TeX:=
  proc(x, y)
    local imaginary, Sign, op2, res;
  begin
    if x = I then
      return(generate::TeXdefs[I]);
    end_if;
    if op(x,2) < 0 then
      op2 := -op(x, 2);  Sign := TRUE;
    else
      op2 := op(x, 2);  Sign := FALSE;
    end_if;
    if op2 = 1 then
      imaginary := generate::TeXdefs[I];
    elif domtype(op2) = DOM_RAT then
      if op(op2, 1) = 1 then
        imaginary := "\\frac{".generate::TeXdefs[I]."}{".op(op2, 2)."}";
      else
        imaginary := "\\frac{".op(op2, 1).generate::ifTeXmacs("\\"."*", "\\, ").generate::TeXdefs[I]."}{".op(op2, 2)."}";
      end_if
    else
      imaginary := generate::tex(op2, OP::Noop).generate::ifTeXmacs("\\"."*", "\\, ").generate::TeXdefs[I];
    end_if;

    if iszero(op(x, 1)) then
      if Sign = TRUE then
        imaginary := "-".imaginary
      else
        imaginary
      end_if;
    else
      if Sign = TRUE then
        res := generate::tex(op(x, 1), OP::Noop)." - ".imaginary;
      else
        res := generate::tex(op(x, 1), OP::Noop)." + ".imaginary;
      end_if;
      generate::TeXquote(res, y, OP::Plus)
    end_if;
  end_proc:

DOM_TABLE::TeXentry :=
proc(x)
begin
  if type(op(x, 1)) = "_exprseq" then
    generate::TeXseq("", "", op(x, 1))
  else
    generate::tex(op(x, 1), OP::Relation)
  end_if . " & " .
  if type(op(x, 2)) = "_exprseq" then
    generate::TeXseq("", "", op(x, 2))
  else
    generate::tex(op(x, 2), OP::Relation)
  end_if . "\\\\\n"
end_proc:
  
DOM_TABLE::TeX := proc(x) 
                 begin
                   _concat("\\begin{array}{r|l}\n\\hline\n",
                           op(map([op(x)], DOM_TABLE::TeXentry)),
                           "\\end{array}")
                 end_proc:
  
DOM_ARRAY::TeXentry :=
proc(e)
  local t;
begin
  t := expr2text(e);
  if type(e) = "_index" and
     length(t) > 6 and
     t[1..6] = "array(" then
    "?".generate::TeXseq("\\left[", "\\right]", op(e, 2..nops(e)));
  else
    generate::tex(e, OP::Exprseq);
  end_if;
end_proc:
DOM_ARRAY::TeXentry() := "\\mathrm{Null}":

DOM_ARRAY::TeX:= proc(a)
                   local n, m, p, i, j, str;
                 begin
                   if op(a,[0,1]) = 1 then
                     n:= op(a,[0,2,1]);
                     m:= op(a,[0,2,2]);
                     _concat("\\left(\\begin{array}{", ("c" $ (m - n + 1)), "}\n",
                             DOM_ARRAY::TeXentry(a[n]),
                             ((" & ",
                               DOM_ARRAY::TeXentry(a[i])) $ i=(n+1)..m),
                             "\n\\end{array}\\right)")
                   elif op(a,[0,1]) = 2 then
                     n:= op(a,[0,3,1]);
                     m:= op(a,[0,3,2]);
       if n = 0 or m = 0 then
         str := "\\left(\\begin{array}{}\n\\end{array}\\right)"
       else
         str :=
         _concat("\\left(\\begin{array}{", ("c" $ (m - n + 1)), "}\n",
           ((DOM_ARRAY::TeXentry(a[j,n]),
                               ((" & ", DOM_ARRAY::TeXentry(a[j,i]))
                                $ i=(n+1)..m), "\\\\\n")
            $ j=op(a,[0,2])));
         if str[-3..-1] = "\\\\\n" then
           str := str[1..-4]."\n";
         end_if;
         str."\\end{array}\\right)";
       end_if;
                   else
                     p:= proc(x)
                           local i;
                         begin
                           if nops(x) = op(a,[0,1]) then
                             "(".generate::tex(op(x),OP::Relation).") & = & ".
                             generate::tex(a[op(x)],OP::Relation)."\\\\\n"
                           else
                             _concat(p(x.[i]) $ i=op(a,[0,nops(x)+2]))
                           end_if
                         end_proc:
                     "\\text{array}\\left(\\begin{eqnarray*}\n".p([])."\\end{eqnarray*}\\right)"
                   end_if
                 end_proc:

DOM_HFARRAY::TeX:= proc(a)
                   local n, m, i, j, str;
                 begin
                   if op(a,[0,1]) = 1 then
                     n:= op(a,[0,2,1]);
                     m:= op(a,[0,2,2]);
                     _concat("\\left(\\begin{array}{", ("c" $ (m - n + 1)), "}\n",
                             DOM_ARRAY::TeXentry(a[n]),
                             ((" & ",
                               DOM_ARRAY::TeXentry(a[i])) $ i=(n+1)..m),
                             "\n\\end{array}\\right)")
                   elif op(a,[0,1]) = 2 then
                     n:= op(a,[0,3,1]);
                     m:= op(a,[0,3,2]);
                     str :=
                     _concat("\\left(\\begin{array}{", ("c" $ (m - n + 1)), "}\n",
                     ((DOM_ARRAY::TeXentry(a[j,n]),
                     ((" & ", DOM_ARRAY::TeXentry(a[j,i]))
                     $ i=(n+1)..m), "\\\\\n")
                     $ j=op(a,[0,2])));
                     if str[-3..-1] = "\\\\\n" then
                       str := str[1..-4]."\n";
                     end_if;
                     str."\\end{array}\\right)";
                   else
                     "\\text{hfarray}\\left(".op(map([op(a, 0)][2..-1], generate::TeX)).", ".generate::TeX([op(a)])."\\right)"
                   end_if
                 end_proc:

DOM_FLOAT::TeX:= proc(x, y)
                   local p;
                 begin
                   
                   if x = RD_INF then
                     return("\\tilde{\\infty}");
                   elif x = RD_NINF then
                     return("-\\tilde{\\infty}");
                   elif x = RD_NAN then
                     return("RD\\_NAN");
                   end_if;
                   
                   // p removes trailing zeros
                   p:= proc(s)
                         local m;
                       begin
                         if strmatch(s, "\\.") then
                           s:= text2list(s, ["."]);
                           m:= s[3];
                           while m[-1] = "0" do
                             if length(m) = 1 then break end_if;
                             m:= m[1..-2]
                           end_while;
                           s[1].".".m
                         else
                           s
                         end_if
                       end_proc:
    
                   x:= expr2text(x);
                   if strmatch(x, "e") then
                     x:= text2list(x, ["e"]);
                     x[1]:= p(x[1]);
                     generate::TeXquote(_concat(x[1],"\\cdot 10^",
                                                (if length(x[3]) = 1 then
                                                   x[3]
                                                 else
                                                   "{".x[3]."}"
                                                 end_if)),
                                        y, OP::Mult)
                   else
                     p(x)
                   end_if
                 end_proc:

DOM_STRING::TeX:= proc(x)
                  begin
		    x := stringlib::subs_regex(x,
					 "\\\\" = "\\ensuremath\\backslash ",
					 "{" = "\\{",
					 "}" = "\\}",
					 "\\^" = "\\^{}",
					 "_" = "\\_",
					 "#" = "\\#",
					 "\\$" = "\\$",
					 "&" = "\\&");
		       "\\text{\"".x."\"}";
                  end_proc:

DOM_IDENT::TeX:=
proc(s)
  local recurse, res;
begin
  s := expr2text(s);
  if s[1] = "`" then
    s := s[2..-2];
  end_if;

  recurse :=
  proc(s : DOM_STRING)
    local st, i, j, res, push, pop, popIdent, item,
          nextToken, token, token2, symb, lastWasSimpleChar;
  begin
    lastWasSimpleChar := FALSE;
    st := adt::Stack();
    push := st::push;
    pop := proc()
             local p;
           begin
             p := st::pop();
             if p = FAIL then
               ""
             else
               p
             end_if;
           end_proc;
    popIdent := proc(wantBraces = FALSE)
                  local p;
                begin
                  p := pop();
                  if length(p)>1 then 
                    if strmatch(p, "[{}^_]") then
                      if wantBraces then
                        p := "{".p."}";
                      end_if;
                    else
                      p := "\\mathrm{".p."}"
                    end_if;
                  end_if;
                  p;
                end_proc;
    nextToken :=
    proc(s : DOM_STRING) : DOM_LIST
      local i, open;
    begin
      if s = "" then
        error("no more tokens");
        return("")
      end_if;
      if s[1] = "{" then
        open  := 0;
        for i from 1 to length(s) do
          if s[i] = "{" then
            open := open + 1;
          elif s[i] = "}" then
            open := open - 1;
            if open = 0 then
              return([i, s[2..i-1]])
            end_if;
          end_if;
        end_for;
        error("unmatched parentheses");
      else
        [1, s[1]]
      end_if;
    end_proc;
    
    // first we split the string
    i := 1;
    while i <= length(s) do
      if s[i] = "{" then
        token := nextToken(s[i..-1]);
        push(recurse(token[2]));
        i := i + token[1];
        next;
      end_if;
      if s[i] = "\\" then
        push("\\backslash{}");
        i := i+1;
        next;
      end_if;
      if s[i] = "\"" then
        push("\\\"{}");
        i := i+1;
        next;
      end_if;
      if s[i] = "#" then
        push("\\#{}");
        i := i+1;
        next;
      end_if;
      if s[i] = "~" then
        push("\\mathasciitilde{}");
        i := i+1;
        next;
      end_if;
      if s[i] = "&" then
        j := strmatch(s[i..-1], ";", Index)[1] + i - 1;
        if contains(generate::TeXaccents, s[i+1..j-1]) then
          push("\\".generate::TeXaccents[s[i+1..j-1]]."{".popIdent()."}");
        elif s[i..j] = "&UnderLine;" then
          push("\\_")
        elif contains(generate::TeXidents, s[i+1..j-1]) then
          push(generate::TeXidents[s[i+1..j-1]]);
        else
          push(stringlib::subs(s[i..j], "&"="\\&"))
        end_if;
        i := j+1;
        next;
      end_if;
      if contains({"_", "^", "$"}, s[i]) then
        lastWasSimpleChar := FALSE;
        symb := s[i];
        token := nextToken(s[i+1..-1]);
        i := i + 1 + token[1];
        token := token[2];
        if symb = "_" then
          push(popIdent(TRUE)."_{".recurse(token)."}");
          if i <= length(s) and s[i] in {"_", "^", "$"} then
            push("{".pop()."}");
          end_if;
        elif symb = "^" then
          push(popIdent(TRUE)."^{".recurse(token)."}");
          if i <= length(s) and s[i] in {"_", "^", "$"} then
            push("{".pop()."}");
          end_if;
        elif symb = "$" then
          token2 := nextToken(s[i..-1]);
          i := i + token2[1];
          push(popIdent(TRUE)."_{".recurse(token)."}^{".recurse(token2[2])."}");
          if i <= length(s) and s[i] in {"_", "^", "$"} then
            push("{".pop()."}");
          end_if;
        end_if;
        
        next;
      end_if;
      
      // unprocessed character, put it on the stack
      if lastWasSimpleChar then
        st::push(st::pop().s[i]);
      else
        st::push(s[i]);
      end_if;
      lastWasSimpleChar := TRUE;
      i := i+1;
    end_while;
    
    res := "";
    item := popIdent();
    while item <> "" do
      res := item.res;
      item := popIdent();
    end_while;
    res;
  end_proc:
      
  if Content::isOperator(hold(``).s) or strmatch(s, "^DOM_") or
     traperror((res := recurse(s))) <> 0 then
    // some syntax error during processing, print ident in literal form
    res := stringlib::subs_regex(s,
					"\\\\" = "\\ensuremath\\backslash ",
					"{" = "\\{",
					"}" = "\\}",
					"\\^" = "\\^{}",
					"#" = "\\#",
					"_" = "\\_",
					"\\$" = "\\$");
		if length(s) > 1 then
		  res := "\\mathrm{".res."}";
		end_if;
  end_if;
  res;
end_proc;



DOM_EXPR::TeX:= proc(x, y)
                  local f, t;
                begin
                  f:= op(x,0);
                  if domtype(f) = DOM_IDENT then
                    f := eval(f);
                    if domtype(f) = DOM_FUNC_ENV then
                      if f::TeX <> FAIL then
                        return(f::TeX(f, x, y))
                      elif op(f, 2) <> NIL and
                           domtype(op(f, 2)) <> DOM_EXEC then
                        t := op(f, 2)(x);
                        if domtype(t) = DOM_EXPR then
                          x := t;
                        end_if;
                      end_if;
                    end_if;
                    generate::TeXfun(generate::tex(op(x,0), OP::Fconcat), y, op(x))
                  else
                    generate::TeXfun(generate::tex(f, OP::Fconcat), y, op(x))
                  end_if
                end_proc:
                  
DOM_DOMAIN::TeX:= proc(x, y)
                  begin
                    if x::TeXrep <> FAIL then
                      x::TeXrep();
                    elif x::Name <> FAIL then
                      if domtype(x::Name) = DOM_STRING then
                        generate::tex(text2expr(x::Name), y)
                      else
                        generate::tex(x::Name, y)
                      end_if
                    elif x::key <> FAIL then
                      if domtype(x::key) = DOM_STRING then
                        generate::tex(text2expr(x::key), y)
                      else
                        generate::tex(x::key, y)
                      end_if
                    else
                      generate::tex(hold(newDomain)(), y)
                    end_if
                  end_proc:

DOM_PROC::TeX :=
proc(p, prio)
  local l, i, nam;
begin
  if contains({op(p, 3)}, hold(arrow)) then
    if nops(op(p, 1)) <> 1 then
      l := generate::TeXseq("(", ")", op(p,1));
    elif op(p, 1) = NIL then
      l := "()";
    else // one var
      l := generate::tex((op(p, 1)), 0);
    end_if;
    l . "\\rightarrow " . // MuPAD does not use a proper \mapsto in its own output
    generate::tex(subs(op(p, 4), 
                       DOM_VAR(0, i+1)=op(p, [1, i]) $ 
                       i=1..nops(op(p, 1)), 
                       Unsimplified), 10);
  elif op(p, 6) <> NIL then
    nam := op(p, 6);
    if contains(generate::TeXdefs, nam) then
      return(generate::TeXdefs[nam]);
    else
      return(generate::tex(nam, prio));
    end_if;
  else
    stringlib::subs(DOM_PROC::print(p), "``"="", " "="\\ ");
  end_if;
end_proc:
  
DOM_FUNC_ENV::TeX :=
proc(f, prio)
  local arr, nam;
begin
  if domtype(op(f,1)) = DOM_EXEC then
    return("\\mathrm{".expr2text(f)."}");
  end_if;
  arr := has([op(f, [1,3])], hold(arrow));
  nam := op(f, [1,6]);
  if not arr and
     nam <> NIL then
    if contains(generate::TeXdefs, nam) then
      return(generate::TeXdefs[nam]);
    else
      DOM_IDENT::TeX(nam);
    end_if;
  else
    if f::print <> FAIL then
      return(generate::tex(f::print(f), prio));
    end_if;
    DOM_PROC::TeX(op(f, 1), prio);
  end_if;
end_proc:
  
// ------------------- TeX formatting of func-env's ------------------- 
/*--
Each TeX-formatting routine must have the form
        myformatfunc(fenv, expr, prio)
where fenv is the func-env of this routine, expr is the original expression
causing this call (the operator of expr was evaluated to fenv) and prio is
the priority of the enclosing expression.
--*/

_mult::TeX :=
proc(arg1, arg2, arg3)
  local list, str, prio, denominator, numerator, paren;
begin
  paren := FALSE;
  list := generate::splitProduct(arg2);
  if op(list,1) = -1 then
    prio := OP::Noop;
  else
    prio := arg3;
  end_if;

  if nops(op(list, 2)) = 1 then
    numerator := generate::tex(op(op(list,2)), OP::Noop);
  else
    paren := TRUE;
    numerator := generate::TeXoperator(generate::ifTeXmacs("\\"."*", "\\, "),
                                 prio, OP::Mult, op(op(list,2)));
  end_if;

  if nops( op(list, 3)) <> 0 then
    if nops(op(list, 3)) = 1 then
      denominator := generate::tex(op(op(list,3)), OP::Noop);
    else
      denominator :=
         generate::TeXoperator(generate::ifTeXmacs("\\"."*", "\\, "),
                               prio, OP::Mult, op(op(list,3)));
    end_if;
    str := _concat("\\frac{", numerator, "}{", denominator, "}");
    if op(list,1) = -1 then
      str := "-".generate::whitespaceBeforeFractions("\\,","").str
    end_if;
  else
    str := numerator;
    if op(list,1) = -1 then
      paren := TRUE;
      str := generate::TeXquote( "- ".str , arg3, OP::Plus);
    end_if;
  end_if;

  if not paren then
    str := generate::TeXquote( str , arg3, OP::Mult);
  end_if;
  str;
end_proc:

_divide::TeX := 
proc( arg1, arg2, arg3 )
  local str;
begin
  str := _concat( "\\frac{".
                 generate::TeXoperator(generate::ifTeXmacs("\\*", "\\, "),
                                       arg3, OP::Mult, op(arg2,1)).
               "}{".
                 generate::TeXoperator(generate::ifTeXmacs("\\*", "\\, "),
                                       arg3, OP::Mult, op(arg2,2)).
               "}");
  generate::TeXquote(str , arg3, OP::Mult);
end_proc:

_negate::TeX :=
proc( arg1, arg2, arg3 )
  local str;
begin
  str := generate::tex(op(arg2,1), OP::Plus);
  generate::TeXquote( "- ".str , arg3, OP::Plus); 
end_proc:

_subtract::TeX :=
proc( arg1, arg2, arg3 )
  local str;
begin
  str := _concat( 
               generate::tex(op(arg2, 1), OP::Plus).
               " - ".
               generate::tex(op(arg2, 2), OP::Plus) 
               );
  generate::TeXquote( str , arg3, OP::Plus);
end_proc:

_invert::TeX :=
proc( arg1, arg2, arg3 )
  local str;
begin
  str := _concat( "\\frac{1}{".
               generate::TeXoperator(" ",arg3, OP::Mult, op(arg2,1)).
               "}");
end_proc:

_plus::TeX :=
proc(arg1, arg2, arg3)
  local tmp, s, i, o, res;
begin
  case nops(arg2)
    of 0 do return("");
        of 1 do return(generate::tex(op(arg2,1), arg3));
  end_case;
  tmp:= generate::sortSum(arg2);
             
  s:= "";   /* generate::tex(op(tmp, 1), OP::Plus); */
  for i from 1 to nops(tmp) do
    o:= op(tmp, i);
    if generate::isNeg(o) then
      res:= generate::tex(- o, OP::Plus);
      if length(res) > 5 and
         res[1..5] = "\\frac" then
          s:= s." -". generate::whitespaceBeforeFractions("\\,"," ").res; 
      else
          s:= s." - ".res; 
      end_if;
      next;
    else
      if i > 1 then
        s:= s." + ";
      end_if;
      s := s.generate::tex(o, OP::Plus)
    end_if;
  end_for;
             
  generate::TeXquote(s, arg3, OP::Plus)
end_proc:

sqrt::TeX := (fn, x, prio) -> "\\sqrt{".generate::tex(op(x,1), OP::Noop)."}":

_power::TeX :=
proc(arg1, arg2, arg3)
  local s, base;
begin
   if nops(arg2) <> 2 then
     return (generate::TeXfun("".arg1, OP::Noop, op(arg2)));
   end_if;
   if testtype(op(arg2,2), Type::Real) then
    if op(arg2,2) = -1 then
      return("\\frac{1}{".generate::tex(op(arg2,1), OP::Noop)."}")
    end_if;
    if op(arg2,2) < 0 then
      return("\\frac{1}{".
             _power::TeX(_power,
                         hold(_power)(op(arg2,1),(-op(arg2, 2))), OP::Noop)."}")
    end_if;
    if op(arg2,2) = 1/2 then
      return("\\sqrt{".generate::tex(op(arg2,1), OP::Noop)."}")
    end_if;
  end_if;
  s:= generate::tex(op(arg2, 2), OP::Noop);
  if length(s) > 1 then
    s := "{".s."}";
  end_if;
  base := generate::tex(op(arg2, 1), OP::Power);
  if length(base)>2 then
    base := "{".base."}";
  end_if;
  generate::TeXquote(base."^".s, arg3, OP::Power)
end_proc:

diff::TeX :=
proc(arg1, arg2, arg3)
  local res, l, t, x, i;
begin
  l:= [ op(arg2, 2..nops(arg2)) ];
  t:= table();
  for x in l do
    if contains(t, x) then
      t[x]:= t[x] + 1
    else
      t[x]:= 1
    end_if
  end_for;

  t:= _concat(("\\partial ", generate::TeX(op(t,[i,1])^op(t,[i,2])))
              $ i=1..nops(t));

  if nops(l) = 1 then
    res := "\\frac{\\partial"
  else
    res := "\\frac{\\partial^".generate::TeX(nops(l))
  end_if;
  generate::TeXquote(res."}{".t."} ".
                     generate::tex(op(arg2,1), OP::Prime),
                     arg3, OP::Prime)
end_proc:

D::TeX :=
  proc(arg1, arg2, arg3)
  begin
    generate::TeXquote(if nops(arg2) = 1 then
                         generate::tex(op(arg2,1), OP::Prime-1)."'"
                       else
                         "\\mathrm{D}_{".
                         generate::tex(op(op(arg2,1)), OP::Noop)."} ".
                         generate::tex(op(arg2,2), OP::Prime)
                       end_if, arg3, OP::Prime);
  end_proc:

int::TeX :=
  proc(arg1, arg2, arg3)
    local intop;
  begin
    intop := "\\int";
    if contains([op(arg2)], PrincipalValue) > 0 then
      intop := "c.v.\\int"; // TODO: Better sign
    end_if;
    generate::TeXquote(
                       if type(op(arg2,2)) = "_equal" then
                         _concat(intop, "_{",
                                 generate::tex(op(arg2, [2,2,1]), OP::Noop),
                                 "}^{",
                                 generate::tex(op(arg2, [2,2,2]), OP::Noop),
                                 "} ",
                                 generate::tex(op(arg2, 1), OP::Noop),
                                 " \\,\\mathrm{d} ",
                                 generate::tex(op(arg2, [2,1]), OP::Fconcat),
                                 generate::ifTeXmacs("\\bignone", ""))
                       else
                         _concat(intop, " ",
                                 generate::tex(op(arg2, 1), OP::Noop),
                                 " \\,\\mathrm{d} ",
                                 generate::tex(op(arg2, 2), OP::Fconcat),
                                 generate::ifTeXmacs("\\bignone", ""))
                       end_if, arg3, OP::Fconcat);
  end_proc:
         
sum::TeX :=
  proc(arg1, arg2, arg3)
  begin
    generate::TeXquote(
                       if type(op(arg2, 2)) = "_equal" then
                         _concat("\\sum_{",
                                 generate::tex(op(arg2,
                                                  [2,1])=op(arg2, [2,2,1]),
                                               OP::Noop),
                                 "}^{",
                                 generate::tex(op(arg2, [2,2,2]), OP::Noop),
                                 "} ",
                                 generate::tex(op(arg2, 1), OP::Plus),
                                 generate::ifTeXmacs("\\bignone", ""))
                       else
                         _concat("\\sum_{",
                                 generate::tex(op(arg2, 2), OP::Noop), "} ",
                                 generate::tex(op(arg2, 1), OP::Plus),
                                 generate::ifTeXmacs("\\bignone", ""))
                       end_if, arg3, OP::Plus);
  end_proc:
  
product::TeX :=
  proc(arg1, arg2, arg3)
  begin
    generate::TeXquote(
                       if type(op(arg2, 2)) = "_equal" then
                         _concat("\\prod_{",
                                 generate::tex(op(arg2,
                                                  [2,1])=op(arg2, [2,2,1]),
                                               OP::Noop),
                                 "}^{",
                                 generate::tex(op(arg2, [2,2,2]), OP::Noop),
                                 "} ",
                                 generate::tex(op(arg2, 1), OP::Mult),
                                 generate::ifTeXmacs("\\bignone", ""))
                       else
                         _concat("\\prod_{",
                                 generate::tex(op(arg2, 2), OP::Noop), "} ",
                                 generate::tex(op(arg2, 1), OP::Mult),
                                 generate::ifTeXmacs("\\bignone", ""))
                       end_if, arg3, OP::Mult);
  end_proc:

_index::TeX :=
  proc(arg1, arg2, arg3)
    local s, i;
  begin
    s:= generate::TeXseq("", "", op(arg2,i) $ i=2..nops(arg2));
    if length(s) <> 1 then
      s:= "{".s."}"
    end_if;
    generate::TeXquote(generate::tex(op(arg2, 1), OP::Index)."_".s,
                       arg3, OP::Index)
  end_proc:
  
conjugate::TeX :=
  (arg1, arg2, arg3) ->
    "\\overline{".generate::tex(op(arg2), OP::Noop)."}":
  
abs::TeX :=
  (arg1, arg2, arg3) ->
    "\\left|".generate::tex(op(arg2), OP::Noop)."\\right|":

_seqgen::TeX :=
  (arg1, arg2, arg3) ->
  generate::TeXquote(generate::tex(op(arg2, 1), OP::Seqgen)."\\$ ".
                     generate::tex(op(arg2, 2), OP::Seqgen).
                     _if(nops(arg2)=2,"","=".generate::tex(op(arg2,3), OP::Seqgen)),
                     arg3, OP::Seqgen):
  
_seqin::TeX :=
  (arg1, arg2, arg3) ->
  generate::TeXquote(generate::tex(op(arg2, 1), OP::Seqgen)."\\$ ".
                     generate::tex(op(arg2, 2), OP::Seqgen)." in ".
                     generate::tex(op(arg2,3), OP::Seqgen),
                     arg3, OP::Seqgen):
  
_not::TeX :=
  (arg1, arg2, arg3) ->
    generate::TeXquote("\\neg ".generate::tex(op(arg2, 1), OP::Not),
                       arg3, OP::Not):

_stmtseq::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator("; ", arg3,
                                              OP::Stmtseq, op(arg2)):
_exprseq::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator(", ", arg3,
                                              OP::Exprseq, op(arg2)):
_or::TeX        :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\vee ", arg3, OP::Or,
                                               op(arg2)):
_and::TeX       :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\wedge ", arg3, OP::And,
                                              op(arg2)):
  
_xor::TeX       :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\mathbin{\\text{xor}} ", arg3, OP::Xor,
                                              op(arg2)):
  
_less::TeX      :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" < ", arg3, OP::Relation,
                                              op(arg2)):
_leequal::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\leq ", arg3, OP::Relation,
                                              op(arg2)):
_equal::TeX     :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" = ", arg3, OP::Relation,
                                              op(arg2)):
_unequal::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\neq ", arg3, OP::Relation,
                                              op(arg2)):
_range::TeX     :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" .. ", arg3, OP::Range,
                                               op(arg2)):
_div::TeX       :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\mathbin{\\text{div}} ", arg3, OP::Mod,
                                              op(arg2)):
_union::TeX     :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\cup ", arg3, OP::Union,
                                              op(arg2)):
_minus::TeX     :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\setminus ", arg3, OP::Minus,
                                              op(arg2)):
_intersect::TeX :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\cap ", arg3, OP::Intersect,
                                              op(arg2)):
_fconcat::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" \\circ ", arg3, OP::Fconcat,
                                              op(arg2)):
_fnest::TeX   :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" {\\circ\\circ} ", arg3, OP::Fnest,
                                              op(arg2)):
_concat::TeX    :=
  (arg1, arg2, arg3) -> generate::TeXoperator(" . ", arg3, OP::Concat,
                                               op(arg2)):
slot::TeX     :=
  (arg1, arg2, arg3) -> if nops(arg2) = 2 and domtype(op(arg2,2)) = DOM_STRING and 
                                                stringlib::validIdent(op(arg2,2)) then
        generate::TeXoperator("{::}", arg3, OP::Slot, op(arg2,1),hold(``).op(arg2,2)):
    else
        generate::TeXfun(generate::tex(op(arg2,0)), arg3, op(arg2))
 
    end_if:

unalias(OP):

// end of file 
