/*-
 new - generates an undirected or directed Graph.

 new(Vertices, Edges <, Directed> <, VertexWeights=[v1,..,vn]>) <, EdgeDescriptions=[d1,..,dm]> <, EdgeWeights=[t1,..,tm]> <, EdgeCosts=[c1,..,cm]>)

 Parameters:
 -----------
   mandatory:
   ----------
   Vertices         - list of one or more vertices 
   Edges            - list of lists of one or more edges
   
   optional:
   ---------
   Directed         - if set, the graph will be directed. Default is undirected
   VertexWeights    - list of maximal weights used for one or more vertices.
   EdgeDescriptions - list of descriptions for one ore more edges.
   EdgeWeights      - list of weights for one or more edges.
   EdgeCosts        - list of costs for one or more edges.   

 ===========================================================================================================
 As long as no vertex is set with an explicit weight, the call VertexWeights will return FAIL (to make some algorithms faster)
 As long as no edge is set with an explicit weight, the call edgeWeight will return FAIL (to make some algorithms faster)
 As long as no edge is set with an explicit cost, the call EdgeCosts will return FAIL (to make some algorithms faster)
 As long as no edge is set with an explicit description, the call EdgeCosts will return FAIL (to make some algorithms faster)
-*/
Graph::new :=
proc(Vertices : DOM_LIST, Edges  : Type::ListOf(DOM_LIST))
  local _edges, _edgesEntering, _edgesLeaving, leftOperator,
	rightOperator, _vertexWeights, _edgeWeights, _edgeCosts,
	_edgeDescriptions, _directed, _argument, _arguments,
	i, j, tmpList, tbl2l, fn;
  begin
    // initialize the Lists
    _vertexWeights := FAIL; 

    _edgeWeights := FAIL;
    _edgeCosts := FAIL;
    _edgeDescriptions := FAIL;

    // Default is not directed
    _directed := FALSE;

    _arguments := [args()];
    
    while ((i := contains(_arguments, hold(Directed)))) <> 0 do
      _directed := TRUE;
      delete _arguments[i];
    end_while;
    while ((i := contains(_arguments, hold(Undirected)))) <> 0 do
      delete _arguments[i];
    end_while;
    
    // Duplicate edges
    tmpList:= [op({op(Edges)})];
    if nops(Edges) <> nops(tmpList) and nops(tmpList) > 0 then
      error("Duplicate edges were given.");     
    end_if;
    
    // Duplicate edges if graph is undirected. 
    // Sort edges with sysorder and then check again for duplicates. (Normal check was already done !)
    if not _directed then
      tmpList := Edges;
      for i from 1 to nops(tmpList) do
        tmpList[i] := sort(tmpList[i]);
      end_for;
      tmpList:= [op({op(tmpList)})];
      if nops(Edges) <> nops(tmpList) and nops(tmpList) > 0 then
        error("Duplicate edges were given.");
      end_if;
    end_if;
    
    // Duplicate vertices
    tmpList:= [op({op(Vertices)})];
    if nops(Vertices) <> nops(tmpList) then
      error("Duplicate vertices were defined.");      
    end_if;
      
    tmpList := Graph::checkForVertices(Edges, Vertices);
    if not tmpList = [] then
      error("One or more edges contain vertices that are not in list: " . expr2text(tmpList) . "\n" );
    end_if;
    
    // first two parameters (Vertices and Edges) are mandatory, whereas all the rest is optional
    for i from 3 to nops(_arguments) do
      _argument := _arguments[i];
      if not (type(_argument) = "_equal") then
        error("Wrong type of ".expr2text(i).". argument: equation expected!");
      end_if;
      leftOperator := op(_argument,1); 
      rightOperator := op(_argument,2);

      if domtype(rightOperator) <> DOM_LIST then
        error("Wrong argument type for " . expr2text(leftOperator) . ": list expected!");
      end_if;

      case leftOperator
        of hold(VertexWeights) do
          if nops(rightOperator) <> nops(Vertices) then
            error("Number of vertex weights differs from number of vertices");
            //rightOperator := rightOperator.[0 $ j=1..(nops(Vertices)-nops(rightOperator))];
          end_if;
          if {op(map(rightOperator, x->bool(x=hold(None) or x=infinity or x=-infinity or domtype(float(x))=DOM_FLOAT)))} <> {TRUE} then
            error("Vertex weights must be real or +/-infinity");                
          end_if;
          _vertexWeights := table(Vertices[j] = rightOperator[j] $ j=1..nops(Vertices));
          //_vertexWeights := _vC;
          for j in _vertexWeights do
             if op(j,2) = hold(None) then
               delete _vertexWeights[op(j,1)];
             end_if;
          end_for;
          break;
        of hold(EdgeWeights) do
          if nops(rightOperator) <> nops(Edges) then
            error("Number of edge weights differs from number of edges");
            //rightOperator := rightOperator.[0 $ (nops(Edges)-nops(rightOperator))];
          end_if;
          _edgeWeights := table(Edges[j] = rightOperator[j] $ j=1..nops(Edges));
          // remove redundant elements
          _edgeWeights := select(_edgeWeights, x->not op(x,2)=hold(None));

          // If graph is undirected also add the reversed edges to it.
          if _directed = FALSE then        
            for j in _edgeWeights do
              _edgeWeights[revert(op(j,1))] := op(j,2);
            end_for;
          end_if;
          break;
        of hold(EdgeCosts) do
          if nops(rightOperator) <> nops(Edges) then
            error("Number of edge costs differs from number of edges");
            //rightOperator := rightOperator.[0 $ (nops(Edges)-nops(rightOperator))];
          end_if;
          _edgeCosts := table(Edges[j] = rightOperator[j] $ j=1..nops(Edges));
          _edgeCosts := select(_edgeCosts, x-> not op(x,2)=hold(None));
          //_edgeCosts := select(_edgeCosts, () -> not bool( rhs(args() ) = None) );
          // if graph is undirected also add reverted edges to it.
          if _directed = FALSE then        
            for j in _edgeCosts do
              _edgeCosts[revert(op(j,1))] := op(j,2);
            end_for;
          end_if;
          if nops(_edgeCosts) = 0 then
            _edgeCosts := FAIL;
          end_if;
          break;
        of hold(EdgeDescriptions) do
          if nops(rightOperator) <> nops(Edges) then
            error("Number of edge descriptions differs from number of edges");
          end_if;
          _edgeDescriptions := table(Edges[j] = rightOperator[j] $ j=1..nops(Edges));
          _edgeDescriptions := select(_edgeDescriptions, x-> not op(x,2)=hold(None));
          // if graph is undirected also add reverted edges to it.
          if _directed = FALSE then        
            for j in _edgeDescriptions do
              _edgeDescriptions[revert(op(j,1))] := op(j,2);
            end_for;
          end_if;
          if nops(_edgeDescriptions) = 0 then
            _edgeDescriptions := FAIL;
          end_if;
          break;        
        otherwise
          error("Illegal Argument: " . expr2text(leftOperator) . ". Please consult help for more information.");
      end_case;
    end_for;

   if _directed then
     _edges := Edges;
   else
     _edges := Edges . map(Edges, revert);
  end_if;

   _edges := listlib::removeDuplicates(_edges, KeepOrder);
    
   /*-
    Generate the adjacency lists
   -*/
    _edgesEntering := table((Vertices[i]=table()) $ i = 1..nops(Vertices));
    _edgesLeaving := _edgesEntering;
    for i in _edges do
      _edgesEntering[op(i,2)][op(i,1)] := TRUE;
      _edgesLeaving[op(i,1)][op(i,2)] := TRUE;
    end_for;
    tbl2l := t -> sort(map([op(t)], op, 1));
    _edgesEntering := map(_edgesEntering,
			  tbl2l);
    _edgesLeaving := map(_edgesLeaving,
			 tbl2l);

   // edges are of form [[a, b], [c, d], ... ] (for example _edge = ([1,2], [1,3])
   // _edges... are of form [a=[], b=[], ... ] (for example _edgesEntering = (1=[] 2=[1] 3=[])
    
    Vertices := sort(Vertices);
    fn := (x, y) -> (if op(x,1) = op(y,1) then
		       sysorder(op(x,2), op(y,2));
		     else
		       sysorder(op(x,1), op(y,1));
		     end_if);
    _edges := sort(_edges, fn);

   new(Graph, Vertices, _edges, _vertexWeights, _edgeDescriptions, _edgeWeights,
                  _edgeCosts, _edgesEntering, _edgesLeaving, _directed);
end_proc:

// End of file
null():

