/*
 Graph::stronglyConnectedComponents

  Decompose a Graph into strongly connected components.

  Input: A graph G

  Output: A list of graphs, which are the strongly connected
          components of G.  

  Algorithm: Chapter 23.5 of "Introduction to Algorithms", Cormen, Leiserson, Rivest.
  This proc exists only to have a file-scope static variable DFS
  which stores a helper function
*/

Graph::stronglyConnectedComponents := proc(G : Graph)
  //option escape;
  local DFS, dSCC;

begin
  // depth-first-search returning
  // the discovery and finish times and the respective predecessors
  DFS :=  proc(G : Graph, backwards = FALSE : DOM_BOOL)
    local DFS_visit, vertices, u, d, flow, pre, now, get_children, components;
  begin

    if backwards then
      get_children := Graph::getEdgesEntering(G);
    else
      get_children := Graph::getEdgesLeaving(G);
    end_if;
    
    DFS_visit := proc(u, vertices_in_component)
      local v;
    begin
      now := now + 1;
      d[u] := now;
      vertices_in_component := vertices_in_component.[u];
      for v in get_children[u] do
        if iszero(d[v]) then
          pre[v] := u;
          vertices_in_component := DFS_visit(v, vertices_in_component);
        end_if;
      end_for;
      now := now + 1;
      flow[u] := now;
      vertices_in_component;
    end_proc;
    
    if args(0) > 2 then
      vertices := args(3);
    else
      vertices := Graph::getVertices(G);
    end_if;
    d := table((u=0)$u in vertices);
    flow := table((u=0)$u in vertices);
    pre := table((u=FAIL)$u in vertices);
    now := 1;
    components := [];
    for u in vertices do
      if iszero(d[u]) then // vertex not visited yet
        components := components.[DFS_visit(u, [])];
      end_if;
    end_for;
    [d, flow, pre, components];
  end_proc;

  dSCC := proc(G : Graph)
    local fu, components;
  begin
    // call DFS(G) to compute finishing times flow[u] for each vertex u
    fu := DFS(G, FALSE)[2];
    
    // call DFS(Nt), but in the main loop of DFS, consider the vertices
    // in order of decreasing fu (as computed above)
    fu := sort([op(fu)], (a, b) -> rhs(a) > rhs(b));
    fu := map(fu, lhs);
    components := DFS(G, TRUE, fu)[4];

    // discard one-element components
    // components := select(components, c -> nops(c) > 1);

    // output the vertices of each tree in the depth-first forest
    // of the preceding step as a strongly connected component
    map(components, c -> Graph::getSubGraph(G, c));
  end_proc;

  dSCC(G);

end_proc:

// End of file
null():
