/*
isEmpty - checks if the graph has no vertices and edges
isEmpty(G)
Parameters:
  G - Graph
*/
Graph::isEmpty := proc(G : Graph)
begin
  if Graph::getEdgeNumber(G) = 0 and Graph::getVertexNumber(G) = 0 then
  	 TRUE;
  else
    FALSE;
  end_if;
end_proc:

/* createMinSort - creates a List that is sorted either with Costs or Weights in ascending order.
Graph::createMinSort := proc(T : DOM_TABLE)
Parameters:
 Weights|Costs - extracts 
*/
Graph::createMinSort := proc(T : DOM_TABLE)
begin
  // Return a sorted list of edges
  // Example:
  // table([a,d]=4,[a,b]=3,[a,c]=2,[a,a]=1)
  // returns
  // [[a, a], [a, c], [a, b], [a, d]]
  //sort([op(T)], () -> bool(rhs(args(1)) < rhs(args(2))));
  map(sort([op(T)], () -> bool(rhs(args(1)) < rhs(args(2)))), () -> lhs(args(1)));
end_proc:

/*-
 getVertices - returns a list with all vertices of a Graph
 getVertices(G)
 Parameters:
   G - Graph
-*/
Graph::getVertices := proc(G : Graph)
begin
   extop(G,1);
end_proc:

/*-
 getEdges - returns a list with all edges of a Graph
 getEdges(G)
 Parameters:
   G - Graph
-*/
Graph::getEdges := proc(G : Graph)
begin
   extop(G,2);
end_proc:

/*-
 getVertexWeights - returns a list with all vertice weights of a Graph
 getVertexWeights(G)
 Parameters:
   G - Graph
-*/
Graph::getVertexWeights := proc(G : Graph) 
begin
   extop(G,3);
end_proc:

/*-
 getEdgeDescriptions - returns a list with all the descriptions of the edges of a Graph
 getEdgeDescriptions(G)
 Parameters:
   G - Graph
-*/
Graph::getEdgeDescriptions := proc(G : Graph)
begin
   extop(G,4)
end_proc:

/*-
 getEdgeWeights - returns a list with all edge weights of a Graph
 getEdgeWeights(G)
 Parameters:
   G - Graph
-*/
Graph::getEdgeWeights := proc(G : Graph)
begin
   extop(G,5);
end_proc:

/*-
 getEdgeCosts - returns a list with the edge costs of a Graph
 getEdgeCosts(G)
 Parameters:
   G - Graph
-*/
Graph::getEdgeCosts := proc(G : Graph)
begin
   extop(G,6)
end_proc:

/*-
 getEdgesEntering - returns the adjacency list for incoming edges of a Graph
 getEdgesEntering(G)
 Parameters:
   G - Graph
-*/
Graph::getEdgesEntering := proc(G : Graph)
begin
   extop(G,7)
end_proc:

/*-
 getEdgesLeaving - returns the adjacency list for outgoing edges of a Graph
 getEdgesLeaving(G)
 Parameters:
   G - Graph
-*/
Graph::getEdgesLeaving := proc(G : Graph)
begin
   extop(G,8)
end_proc:

/*-
 isDirected - returns TRUE if the Graph is directed, FALSE otherwise
 isDirected(G)
 Parameters:
   G - Graph
-*/
Graph::isDirected := proc(G : Graph)
begin
   extop(G,9)
end_proc:

/*-
 isVertex - checks whether a vertex is contained in a Graph

 isVertex(G, Vertex)
 Parameters:
   G      - Graph
   Vertex - a list of one or more vertices
-*/
Graph::isVertex := proc(G : Graph, Vertex : DOM_LIST)
local _vertices, i;
begin
  if args(0) > 2 then
    error("Wrong number of arguments! See help for more details."):
  end_if;

  _vertices := Graph::getVertices(G);

  for i in Vertex do
    if not bool(contains(_vertices, i) <> 0) then
      return (FALSE);
    end_if;  	
  end_for;
  TRUE;
end_proc:

/*-
 isEdge - checks whether an edge is contained in a Graph
 Because of the new data structure the check is done in O(1).
 If the graph is directed just check if 
 Table[x,y] is not NIL (if there is only one edge allowed between two vertices)
 If the graph is undirected check:
 if y < x switch x and y and check if Table[x,y] is not NIL.

 isEdge(G,_edge)
 Parameters:
   G    - Graph
   Edge - a list of lists of one or more edges
   
-*/
Graph::isEdge := proc(G : Graph, Edge : Type::ListOf(DOM_LIST))
local i, _edges;
begin
   /*- Because of the data-structure (undirected graphs are handled like directed graphs) if the graph is undirected the
       edge (a, b) has to be checked and the edge (b, a), too.
   -*/
  if args(0) > 2 then
    error("Wrong number of arguments! See help for more details."):
  end_if;

   _edges := Graph::getEdges(G);

   if Graph::isDirected(G) then
      // directed graph, only check the edge given.
      for i in Edge do
        if not bool(contains(_edges, i) <> 0) then
        	 return (FALSE);
        end_if;      	
      end_for;
   else
      // undirected graph, check the edge (a, b) and the edge (b, a)
     for i in Edge do
       if not bool(contains(_edges, i) <> 0 and contains(_edges, revert(i)) <> 0) then
     	   return (FALSE);
       end_if;
     end_for;
   end_if;

   TRUE;
end_proc:

/*-
 graphCheck - checks if G has the type Graph
 Parameters:
   G - Graph
-*/
Graph::graphCheck := proc(G)
begin
  if not testtype(G, Graph) then
// ccr: Try
// context(hold(error)("Wrong argument type: Graph expected!"))
// and find out the difference.
    error("Wrong argument type: Graph expected!");
  end_if;
end_proc:

/*
 returns the amount of vertices in G.
*/
Graph::getVertexNumber := proc(G : Graph)
begin
  nops( Graph::getVertices(G) );
end_proc:

/*
  returns the amount of edges in G. (Remember to divide the number by 2 if the graph is undirected !)
*/
Graph::getEdgeNumber := proc(G : Graph)
begin
  nops( Graph::getEdges(G) );
end_proc:

/*
  getAdjacentEdgesEntering(Vertex)
  Returns the vertices to which the edges lead from the given vertex
  Parameters:
  G      - Graph
  Vertex - The vertex which entering edges should be returned
*/
Graph::getAdjacentEdgesEntering := proc(G : Graph, Vertex: DOM_LIST)
local _edgesEntering;
begin
  if nops(Vertex) > 1 then
  	 error("Only one specified vertex allowed.");
  end_if;
  _edgesEntering := Graph::getEdgesEntering(G);
  _edgesEntering[op(Vertex)];
end_proc:

/*
  getAdjacentEdgesLeaving(Vertex)
  Parameters:
  G      - Graph
  Vertex - The vertex which leaving edges should be returned
*/
Graph::getAdjacentEdgesLeaving := proc(G : Graph, Vertex: DOM_LIST)
local _edgesLeaving;
begin
  if nops(Vertex) > 1 then
  	 error("Only one specified vertex allowed.");
  end_if;
  _edgesLeaving := Graph::getEdgesLeaving(G);
  _edgesLeaving[op(Vertex)];
end_proc:

// End of file
null():

