// 

// interval::matlinsolve(A, b)
// returns a matrix X of the same type as A such that
// b \subset A*X.
// Uses interval Gauss-Seidel elimination
// TODO: Pivoting
// TODO: Fix return type when resorting to preconditioning
// TODO: Post-iteration using Gauss-Seidel?

// This is a modified copy of numeric::matlinsolve

interval::matlinsolve :=
proc(A_in,b_in)
local A, b, C, saveDIGITS, returnType, rb, result_with_precond,
      m, neqs,n,method,coefftypes,use_normal,i,j,tmp,B,cb,check,
      Amap, bmap, Aop, L, list, showassumptions, PivotAssumptions,dowarn,
      x,y,X,X0,Xvars,nTerms,koeff,np2,np2mj,p,kdim,Kernel,
      piv,pivot,pivindex,constraints, doPreconditioning,
      lowerbands,jpl,minlength, count, result, triv, kernindex;
save DIGITS;
begin
  if args(0)<2 then error("expecting at least two arguments"); end_if;
  if args(0)>7 then error("expecting no more than seven arguments"); end_if;
  
  A := A_in;
  b := b_in;
  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  if domtype(A) = DOM_ARRAY then
    if op(A, [0, 1]) <> 2 then
      error("first argument: expecting a 2-dimensional array");
    end_if;
    [neqs, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
    returnType:= DOM_ARRAY;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
    [neqs, n]:= A::dom::matdim(A):
    returnType:= Dom::Matrix();
    if A::dom::constructor = Dom::DenseMatrix then
      returnType:= Dom::DenseMatrix();
    elif A::dom <> Dom::Matrix() then
      A:= Dom::Matrix()(A);
    end_if;
  else
    error("first argument: expecting an array or ".
          "a matrix of category 'Cat::Matrix'"):
  end_if;
  
  if domtype(b) = DOM_ARRAY then
    case op(b, [0, 1])
      of 1 do
        [rb, cb]:= [op(b, [0, 2, 2]), 1];
        b:=array(1 .. rb,1..1,[[b[i]]$ i=1..rb]);
        break;
      of 2 do
        [rb, cb]:= [op(b, [0, 2, 2]), op(b, [0, 3, 2])];
        break;
      otherwise
        error("second argument: expecting a 1-dimensional array ".
              "(a vector) or a 2-dimensional array (a matrix)");
    end;
  elif domtype(b) = DOM_LIST then
    [rb, cb]:= [nops(b), 1];
    b:=array(1 .. rb, 1..1,[[b[i]]$ i=1..rb]);
  elif b::dom::hasProp(Cat::Matrix) = TRUE then
    [rb, cb]:= b::dom::matdim(b):
    if b::dom <> Dom::DenseMatrix() then
         b:= Dom::DenseMatrix()(b);
    end_if;
  else error("second argument: expecting a list, an array, or ".
             "a matrix of category 'Cat::Matrix'"):
  end_if;
  
  if neqs <> rb  then
    error("dimensions of matrices/vectors not compatible");
  end_if;
  
  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  showassumptions:= FALSE;
  dowarn:= TRUE;
  doPreconditioning := UNKNOWN;
  
  x:=genident("x ");
  y:=genident("y ");
  
  for i from 3 to args(0) do
    case args(i)
      of Symbolic do method:=1; // symbolic method
        break;
      of ShowAssumptions do
        showassumptions:= TRUE;
        break;
      of NoWarning do
        dowarn:= FALSE;
        break;
      otherwise
        if type(args(i)) = "_equal" then
          if lhs(args(i)) = ReturnType then
            returnType:= rhs(args(i));
            if not has({DOM_ARRAY, Dom::DenseMatrix(), Dom::/*Sparse*/Matrix()}, returnType) then
              error("illegal return type ".expr2text(args(i)).
                    " specified. Choose between DOM_ARRAY, ".
                    "Dom::Matrix() or Dom::DenseMatrix()"):
            end_if;
          elif lhs(args(i)) = Preconditioning then
            if rhs(args(i)) = TRUE or rhs(args(i)) = FALSE then
              doPreconditioning := rhs(args(i));
            else
              error("illegal argument ".expr2text(rhs(i))
                    ." for Preconditioning");
            end_if;
          else error("unknown option ".expr2text(args(i)));
          end_if;
        else error("unknown option ".expr2text(args(i)));
        end_if;
    end_case;
  end_for:

  if doPreconditioning = TRUE then
    if A::dom::hasProp(Cat::Matrix)<>TRUE then
      A := /*dense*/matrix(A);
    end_if;
    if b::dom::hasProp(Cat::Matrix)<>TRUE then
      b := /*dense*/matrix(b);
    end_if;
    // precondition using an approximate midpoint inverse
    saveDIGITS := DIGITS;
    DIGITS := 3;
    C := numeric::inverse(map(A, DOM_INTERVAL::center));
    if C = FAIL then
      warning("Can't do preconditioning: singular matrix?");
    else
      A := C*A;
      b := C*b;
    end_if;
  end_if;
  
  if A::dom::hasProp(Cat::Matrix)=TRUE and
    A::dom::constructor = Dom::/*Sparse*/Matrix then
    Amap:= A::dom::mapNonZeroes;
    Aop := A::dom::nonZeroOperands;
  else
    Amap:= map;
    Aop := op;
  end_if;
  
  // convert dense matrices to domtype DOM_ARRAY for speed
  if A::dom::hasProp(Cat::Matrix)=TRUE then
    if A::dom::constructor = Dom::/*Sparse*/Matrix then
      if A::dom <> Dom::/*Sparse*/Matrix() then
        A:= Amap(A, expr):
        A:= (Dom::/*Sparse*/Matrix())::create(extop(A));
      end_if;
    else
      A:= expr(A)
    end_if;
  end_if;
  
  if b::dom::hasProp(Cat::Matrix)=TRUE and
    b::dom::constructor = Dom::/*Sparse*/Matrix then
    bmap:= b::dom::mapNonZeroes;
  else
    bmap:= map;
  end_if;
  
  // convert dense rhs to domtype DOM_ARRAY for speed
  if b::dom::hasProp(Cat::Matrix)=TRUE then
    if b::dom::constructor = Dom::/*Sparse*/Matrix then
      if b::dom <> Dom::/*Sparse*/Matrix() then
        b:= bmap(b, expr):
        b:= (Dom::/*Sparse*/Matrix())::create(extop(b));
      end_if;
    else
      b:= expr(b);
      if not contains({1,2},op(b,[0,1])) then
        error("second argument: wrong format");
      end_if;
    end_if;
  end_if;

  // force evaluation of A and b (both are arrays or Dom::/*Sparse*/Matrix now) 
  A:= Amap(A, eval):
  b:= bmap(b, eval);
  
  use_normal:=FALSE;  // default
  
  if method=2 then
    tmp:= Amap(A, interval):
    B:= bmap(b, interval):
    coefftypes := map({Aop(tmp)}, domtype);
    if coefftypes // union {op(map(B,domtype))}
      minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_INTERVAL} <> {} then
      if dowarn then
        warning("symbolic coefficient found, switching to symbolic mode");
      end_if;
      method:=1;
      B:=NIL;
    else
      A:=tmp;
      b:=B;
      B:=NIL:
    end_if;
  end_if;
  
  use_normal := FALSE;
  if method=1 then
    coefftypes := map({op(A)}, domtype);
    if coefftypes minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_RAT,DOM_INTERVAL} <> {}
      then use_normal:=TRUE
    end_if;
  end_if;
  
  userinfo(10,
           if method = 2 then
             "using numerical mode"
           elif use_normal then
             "using symbolic mode with normalization"
           else "using symbolic mode without normalization"
           end_if
           );
  
  if method=1 then
    userinfo(1,"symbolic mode: no pivoting unless necessary");
  end_if;
  if method=2 then userinfo(1,"numerical mode: using column pivoting"); end_if;

  // -- initialization done --- 
  
  kdim:=0;  // dimension of kernel
  np2:=n+2;
  lowerbands:=0:       // count sudiagonal bands and store in lowerbands:
  
  //-----------------------------------------------
  // convert A from polys = columns to polys = rows.
  // Use p as container. Store the j-th element in p[j]
  // as the coefficient of x^(n+2-j) (j=1.. n )
  //-----------------------------------------------
  m:= neqs; // dimension(A) = m x n
  // table p = container for the rows
  // Initialize as a table
  p := table();
  if A::dom::hasProp(Cat::Matrix)=TRUE and
    A::dom::constructor = Dom::/*Sparse*/Matrix then
      // In analogy to (Dom::/*Sparse*/Matrix())::transpose:
      // For sparse matrices the version using L:= poly2list(x[j])
      // is faster than the version using coeff(x[i], j).
      // For dense matrices it is the other way round.
      // Regard a matrix as 'reasonably sparse', if at most
      // half its entries are nonzero:
    
    if A::dom::nonZeroes(A) < m*n/2
      then //sparse case
      A:= extop(A, 3);
      for j from 1 to n do
        L:= poly2list(A[j]);
        for list in L do
            // fill the rows
          p[list[2]][j]:= [list[1], np2 - j];
          if list[2] - j >lowerbands then
            lowerbands:=  list[2] - j;
          end_if;
        end_for;
      end_for;
        // So far, the row p[i] is a table created implicitly by
        // p[list[2]][j]:= [list[1], np2 - j];
        // Convert the rows to polynomials.
      for i from 1 to m do
        if type(p[i]) = "_index" then
             // p[i] was not implictly created. This row is empty
          p[i] := poly(0, [x]);
        else
          // convert table p[i] to a list p[i]:
          p[i]:= map([op(p[i])], op, 2);
             // convert list p[i] to a poly p[i]:
          p[i]:= poly(p[i], [x]);
          if use_normal then p[i]:=mapcoeffs(p[i],normal);end_if;
        end_if;
      end_for;
    else //dense case
        // build up row for row to avoid doubling of memory
        // due to doubly nested sequences
      A:= extop(A, 3);
      for i from 1 to m do
        p[i] := poly([[coeff(A[j],i), np2-j] $j=1..n],[x]);
        if use_normal then p[i]:=mapcoeffs(p[i],normal);end_if;
      end_for;
      lowerbands:= m - 1;
    end_if;
  else // A is a DOM_ARRAY
    for i from 1 to neqs do // convert equations to univariate polynomials
      tmp:= [[0,np2-j]$j=1..n]:
      for j from 1 to n do  // x[j] -> x^(n+2-j),j=1.. n
        if not iszero(A[i,j]) then
          tmp[j][1]:=A[i,j]:
          if i-j>lowerbands then lowerbands:=i-j; end_if;
        end_if;
      end_for;
      p[i]:=poly(tmp,[x]):
      if use_normal then p[i]:=mapcoeffs(p[i],normal);end_if;
    end_for;
  end_if;

  //-----------------------------------------------
  // convert b from polys = columns to polys = rows.
  // Use B as container. Store the j-th element in B[j]
  // as the coefficient of y^(j-1) (j=1.. n )
  //-----------------------------------------------
  // B = container for the rows of the rhs
  B := table();
  if b::dom::hasProp(Cat::Matrix)=TRUE and
    b::dom::constructor = Dom::/*Sparse*/Matrix then
    if b::dom::nonZeroes(b) < m*cb/2
      then //sparse case
      b:= extop(b, 3);
      for j from 1 to cb do
        L:= poly2list(b[j]);
        for list in L do
            // fill the rows
          B[list[2]][j]:= [list[1], j-1];
        end_for;
      end_for;
        // So far, the row B[i] is an implicitly created table.
        // Convert the rows to polynomials.
      for i from 1 to m do
        if type(B[i]) = "_index" then
             // B[i] was not implictly created. This row is empty
          B[i] := poly(0, [y]);
        else
          // convert table B[i] to a list B[i]:
          B[i]:= map([op(B[i])], op, 2);
          B[i]:= poly(B[i], [y]);
        end_if;
      end_for;
      else //dense case
        // build up row for row to avoid doubling of memory
        // due to doubly nested sequences
        b:= extop(b, 3);
        for i from 1 to m do
          B[i] := poly([[coeff(b[j],i), j-1] $j=1..cb],[y]);
        end_for;
    end_if;
  else // b is a DOM_ARRAY
    for i from 1 to neqs do  // convert equations to univariate polynomials
      tmp:= [[b[i,j],j-1]$j=1..cb];
      B[i]:=poly(tmp,[y]):
    end_for;
  end_if;
  
  //----------------------------------------------------
  // Row compression: get rid of all trivial equations
  //----------------------------------------------------
  count:= 0;
  m:= neqs;
  for i from 1 to neqs do
    if iszero(p[i]) and iszero(B[i]) then
      m:= m - 1;
    else
      count:= count + 1;
      p[count]:= p[i];
      B[count]:= B[i];
    end_if;
  end_for;
  for i from m+1 to neqs do
    delete p[i], B[i];
  end_for;
  neqs:= m;
  
  //----------------------------------------------------
  // check = utility to check rhs of equations 0 = b:
  //----------------------------------------------------
  constraints:= {};

  check:=
  proc(b) // routine to check right hand side of equations 0=b
    local koeff, tmp;
  begin //b:=subs(b,float(0)=0);
    if iszero(b) then return(null()) end_if;
    if showassumptions=FALSE then return(FAIL); end_if;
    // Now showassumptions = TRUE and iszero(b) = FALSE.
    // Investigabe, whether b has symbolic parameters.
    // If b is numeric, then return FAIL (the system is not solvable),
    // otherwise insert contraint 0=b into constraints.
    koeff:=[coeff(b)]:     // b is a polynomial in y
    for tmp in koeff do
      if not iszero(tmp) then
        if hastype(tmp,{DOM_EXPR,DOM_IDENT,"_index"})
          then constraints:=constraints union {numer(tmp)};
        else return(FAIL):
        end_if;
      end_if;
    end_for;
    return(b);
  end_proc;
  
  //  Conversion of matrices to sparse univariate poly representation done
  
  PivotAssumptions:= {};
  triv:= table();
  kernindex:= table();
  
  //------------------------------
  // Start Gauss-Elimination
  //------------------------------
  
  //for j from 1 to n do // eliminate column j
  j:= 0;
  while j < min(neqs, n) do
    j:= j + 1;
    np2mj:=np2-j;
    jpl:=min(j+lowerbands,neqs):

   // ------------------- search for pivot element ------------------ 
   // ------- initialization: collect piv[j],..,piv[jpl] -------- 

    pivot:=0;pivindex:=j; minlength:=RD_INF;
    (piv[i]:= 0;) $ i=j..max(neqs,n):
    /* store element j from p[i] = i.th row in piv[i] and
      truncate p[i] to become the remainder of the row */
    for i from j to jpl do
      piv[i]:= coeff(p[i],np2mj);
      // Make sure, vanishing piv[i] are identified as 0:
      if use_normal and j=1 and not iszero(piv[i])
        then piv[i]:=normal(piv[i]);
        mapcoeffs(p[i], normal);
      end_if;
        /* Must cut piv[i] from p[i], even if iszero(piv[i]). Note that
          piv[i]=float(0)<>0 might happen. */
      if not iszero(piv[i]) then p[i]:=lmonomial(p[i],Rem)[2]: end_if;
    end_for;

    // ------- search for best pivot element----------------- 
    // piv[j],..,piv[jpl] are candidates for the pivot elements.
    for i from j to jpl do

      if method=1 then  /*symbolic mode: take the "simplest" nonsymbolic
                           element<>0 as pivot */
        if (not iszero(piv[i])) then
          // look for non-symbolic Pivot elements only
          if indets(piv[i],PolyExpr)={} then
            if iszero(p[i])
              then /*remaining row is trivial. This piv[i] is the ideal
                     candidate for a pivot element: accept it! */
              pivindex:=i; break;   //stop pivot search
            else /*remaining row is nontrivial. Look for rows
                     with pivot element of smallest complexity */
              length(piv[i]):
              if last(1)<minlength
                then minlength:=last(1); pivindex:=i:
              end_if;
            end_if;
          end_if;
        end_if;
      end_if;

      if method=2 then //numeric mode: do row pivoting
        if not iszero(piv[i]) then
          pivindex:=i; break;  //stop pivot search
        end_if;
      end_if;

    end_for;
    
    /* so far we tried to avoid a symbolic pivot element. If no such
      pivot element exists, we have to start again. This time we must
      accept symbolic elements. */
    if method=1 and iszero(piv[pivindex]) then
      pivot:=0; pivindex:=j; minlength:=RD_INF;
      for i from j to jpl do
        if not iszero(piv[i]) then
          if iszero(p[i]) then pivindex:=i; break;
          else length(piv[i]):
            if last(1)<minlength
              then minlength:=last(1); pivindex:=i:
            end_if;
          end_if;
        end_if;
      end_for;
    end_if;

    //------------------------------------------------
    // optimal Pivot element found, now exchange rows 
    //------------------------------------------------

    if j<>pivindex then
      userinfo(10, "pivoting: swap rows ".j." and ".pivindex);
      ([p[j],p[pivindex]]):=[p[pivindex],p[j]]:
      ([B[j],B[pivindex]]):=[B[pivindex],B[j]]:
      ([piv[j],piv[pivindex]]):=[piv[pivindex],piv[j]]:
    end_if;

/* ----- if symbolic pivot must be used, this is assumed to be <>0.
        Store this assumption in PivotAssumptions: 
--------*/

    if method=1 and showassumptions=TRUE and
      indets(piv[j],PolyExpr)<>{} then
      tmp:= numer(piv[j]);
      if indets(float(tmp),PolyExpr)<>{} then
        PivotAssumptions:= PivotAssumptions union {tmp};
      end_if;
    end_if;
    
// ----- check if non-trivial pivot element was found ------- 
    if iszero(piv[j]) then
      kdim:= kdim + 1;
      triv[j]:= TRUE;
      kernindex[j]:= kdim;
      
      // Entzerrungsschritt: fuege eine weitere Gleichung ein
      // j may have become larger than the number of
      // input equations, i.e., p[j] and B[j] have not
      // been set before:

//!!!!!!!!!!!!!!!1
      if not(iszero(p[j]) and iszero(B[j])) then
//!!!!!!!!!!!!!!!1
        neqs:=neqs+1;
        piv[neqs]:=0;
        p[neqs]:=p[j]:
        B[neqs]:=B[j];
        // the number of lower bands was increased
        lowerbands:=max(lowerbands,neqs-j-1);
        jpl:=min(j+lowerbands,neqs):
//!!!!!!!!!!!!!!!1
      end_if;
//!!!!!!!!!!!!!!!1
    else // do the elimination. Column j is stored in piv[j],..,piv[neqs]
      for i from j+1 to jpl do
        if not iszero(piv[i]) then
          tmp:=piv[i]/piv[j]:
          case tmp
            of  1 do  p[i]:=p[i]-p[j];B[i]:=B[i]-B[j];break;
            of -1 do  p[i]:=p[i]+p[j];B[i]:=B[i]+B[j];break;
            otherwise p[i]:=p[i]-mapcoeffs(p[j],_mult,tmp):
              B[i]:=B[i]-mapcoeffs(B[j],_mult,tmp):
          end_case;
          if use_normal then
            p[i]:=mapcoeffs(p[i],normal):
            B[i]:=mapcoeffs(B[i],normal):
          end_if;
        end_if;
      end_for:
    end_if;
  end_while; // while j < min(neqs, n)
  //end_for; // for j = 1..n do

  //-------------------------------------------------
  // we now have neqs equations for n unknowns, where 
  // the first n equations are upper triangular. 
  // Check that the remaining equations are trivial
  //-------------------------------------------------

  for j from n+1 to neqs do
    if check(B[j])=FAIL then
      if showassumptions=FALSE then
        return([FAIL, NIL]);
      else return([FAIL, NIL, [], []]);
      end_if;
    end_if;
  end_for;
  constraints:=[op(constraints)];

  //-----------------------------
  // Fill in trivial equations to
  // obtain a quadratic scheme:
  //-----------------------------
  for j from neqs + 1 to n do
    triv[j]:= TRUE;
    piv[j]:= 0:
    p[j]:= poly(0, [x]):
    B[j]:= poly(0, [y]):
    kdim:= kdim + 1;
    kernindex[j]:= kdim;
  end_for;

/*--------------------
  // this is a little utility needed during
  // the design of this function. It checks
  // that above no unexpected indexed calls to
  // rows occured that where not properly
  // initialized
// if nops(select(p, hastype, DOM_TABLE)) <> 0 then 
//    warning("p has DOM_TABLE"): 
// end_if;
// if nops(select(B, hastype, DOM_TABLE)) <> 0 then 
//    warning("B has DOM_TABLE"): end_if;
// if nops(select(p, hastype, DOM_TABLE)) <> 0 or
//    nops(select(B, hastype, DOM_TABLE)) <> 0 then
//   print("-------------"): 
//   print(p, B);
//   error(" p or B has DOM_TABLE"): 
// end_if;
------------------------*/

  //--------------------------------------------------------
  // Backsubstitution:
  // Now the system is in upper triangular form. The diagonal
  // is stored in piv[1],..,piv[n]
  //--------------------------------------------------------

  // convert univariate polynomials to expressions
  // for fast sparse backsubstition:  
  // coeff*x^(n+2-i) <-> coeff*x[i] 
  if triv[n] = TRUE then
    X[n]:=y^(cb-1+kernindex[n]);
  else
    X[n]:= op(B[n],1)/piv[n];
    if use_normal then X[n]:=normal(X[n]): end_if:
  end_if;
  for i from n-1 downto 1 do
    if triv[i] = TRUE then
      X[i]:=y^(cb-1+kernindex[i]);
    else // pick out only non-trivial coefficients of the poly:
      nTerms:=nterms(p[i]):
      Xvars:=[X[np2-op(nthterm(p[i],j),[1,2])]$ j=1..nTerms]:
      koeff:=[coeff(p[i])]:
      p[i]:=_plus( koeff[j]*Xvars[j] $ j=1..nTerms):
      if piv[i]=1
        then X[i]:=(op(B[i],1)-p[i]);
      else X[i]:=(op(B[i],1)-p[i])/piv[i]:
      end_if;
    end_if;
  end_for;

  //-------------------------------------------------------------
  // The backsubstituion is finished.
  // Now, X[i] = polynomial expression in y. We need to pick out the
  // various y powers. Reconvert expressions to polys in y for
  // faster access via coeff. 
  // Do not normalize X here, because poly would undo normal   
  //-------------------------------------------------------------

  for i from 1 to n do
    X[i]:=poly(X[i], [y]);
  end_for;
  //-------------------------------------------------------------
  // The polynomials X[i] store the special solution X0[i, j]
  // (i=1..n, j=1..cb) as the coefficients of y^0, .. , y^{cb -1}
  // and the kernel elements as the coefficients of
  // y^cb, .. ,y^cb + kdim -1
  /* Dense extraction:
     X0:=array(1..n,1..cb,[[coeff(X[i],j-1)$ j=1..cb]$ i=1..n]);
     Kernel:=array(1..n,1..kdim,
           [[coeff(X[i],cb-1+j)$ j=1..kdim]$ i=1..n]);
  */
  //-------------------------------------------------------------

  // sparse extraction:
  if returnType = Dom::/*Sparse*/Matrix() then
    // use a table (sparse structure!) to write into
    // when extracting the coeffs of the row polys.
    // Further down, the table is passed to
    // (Dom::/*Sparse*/Matrix())::new for conversion
    // to a sparse matrix
    X0:= table();
    if kdim > 0 then
      Kernel:= table();
    else Kernel:= 0; // array(1..n,1..1,[[0]$n])
    end_if;
  else // if arrays or Dom::Matrices are to be returned,
       // it is the cheapest to create arrays right here
       // Note that array -> Dom::DenseMatrix is extremely
       // cheap via (Dom::DenseMatrix())::create
    X0:=array(1..n,1..cb, [[0$cb]$n]);
    if kdim > 0 then
      Kernel:=array(1..n,1..kdim, [[0$kdim]$n]);
    else Kernel:= 0; // array(1..n,1..1,[[0]$n])
    end_if;
  end_if;

  // sparse extraction:
  for i from 1 to n do
    L:= poly2list(X[i]);
    for list in L do
      if list[2] < cb then
        X0[i, list[2] + 1]:= list[1];
      else Kernel[i, list[2] + 1 - cb]:= list[1];
      end_if;
    end_for;
  end_for;
  if method = 2 and kdim > 0 then
    // for kdim = 0, return 0, not float(0)!
    Kernel:= map(Kernel, float);
  end_if;
  X:= X0:

  if method = 2 then
    X:= map(X, interval);
  end_if;
   
  //----------------------------------
  // That's it. The special solution X
  // and the kernel 'Kernel' are extracted.
  // Final simplifications: 
  //----------------------------------
  if use_normal then
    X:=map(X,normal);
    Kernel:=map(Kernel,normal);
  end_if:
   
  constraints:=map(constraints,
                   proc(x)
                   begin
                     if iszero(x) then null() else x end_if
                   end_proc);

  if showassumptions=FALSE then
    result:= [X, Kernel];
  else
    constraints:=map([op(constraints)],
                        proc() begin args(1)=0 end_proc);
    PivotAssumptions:=map([op(PivotAssumptions)],
                          proc() begin args(1)<>0 end_proc);
    result:= [X, Kernel, constraints, PivotAssumptions];
  end_if;

  //-------------------------
  //------- return ----------
  //-------------------------
  case returnType
  of Dom::/*Sparse*/Matrix() do
       if result[1] <> FAIL then
          result[1]:= returnType(n, cb, result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType(n, kdim, result[2]);
       end_if;
       break;
  of Dom::DenseMatrix() do
       if result[1] <> FAIL then
          result[1]:= returnType::create(result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType::create(result[2]);
       end_if;
       break;
  of DOM_ARRAY do
       // nothing to do
       break;
  otherwise
       error("unexpected return type"):
  end_case;

  if doPreconditioning = UNKNOWN and has(result, [RD_INF, RD_NINF]) then
    userinfo(10, "Solution without preconditioning contains infinity,\ntrying with preconditioning...");
    result_with_precond := interval::matlinsolve(args(), Preconditioning=TRUE);
    result[1] := array(op(result[1], [0, 2..-1]), zip([op(result[1])], [op(result_with_precond[1])], _intersect));
    if result[2] = 0 then
      result[2] := result_with_precond[2];
    elif result_with_precond[2] <> 0 then
      case returnType
      of Dom::Matrix() do
      of Dom::DenseMatrix() do
        result[2] := zip(interval(result[2]), interval(result_with_precond[2]), _intersect);
        break;
      of DOM_ARRAY do
        result[2] := array(op(result[2], 0),
          zip(interval([op(result[2])]), interval([op(result_with_precond[2])]),
            _intersect));
        break;
      end_case;
    end_if;
    if showassumptions then
      result[3] := [op({op(result[3]), op(result_with_precond[3])})];
    end_if;
  end_if;

  return(result):
end_proc:

