// 

// interval evaluation of zeta(z)

// for Re(z) >= 1/2, we use Algorithm 2 of Borwein's 1995 paper
// "An Efficient Algorithm for the Riemann Zeta Function"
DOM_INTERVAL::_zeta_borwein_coeffs :=
proc(n, prec)
  option remember;
  local d, k;
begin
  d := [0$n+1];
  d[1] := hull(0);
  for k from 1 to n do
    d[k+1] := d[k] + hull(n * (n+k-1)!*4^k/((n-k)!*(2*k)!));
  end_for;
  d;
end_proc:
	
DOM_INTERVAL::_zeta_borwein :=
proc(iv, prec)
  local d, n, k, im;
begin
  if iv = {} then
    return({});
  end_if;
  im := Im(iv);
  
  // rough approx.
  n := ceil(1.3*prec// +
//	    max(1.7*(ln(max(abs(2^-iv))) + ln(max(abs(2^iv-2)))
//		     - ln(2*min(abs(im))+1) - 1 - min(abs(im))*1.5),
//		0)
	   );
  
  d := DOM_INTERVAL::_zeta_borwein_coeffs(n, prec);
  
  -1/(d[-1]*(1-2^(1-iv))) *
  _plus((-1)^k*(d[k+1]-d[-1])/(k+1)^iv $ k=0..n-1) +
  // remainder bound
  (if iszero(im) then
     (-1...1) * hull((3/(3+sqrt(8))^n) * 1/abs(1-2^(1-iv)))
   else
     (-1-I...1+I) * hull((3/(3+sqrt(8))^n) * ((1+2*abs(im))*exp(abs(im)*PI/2))/abs(1-2^(1-iv)))
   end_if);
end_proc:

DOM_INTERVAL::zeta :=
proc(z)
  local iv;
  save DIGITS;
begin
  iv := interval(z);
  if domtype(iv) <> DOM_INTERVAL then
    if iv = {} then return(iv); end_if;
    return(FAIL);
  end_if;
  
  if op(iv,0) = hold(_union) then
    _union(map(op(iv), DOM_INTERVAL::zeta));
  elif domtype(z) = DOM_INT and
       z > 0 and z < 200 and
       z mod 2 = 0 then
    hull(-(2*PI*I)^z*DOM_INTERVAL::_bernoulli(z, DIGITS)/(2*z!));
  elif domtype(z) = DOM_INT and z < 0 then
    if z mod 2 = 0 then
      0...0
    elif z > -200 then
      -DOM_INTERVAL::_bernoulli(1-z, DIGITS)/(1-z);
    end_if;
  elif Re(iv) >= 1/2 then
    DOM_INTERVAL::_zeta_borwein(iv, DIGITS);
  else
    // zeta(1-s) = 2*(2*PI)^(-s) * cos(s*PI/2) * gamma(s) * zeta(s)
    (s -> DOM_INTERVAL::_zeta_borwein(s, DIGITS) *
	 2*interval((2*PI)^(-s)) *
	 DOM_INTERVAL::cos(s*PI/2) *
	 DOM_INTERVAL::gamma(s))
    ((1-iv) intersect
		   (1/2+RD_NINF*I ... RD_INF+RD_INF*I))
    union
    DOM_INTERVAL::_zeta_borwein(iv intersect
				(1/2+RD_NINF*I ... RD_INF+RD_INF*I),
				DIGITS);
  end_if;
end_proc:
