// Arithmetic on Divisors
//
// A divisor is a formal combination of places on an algebraic
// curve, needed for finding the logarithmic part of an integral
// over such a curve, defined by y^n=f(x).
// A semi-reduced divisor D is represented by two polynomials h(x,y) and A(x)
// such that (cf. Trager, Integration of Algebraic Functions, 1984)
//  order(h) = order(D) at all placed over the roots of A(x)
//  order(D) = 0 at all other places
//  both h and A are multiples of D except at infinity
// The notation below follows Bertrand, Computing a Hyperelliptic Integral
// using Arithmetic in the Jacobian of the Curve, in writing divisors
// as *additive* combinations of places. Trager uses a multiplicative
// notation. This makes o mathematical difference, though.
//
// In addition to h and A, our divisors also store an expression ex,
// such that they actually encode, using the notation of Bertrand,
// [h, A] + div(ex). This makes the callers' code significantly easier.

// NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE
// currently, the implementation is only valid for hyperelliptic curves!

// The implementation for the hyperelliptic case follows
// L. Bertrand, Computing a Hyperelliptic Integral using Arithmetic
// in the Jacobian of the Curve, Section 2.2

domain intlib::algebraic::Divisor(ts, diffs, algs, modulus = 0)
  local x, f;
  inherits Dom::BaseDomain;

  modred :=
  if iszero(modulus) then
    p -> misc::maprec(p,
      {DOM_POLY} = (p -> mapcoeffs(p, c -> normal(c, Expand=TRUE, Recursive=FALSE))), // the normal is very useful when having sqrt(3) etc. in the inputs
      PostMap)
  else
    p -> misc::maprec(p,
      {DOM_POLY} = (p -> mapcoeffs(p, c -> dom::modred(normal(c, Expand=TRUE, Recursive=FALSE)))), // the normal is very useful when having sqrt(3) etc. in the inputs
      {"_power"} = (ex -> subsop(ex, 1=dom::modred(op(ex, 1)))),
      {DOM_RAT, DOM_INT} = (x -> x mod modulus),
      {DOM_COMPLEX} = (z -> (Re(z) mod modulus) + I*(Im(z) mod modulus)),
      PostMap)
  end_if;

  gcdex :=
  if iszero(modulus) then gcdex
  else
    proc(a, b)
    begin
      [a, b] := map([a, b], poly, Dom::ExpressionField(dom::modred@(x -> normal(x, Expand = TRUE))));
      map(gcdex(a, b), poly, Expr);
    end_proc;
  end_if;

  new := 
  proc(h, A)
  begin
    h := poly(h, [x]);
    A := divide(poly(A, [x]), poly(h, [x]), Rem);
    assert(A <> FAIL);
    new(dom, dom::modred(h), dom::modred(A), 1);
  end_proc;
  
  print := if iszero(modulus) then
    d -> expr(d[1..3])
  else
    d -> hold(_mod)(expr(d[1..2]), modulus)
  end_if;
  
  _negate := D -> subsop(D, 2 = -extop(D, 2), 3 = 1/(expr(extop(D, 1))*extop(D, 3)));
  
  zero := 0;
  
  _plus :=
  proc(d1, d2)
    local L, i, a, b, a1, a2, b1, b2, d, h1, h2, h12, h3, divs;
  begin
    if args(0) = 1 or d2 = 0 then
      return(d1);
    end_if;
    if d1 = 0 then
      return(d2);
    end_if;
    if args(0) > 2 then
      L := [args()];
      while nops(L) > 2 do
        L := L.[0];
        L := [L[i]+L[i+1] $ i = 1..nops(L)-1 step 2];
      end_while;
      [d1, d2] := L;
    end_if;
    if domtype(d1) <> dom then
      if domtype(d2) <> dom then
        return(d1 + d2);
      end_if;
      return(subsop(d2, 3=extop(d2, 3)+d1));
    end_if;
    if domtype(d2) <> dom then
      return(subsop(d1, 3=extop(d1, 3)+d2));
    end_if;
    // Follow Bertrand, Proposition 2.4.
    [a1, b1] := d1[1..2];
    [a2, b2] := d2[1..2];
    [d, h1, h2]  := [dom::gcdex(a1, a2)];
    [d, h12, h3] := [dom::gcdex(d, b1+b2)];
    [h1, h2] := h12*[h1, h2];
    assert(iszero(dom::modred(divide(a1*a2, d^2, Rem))));
    a := dom::modred(divide(a1*a2, d^2, Quo));
    a := dom::modred(multcoeffs(a, 1/lcoeff(a)));
    b := dom::modred(h1*a1*b2+h2*a2*b1+h3*(b1*b2+f));
    b := dom::modred(divide(b, a, Rem));
    // b := b/d mod a
    b := dom::modred(divide(b*[dom::gcdex(d, a)][2], a, Rem));

    divs := d1[3]*d2[3];
    if degree(d) > 0 then
      divs := divs*expr(d);
    end_if;
    dom::reduce(new(dom, a, b, divs));
  end_proc;
  
  intmult := domains::repeatedSquaring(dom, "_plus", "zero", "_negate");
  
  _mult := 
  proc()
    local these, other, dummy;
  begin
    [these, other, dummy] := split([args()], testtype, dom);
    other := _mult(op(other));
    if domtype(other) <> DOM_INT or nops(these) <> 1 then
      error("bad usage of divisors");
    end_if;
    dom::intmult(these[1], other);
  end_proc;
  
  multDiv := (d, e) -> subsop(d, 3=extop(d, 3)*e);
  
  reduce :=
  proc(d)
    local y, a, b, g, h;
  begin
    // Reduce using Lemma 2.6 of Bertrand
    y:= ts[-1];
    [a, b, h] := d[1..3];
    g := (degree(f)-1)/2;
    while degree(a) > g do
      if not iszero(dom::modred(divide(f-b^2, a, Rem))) then
        error("not a semi-reduced representative!");
      end_if;
      a := dom::modred(divide(f - b^2, a, Quo));
      a := mapcoeffs(a, normal, Expand=TRUE);
      a := dom::modred(multcoeffs(a, 1/lcoeff(a)));
      h := h*(expr(b)-y)/expr(a);
      b := dom::modred(divide(-b, a, Rem));
      b := mapcoeffs(b, normal, Expand=TRUE);
    end_while;
    if iszero(modulus) then
      h := normal(h, Expand=FALSE);
    else // ignore h
      h := 1;
    end_if;
    new(dom, a, b, h);
  end_proc;
  
  isPrincipal := D -> bool(expr([extop(dom::reduce(D), 1..2)]) = [1, 0]);
  
  _index := (d, i) -> if type(i) = "_range" then [extop(d, i)] else extop(d, i) end_if;
  
  // return D mod p if the reduction is good, FAIL otherwise.
  goodReduction :=
  proc(d, p)
    local a, b, h, nf, na, nb, nh, cnt, substs, nd;
  begin
    if iszero(lcoeff(f) mod p) then
      return(FAIL);
    end_if;
    
    nd := intlib::algebraic::Divisor(ts, diffs, algs, p);
    [a, b, h] := [extop(d)];
    if traperror((
        na := nd::modred(a);
        nb := nd::modred(b);
        nf := nd::modred(f);
      )) <> 0 or has([na, nb, nf], [FAIL]) or degree(nf) < degree(f)
    then return(FAIL); end_if;

    // TODO: Trager, p. 68, says we can replace a non-irreducible f with its
    // smallest factor at this place. Analyze that claim.
    if not iszero(degree(gcd(nf, nf'))) then return(FAIL); end_if;
    if not iszero(degree(gcd(na, nb))) then return(FAIL); end_if;
    
    nd(na, nb);
  end_proc;
  
  // order of D mod p, or FAIL if D mod p is not a good reduction
  orderMod :=
  proc(d, p)
    local nd, i, nd_i;
  begin
    nd := dom::goodReduction(d, p);
    if nd = FAIL then return(FAIL); end_if;
    i := 1;
    nd_i := nd::dom::reduce(nd);
    while not nd::dom::isPrincipal(nd_i) do
      // theory says this terminates.
      i := i+1;
      nd_i := nd::dom::reduce(nd_i + nd);
    end_while;
    i;
  end_proc;
  
  // compute order via good reductions. Returns a positive integer or infinity.
  // the returned value may not really be the order of D,
  // but if it is not, D is of infinite order.
  order :=
  proc(d)
    local p, orderModP, i, g, k, l;
  begin
    // find to primes p[1] and p[2] with good reductions
    p := [3,3];
    orderModP := [0,0];
    for i from 1 to 2 do
      if i = 2 then p[i] := nextprime(p[i-1]+1); end_if;
      while traperror((orderModP[i] := dom::orderMod(d, p[i]))) <> 0 or orderModP[i] = FAIL do
        p[i] := nextprime(p[i]+1);
      end_while;
    end_for;
    // ignore the factor p[i] in orderModP[i]
    for i from 1 to 2 do
      while testtype(orderModP[i], DOM_INT) do
        orderModP[i] := orderModP[i]/p[i];
      end_while;
      orderModP[i] := orderModP[i]*p[i];
    end_for;

    // if order(d)=o, then o=gcd(degModP[1],degModP[2])*p[1]^k*p[2]^l
    g := gcd(orderModP[1], orderModP[2]);
    k := log(p[1], orderModP[2]/g);
    l := log(p[2], orderModP[1]/g);
    if {domtype(k), domtype(l)} <> {DOM_INT} then
      infinity;
    else
      g*p[1]^k*p[2]^l;
    end_if;
  end_proc;
  
  
begin
  assert(algs[-1] <> [] and algs[-1][1] = "simpleradical");
  x := ts[-2];
  f := algs[-1][3];
  if domtype(f) <> DOM_POLY then
    f := poly(algs[-1][3], [x]);
  end_if;
  assert(f <> FAIL);
end_domain:
