// Hermite reduction, quadratic version
//
// Given a derivation on k(t) and f in k(t), return g,h,r in k(t)
// such that f=D(g)+h+r, h is simple and r is reduced.
//
// following Bronstein, Symbolic Integration I, pp. 44, 139.

intlib::algebraic::Hermite :=
proc(fnum, fden, ts, diffs, algs)
  local d, fp, fs, fn, t, a, dd, di, i, j, g, u, v, b, c, q, r;
begin
  if algs[-1] <> [] then
    if algs[-1][1] = "simpleradical" then
      return(intlib::algebraic::simpleRadicalHermite(fnum, fden, ts, diffs, algs));
    end_if;
    error("internal error: bad algebraics specification");
  end_if;
  d := intlib::algebraic::diff(ts, diffs, algs);
  t := ts[-1];
  [fp, fs, fn] := intlib::algebraic::canonicalRepresentation(fnum, fden, ts, diffs, t);
  [a, dd] := fn;
  di := coerce(polylib::sqrfree(dd), DOM_LIST);
  a := multcoeffs(a, 1/di[1]);
  dd := multcoeffs(dd, 1/di[1]);
  di := [[di[2*i], di[2*i+1]] $ i = 1..(nops(di)-1)/2]; // [[d1, e1], ...] ~ d1^e1*...
  g := 0;
  for v in di do
    [v, i] := v;
    u := divide(dd, v^i, Exact); // known to be exact, since di was created that way.
    if has(u, [FAIL]) then return([FAIL $ 3]); end_if; // mathematically, can't happen. see below.
// TODO: This is the line that should be used, but there are
//       examples in our tests that break with timeouts then.
//    u := divide(dd, v^i, Quo); // known to be exact, since di was created that way.
    for j from i-1 downto 1 do
      [b, c] := intlib::algebraic::extendedEuclidean(u*d(v), v, multcoeffs(a, -1/j));
      g := g + expr(b)/expr(v^j);
      a := multcoeffs(c, -j) - u*d(b);
    end_for;
    dd := u*v;
  end_for;
  [q, r] := [divide(a, dd)];
  return([g,
    intlib::algebraic::normal(expr(r)/expr(dd), [t]),
    intlib::algebraic::normal(expr(q)+expr(fp)+expr(fs[1])/expr(fs[2]), [t])]);
end_proc:


/*
// This was an attempt at a linear version,
// but it seems to use the wrong types of derivative at the wrong places
// and fails one of the tests in Hermite.tst

intlib::algebraic::Hermite :=
proc(fnum, fden, ts, diffs)
  local d, fp, fs, fn, fn_num, fn_den, Dbar, Dast, Dbar2, DbarAst, B, C, g, t, q, r;
begin
  d := intlib::algebraic::diff(ts, diffs, algs);
  t := ts[-1];
  [fp, fs, fn] := intlib::algebraic::canonicalRepresentation(fnum, fden, ts, diffs, t);
  [fn_num, fn_den] := fn;
  assert(op(fn_num, 2) = [t]);
  g := 0;
  Dbar := gcd(fn_den, diff(fn_den, t)); // not d(fn_den), this is for the squarefree factorization
  Dast := divide(fn_den, Dbar, Quo);
  while degree(Dbar) > 0 do
    Dbar2 := gcd(Dbar, diff(Dbar, t));
    DbarAst := divide(Dbar, Dbar2, Quo);
    [B, C] := intlib::algebraic::normal(-expr(Dast)*d(expr(Dbar))/expr(Dbar), [t]);
    assert(degree(C)=0 and not iszero(C));
    B := multcoeffs(B, 1/lcoeff(C));
    [B, C] := intlib::algebraic::extendedEuclidean(B, DbarAst, fn_num);
    fn_num := C - d(B)*divide(Dast, DbarAst, Quo);
    g := g + expr(B)/expr(Dbar);
    Dbar := Dbar2;
  end_while;
  [q, r] := [divide(fn_num, Dast)];
  [g, intlib::algebraic::normal(expr(r)/expr(Dast), [t]),
   intlib::algebraic::normal(expr(q)+expr(fp)+expr(fs[1])/expr(fs[2]), [t])];
end:
*/
