// Generate a system for the constant solutions

// Given a differential field (K, d) with constand field C,
// a matrix (a hash, an array, or a list of lists) A and a vector (a list) u with
// coefficients in K, return a matrix (a hash) B with
// entries in C and a vector (a list) v such that either v
// has coefficients in C, in which case the solutions in C
// of A*x=u are exactly all the solutions of B*x=v, or v
// has a nonconstant coefficient, in which case A*x=u
// has no constant solution.

// Follows Bronstein, Symbolic Integration I, p. 225

intlib::algebraic::rde::constantSystem :=
proc(Aa, u, ts, diffs, algs)
  local A, B, m, n, i, j, k, d, dA, du;
begin
  if domtype(Aa) = DOM_TABLE then
    A := Aa;
    delete A["m"], A["rows"], A["cols"];
    A := matrix(Aa["rows"], Aa["cols"], A);
    A := linalg::gaussJordan(A.u);
  else
    A := linalg::gaussJordan(matrix(Aa).u);
  end_if;
  if iszero(linalg::ncols(A)) or iszero(linalg::nrows(A)) then
    A := table(0);
    if domtype(Aa) = DOM_TABLE then
      A["m"] := Aa["m"];
      A["cols"] := Aa["cols"];
    else
      A["m"] := linalg::ncols(matrix(Aa));
      A["cols"] := A["m"];
    end_if;
    A["rows"] := 0;
    return([A, []]);
  end_if;
  u := [op(linalg::col(A, linalg::ncols(A)))];
  if linalg::ncols(A) > 1 then
    A := linalg::delCol(A, linalg::ncols(A));
  else
    A := matrix(linalg::nrows(A), 0);
  end_if;
  m := linalg::nrows(A);
  n := linalg::ncols(A);
  d := intlib::algebraic::diff(ts, diffs, algs);
  dA := map(A, d);
  du := map(u, d);

  // while A is not constant
  while not iszero(dA) do
    for j from 1 to n do
      if not iszero(linalg::col(dA, j)) then
        break;
      end_if;
    end_for;
    assert(j <= n and not iszero(linalg::col(dA, j)));
    for i from 1 to m do
      if not iszero(dA[i,j]) then
        break;
      end_if;
    end_for;
    B := map(linalg::row(dA, i), _mult, 1/dA[i,j]);
    B := map(B, normal, NoGcd);
    A := linalg::stackMatrix(A, B);
    u := u.[du[i]/dA[i,j]];
    for k from 1 to m do
      u[k] := normal(u[k] - A[k, j]*u[m+1]);
      A := linalg::addRow(A, m+1, k, -A[k,j]);
      A := map(A, normal);
    end_for;
    m := m+1;
    dA := map(A, d);
    du := du.[d(u[m])];
  end_while;

  // delete zero rows
  for i from linalg::nrows(A) downto 1 do
    if iszero(u[i]) and iszero(linalg::row(A, i)) then
      delete u[i];
      A := linalg::delRow(A, i);
    end_if;
  end_for;
    
  // convert to the table form we use in the parametric RDE solver
  if A = NIL then
    // bug in linalg::delRow ...
    A := table(0);
    if domtype(Aa) = DOM_TABLE then
      A["m"] := Aa["m"];
      A["cols"] := Aa["cols"];
    else
      A["m"] := linalg::ncols(matrix(Aa));
      A["cols"] := A["m"];
    end_if;
    A["rows"] := 0;
    return([A, []]);
  end_if;
    
  m := linalg::nrows(A);
  n := linalg::ncols(A);
  A := coerce(A, DOM_TABLE);
  A := subsop(A, 0=0);
  if domtype(Aa) = DOM_TABLE then
    A["m"] := Aa["m"];
  else
    A["m"] := n;
  end_if;
  A["rows"] := m;
  A["cols"] := n;
  return([A, u]);
end:
