// Risch differential equation

// Given a differential field K of chacteristic 0, f, g in K,
// decide whether the equation y' + f*y = g has a solution in K
// and to find one if there are some.

// This is the high-level routine combining the individual
// parts given by Brontein in Symbolic Integration I, chapter 6.

intlib::algebraic::rde::new :=
proc(fNum, fDen, gNum, gDen, ts, diffs, algs)
  local q, dq, gc, f, g, y, a, b, c, hn, hs, r, n, m, aa, bb, dt;
begin
  // q in k[t] s.t. f-D(q)/q is weakly normalized
  q := intlib::algebraic::rde::weakNormalizer(fNum, fDen, ts, diffs, algs);
  if q = FAIL then
    return(FAIL);
  end_if;

  dq := [intlib::algebraic::diff(ts, diffs, algs)(q), q];
  gc := gcd(dq[1], dq[2]);
  dq := map(dq, intlib::algebraic::polyDivide, gc);
  
  // f := f - dq
  f := [fNum*dq[2]-fDen*dq[1], fDen*dq[2]];
  gc := gcd(f[1], f[2]);
  f := map(f, intlib::algebraic::polyDivide, gc);
  
  // g := g*q
  g := [gNum*q, gDen];
  gc := gcd(g[1], g[2]);
  g := map(g, intlib::algebraic::polyDivide, gc);
  
  if iszero(fNum) then
    return(intlib::algebraic::normal(
      intlib::algebraic::inField(gNum, gDen, ts, diffs, algs), [ts[-1]]));
  end_if;
  
  // next, compute the normal part of any solution:
  [a, b, c, hn] := intlib::algebraic::rde::normalDenominator
    (f[1], f[2], g[1], g[2], ts, diffs, algs);
  if contains([a, b, c, hn], FAIL)>0 then return(FAIL); end_if;
  // now, for any y in k(t) s. t. D(y)+f*y=g,
  // qq = y*hn in k<t> satisfies a*D(qq)+b*qq=c.
  
  // next, the same for the special part of the denominator
  [a, b, c, hs] := intlib::algebraic::rde::specialDenominator
    (a, b[1], b[2], c[1], c[2], ts, diffs, algs);
  if contains([a, b, c, hs], FAIL)>0 then return(FAIL); end_if;
  // at this place, we only need to look for r in k[t]
  // satisfying a*D(r)+b*r=c.
  
  // r has a degree of at most this:
  n := intlib::algebraic::rde::degreeBound(a, b, c, ts, diffs, algs);

  // and the problem can be reduced to a=1:
  [b, c, m, aa, bb] := intlib::algebraic::rde::SPDE(a, b, c, ts, diffs, algs, n);
  if contains([b, c, m, aa, bb], FAIL)>0 then return(FAIL); end_if;

  r := intlib::algebraic::rde::polyRDENoCancel(b, c, ts, diffs, algs, n);
  if r=FAIL then
    // we are in a cancellation situation
    dt := poly(diffs[-1], [ts[-1]]);
    case degree(dt)
    of 0 do
      r := intlib::algebraic::rde::polyRDECancelPrim(b, c, ts, diffs, algs, n);
      break;
    of 1 do
      if iszero(coeff(dt, 0)) and iszero(degree(b)) then
        r := intlib::algebraic::rde::polyRDECancelExp(b, c, ts, diffs, algs, n);
      end_if;
      break;
    otherwise
      if iszero(coeff(dt, 1)) and coeff(dt,0)=coeff(dt,2) then
        r := intlib::algebraic::rde::polyRDECancelTan(b, c, ts, diffs, algs, n);
      end_if;
    end_case;
  elif domtype(r) = DOM_LIST then
    // r = [h,b0,c0], such that 
    // h in k[t], b0 in k, c0 in k and for any solution q in k[t] of degree 
    // at most n, y = q-h is a solution of Dy+b0*y=c0.
    // TODO: Implement something reasonable here!
    return(FAIL);
  end_if;
  
  if r=FAIL then return(FAIL); end_if;
  
  // lastly, combine the various factors collected above
  y := expr(aa*r+bb)/expr(q*hn*hs);
  y := intlib::algebraic::normal(y, ts);
  y := map(y, p -> poly(expr(p), [ts[-1]]));
end_proc:

