// parametric Risch differential equation

// Given a differential field k(t) of chacteristic 0, f, g1,...,gm in k(t),
// compute f1,...,fr in k[t], a matrix (a hash) M with m+r columns,
// entries in Const(k) and h in k[t] such that any y solving d(y) + f*y = 
// sum(c[i]*g[i], i=1..m) with c[i] in Const(K) can be written as 
// y = sum(d[i]*f[i], i=1..r)/h with d[i] in Const(K) and 
// M*transpose(c[1],...,c[m],d[1],...,d[r])=0.

// This is the high-level routine combining the individual
// parts given by Bronstein in Symbolic Integration I, chapter 7.

// TODO: Check the [FAIL, FAIL] places. y=0 is always a solution,
// so at least that should always be returned.

intlib::algebraic::rde::parametricRDE :=
proc(fNum, fDen, g : DOM_LIST, ts, diffs, algs)
  local q, dq, gc, f, a, b, gg, hn, hs, r, n, i,
    M, M2, dummy, aprime;
begin
  // q in k[t] s.t. f-D(q)/q is weakly normalized
  q := intlib::algebraic::rde::weakNormalizer(fNum, fDen, ts, diffs, algs);

  dq := [intlib::algebraic::diff(ts, diffs, algs)(q), q];
  gc := gcd(dq[1], dq[2]);
  dq := map(dq, intlib::algebraic::polyDivide, gc);
  
  // f := f - dq
  f := [fNum*dq[2]-fDen*dq[1], fDen*dq[2]];
  gc := gcd(f[1], f[2]);
  f := map(f, intlib::algebraic::polyDivide, gc);
  
  // g := g*q
  g := map(g,
    gi -> (gi := [gi[1]*q, gi[2]];
           gc := gcd(gi[1], gi[2]);
           gi := map(gi, intlib::algebraic::polyDivide, gc)));
  
  // next, compute the normal part of any solution:
  [a, b, gg, hn] := intlib::algebraic::rde::parametricNormalDenominator
    (f[1], f[2], g, ts, diffs, algs);
  if a=FAIL then return([FAIL, FAIL]); end_if;
  // now, for any y in k(t) s. t. D(y)+f*y=sum(c[i]*g[i]),
  // qq = y*hn in k<t> satisfies a*D(qq)+b*qq=sum(c[i]*gg[i]).
  
  // next, the same for the special part of the denominator
  [a, b, gg, hs] := intlib::algebraic::rde::parametricSpecialDenominator
    (a, b[1], b[2], gg, ts, diffs, algs);
  if a = FAIL then return([FAIL, FAIL]); end_if;
  // at this place, we only need to look for r in k[t]
  // satisfying a*D(r)+b*r=sum(c[i]*gg[i]).
  
  // ensure that a and b are coprime
  gc := gcd(a, b);
  if degree(gc) > 0 then
    a := intlib::algebraic::polyDivide(a, gc);
    b := intlib::algebraic::polyDivide(b, gc);
    gg := map(gg, g -> [g[1], g[2]*gc]);
    // TODO: Is full normalization rally necessary?
    gg := map(gg, g -> intlib::algebraic::normal(expr(g[1])/expr(g[2]), [ts[-1]]));
  end_if;
  
  // next, reduce to a problem where gg[i] in k[t]
  // This gives us linear constraints on the c[i]
  [gg, M] := intlib::algebraic::rde::linearConstraints(a, b, ts, diffs, algs, gg);
  [M, dummy] := intlib::algebraic::rde::constantSystem(M, [0$M["rows"]], ts, diffs, algs);
  
  if freeIndets(M) intersect {op(ts)} <> {} then
    // no constant solution possible
    return([FAIL, FAIL]);
  end_if;
  
  // TODO: Use M to reduce the number of g-s. Cf. Bronstein, p.224, point 4.
  // or not? It changes the problem ...
  
  // r has a degree of at most
  n := intlib::algebraic::rde::parametricDegreeBound(a, b, gg, ts, diffs, algs);

  // and the problem can be reduced to a=1:
  [aprime, b, gg, r, M2, n] := intlib::algebraic::rde::parametricSPDE
    (a, b, gg, ts, diffs, algs, n);
  if b=FAIL then return([FAIL, FAIL]); end_if;
  M := intlib::algebraic::rde::combineHashMatrices(M, M2);

  [f, M2] := intlib::algebraic::rde::parametricPolyRDE(b, gg, n, ts, diffs, algs);
  M := intlib::algebraic::rde::combineHashMatrices(M, M2);
  
  assert(nops(f) + nops(g) = M["cols"]);
  
  // now, for M*transpose(c1,...,cm, d1,...,dr)=0,
  // q=(aprime*sum(d[i]*f[i])-sum(c[i]*r[i]))/(hn*hs) is a solution
  // of d(q)+f*q=sum(c[i]*q[i]). Combine r and f into one list
  // and return:
  r := select([[i, r[i]] $ i = 1..nops(r)], l -> not iszero(l[2]));
  // add conditions for non-zero r to M
  if r <> [] then
    M2 := table(0);
    M2["rows"] := nops(r);
    M2["cols"] := nops(r) + nops(g);
    M2["m"] := nops(g);
    for i from 1 to nops(r) do
      M2[[i, r[i][1]]] := -1;
      M2[[i, nops(g)+i]] := 1;
    end_for;
    M := intlib::algebraic::rde::combineHashMatrices(M, M2);
  end_if;

  f := map(f, _mult, aprime) . map(r, op, 2);
  
  return([f, M, hn*hs*q]);
end_proc:
