// Find the special part of the demoninator of the solution of the RDE
//
// Given a derivation d on k[t] and a in k[t], b in k<t> and 
// g1, ..., gn in k(t), a<>0 and gcd(a,t)=1, return [a', b',
// [numer(g1'), numer(g2'), ...], [denom(g1'), denom(g2'), ...], h]
// such that a', b', h in k[t], g1', g2' in k(t), and for any
// solution c1, ..., cm in Const(k) and q in k<t> of 
// a*d(q)+b*q = sum(c[i]*g[i], i=1..m), r=q*h in k[t]
// satisfies a'*d(r)+b'*r=sum(c[i]*g[i]', i=1..m)
// 

intlib::algebraic::rde::parametricSpecialDenominator :=
proc(a, bNum, bDen, g, ts, diffs, algs, tType=UNKNOWN)
  local t, dt, nb, nc, n, alpha, beta, m, N, i;
begin
  t := ts[-1];
  dt := poly(diffs[-1], [t]);
  if tType=UNKNOWN then
    if iszero(degree(dt)) then
      tType := "Primitive";
    elif degree(dt)=1 and iszero(coeff(dt, 0)) then
      tType := "Hyperexponential"
    elif degree(dt)=2 and coeff(dt, 2)=coeff(dt, 0) and iszero(coeff(dt, 1)) then
      tType := "Hypertangent"
    else
      error("Can't handle monomial ".expr2text(t).
            " with derivative ".expr2text(expr(dt)));
    end_if;
  end_if;
  case tType
  of "Primitive" do
    // k<t> = k[t]
    assert(degree(bDen)=0);
    return([a, multcoeffs(bNum, 1/lcoeff(bDen)),
            g, poly(1, [t])]);
  of "Hyperexponential" do
    // follows Bronstein, Symbolic Integration I, p. 221
    assert(not iszero(coeff(a, 0)));
    nb := intlib::algebraic::orderAt(poly(t,[t]), bNum, bDen);
    nc := min(intlib::algebraic::orderAt(poly(t,[t]), g[i][1], g[i][2]) $ i = 1..nops(g));
    n := min(0, nc-min(0, nb));
    if iszero(nb) then
      assert(not iszero(coeff(bDen, 0)*coeff(a, 0)));
      alpha := intlib::algebraic::normal(-coeff(bNum, 0)/
                                         (coeff(bDen, 0)*coeff(a, 0)), [ts[-2]]);
      if (m := intlib::algebraic::rde::parametricLogarithmicDerivative
        (alpha[1], alpha[2], ts, diffs, algs))[1] = 1 then
        n := min(n,m[2]);
      end_if;
    end_if;
    N := max(0, -nb);
    // g[i] := g[i]*t^(N-n)
    g := map(g,
      proc(gi)
        local expo;
      begin
        // note: b,c in k<t>, i.e., bNum and dNum are powers of t, 
        // since t is a hyperexponential monomial.
        // find the exponent of the trailing term:
        expo := poly2list(gi[2])[-1][2];
        if expo < N - n then
          [gi[1]*poly(t^(N-n-expo), [t]),
           // gi[2]/poly(t^expo, [t])
           poly(map(poly2list(gi[2]), l -> subsop(l, 2=l[2]-expo)), op(gi[2], 2..3))]
        else
          [gi[1],
           // gi[2]/poly(t^(N-n), [t])
           poly(map(poly2list(gi[2]), l -> subsop(l, 2=l[2]-N-n)), op(gi[2], 2..3))]
        end_if
      end_proc);
    return([poly(a*t^N, [t]), 
            poly(expr(bNum)/expr(bDen)*t^N+n*expr(a)*diffs[-1]*t^(N-1), [t]),
            g, poly(t^(-n), [t])]);
  of "Hypertangent" do
    // follows Bronstein, Symbolic Integration I, p. 222
    // is only permissible over fields not containing sqrt(-1):
    assert(not has([a, bNum, bDen, g, diffs, algs], I));
    assert(gcd(a, poly(t^2+1, [t]))=poly(1, [t]));
    nb := intlib::algebraic::orderAt(poly(t^2+1,[t]), bNum, bDen);
    nc := min(intlib::algebraic::orderAt(poly(t^2+1,[t]), g[i][1], g[i][2]) $ i = 1..nops(g));
    n := min(0, nc-min(0, nb));
    if iszero(nb) then
      alpha := -bNum(I)/(bDen(I)*a(I));
      // careful rectform:
      assert(not has(alpha, t));
      alpha := intlib::algebraic::normal(subs(alpha, I=t), [t]);
      while degree(alpha[2]) > 1 do
        alpha[2] := poly(subs(eval(alpha[2](I)), I=t), [t]);
      end_while;
      if degree(alpha[2]) > 0 then
        alpha[1] := alpha[1]*poly(alpha[2](-t), [t]);
        alpha[2] := coeff(alpha[2],0)^2-coeff(alpha[2],1)^2;
      end_if;
      while degree(alpha[1]) > 1 do
        alpha[1] := poly(subs(eval(alpha[1](I)), I=t), [t]);
      end_while;
      beta := coeff(alpha[1],0)/alpha[2];
      alpha := coeff(alpha[1],1)/alpha[2];
      // end of careful rectform
      
      if (m := intlib::algebraic::rde::parametricLogarithmicDerivative
        (2*beta, t, ts, diffs, algs)) <> FAIL then
        error("not yet implemented");
        /* if ??? then
          n := min(n,m[2]);
        end_if; */
      end_if;
    end_if;
    N := max(0, -nb);
    // TODO: Simplify calculation
    g := map(g,
      gi -> expr(gi[1])/expr(gi[2])*(t^2+1)^(N-n));
    g := map(g, intlib::algebraic::normal, [t]);
    return([poly(a*(t^2+1)^N, [t]), 
            poly((expr(bNum)/expr(bDen)+n*a*diffs[-1]/(t^2+1))*(t^2+1)^N, [t]),
            g, poly((t^2+1)^(-n), [t])]);
  otherwise error("Unknown monomial type '".expr2text(tType)."'");
  end_case;
end:
