// Polynomial Risch DE, cancellation -- primitive case

// Given a derivation D on k[t] with D(t) in k, 
// n either an integer or infinity,
// b in k and c in k[t],
// return either FAIL, in which case the equation
// D(q)+b*q=c has no solution of degree at most n
// in k[t], or a solution q in k[t] of this equation
// with degree(q) <= n.

// Follows Bronstein, Symbollic Integration I, p. 212

intlib::algebraic::rde::polyRDECancelPrim :=
proc(b, c, ts, diffs, algs, n)
  local bb, z, p, q, m, s, lcc;
begin
  if nops(ts)=1 then return(FAIL); end_if;
  
  if iszero(c) then return(poly(0, [ts[-1]])); end_if;
  if iszero(b) then
    q := intlib::algebraic::inField(c, poly(1, [ts[-1]]), ts, diffs, algs);
    q := intlib::algebraic::normal(q, [ts[-1]]);
    if q[1]=FAIL or degree(q[2])>0 or degree(q[1])>n then
      return(FAIL);
    end_if;
    return(multcoeffs(q[1]), 1/lcoeff(q[2]));
  end_if;
  
  bb := intlib::algebraic::normal(expr(b), [ts[-2]]);
  z := intlib::algebraic::logarithmicDerivative(bb[1], bb[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
  if z <> FAIL and z[1] <> FAIL and 
    degree(z[1])=0 and degree(z[2])=0 then // z in k
    p := intlib::algebraic::inField(
      multcoeffs(c, expr(z[1])/expr(z[2])),
      poly(1,[ts[-1]]), ts, diffs, algs);
    p := intlib::algebraic::normal(p, [ts[-1]]);
    if p[1] <> FAIL and degree(p[2])=0 and degree(p[1])<=n then
      return(multcoeffs(p[1], lcoeff(z[2])/(lcoeff(p[2])*lcoeff(p[1]))));
    else
      return(FAIL);
    end_if;
  end_if;
  if n < degree(c) then return(FAIL); end_if;
  
  q := poly(0, [ts[-1]]);
  while not iszero(c) do
    m := degree(c);
    if n<m then return(FAIL); end_if;
    // D(s)+b*s=lcoeff(c):
    lcc := intlib::algebraic::normal(lcoeff(c), [ts[-2]]);
    s := intlib::algebraic::rde(bb[1], bb[2], lcc[1], lcc[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
    if s[1]=FAIL then return(FAIL); end_if;
    s := poly(expr(s[1])/expr(s[2])*ts[-1]^m,[ts[-1]]);
    q := q+s;
    n := m-1;
    c := c - multcoeffs(s, b) - intlib::algebraic::diff(ts, diffs, algs)(s);
  end_while;
  return(q);
end:
