// polynomial Risch DE (Dq+bq=c), no cancellation

// Given a differential field k, n either an integer or infinity,
// b,c in k[t] such that one of the following holds:
//
// - b <> 0, either k=const(k) or degree(b) > max(0, degree(D(t))-1)
// - degree(b) < degree(D(t))-1 and either k=const(k) or degree(D(t))>1
// - degree(b) = degree(D(t))-1 and degree(D(t))>1,
// 
// return either FAIL, in which case Dq+b*q=c has no solution q in k[t]
// of degree at most n, or such a solution, or a list [h,b0,c0], such that 
// h in k[t], b0 in k, c0 in k and for any solution q in k[t] of degree 
// at most n, y = q-h is a solution of Dy+b0*y=c0.

// Follows Bronstein, Symbolic Integration I, pp. 208, 209, 210

intlib::algebraic::rde::polyRDENoCancel :=
proc(b, c, ts, diffs, algs, n)
  local q, m, p, t, dt, M, lt, u;
begin
  t := ts[-1];
  dt := poly(diffs[-1], [ts[-1]]);
  lt := lcoeff(dt);
  q := poly(0, [t]);
  if not iszero(b) and (nops(ts)=1 or degree(b)>max(0,degree(dt)-1)) then
    // case 1, p. 208
    while not iszero(c) do
      m := degree(c)-degree(b);
      if n<0 or m<0 or m>n then return(FAIL); end_if;
      p := poly(lcoeff(c)/lcoeff(b)*t^m, [t]);
      q := q+p;
      c := c - intlib::algebraic::diff(ts, diffs, algs)(p) - b*p;
      c := mapcoeffs(c, normal);
    end;
  elif (iszero(b) or degree(b)<degree(dt)-1) and (nops(ts)=1 or degree(dt)>1) then
    // case 2, p. 209
    while not iszero(c) do
      if iszero(n) then
        m := 0;
      else
        m := degree(c) - degree(dt) + 1;
      end_if;
      if n < 0 or m < 0 or m > n then
        return(FAIL);
      end_if;
      if m>0 then
        p := poly(lcoeff(c)/(m*lt)*t^m, [t]);
      else
        if degree(b) <> degree(c) then return(FAIL); end_if;
        if degree(b)=0 then return([q, b, c]); end_if;
        p := poly(lcoeff(c)/lcoeff(b), [t]);
      end_if;
      q := q+p;
      n := m-1;
      c := c - intlib::algebraic::diff(ts, diffs, algs)(p) - b*p;
      c := mapcoeffs(c, normal);
    end_while;
  elif degree(b)=degree(dt)-1 and degree(dt)>1 then
    // case 3, p. 210
    M := normal(-lcoeff(b)/lt);
    if not testtype(M, Type::PosInt) then M := -1; end_if;
    while not iszero(c) do
      m := max(M, degree(c)-degree(dt)+1);
      if n<0 or m<0 or m>n then return(FAIL); end_if;
      u := m*lt+lcoeff(b);
      if iszero(u) then return([q,m,c]); end_if;
      if m>0 then
        p := poly(lcoeff(c)/u*t^m, [t]);
      else
        if degree(c) <> degree(dt)-1 then return(FAIL); end_if;
        p := lcoeff(c)/lcoeff(b);
      end_if;
      q := q+p;
      n := m-1;
      c := c - intlib::algebraic::diff(ts, diffs, algs)(p) - b*p;
      c := mapcoeffs(c, normal);
    end_while;
  else
    // none of the no-cancellation cases.
    q := FAIL;
  end_if;
  mapcoeffs(q, normal);
end:
