// integration of hyperexponential functions

// given a hyperexponential monomial t over k and f in k(t), return
// g elementary over k(t) and a Boolean b such that f-d(g) in k
// if b = TRUE and f-d(g) (and therefore f) does not have an elementary
// integral over k(t) if b = FALSE.

// Follows Bronstein, Symbolic Integration I, pp. 163, 162

intlib::algebraic::integrateHyperexponential :=
proc(fNum, fDen, ts, diffs, algs)
  local g1, h, r, g2, q, good, i, t, dlnt, vtp, vinfp, a, v, remn, shift, p;
begin
  [g1, h, r] := intlib::algebraic::Hermite(fNum, fDen, ts, diffs, algs);
  if g1 = FAIL then return([0, FALSE]); end_if;
  [g2, good] := intlib::algebraic::residueReduce(h[1], h[2], ts, diffs, algs);
  if good = FALSE then return([g1+g2, FALSE]); end_if;
  
  // remn := h-D(g2)+r
  t := ts[-1];
  remn := intlib::algebraic::normal(expr(h[1])/expr(h[2]) 
     + expr(r[1])/expr(r[2])
     -intlib::algebraic::diff(ts, diffs, algs)(g2), [t]);
  
  shift := degree(remn[2]);
  if not iszero(mapcoeffs(divide(remn[1]*poly([[1, shift]], [t]), remn[2], Rem), normal)) then
    return([g1+g2, FALSE]);
  end_if;
  p := intlib::algebraic::polyDivide(remn[1]*poly([[1, shift]], [t]), remn[2]);
  
  dlnt := intlib::algebraic::normal(diffs[-1]/t, [ts[-2]]);
  q := 0;
  vtp := intlib::algebraic::orderAt(poly(t, [t]), remn[1], remn[2]);
  vinfp := intlib::algebraic::orderAt(infinity, remn[1], remn[2]);
  if vtp <= -vinfp then // necessary to handle cases of infinity
    for i from vtp to -vinfp do
      if iszero(i) then next; end_if;
      a := coeff(p, i+shift);
      v := intlib::algebraic::rde(
        multcoeffs(dlnt[1], i), dlnt[2],
        op(intlib::algebraic::normal(a, [ts[-2]])),
        ts[1..-2], diffs[1..-2], algs[1..-2]);
      if v=FAIL then
        good := FALSE;
      else
        v := expr(v[1])/expr(v[2]);
        q := q+v*t^i;
      end_if
    end_for;
  end_if;
  return([g1+g2+q, good]);
end:
