// 'parallel' integration, a heuristic integrator based on 
// an extended abstract and rudimentary maple code by Manuel Bronstein
// alias(normal = (x -> normal(x, Expand = TRUE)));
// alias(numer  = (x -> numer(x, Expand = TRUE)));
// alias(denom  = (x -> denom(x, Expand = TRUE)));

intlib::algebraic::parallel :=
proc(f, x, options=table())
  local ts, vars, algs, diffs, ff, q, d, dbar, darbouxPolys,
        substs, res, final_substs, badfn, ft;
begin
  badfn := FALSE:
  misc::maprec(f,
    // can't handle any of these
    {"Re", "Im", "sign", "abs", "dirac", "heaviside", RootOf, "sum"} =
      (() -> (if has([args()], [x]) then badfn := TRUE; misc::breakmap(); end_if; args())),
//    {"sum"} =
//      (s -> (if testtype(op(s, [2, 2]), RootOf) and has(s, [x]) then badfn := TRUE; misc::breakmap(); end_if; args())),
    // don't like these (performance killer)
    {"_power"} =
      (ex -> (if has(ex, [x]) and not testtype(op(ex, 2), DOM_INT) then  badfn := TRUE; misc::breakmap(); end_if; args())),
    Unsimplified);
  if badfn then
    return(FAIL);
  end_if;
  
  // find differential field in which the function is rational
  [ts, vars, diffs, algs, ff, final_substs] :=
    intlib::algebraic::fieldTower(f, x, table(options, "TranscendentalsOnly"=TRUE));
  if has([ts, vars, diffs, ff, final_substs], [FAIL, undefined]) // non-transcendental extension, sorry, folks
    or algs <> [[] $ nops(ts)]
    or length([ff, diffs])*nops(ts) > MAXEFFORT then
    return(FAIL);
  end_if;
  substs := revert(zip(vars, ts, _equal));
  ff := normal(ff);
  
  // the lcm of the denominator of the derivatives
  q := lcm(op(map(diffs, denom)));
  
  // our derivations
  d := intlib::algebraic::diff(ts, diffs, algs);
  dbar := intlib::algebraic::diff(ts, map(diffs, normal@_mult, q));

// this routine could be useful to find Darboux polynomials manually
//  return(ex -> Simplify(dbar(subsex(ex, revert(zip(vars, ts, _equal))))/
//                        subsex(ex, revert(zip(vars, ts, _equal)))));
  
  darbouxPolys := map(vars, 
                      intlib::algebraic::parallel::knownDarboux,
                      substs);
  darbouxPolys := darbouxPolys.intlib::algebraic::parallel::alsoDarboux(q, dbar, ts, substs);
                      
  // actual integration
  res := intlib::algebraic::parallel::pmIntegrate(ff, d, dbar, q, ts,
                                                  vars, diffs, darbouxPolys, options);

  // simplifications
  res := subs(res, final_substs);
  if res <> FAIL and options[hold(DontSimplifyComplexIntegral)] <> TRUE then
    if traperror((ft := intlib::simplifyComplexIntegral(res, x, Unique)),
      MaxSteps = intlib::simplifyMaxSteps(f, res, "simpCint")) = 0 then
      if intlib::Simplify::defaultValuation(ft) <
        intlib::Simplify::defaultValuation(res) then
        res := ft;
      end_if;
    end_if;
  end_if;

  intlib::Simplify(res, options);
end_proc:


intlib::algebraic::parallel := funcenv(intlib::algebraic::parallel):

// known Darboux polynomials.
// The Boolean in the second entry determines if this
// polynomial is used in the heuristic guess for s
intlib::algebraic::parallel::knownDarboux :=
proc(f, substs)
begin
  case op(f, 0)
    of hold(tan) do return([subs(f, substs)^2+1, FALSE]);
    of hold(tanh) do return([1+subs(f, substs), FALSE],
                            [1-subs(f, substs), FALSE]);
    of hold(exp) do return([subsex(f, substs), FALSE]);
    of hold(lambertW) do return([subs(f, substs), TRUE]);
  end_case;
  null();
end_proc:

// the same, but not given some subexpression f but the lcm of denoms
intlib::algebraic::parallel::alsoDarboux :=
proc(q, dbar, ts, substs)
  local dq;
begin
  dq := poly(dbar(q), ts);
  if dq = FAIL then
    return([]);
  end_if;
  dq := gcd(dq, poly(q, ts));
  if degree(dq) > 0 then
    return([[expr(q), FALSE]]);
  end_if;
  [];
end:

intlib::algebraic::parallel::combineFractions := (x -> normal(x, Expand = FALSE)):

// split into numerator and denominator, without normailzation
intlib::algebraic::parallel::numDen :=
proc(ex)
  local num, den, fac;
begin
  case type(ex)
  of "_mult" do
    [fac, num, den] := generate::splitProduct(ex);
    return([_mult(fac, op(num)), _mult(op(den))]);
  of "_power" do
    if op(ex, 2) < 0 then
      return([1, 1/ex]);
    end_if;
    break;
  end_case;
  [ex, 1];
end_proc:


// the core routine, which includes all the heuristics
intlib::algebraic::parallel::pmIntegrate :=
proc(f, d, dBar, q, ts, vars, diffs, knownDarboux, options)
  local i, qs, qn, s, ff, denF, denFs, denFn, candDen, maxDeg,
        monomials, t, tmin, tmax, b, cand, Dcand, factors, triedFactors,
        addTerm, guessMoreTerms, coeffs, sol, target, targetfactor;
begin
  [qs, qn] := intlib::algebraic::parallel::splitFactor(q, dBar, ts);
  s := _mult(qs, op(map(select(knownDarboux, op, 2), op, 1)));
  ff := normal(f, Expand = TRUE);
  denF := denom(ff);
  if denF = FAIL or poly(denF, ts) = FAIL or
    has([coeff(poly(denF, ts))], ts) = TRUE or
    poly(numer(ff), ts) = FAIL or
    has([coeff(poly(numer(ff), ts))], ts) = TRUE then
    return(FAIL);
  end_if;
  [denFs, denFn] := intlib::algebraic::parallel::splitFactor(denF, dBar, ts);
  candDen := collect(s * denFs * intlib::algebraic::parallel::deflation(denFn, ts), ts);
  candDen := normal(candDen);
  // cf. Bronstein, Symbolic Integration I, p. 307
  // increased by 2 to allow slight cancellations - solves a couple of examples
  maxDeg := 3 + degree(numer(ff), ts) + 
    max(0, degree(q, ts) - max(map(ts,
      proc(t)
        local dbt;
      begin
        // degree(dBar(t), ts), but dBar(t) may not look like a polynomial
        dbt := dBar(t);
        degree(numer(dbt), ts) - degree(denom(dbt), ts);
      end_proc)));
  maxDeg := min(maxDeg,
    1 + degree(candDen, ts) + max(degree(numer(ff), ts), degree(denF, ts)));
//  maxDeg := min(maxDeg, 12);
  if has(maxDeg, FAIL) then
    return(FAIL);
  end_if;

  userinfo(5, "guess for degree bound: ".expr2text(maxDeg));

  // given f, Dbar, q, the substituted expressions,
  // the monomials D(s[i])/s[i], D(p[i])/p[i], a candidate
  // denominator and a max degree for the candidate enumerator,
  // find a solution of eq. 1 in M. Bronstein: Parallel Integration,
  // Programming and Computer Software, Vol. 32, No. 1.

    monomials := map(combinat::compositions(maxDeg, Length=nops(ts)+1, MinPart=0,
    // reduce the amount of time spent in normalizing, and the number of indeterminates,
    // at the expense of missing candidates:
    // TODO: Find a better way of selecting promising monomials!
      "StopAfter" = 200), // undocumented option exactly for this purpose
      l -> expr(poly([[1, l[1..-2]]], ts)));
                     
    triedFactors := table():

      factors := coerce(factor(qn), DOM_LIST)[2..-1] .
                 coerce(factor(denFs), DOM_LIST)[2..-1] .
                 coerce(factor(denFn), DOM_LIST)[2..-1];
      factors := [op({factors[2*i-1] $ i = 1..nops(factors)/2,
                      op(map(knownDarboux, op, 1))})];

      if triedFactors[factors]=TRUE then next; end_if;
      triedFactors[factors] := TRUE;

      t := solvelib::getIdent(Any, indets([ts, ff, vars]));
      tmax := select(indets([ts, ff, vars]), x -> strmatch("".x, "^".t."\\d+$"));
      tmax := map(tmax, x -> text2expr(("".x)[length("".t)+1..-1]));
      tmax := max(0, op(tmax));
      tmin := tmax;
      b := _plus(t.(tmax+i)*monomials[i] $ i=1..nops(monomials));
      tmax := tmax + nops(monomials);
      cand := b/candDen;

      /*
      // gcd gets *much* slower when the input contains I.
      // We don't need real gcd just yet, hide the I:
      Dcand := subs(d(cand), I=t.(tmax+1));
      Dcand := intlib::algebraic::normal(d(cand), ts);
      Dcand := subs(expr(Dcand[1])/expr(Dcand[2]), t.(tmax+1)=I);
      */
      Dcand := d(cand);
      // normalization sometimes takes a lot of time.
      // give up early
      if length(Dcand) > 5000 then
        return(FAIL);
      end_if;
      Dcand := intlib::algebraic::parallel::combineFractions(Dcand);
      
      // target: f = Dcand <=> numer(f)*denom(Dcand)-denom(f)*numer(Dcand)=0
      // TODO: How much of denF or denFs can be left out?
      if type(Dcand)="_mult" then
        target := generate::splitProduct(Dcand);
        target := [_mult(target[1], op(target[2])), _mult(op(target[3]))];
      else
        target := [Dcand, 1];
      end_if;
      targetfactor := denF*target[2];
      target := numer(ff)*target[2] - denF*target[1];

      addTerm :=
      proc(ex, Dex)
        name intlib::algebraic::parallel::pmIntegrate::addTerm;
      begin
        Dex := intlib::algebraic::normal(Dex, ts);
        if has(map(Dex, coeff), ts) then
          return();
        end_if;
        userinfo(10, "adding ".expr2text(ex));
        tmax := tmax+1;
        cand := cand + t.tmax*ex;
        target := target*expr(Dex[2]) - t.tmax*expr(Dex[1])*targetfactor;
        targetfactor := targetfactor * expr(Dex[2]);
        Dex := expr(Dex[1])/expr(Dex[2]);
        Dcand := Dcand + t.tmax*Dex;
      end;
    
      // in addition to Bronstein's paper, we also try arctan terms,
      // Ei terms, Li, and erf terms:
      guessMoreTerms := 
      proc(ex)
        local t, pex, a, b, c, dis, i, e, te, cont, eIsSquare, sqrtE;
      begin
        addTerm(ln(ex), d(ex)/ex);
        i := contains(ts, ex);
        if i>0 then
          te := type(vars[i]);
          e := subs(op(vars[i]), revert(zip(vars, ts, _equal)));
        else
          te := type(ex);
          e := op(ex);
        end_if;
        // check if the argument is a square.
        eIsSquare := FALSE;
        t := [op(indets([e]) intersect {op(ts)})];
        if type(e) <> "_exprseq" and nops(t) = 1 and
          testtype(e, Type::RatExpr(op(t), Type::IndepOf(op(t))))=TRUE then
          a := intlib::algebraic::normal(e, t);
          a := map(a, polylib::sqrfree);
          a := map(a, coerce, DOM_LIST);
          cont := TRUE;
          for i from 3 to nops(a[1]) step 2 do
            if not testtype(a[1][i]/2, DOM_INT) then
              cont := FALSE;
              break;
            end_if;
          end_for;
          if cont then
            for i from 3 to nops(a[2]) step 2 do
              if not testtype(a[2][i]/2, DOM_INT) then
                cont := FALSE;
                break;
              end_if;
            end_for;
            if cont then
              eIsSquare := TRUE;
              sqrtE := [sqrt(a[1][1]/a[2][1]),
                        _mult(expr(a[1][2*i])^(a[1][2*i+1]/2) 
                               $ i = 1..(nops(a[1])-1)/2) /
                        _mult(expr(a[2][2*i])^(a[2][2*i+1]/2) $ i = 1..(nops(a[2])-1)/2)];
            end_if;
          end_if;
        end_if;

        case te
        of "ln" do
          addTerm(Li(e), d(e)/ex);
          addTerm(dilog(e), -d(e)*ex/(e-1));
          break;
        of "exp" do
          addTerm(Ei(e), ex*d(e)/e);
          if eIsSquare then
            addTerm(sqrt(PI)/(2*sqrtE[1])*erf(sqrtE[1]*sqrtE[2]),  d(sqrtE[2])/ex);
            addTerm(sqrt(PI)/(2*sqrtE[1])*erfi(sqrtE[1]*sqrtE[2]), d(sqrtE[2])*ex);
          end_if;
          break;
        of "cos" do
          addTerm(Ci(e), ex*d(e)/e);
          if eIsSquare then
            addTerm(sqrt(PI/2)*fresnelC(sqrt(2/PI)*sqrtE[1]*sqrtE[2]),
                    ex*d(sqrtE[1]*sqrtE[2]));
          end_if;
          break;
        of "sin" do
          addTerm(Si(e), ex*d(e)/e);
          if eIsSquare then
            addTerm(sqrt(PI/2)*fresnelS(sqrt(2/PI)*sqrtE[1]*sqrtE[2]),
                    ex*d(sqrtE[1]*sqrtE[2]));
          end_if;
          break;
        of "polylog" do
          if type(e) <> _exprseq or nops(e) <> 2 then break; end_if;
          [a, b] := [e];
          if iszero(d(a)) then
            addTerm(polylog(a+1, b), d(b)*ex/b);
          end_if;
          break;
        end_case;
      
        t := [op(indets([ex]) intersect {op(ts)})];
        if nops(t) = 1 then
          pex := poly(ex, t);
          if pex<>FAIL and degree(pex) = 2 then
            a := coeff(pex, 2);
            b := coeff(pex, 1);
            c := coeff(pex, 0);
            dis := 4*a*c-b^2;
            if not iszero(dis) and indets([dis]) intersect {op(ts)} = {} then
              addTerm(arctan((2*a*t[1]+b)/sqrt(dis)),
                      d(t[1])*sqrt(dis)/(2*ex))
            end_if;
          end_if;
        end_if;
      end:

      map({op(factors), op(ts)}, guessMoreTerms);
    
      sol := FAIL;
      
      // Don't start normalizing expressions with very many indets
      if tmax-tmin > 200 or length(target) > 8000 then
        return(FAIL);
      end_if;

      // fprint(0, cand, Dcand, target);
      userinfo(2, "set up candidate with ".(tmax-tmin)." free parameters");
      
      coeffs := [coeff(target, ts)];
      // print(coeffs);
      if has(coeffs, ts) then
        error("internal logic error");
      end_if;
      sol := intlib::algebraic::linsolve(coeffs, [t.i$i=tmin+1..tmax]);
      sol := map(sol, map, intlib::Simplify, options, Steps=10);

    if sol <> FAIL then
      sol := subs(cand, sol);
      // new indets, indets not covered by the solution (e.g., not occuring in d(cand))
      sol := subs(sol, map(numeric::indets(sol) minus numeric::indets(f)
                           minus numeric::indets(vars) minus {op(ts)}, _equal, 0));
      // since we've introduced new indets, we must do the backsubstitution here
      // result treatment
      return(subs(sol, zip(ts, vars, _equal)));
    end_if;
  FAIL;
end:

// splitting-factorization of a polynomial w.r.t. a derivation:
// return ps*pn, s.t. ps is a Darboux polynomial (i.e.,
// d(ps) divides ps) and no irreducible factor of pn is Darboux.
intlib::algebraic::parallel::splitFactor :=
proc(p, d, ts)
  local x, cont, ppart, splrek, s, splrek2;
begin
  if ts = [] then return([1, p]); end_if;
  x := ts[1];
  while not has(p, x) do
    delete ts[1];
    if ts = [] then return([1, p]); end_if;
    x := ts[1];
  end_while;
  cont := content(p, [x]);
//  ppart := multcoeffs(p, 1/cont);
//  ppart := normal(p/cont);
  ppart := expr(poly(p/cont, [x]));
  splrek := intlib::algebraic::parallel::splitFactor(cont, d, ts[2..-1]);
  s := normal(gcd(ppart, d(ppart))/gcd(ppart, diff(ppart, x)));
  if iszero(degree(s)) then
    return([splrek[1], ppart*splrek[2]]);
  end_if;
  splrek2 := intlib::algebraic::parallel::splitFactor(normal(ppart/s), d, ts[2..-1]);
  return([splrek[1]*splrek2[1]*s, splrek[2]*splrek2[2]]);
end:

// deflation: given p=product(pp[i]^i, i=1..n), return
// product(pp[i]^(i-1),i=2..n)
intlib::algebraic::parallel::deflation :=
proc(p, ts)
  local x, cont, ppart;
begin
  if ts = [] then return(p); end_if;
  x := ts[1];
  while not has(p, x) do
    delete ts[1];
    if ts = [] then return(p); end_if;
    x := ts[1];
  end_while;    
  cont := content(p, [x]);
  ppart := poly(p/cont, [x]);
  return(intlib::algebraic::parallel::deflation(cont, ts[2..-1]) *
         expr(gcd(ppart, diff(ppart, x))));
end:
