// integration of rational functions
//
// Uses the Horowitz-Ostrogradsky algorithm to compute the rational part
// of the integral

intlib::algebraic::rational := funcenv(
proc(ex, x, options=table())
  local f, p, a, d, a2, d2, integral, poly_integral, minpolys, T, sol, rat, done, usedIdents;
begin
  assert(testtype(ex, Type::RatExpr(x, Type::IndepOf(x))));
  
  if type(ex) = "_plus" then
    f := poly(ex, [x]);
    if f <> FAIL then
      return(expr(intlib::algebraic::rational::polynomial(f)));
    else
      // faster than normalizing first, and the summands
      // are integrable in any case
      return(map(ex, intlib::algebraic::rational, x));
    end_if;
  end_if;
  
  f := intlib::algebraic::normal(combine(ex), [x]);
  d := f[2];
  [p, a] := [divide(f[1], d)];
  poly_integral := expr(intlib::algebraic::rational::polynomial(p));
  if iszero(a) then return(poly_integral); end_if;
  // normalize a/d. To avoid excessive computations, rationalize the coefficients first:
  usedIdents := indets([poly2list(a), poly2list(d)], All);
  [a, rat, minpolys] := [rationalize([poly2list(a), poly2list(d)],
    MinimalPolynomials, FindRelations = ["exp", "_power"])];
  if minpolys <> {} and
    traperror((T := fp::fold((p, d) ->
        Dom::AlgebraicExtension(d, poly(p, [op(indets(p) minus usedIdents)], d)),
      Dom::Fraction(Dom::Polynomial(Dom::Rational)))(op(minpolys)))) = 0 and
    not has((d := map(a, poly, op(p, 2), T)), [FAIL]) then
      [a, d] := d;
  else
    [a, d] := map(a, poly, op(d, 2..3));
  end_if;
  
  done := FALSE;
  repeat
    integral := poly_integral;
    
    p := gcd(a, d);
    a := divide(a, p, Quo); // known to be exact
    d := divide(d, p, Quo); // known to be exact
  
    [sol, a2, d2] := intlib::algebraic::rational::rationalPart(a, d, options, usedIdents);
    integral := integral + sol;
  
    // yet to integrate: a/d, d is squarefree
    integral := integral +
      intlib::algebraic::rational::logPart(poly(a2, Expr), poly(d2, Expr),
        options, usedIdents, op(a2, 3));
        
    if rat = {} or traperror((
      repeat
        a2 := integral;
        integral := subs(integral, rat, EvalChanges);
//        integral := combine(integral); // handling square roots
      until a2 = integral end_repeat
    )) = 0 then
      done := TRUE;
    else
      // it seems the rationalizing done above was a bad idea.
      // retry without it:
      [a, d] := subs(map([a, d], poly, Expr), rat);
      rat := {};
      done := FALSE;
    end_if;
  until done end_repeat;
  integral;
end):


intlib::algebraic::rational::polynomial :=
proc(p)
  local l;
begin
  l := poly2list(p);
  l := map(l, t -> [t[1]/(t[2]+1), t[2]+1]);
  poly(l, op(p, 2..3));
end_proc:

// Horowitz-Ostrogradsky, following Bronstein, Symbolic Integration I, p.46
intlib::algebraic::rational::rationalPart :=
proc(a, d, options, usedIdents)
  local x, dBar, dAst, t, m, n, b, c, H, sol, i, g;
begin
  x := op(op(a, 2));
  dBar := gcd(d, d');
  dAst := divide(d, dBar, Quo); // known to be exact
  if op(a, 3) = Expr then
    dAst := mapcoeffs(dAst, normal, Expand=FALSE);
  end_if;
  m := degree(dAst)-1;
  n := degree(dBar)-1;
  t := solvelib::getIdent(Any, indets({coeff(a), coeff(d), x, usedIdents}, All));
  b := poly([[t[i], i] $ i = 0..n], [x], op(a, 3));
  c := poly([[t[i+n+1], i] $ i = 0..m], [x], op(a, 3));
  H := a - b'*dAst + divide(b*dAst*dBar', dBar, Quo) - c*dBar;
  if op(a, 3) = Expr then
    H := mapcoeffs(H, normal, Expand=FALSE);
  end_if;
  sol := intlib::algebraic::linsolve([coeff(H, i) $ i = 0..degree(d)], [t[i] $ i = 0..m+n+1]);
  c := subs(c, sol);
  g := gcd(c, dAst);
  if not iszero(degree(g)) then
    c    := divide(c, g, Quo); // exact
    dAst := divide(dAst, g, Quo); // exact
  end_if;
  [expr(subs(b, sol))/expr(dBar), c, dAst];
end:

// Lazard-Rioboo-Trager algorithm,
// following Bronstein, Symbolic Integration I, p. 51

/*

 Notes: In some examples, the code below returns expressions
  considerably more complex than other algorithms would.
  E.g., for an irreducible denominator as in
  int(1/(y^15+a*y+b), y), Maple returns its equivalent of
  sum(ln(y-z)/(15*z^14+a), z in RootOf(y^15+a^y+b, y))

*/

intlib::algebraic::rational::logPart :=
proc(a, d, options, usedIdents, T)
  local x, t, R, q, i, j, k, s, integral, A, B, z, tmp, b, c, sq, coeffs, lns;
begin
  if iszero(a) then return(0); end_if;
  x := op(op(a, 2));
  t := solvelib::getIdent(Any, indets({coeff(a), coeff(d), x, usedIdents}, All));
  z := solvelib::getIdent(Any, indets({coeff(a), coeff(d), x, t, usedIdents}, All));
  R := polylib::subresultant(d, poly(expr(a-t*d'), [x], op(a, 3)));
  if T <> Expr then
    /// just mapping T to all coefficients results in much more complicated outputs
    /// e.g., try int(1/(2*m^2 + 2*m)*(3*m - a + 2*(1/4*a^2 - a + b + 1)^(1/2) - a*m + 1), m)
    // R[0] := poly(poly(R[0], T), Expr);
    /// but we need some simplification for correct results:
    while not iszero(R[0]) and iszero(T(lcoeff(R[0]))) do
      R[0] := R[0] - lterm(R[0]);
    end_while;
  end;
  // In the following call, it is important to use normal(.., Expand = TRUE), 
  // sincd  normal( expand((x + I)^2)/(x + I), Expand = FALSE) produces
  //    (x^2 + 2*I*x - 1) / (x + I)
  // whilst a polynomial is expected.
  // In contrast to this, 
  // normal(expand((x+I)^2)/(x + I), Expand = TRUE) produces x + I.
  R[0] := mapcoeffs(R[0], normal, Expand=TRUE);
//  q := coerce(factor(poly(R[0], [t])), DOM_LIST);
  q := coerce(polylib::sqrfree(poly(R[0], [t], op(a, 3))), DOM_LIST);
  integral := 0;
  for i from 2 to nops(q)-1 step 2 do
    if q[i+1] = degree(d) then
      s := poly(d, [t], op(a, 3));
    else
//      m := contains(degR, q[i+1]);
//      assert(m > 0);
//      s := R[2][m];
      s := R[q[i+1]];
      assert(s <> 0);
      A := coerce(polylib::sqrfree(poly(lcoeff(expr(s), [x]), [t], op(a, 3))), DOM_LIST);
      for j from 2 to nops(A)-1 step 2 do
        s := multcoeffs(s, 1/expr(gcd(A[j], q[i])^A[j+1]));
      end_for;
      s := mapcoeffs(s, normal);
      s := poly(s, [t], op(a, 3));
    end_if;
    case degree(expr(q[i]), t)
    of 1 do
      tmp := poly(expr(q[i]), [t], op(a, 3));
      tmp := -coeff(tmp, 0)/coeff(tmp, 1);
      integral := integral + tmp*ln(s(tmp));
      break;
    of 2 do
//      if degree(expr(s), [t]) = 1 then
        tmp := poly(expr(q[i]), [t], op(a, 3));
        tmp := multcoeffs(tmp, 1/lcoeff(tmp));
        b := coeff(tmp, 1)/2;
        c := coeff(tmp, 0);
        sq := normal(c-b^2);
        // we use complex conjugated pairs anyway,
        // so don't write sqrt(a^2)
        sq := specfunc::Wurzelbehandlung(sq);

        A := s(-b+I*sq);
        B := s(-b-I*sq);
        A := poly(subs(poly(subs(A, I=z), [z])(I), I=z), [z], op(a, 3));
        B := poly(subs(poly(subs(B, I=z), [z])(I), I=z), [z], op(a, 3));
        if degree(A) = 1 and degree(B) = 1 and
           coeff(A, 0) = coeff(B, 0) and
           coeff(A, 1) = -coeff(B, 1) then
          B := coeff(A, 1);
          A := coeff(A, 0);
          integral := integral - b * ln(s(-b+sqrt(b^2-c)) * s(-b-sqrt(b^2-c)))
            + sq*intlib::algebraic::rational::logToArctan
              (poly(A, [x], op(a, 3)), poly(B, [x], op(a, 3)), usedIdents);
//            +I*sq*ln((A+I*B)/(A-I*B));
          break;
        end_if;
//      end_if;
    otherwise
      coeffs := solve(expr(q[i]), t, IgnoreSpecialCases,
        if options[IgnoreAnalyticConstraints]=TRUE then IgnoreAnalyticConstraints end_if);
      if domtype(coeffs) = DOM_SET then
        coeffs := [op(coeffs)];
        coeffs := map(coeffs, intlib::Simplify, options, Steps=15);
        lns:= map(coeffs, s);
        // we do *not* want to do a map(lns, expand) here: can cause huge expression swell
        // experiments with map(lns, normal, Expand = FALSE) were not convincing, too
        // finally, one might try something like
        // tryExpand:= proc(X) local a; begin a:= expand(X); if length(a) < 10*length(X) then a else X end_if end_proc;
        // lns:= map(lns, tryExpand);        
        // which makes some xamples better and others worse

        // look for terms to be combined to arc tangents
        j := 1;
        while j < nops(coeffs) do
          assert(not iszero(coeffs[j]));
          k := contains(coeffs, -coeffs[j]);
          if k > 0 and k <> j then
            A := (lns[j] - lns[k])/I;
            B := lns[j] + lns[k];
// TODO: This should be activated. But it breaks examples like
// eq := (diff(y(x), x) + g) - k*y(x)^2; sol := solve(ode(eq, y(x)))
// where solve handles the arctan much better than the two lns.
//            if has([lns[j], lns[k]], I) or not has(A/B, I) then
              integral := integral + 2*I*coeffs[j] * arctan(intlib::Simplify(A/B, options, Steps=5));
              assert(k>j);
              delete coeffs[k], coeffs[j];
              delete lns[k], lns[j];
//            else
//              j := j+1;
//            end_if;
          else
            j := j+1;
          end_if;
        end_while;
        
        integral := _plus(integral, op(zip(coeffs, lns, (c, z) -> c*ln(z))));
      elif type(coeffs) = "_union" and
         map({op(coeffs)}, domtype) minus {DOM_SET, RootOf} = {} then
        // TODO: Also here, combine logs to arctans.
        integral := _plus(integral,
          op(map([op(coeffs)], 
            br -> if type(br) = DOM_SET then
                    _plus(z*ln(s(z)) $ z in br)
                  else
/*
 Note: Here, we could replace s by divide(s, <poly in br>, Rem).
*/
                    hold(sum)(z*ln(s(z)), z in br)
                  end)));
      else
/*
 Note: Here, we could replace s by divide(s, q[i], Rem).
*/
        integral := integral + hold(sum)(z*ln(s(z)), z in RootOf(expr(q[i]), t));
      end_if;
    end_case;
  end_for;
  integral;
end:

// Rioboo's conversion of complex logarithms to real arc-tangents,
// cf. Bronstein, Symblic Integration I, p. 63
intlib::algebraic::rational::logToArctan :=
proc(A, B)
  local g, c, d;
begin
  if iszero(divide(A, B, Rem)) then
    return(2*arctan(expr(A)/expr(B)));
  end_if;
  if degree(A) < degree(B) then
    return(intlib::algebraic::rational::logToArctan(-B, A));
  end_if;
  [g, d, c] := [gcdex(B, -A)]; // B*d - A*c = g
  assert(iszero(mapcoeffs(B*d-A*c-g, Simplify)));
  d := mapcoeffs(d, Simplify);
  c := mapcoeffs(c, Simplify);
  return(2*arctan(intlib::Simplify(expr(A*d+B*c)/expr(g), table(), Steps=5)) +
         intlib::algebraic::rational::logToArctan(d, c));
end:

