//
// Lazard-Rioboo-Rothstein-Trager resultant reduction and residue criterion
// following Bronstein, Symbolic Integration I, p. 153
//
// Input: A simple rational function f in k(t),
//  given as numerator and denominator, and a derivation.
// Returns: A list containing an expression g elementary over k(t) and a Boolean
//  b stating whether f could have an elementary integral over k(t).
//  If b = TRUE, f-Dg in k[t].

intlib::algebraic::residueReduce :=
proc(fNum, fDen, ts, diffs, algs)
  local t, d, p, z, R, n, s, i, j, S, integral, A, B, alpha, tmp, b, c, sq, r;
begin
  integral := 0;
  
  d := intlib::algebraic::diff(ts, diffs, algs);
  
  // normalize denominator
  t := lcoeff(fDen);
  fNum := multcoeffs(fNum, 1/t);
  fDen := multcoeffs(fDen, 1/t);

  // f must be simple, i.e., fDen must be normal
  if degree(gcd(fDen, d(fDen)))>0 then
    return([0, FALSE]);
  end_if;
  
  
  t := ts[-1];
  assert(pindets(fNum) = [t] and pindets(fDen) = [t]);
  [p, fNum] := [divide(fNum, fDen)];
  z := solvelib::getIdent(Any, indets([poly2list(fNum), poly2list(fDen), ts, diffs, algs]));
  alpha := solvelib::getIdent(Any, indets([poly2list(fNum), poly2list(fDen), ts, diffs, algs, z]));
  tmp := mapcoeffs(fNum-poly(z*d(expr(fDen)), [t]), normal);
  if traperror((
    if degree(d(fDen)) < degree(fDen) then
      R := intlib::algebraic::subresultant(fDen, tmp);
    else
      R := intlib::algebraic::subresultant(tmp, fDen);
    end_if),
    MaxSteps = intlib::subresMaxSteps(tmp, fDen)) <> 0 then
    // returning failure is one option, but if this fails once,
    // we probably get another timeut in the next loop in
    // intlib::algebraic::parametricLogarithmicDerivative, too
    // return([FAIL, FALSE]);
    // error is trapped above
    error("subresultant computation takes too long");
  end_if;
  // actually, we only need mapcoeffs(..., i::a::diff(ts, diffs)).
  // This is functionally identical, though, since z is not in ts:
  R[1] := intlib::algebraic::poly(expr(R[1]), [z]);
  // experiments show that this sometimes decreases the running time
  if not iszero(R[1]) then R[1]:= normal(multcoeffs(R[1], 1/gcd(coeff(R[1]))), Expand = FALSE) end_if;
  [n, s] := intlib::algebraic::splitSquarefreeFactor(R[1], ts, diffs, algs);
  R[2] := table((degree(r[2]) = r[2]) $ r in R[2]);
  for i from 1 to nops(n) do
    if degree(s[i]) = 0 then
      next;
    end_if;
    if i=degree(fDen) then
      S := poly(expr(fDen), [z]);
    else
      S := R[2][i];
      assert(S::dom = DOM_POLY);
      A := polylib::sqrfree(intlib::algebraic::poly(lcoeff(S), [z]), "UseAlgebraicExtension");
      A := coerce(A, DOM_LIST)[2..-1];
      S := intlib::algebraic::poly(expr(S), [z]);
      if A <> [] then
        A := map([$1..A[-1]],
            expo -> if contains(A, expo)>0 then
                      A[contains(A, expo)-1]
                    else
                      poly(1, [z])
                    end_if);
      end_if;
      for j from 1 to nops(A) do
        assert(testeq(expr(divide(S, gcd(A[j], s[i]), Rem)), 0));
        S := divide(S, gcd(A[j], s[i]), Quo);
      end_for;
    end_if;
    // reduce mod s[i]
    S := divide(S, s[i], Rem);
    
    S := mapcoeffs(S, normal);
    case degree(s[i])
    of 1 do
      tmp := -coeff(s[i], 0)/coeff(s[i], 1);
      integral := integral + tmp*ln(S(tmp));
      break;
    of 2 do
//      if degree(S) = 1 then
        tmp := multcoeffs(s[i], 1/lcoeff(s[i]));
        b := coeff(tmp, 1)/2;
        c := coeff(tmp, 0);
        sq := normal(c-b^2);
        sq := [numer(sq), denom(sq)];
        if generate::isNeg(sq[1]) then
          sq := map(sq, _negate);
        end_if;
        sq := sqrt(sq[1])/sqrt(sq[2]);
        A := S(-b+I*sq);
        B := S(-b-I*sq);
        A := poly(subs(normal(A), I=alpha), [alpha]);
        B := poly(subs(normal(B), I=alpha), [alpha]);
        if degree(A) = 1 and degree(B) = 1 and
           coeff(A, 0) = coeff(B, 0) and
           coeff(A, 1) = -coeff(B, 1) then
          B := coeff(A, 1);
          A := coeff(A, 0);
          integral := integral - b * ln(S(-b+sqrt(b^2-c)) * S(-b-sqrt(b^2-c)))
            + sq*intlib::algebraic::rational::logToArctan
              (poly(A, [ts[-1]]), poly(B, [ts[-1]]));
//            +I*sq*ln((A+I*B)/(A-I*B));
          break;
        end_if;
//      end_if;
    otherwise
      // TODO: options
      tmp := solve(expr(s[i]), z, IgnoreSpecialCases);
      if domtype(tmp) = DOM_SET then
        integral := _plus(integral, op(map(tmp, z -> z*ln(S(z)))));
      elif type(tmp) = "_union" and
         map({op(tmp)}, domtype) minus {DOM_SET, RootOf} = {} then
/*
 Note: Here, we could replace s by divide(s, <poly in br>, Rem).
*/
        integral := _plus(integral,
          op(map([op(tmp)], br -> hold(sum)(z*ln(S(z)), z in br))));
      else
        integral := integral + hold(sum)(z*ln(S(z)), z in RootOf(s[i](alpha), alpha));
      end_if;
    end_case;
  end_for;
  [integral,
   bool(_and(op(map(n, p -> degree(p)=0))))]
end:

