// symbolic integration of transcendental functions -- the Risch algorithm

// This is the general controller, distributing work all over
// intlib::algebraic. The return value is either an expression
// (in the indeterminates given) or FAIL.

intlib::algebraic::rischInt :=
proc(fNum, fDen, ts, diffs, algs)
  local integral, h, r, dt, int2, good, gc, d, p, a, c;
begin
  // some simplifications for recursive calls:
  fNum := intlib::algebraic::reduceModAlgebraics(fNum, ts, diffs, algs);
  fDen := intlib::algebraic::reduceModAlgebraics(fDen, ts, diffs, algs);
  
  if iszero(fNum) then
    return(0);
  end_if;
  
  while iszero(degree(fNum)) and iszero(degree(fDen)) and nops(ts) > 1 do
    [fNum, fDen] := intlib::algebraic::normal(expr(fNum)/expr(fDen), [ts[-2]]);
    ts    := ts[1..-2];
    diffs := diffs[1..-2];
    algs  := algs[1..-2];
  end_while;

  if nops(ts) = 1 then
    assert(diffs = [1]);
    return(intlib::algebraic::rational(expr(fNum)/expr(fDen), ts[1], table()));
  end_if;


  d := intlib::algebraic::diff(ts, diffs, algs);

  // Step 1: Hermite reduction
  [integral, h, r] := intlib::algebraic::Hermite(fNum, fDen, ts, diffs, algs);
  if integral = FAIL then return(FAIL); end_if;
  // h is simple, r is reduced.
  
  // Step 2: Polynomial reduction
  dt := poly(diffs[-1], [ts[-1]]);
  if dt=FAIL then
    // we need a derivation on k[t], sorry, folks
    return(FAIL);
  end_if;
  if degree(dt) > 1 then
    if degree(h[1])-degree(h[2]) >= degree(dt) then
      [p, a] := [divide(h[1], h[2])];
      p := mapcoeffs(p, normal, Expand = FALSE);
      p := mapcoeffs(p, intlib::algebraic::reduceModAlgebraics, ts, diffs, algs);
      [int2, p] := intlib::algebraic::polynomialReduce(p, ts, diffs, algs);
      integral := integral + int2;
      h[1] := mapcoeffs(p*h[2]+a, normal);
      gc := gcd(h[1], h[2]);
      h := map(h, intlib::algebraic::polyDivide, gc);
    end_if;
    if degree(r[1])-degree(r[2]) >= degree(dt) then
      [p, a] := [divide(r[1], r[2])];
      p := mapcoeffs(p, normal, Expand = FALSE);
      p := mapcoeffs(p, intlib::algebraic::reduceModAlgebraics, ts, diffs, algs);
      [int2, p] := intlib::algebraic::polynomialReduce(p, ts, diffs, algs);
      integral := integral + int2;
      // Theorem 5.7.2 of Bronstein, Symbolic integration I:
      c := coeff(p, degree(dt)-1)/lcoeff(dt);
      if not iszero(normal(d(c))) then return(FAIL); end_if;
      
      r[1] := mapcoeffs(p*r[2]+a, normal);
      gc := gcd(r[1], r[2]);
      r := map(r, intlib::algebraic::polyDivide, gc);
    end_if;
  end_if;

  //  Step 3: Residue criterion 
  [int2, good] := intlib::algebraic::residueReduce(h[1], h[2], ts, diffs, algs);
  if good=FALSE then return(FAIL); end_if;
  integral := integral+int2;
  h := intlib::algebraic::normal(expr(h[1])/expr(h[2])-d(int2), [ts[-1]]);
  h := map(h, intlib::algebraic::reduceModAlgebraics, ts, diffs, algs);
  h := map(h, mapcoeffs, intlib::Simplify);
  
  h := intlib::algebraic::normal(expr(h[1])/expr(h[2]), [ts[-1]]);
  assert(testeq(expr(h[2])-coeff(h[2], 0), 0, Steps=1) <> FALSE or testeq(expr(h[1]), 0, Steps=1) <> FALSE);
  h := multcoeffs(h[1], 1/coeff(h[2], 0));
  
  // r := h+r
  r[1] := r[1]+h*r[2];
  gc := gcd(r[1], r[2]);
  r := map(r, intlib::algebraic::polyDivide, gc);
  
  good := FALSE;
  if algs[-1] <> [] and algs[-1][1] = "simpleradical" then
    [int2, good] := intlib::algebraic::simpleRadicalLogPart(r[1], r[2], ts, diffs, algs);
  end_if;
  if good = FALSE then // simpleRadicalLogPart not yet complete. Fall back to transcendental code:
    case degree(dt)
    of 0 do
      // primitive
      [int2, good] := intlib::algebraic::integratePrimitive(r[1], r[2], ts, diffs, algs);
      break;
    of 1 do
      // hyperexponential
      [int2, good] := intlib::algebraic::integrateHyperexponential(r[1], r[2], ts, diffs, algs);
      break;
    of 2 do
      // hypertangent
      [int2, good] := intlib::algebraic::integrateHypertangent(r[1], r[2], ts, diffs, algs);
      break;
    otherwise
      // we know very little about this general case.
      // if we are lucky, there is nothing left to do, though:
      if degree(r[1])=0 and degree(r[2])=0 then
        [int2, good] := [0, TRUE];
      else
        // In the case where there are no special polynomials in k[t],
        // such an f is not integrable, cf. Chapter 5.11 of Bronstein.
        // In general, it might be, but we don't know how.
        [int2, good] := [0, FALSE];
      end_if;
    end_case;
  end_if;

  if good=FALSE then return(FAIL); end_if;
  integral := integral + int2;
  r := intlib::algebraic::normal(expr(r[1])/expr(r[2])-expr(d(int2)), [ts[-1]]);
  r := map(r, intlib::algebraic::reduceModAlgebraics, ts, diffs, algs);
  r := map(r, mapcoeffs, simplify);
  if iszero(r[1]) then
    r[2] := poly(1, [ts[-1]]);
  end_if;
  if not (iszero(degree(r[1])) and iszero(degree(r[2]))) then
    return(FAIL);
  end_if;
  if nops(ts) = 1 then
    return(integral + expr(r[1])/expr(r[2])*ts[-1]);
  else
    r := intlib::algebraic::normal(expr(r[1])/expr(r[2]), [ts[-2]]);
    return(integral + intlib::algebraic::rischInt(r[1], r[2], ts[1..-2], diffs[1..-2], algs[1..-2]));
  end_if;
end:
