// intlib::residueInt - definite integration via residue calculus

// currently treats
// r(x), where r(x) in O(1/x^2) for x -> +/-infinity
// exp(I*k*x)*r(x), where r(x) -> 0 for x -> +/-infinity and k in R_
// linear combinations of these
// and things rewrite(f, exp) gets into this form
//
// over x=-infinity..infinity, r(x) a rational function
//
// and
//
// r({sin,cos,tan}(n*x)) over x=a..b, (b-a)/(2*PI) in Z_.

intlib::residueInt :=
proc(f, x, a, b, options)
begin
  // delete assumption that x is real - we want residues in the complex plane
  save x;
  eval(hold(unprotect)(x));
  eval(hold(_delete)(x));
  
  if a = -infinity and b = infinity then
    return(intlib::residueInt_inf(args()));
  end_if;
  
  if is((b-a)/(2*PI) in Z_, Goal=TRUE) then
    return(intlib::residueInt_2PI(args()));
  end_if;
  
  FAIL;
end_proc:

intlib::residueInt_inf :=
proc(f, x, a, b, options)
  local r, x0, num, den, denfac, zeroes, res, exps, i, k;
begin
  if a <> -infinity or b <> infinity then
    return(FAIL);
  end_if;
  // for recursive calls
  if iszero(f) then return(0); end_if;
  // we need to find the residues in the upper plane
  if indets(f) minus Type::ConstantIdents <> {x} then
    return(FAIL);
  end_if;

  if testtype(f, Type::RatExpr(x, Type::IndepOf(x))) then
    [num, den] := intlib::algebraic::normal(f, [x]);
    if degree(num) <= degree(den) - 2 then
      // check for real poles
      if numeric::realroots(expr(den), x=-infinity..infinity) <> [] then
        if options[PrincipalValue] = TRUE then
          return(FAIL);
        else
          return(undefined);
        end_if;
      end_if;
      // ok, use formula
      zeroes := solve(den, x, MaxDegree=3);
      if domtype(zeroes) = DOM_SET then
        zeroes := select(zeroes, x0 -> Im(x0) > 0);
        denfac := polylib::sqrfree(den);
        return(2*PI*I*_plus(intlib::ratResidue(x0, num, den, denfac, x) $ x0 in zeroes));
      else
        return(FAIL);
      end_if;
    end_if;
  end_if;
  
  // check for exp*r(x), sin*r(x), cos*r(x), etc.
  r := rewrite(f, exp);
  exps := misc::subExpressions(r, "exp");
  res := FAIL;
  if nops(exps) > 0 and 
    _lazy_and(op(map(exps, e -> testtype(op(e)/(I*x), Type::Real)))) then
    exps := [op(exps)];
    r := poly(r, exps);
    if _lazy_and(degree(r) = 1,
      op(map({coeff(r)}, testtype, Type::RatExpr(x, Type::IndepOf(x))))) then
      res := intlib::residueInt(coeff(r, [0 $ nops(exps)]), x, a, b, options);
      if res = undefined or res = FAIL then
        // poles could cancel with other terms, so don't return undefined here
        return(FAIL);
      end_if;

      for i from 1 to nops(exps) do
        [num, den] := intlib::algebraic::normal(coeff(r, [0 $ i-1, 1, 0 $ nops(exps)-i]), [x]);
        k := op(exps[i])/(I*x);
        assert(testtype(k, Type::Real));
        if degree(num) < degree(den) then
          // check for real poles
          if numeric::realroots(expr(den), x=-infinity..infinity) <> [] then
            // poles could cancel with other terms, so don't return undefined here
            // example: sin(x)/x, x=-infinity..infinity
            return(FAIL);
          end_if;
          // ok, use formula
          num := poly(num(x/k), [x]);
          den := poly(den(x/k), [x]);
          zeroes := solve(den, x, MaxDegree=4);
          if domtype(zeroes) = DOM_SET then
            zeroes := select(zeroes, x0 -> Im(x0) > 0);
            denfac := polylib::sqrfree(den);
            res := res + 2*PI*I/abs(k)*_plus(intlib::ratResidue(x0, num*exp(I*x), den, denfac, x) $ x0 in zeroes);
          else
            res := FAIL;
          end_if;
        else
          res := FAIL;
        end_if;
        if res = undefined or res = FAIL then
          return(res);
        end_if;
      end_for;
    
    end_if;
  end_if;
  
  res;
end:

intlib::residueInt_2PI :=
proc(f, x, a, b, options)
  local z, z0, num, den, denfac, zeroes;
begin
  if indets(f, All) minus {x} minus Type::ConstantIdents() minus
    hold({exp, _plus, _mult, _power, sin, cos, tan, cot}) = {} then
	  f := rewrite(f, exp);
  end_if;
  if indets(f, All) minus {x} minus Type::ConstantIdents() minus 
    {hold(exp), hold(_plus), hold(_mult), hold(_power)} <> {} then
    return(FAIL);
  end_if;
  z := solvelib::getIdent(Any, indets([args()]));
  f := subs(f, 
    hold(exp) = (ex -> if has(ex, [x]) then z^normal(ex/(I*x)) else exp(ex) end),EvalChanges)/(I*z);
  if has(f, [x]) or not testtype(f, Type::RatExpr(z, Type::IndepOf(z))) then
    return(FAIL);
  end_if;
  [num, den] := intlib::algebraic::normal(f, [z]);
  // check for poles on unit circle
  if numeric::realroots(expr(den) | z = exp(I*x), x=-PI..PI) <> [] then
    return(FAIL);
  end_if;
  // ok, use formula
  zeroes := solve(den, z, MaxDegree=4);
  if domtype(zeroes) = DOM_SET then
    zeroes := select(zeroes, x0 -> abs(x0) < 1);
    denfac := polylib::sqrfree(den);
    return((b-a)*I*_plus(intlib::ratResidue(z0, num, den, denfac, z) $ z0 in zeroes));
  end_if;
  
  FAIL;
end_proc:

// residue of rational expression or exp*rational expression
intlib::ratResidue :=
proc(z0, num, den, denfac, z)
  local f, k, multiplicity, df;
begin
  denfac := coerce(denfac, DOM_LIST);
  k := 0;
  repeat
    k := k+1;
  until
    2*k > nops(denfac) 
    or 0 in interval(denfac[2*k] | z = z0)
    or testeq(denfac[2*k] | z = z0, 0) <> FALSE
  end_repeat;
  if 2*k > nops(denfac) then
    error("f seems continuous at z0");
  end_if;
  multiplicity := denfac[2*k+1];
  df := denfac[2*k];

  f := expr(num)/expr(divide(den, poly((z-z0)^multiplicity, [z]), Quo));
  df := diff(f, z$multiplicity-1) | z=z0;
  df/(multiplicity-1)!
end_proc:
