/*++
  Synopsis: changevar(f,eq) changes the variables in f according to eq.

  Examples:
  >> intlib::changevar(hold(int)(ln(x)*2, x=0..1),x=exp(t));

                     int(2 t exp(t), t = -infinity..0)
++*/

intlib::changevar :=
  proc(i,eq=FAIL,t)
    local _x,xr,xoft,u,tofx,s, foft, old_foft, skip, tmp, nowarn, unique;
  begin
    tmp := [args(3..args(0))];
    nowarn := contains(tmp, NoWarning);
    if nowarn > 0 then
      delete tmp[nowarn];
      nowarn := TRUE;
    else
      nowarn := FALSE;
    end_if;
    unique := contains(tmp, Unique);
    if unique > 0 then
      delete tmp[unique];
      unique := TRUE;
    else
      unique := FALSE;
    end_if;
    if nops(tmp) > 1 then
      error("Unexpected argument ".expr2text(tmp[2]));
    end_if;
    if nops(tmp) > 0 then
      t := tmp[1];
    else
      t := NIL;
    end_if;

    if type(eq)=DOM_FAIL then
      error("Missing equation for substitution");
    elif type(eq)<>"_equal" then
      error("Second argument must be an equation describing the substitution");
    end_if;
    if not(type(i)="int") then
      tmp := args(2..args(0));
      return(misc::maprec(i, {"int"} = (x -> intlib::changevar(x, tmp))));
    end_if;
    if has(i, hold(D)) then
       i:= rewrite(i, diff);
    end_if;
    if has(eq, hold(D)) then
       eq:= rewrite(eq, diff);
    end_if;
        
    if type((_x:=op(i,2)))="_equal" then
      [_x,xr]:=[op(_x)];
      if type(xr) <> "_range" then
        error("invalid range specification ".expr2text(xr));
      end_if;
      assumeAlso(op(xr, 1) < _x < op(xr, 2));
    end_if;
    if not has(eq,_x) then
      error("The old integration variable ".expr2text(_x).
            " is missing in the substitution equation");
    end_if;
    if t = NIL then // try to find the new variable
      t:=indets(eq) minus {_x} minus Type::ConstantIdents;
      if nops(t)<>1 then
        error("Please specify the new integration variable")
      end_if;
      t:=op(t)
    end_if;
    if not has(eq,t) then
      error("The new integration variable ".expr2text(t).
            " is missing in the substitution equation");
    end_if;

    //-----------------------------------------------------
    // Additional code by W. Oevel, 24.9.02:
    // Before inverting xoft = x(t) to tofx = t(x) via
    // solve, we look for situations
    //   int(f(x(t)))*diff(x(t), t), t) = int(f(u), u)
    // (with u = x(t)), where no inverse of x(t) is
    // needed.
    //      -->    int(f(t), t)

    skip:= TRUE; // the transformation may be of the form
    tofx:= FAIL; // x = g@@(-1)(t) instead of t = g(x).
                 // In such a case, the following will
                 // not be successful and we have to go
                 // to the branch below (the old code).
    // eq:  t = g(x)
    if lhs(eq) = t and not has(rhs(eq), t) then
       tofx := rhs(eq); // t(x) = g(x)
       skip:= FALSE;
    elif rhs(eq) = t and not has(lhs(eq), t) then
       tofx:= lhs(eq); // t(x) = g(x)
       skip:= FALSE;
    else
       tmp:=solvelib::discreteSolve(eq,t); //,PrincipalValue;
       if type(tmp) = DOM_SET and 
          tmp <> {} and
          type(op(tmp, 1)) <> RootOf then
          skip:= FALSE;
          tofx:= op(tmp,1);
       end_if;
    end_if;
    if skip = FALSE then
       foft:= op(i, 1)/diff(tofx, _x); // foft = f(x(t))/ (dg(x)/dx)
       // beware: if diff(tofx, x) = diff(g(x), x) is not 
       // cancelled in the expression above, then
       // subsex(%, tofx = t) = subsex(%, g(x) = t) contains
       // a factor 
       // subsex(something/diff(g(x), x), g(x) = t) = 
       // something/diff(t, x).
       // The next step will produce a 'division by zero'
       //   ---> use traperror
       if traperror((foft:= eval(subsex(foft, tofx = t)))) = 0 then 
         old_foft:= foft;
         if has(foft, _x) and testeq(diff(foft, _x), 0, hold(NumberOfRandomRatTests) = 2, NumberOfRandomTests=5, Steps=0) <> FALSE then
           foft:=simplify(foft);
           if foft <> old_foft and has(foft, _x) then
              // try again
              if traperror((foft:= eval(subsex(foft, tofx = t)))) = 0 then
                 foft:=simplify(foft);
              end_if;
           end_if;
         end_if;
         // The transformation was successful if the new integrand foft
         // does not contain the original integration variable _x anymore:
         if not has(foft, _x) then
            if type(op(i, 2)) = "_equal" then
               // definite intral. We need to transform the
               // boundaries x = a..b ---> t = g(a)..g(b)
           //  u:= t = 
           //      eval(subs(tofx, _x = op(i, [2, 2, 1]))) 
           //       ..   
           //      eval(subs(tofx, _x = op(i, [2, 2, 2])));   
               u:=  t = 
                    limit(tofx,_x = op(i, [2,2,1]), Right)
                     ..
                    limit(tofx,_x = op(i, [2,2,2]), Left);
            else
               u:= t:
            end_if;
            if (has(foft, sin) or has(foft, cos)) and 
               has(foft, arctan) then
               tmp:= rewrite(foft, tan):
               if not has(tmp, tan) then
                  foft:= expr(factor(tmp));
                  u:= rewrite(u, tan);
               end_if;
            end_if;
            return(hold(int)(foft, u, op(i, 3..nops(i))));
         end_if;
       end_if:
    end_if;
    //-----------------------------------------------------
    // End of additional code by W. Oevel, 24.9.02
    //-----------------------------------------------------

    //assume(t,Type::Real);
    // invert t = g(x) to x = x(t) by some solver:
    if lhs(eq) = _x then
      xoft:= rhs(eq)
    elif rhs(eq) = _x then
      xoft:= lhs(eq)
    else
      s:=solvelib::discreteSolve(eq,_x); //,PrincipalValue;
      if type(s)<>DOM_SET or s={} or type(op(s, 1)) = RootOf then
        error("Cannot solve the transformation ".expr2text(eq)." for "._x)
      end_if;
      if nops(s)>1 then
        if unique then
          error("Cannot solve the transformation ".expr2text(eq)." for "._x." uniquely");
        end_if;
        if not nowarn then
          warning("Cannot solve the transformation ".expr2text(eq)." for "._x." uniquely, result may be invalid.");
        end_if;
      end_if;
      xoft:=op(s,1);
    end_if:
    if type(op(i,2))="_equal" then
      if lhs(eq) = t then
        tofx:= rhs(eq);
      elif rhs(eq) = t then
        tofx:= lhs(eq);
      else
        tofx:=solvelib::discreteSolve(eq,t); //,PrincipalValue;
        if type(tofx) <> DOM_SET or tofx = {} then
          error("Cannot solve the transformation ".expr2text(eq)." for ".t)
        end_if;
        if nops(tofx)>1 then
          if unique then
            error("Cannot solve the transformation ".expr2text(eq)." for ".t." uniquely");
          end_if;
          if not nowarn then
            warning("Cannot solve the transformation ".expr2text(eq)." for ".t." uniquely, result may be invalid.");
          end_if;
        end_if;
        tofx:=op(tofx,1);
      end_if;
      // The new integration interval for the new integration variable t:
      u:=  t = limit(tofx,_x = op(i, [2,2,1]), Right)
                ..
               limit(tofx,_x = op(i, [2,2,2]), Left);
    else u:= t;
    end_if;
    if tofx = FAIL then
      foft:=eval(subsex(op(i,1), _x=xoft));
    else
      foft:=eval(subsex(op(i,1), [tofx=t, _x=xoft]));
    end_if;
    foft:=simplify(foft*diff(xoft, t));
    if (has(foft, sin) or has(foft, cos)) and 
       has(foft, arctan) then
       tmp:= rewrite(foft, tan):
       if not has(tmp, tan) then
          foft:= expr(factor(tmp));
       end_if;
    end_if;
    hold(int)(foft, u, op(i, 3..nops(i))):
  end_proc:
  
// internal routine, with an outer counter for the depth
proc()
  option escape;
  local depth, seen;
begin
intlib::tryChangeVar :=
proc(i, to_subs)
  local t, i2, er, dis, pe, x;
begin
  // to bail out from pattern lookup, throw an error
  if depth > 10 then 
    // make sure we really have a recursion this deep
    depth := 0;
    pe := _act_proc_env();
    while pe <> NIL do
      if op(pe, 4) = procname then
        depth := depth + 1;
      end_if;
      pe := op(pe, 3);
    end_while;
    if depth > 10 then
      error("deep recursion"); 
    end_if;
  end_if;
  if contains(seen, [i, to_subs]) then error("bad loop"); end_if;
  depth := depth+1;
  seen[[i, to_subs]] := TRUE;
  if type(op(i, 2))= "_equal" then
    x:= op(i, [2, 1])
  else
    x:= op(i, 2)
  end_if;
  er :=
  traperror
  ((i := matchlib::unblock(subs(i, [hold(hold(intlib::intfn))=hold(int),
                                    hold(intlib::intfn)=hold(int)]));
    t := genident("cv");
    i2 := intlib::changevar(i, t=to_subs, t, Unique)));
  delete seen[[i, to_subs]];
  depth := depth - 1;
  if er <> 0 then lasterror(); end_if;
  if has(i2, [x]) then
    error("insufficient substitution")
  end_if;
  dis := discont(to_subs, x);
  if dis <> {} then
    error("bad substitution")
  end_if;
  assume(t=to_subs);
  if contains({args()}, "lookupOnly") then
    if type(op(i, 2))="_equal" then
      i2 := intlib::definite::lookup(subs(op(i2, 1), [t=_X_, _X_=t]),
        op(i2, [2,2,1]), op(i2, [2,2,2]), table());
      i2 := subs(i2, [t=_X_, _X_=t]);
    else
      i2 := subs(i2, hold(int)=hold(intlib::lookup),EvalChanges);
    end_if;
  else
    i2 := eval(i2);
  end_if;
  if i2=FAIL then
    // pattern failure: do not return FAIL, raise an error!
    error("");
  end_if;
  if not hastype(i2, "int") then
    subs(i2, t=to_subs);
  else
    // pattern failure: do not return FAIL, raise an error!
    error("");
  end_if;
end_proc:
  
  depth := 0;
  seen := table();
end_proc():
  
// end of file 
