/*++ ---------------- simpCInt.mu ---------------------
Description:
This file contains functions to simplify complex elementary functions.
							  
Functions: 

 - used Parameter:
    ++ f,A,B :DOM_EXPR
    ++ x     :DOM_IDENT				  
    ++ f     = (complex) elementary function
    ++ A,B   = polynomial expressions in x over a field of characteristic zero
               not containing sqrt(-1)
    ++ x     = indeterminate of f					  

 - intlib::applyEulerIdentity(f,x <, Complete >)
    ++ applies Euler's indentity to all subexpression of f of type "_power"
    ++ which have complex exponents and containing x in the basis as well as
    ++ all subexpressions of type "exp" with complex argument.
    ++ If option Complete is choosen, subexpressions of f of type "_power" with
    ++ arbitrary basis are treated.
 - intlib::complexLnConversion(f,x)
    ++ tries to convert sums of complex logarithms to real functions using
    ++ Rioboo's conversion procedure intlib::lnToArctan and some special
    ++ simplifications.
 - intlib::expandSum(f,x)
    ++ tries to expand any f of type "sum".
 - intlib::hasLnExp(f)
    ++ tests if the function f logarithms containing exponentials 
    ++ and returns TRUE if so, otherwise FALSE.
 - intlib::hasNegativeSquareRoots(f)
    ++ tests if the function f contains a square root with a negative radicand
    ++ and returns TRUE if so, otherwise FALSE.
 - intlib::lnToArctan(A,B,x)
    ++ returns a sum f of arctangents of polynomials over the same coefficient
    ++ field as A and B, such that
    ++                diff(f,x) = diff(I*ln((A+I*B)/(A-I*B)),x)
    ++ where is needed B<>0 and sqrt(-1) is not contained in the coefficient
    ++ field [1, p.63].
 - intlib::removeNegativeSquareRoots(f)
    ++ tries to find and remove (using rectform) square roots with negative
    ++ radicands contained in the function f.
 - intlib::simplifyComplexIntegral(f,x)
 - intlib::simplifyComplexIntegral(f <,x> <,Unique>)
    ++ tries to rewrite an elementary function which contains the complex I
    ++ into an equivalent real form and returns if possible the real part of
    ++ that expression. If intlib::simplifyComplexIntegral is called not from
    ++ within intlib::int and x has no property, x is assumed to be real.
    ++ If option Unique is given, the imaginary part 
    ++ remains in the expression.
    
See:
  [1] Bronstein, M. (1997). Symbolic Integration I: Transcendental Functions.
      Springer.
++*/

intlib::hasLnExp:=
  proc(a /*: Type::Arithmetical */)
    local res, findLnExp;
  begin
    findLnExp:=
      proc()
        local f;
      begin
        f:=op(args(),1);
        if has(f,exp) then
          res:=misc::breakmap();
        end_if;
        ln(f)
      end_proc;
          
    res:=FALSE;
    misc::maprec(a, {"ln"}=findLnExp, PostMap);
    res
  end_proc:

intlib::hasNegativeSquareRoots:=
  proc(a /*: Type::Arithmetical */)
    local res,findNegSqRoot;
  begin
    findNegSqRoot:=
      proc()
        local basis,exponent,den;
      begin
        basis:=op(args(),1);
        exponent:=op(args(),2);
        den:=denom(exponent);
        if domtype(den)=DOM_INT and (den mod 2)=0 and stdlib::hasmsign(basis) then
          // contains a square root with negative sign
          res:=misc::breakmap()
   	  end_if;
        _power(basis,exponent)
      end_proc;
        
    res:=FALSE;
    misc::maprec(a, {"_power"}=findNegSqRoot);
    res
  end_proc:

intlib::removeNegativeSquareRoots:=
  proc(a /*: Type::Arithmetical */)
    local remNegSqRoot;
  begin
    remNegSqRoot:=
      proc()
        local basis,exponent,den,res;
      begin
        basis:=op(args(),1);
        exponent:=op(args(),2);
        den:=denom(exponent);
        if domtype(den)=DOM_INT and (den mod 2)=0 and stdlib::hasmsign(basis) then
          // contains a square root with negative sign
          res:=NIL;
          if traperror((res:=expr(rectform(_power(basis,exponent)))),
                       MaxSteps=intlib::simplifyMaxSteps(a, args(), rectform))=0
             and not( has(res,{Re,Im,abs,sign,arg}) ) then
            return(res)
          end_if;
        end_if;
        _power(basis,exponent);
      end_proc;
        
    misc::maprec(a, {"_power"}=remNegSqRoot,PreMap);
  end_proc:

intlib::simplifyComplexIntegral :=
  proc(integral,x=FAIL,o=" ") // Option: Unique 
    local i, deleteProp, F; // ,ImPart;
  begin
    if has(integral, exp) then
      i:=rewrite(combine(integral,exp),sincos);
      if not has(i,I) and not intlib::hasNegativeSquareRoots(i) then return(i) end_if;
    end_if;
    //F:=integral;
    if x=FAIL then
      x:=null();
    elif x=hold(Unique) then
      o:=x;
      x:=null();
    end_if;
    // if this method is called from within intlib::antiderivative :
    if not calledFromWithinDefInt = hold(calledFromWithinDefInt) then
      o:=hold(Unique)
    end_if;
    // if this method is called not from within intlib::int :
    deleteProp:=FALSE;
    if intlibDepthLevel = hold(intlibDepthLevel) then
      // to prevent an error set option Unique in case of ln(exp(..)..)
      if intlib::hasLnExp(integral) then o:=hold(Unique); end_if;
      // x should be assumed to be real, if possible
      if x<>null() and getprop(x)=x then
        assume(x,Type::Real);
        deleteProp:=TRUE;
      end_if;
    end_if;

    if intlib::hasNegativeSquareRoots(integral) then
      userinfo(6,"...try to remove negative square roots");
      integral:=intlib::removeNegativeSquareRoots(integral);
    end_if;
    userinfo(6,"...apply Euler's identity");
    integral:=intlib::applyEulerIdentity(expand(integral),x);
    F:=integral;
    if has(integral, I) then
      if has(integral, ln) then
        //this is needed to prevent choosing a wrong branch in rectform
        userinfo(6,"...try to convert complex logarithms to real functions");
        integral:=intlib::complexLnConversion(integral,x);
        F:=integral;
      end_if;
      
      if has(integral, I) then
	userinfo(6,"...apply rectform");
	i:=NIL;
	// traperror, since there might be problems within rectform
	if traperror((i:= rectform(integral)),
		     MaxSteps=intlib::simplifyMaxSteps(integral, integral,
						       rectform))=0 then
	  // try to get only the real part 
	  userinfo(10,"    ...test, if rectform worked as needed");
	  //ImPart:=Im(i);
	  if iszero(op(i,3)) then
	    if o<>hold(Unique) then
	      integral:=op(i,1); // take only real part
	    else
	      integral:=expr(i);
	    end_if;
	  end_if;
	end_if;
      end_if;
      
    end_if;
    
    // remove unneccessary additive constants
	 if type(integral)="_plus" then
		integral := select(integral, has, x);
	 end_if;

    // if this method is called not from within intlib::int
    // remove possibly given property from x
    if deleteProp then unassume(x);
    end_if;
    if not has(integral,{Re,Im,abs,sign,arg}) then
      integral
    else
      F
    end_if;
  end_proc:

solvelib::setPreferredIdents("intlib::I", [II, III]):
solvelib::setPreferredPrefix("intlib::I", hold(`I`)):

intlib::applyEulerIdentity :=
  proc(f,x,o=" ") local rationalizeFraction,subsExp,subsPower,J,total,f0;
  begin
    J:=solvelib::getIdent("intlib::I", indets(f) union {x});
    total:=bool(o=hold(Complete));
    
    rationalizeFraction:=  //remove complex unit from denominator
    proc(e,J)
      local num,den,a,b;
    begin
      den:=subs(expand(subs(denom(e),J=I,EvalChanges)),I=J); //doing simplifications
      a:=coeff(den,[J],J,0);
      b:=coeff(den,[J],J,1);
      if a<>FAIL and b<>FAIL and b<>0 then
        num:=subs(expand(subs(numer(e)*(a-b*J),J=I,EvalChanges)),I=J);
        collect(num/(a^2+b^2),[J])
      else
        e
      end_if
    end_proc:
         
    subsPower:=
    proc()
      local basis,exponent,a,b;
    begin
      basis:=op(args(),1);
      exponent:=op(args(),2);
      if (has(basis,x) or total) and has(exponent,J) then
      //if has(basis,x) and has(exponent,J) then
        exponent:=rationalizeFraction(exponent,J);
	a:=coeff(exponent,[J],J,0);
	b:=coeff(exponent,[J],J,1);
        if a<>FAIL and b<>FAIL then
          _mult(basis^a,cos(b*ln(basis))+J*sin(b*ln(basis)))
        else
          _power(basis,exponent)
        end_if;
      else
	_power(basis,exponent)
      end_if;
    end_proc;

    subsExp:=
    proc()
      local exponent,a,b,d,k,l;
    begin
      exponent:=op(args(),1);
     // if has(exponent,J) and has(exponent,x) then
     if has(exponent,J)
        // could be no poly in J! (BUG 1432)
        and poly(exponent, [J]) <> FAIL then
        exponent:=rationalizeFraction(exponent,J);

	//---------------------------------------------------
	// Thilo: 25/07/2002
	// exponent can have exponents of J > 1 
	//---------------------------------------------------
	
	exponent:=expand(exponent);
	d:=degree(exponent,[J],J);
	for k from 4 to d do
		l:= k mod 4;
		exponent:=subs(exponent,J^k=J^l);
	end_for;
	exponent:=subs(exponent,J^2=-1);
        exponent:=subs(exponent,J^3=-J);

	//---------------------------------------------------

	a:=coeff(exponent,[J],J,0);
	b:=coeff(exponent,[J],J,1);
        if a<>FAIL and b<>FAIL then
	  _mult(exp(a),cos(b)+J*sin(b))
        else
          exp(exponent)
        end_if
      else
	exp(exponent)
      end_if;
    end_proc;
    f:=subs(misc::maprec(subs(f,I=J),{"_power"}=subsPower),J=I,EvalChanges);
    if has(f, exp) then
      f:=subs(misc::maprec(subs(f,I=J),{"exp"}=subsExp),J=I,EvalChanges);
    end_if;
    delete subsPower, subsExp;
    //if has(f,I) then simplify(f) else f end_if;
    if has(f,I) then
       // traperror, since there might be problems within simplification
      if traperror((f0:=simplify(f)),
		    MaxSteps=intlib::simplifyMaxSteps(f, f, I))=0 then
        f0
      else f end_if;
    else f end_if;
  end_proc:

/*----------------------------------------------------------------
Note by Walter: the following identities hold
For any complex x one has:   arctan(x) = I/2* ( ln(1-I*x) - ln(1+I*x) )
For any complex x
    with Im(x) < 1 one has:    ln(I-x) = ln(x-I) + I*PI
    with Im(x) > 1 one has:    ln(I-x) = ln(x-I) - I*PI
For any   real x   one has:  arctan(x) =  I/2* ( ln(1-I*x) - ln(1+I*x) )
                                       =  I/2*  ln( (1-I*x)/(1+I*x) )
                                       =  I/2*  ln( (I + x)/(I - x) )
                                       =  I/2* (ln(I+x) - ln(I-x) )
                      arctan(x) - PI/2 =  I/2* (ln(x+I) - ln(x-I) )
                                       <> I/2*  ln( (x+I)/(x-I) )
                       -arctan(1/x)    =  I/2*  ln( (x+I)/(x-I) )

Note:  ln(-z) = ln(z) + signIm(-z)*PI*I
              = ln(z) - signIm( z)*PI*I

Assume A/B is real:

For real A > 0:
   I/2*( ln(A+I*B) - ln(A-I*B) ) 
=  I/2*( ln(A*(1+I*B/A)) - ln(A*(1-I*B/A)) ) 
=  I/2*( ln(A) + ln(1+I*B/A) - ln(A) - ln(1-I*B/A)) ) 
=  I/2*( ln(1+I*B/A) - ln(1-I*B/A)) ) = arctan(-B/A) (#a)

For real A < 0:
   I/2*( ln(A+I*B) - ln(A-I*B) ) 
=  I/2*( ln(-(-A)*(1+I*B/A)) - ln(-(-A)*(1-I*B/A)) ) 
=  I/2*( ln((-A)*(1+I*B/A)) - signIm((-A)*(1+I*B/A))*PI*I
        -ln((-A)*(1-I*B/A)) + signIm((-A)*(1-I*B/A))*PI*I  ) 
=  I/2*( ln(1+I*B/A) + ln(-A) - signIm(1+I*B/A)*PI*I
        -ln(1-I*B/A) - ln(-A) + signIm(1-I*B/A)*PI*I ) 
=  I/2*( ln(1+I*B/A) + ln(-A) - signIm(1+I*B/A)*PI*I
        -ln(1-I*B/A) - ln(-A) + signIm(1-I*B/A)*PI*I ) 
=  I/2*( ln(1+I*B/A) - sign(B/A)*PI*I
        -ln(1-I*B/A) - sign(B/A)*PI*I ) 
=  I/2*( ln(1+I*B/A) -ln(1-I*B/A) ) + sign(B/A)*PI
=  I/2*( ln(1+I*B/A) -ln(1-I*B/A) ) - sign(B)*PI
=  arctan(-B/A) - sign(B)*PI

 +-------------------------------------------------------------
 | For A > 0:                                                 |
 |  I/2*( ln(A+I*B) - ln(A-I*B) ) = arctan(-B/A)              |
 | For A < 0:                                                 |
 |  I/2*( ln(A+I*B) - ln(A-I*B) ) = arctan(-B/A) - sign(B)*PI |
 +-------------------------------------------------------------

For any real or complex A and real or complex B:
   I/2* ln(   (A + I*B)/(A - I*B) )
=  I/2* ln( (1 + I*B/A)/(1 - I*B/A) ) = arctan(-B/A)  (#)
=  I/2* ln(   (A/B + I)/(A/B - I) )   = -arctan(B/A)  (#)
=  I/2* ln(-(1 - I*A/B)/(1 + I*A/B) ) 

For real B > 0:
   I/2*( ln(A+I*B) - ln(A-I*B) ) 
=  I/2*( ln((A/B+I)*B) - ln((A/B-I)*B) ) 
=  I/2*( ln(A/B+I)+ln(B) - ln(A/B-I) -ln(B) ) 
=  I/2*( ln(A/B+I) - ln(A/B-I) ) = arctan(A/B) - PI/2 (##a)

For real B < 0:
   I/2*( ln(A+I*B) - ln(A-I*B) ) 
=  I/2*( ln((A/B+I)*B) - ln((A/B-I)*B) ) 
=  I/2*( ln(-(A/B+I)*(-B)) - ln(-(A/B-I)*(-B)) ) 
=  I/2*( ln((A/B+I)*(-B)) + signIm(-(A/B+I)*(-B))*PI*I
        -ln((A/B-I)*(-B)) - signIm(-(A/B-I)*(-B))*PI*I  )
=  I/2*( ln(A/B+I) + ln(-B) - signIm(A/B+I)*PI*I
        -ln(A/B-I) - ln(-B) + signIm(A/B-I)*PI*I  )
=  I/2*( ln(A/B+I) - PI*I
        -ln(A/B-I) - PI*I  )
=  I/2*( ln(A/B+I) -ln(A/B-I) )  + PI
=  arctan(A/B) - PI/2 + PI =  arctan(A/B) + PI/2 (##b)

 +-----------------------------------------------------------+
 | For any real A, B, B <> 0:                                |
 | I/2*( ln(A+I*B) - ln(A-I*B) ) = arctan(A/B) -sign(B)*PI/2 |
 +-----------------------------------------------------------+

In lnToArctan(A, B), modifications of the identities 
(#) and (##) are used ?!?

Note by ccr: what is done here is to return f such that
diff(f, x) = diff(I*ln((A+I*B)/(A-I*B)), x).
----------------------------------------------------------------*/

intlib::lnToArctan:=
  proc(A,B,x)
    local sB, g;
  begin
    // --------------------------------------------------
    // Walter 5.2.02: added the following processing of
    // general terms (not necessarily polys in the
    // integration variable x), based on the identities
    // above. Note that both A and B are assumed to be real.
    // ------------------------------------------------
    sB:= sign(B);
    case sB
    of 1 do  // B is real and positive:
       return(2*arctan(A/B) - PI);
    of -1 do // B is real and negative:
       return(2*arctan(A/B) + PI);
    end_case;
    case sign(A)
    of 1 do   // A is real and positive:
       return(2*arctan(-B/A));
    of - 1 do // A is real and negative
       return(2*arctan(-B/A) - sign(B)*2*PI);
    end_case;

    // ------------------------------------------------
    // Ok, we did our best for a *clean* rewriting of 
    // sums of logarithms to arctan. The stuff further
    // down below seems to be just a heuristics for the
    // special (but important) case of A, B = polynomials
    // in the integration variable.
    // ------------------------------------------------

    if (not testtype(A, Type::PolyExpr(x))) or
       (not testtype(B, Type::PolyExpr(x))) then
       return(I*( ln(A+I*B) - ln(A-I*B)));
    end_if;

    // ------------------------------------------------
    // Rioboo's algorithm for polynomials
    // ------------------------------------------------
    if divide(A,B,[x],Rem)=0 then
      // ------------------------------------------------
      // This is Winfried's original return value
      // ------------------------------------------------
      return(2*arctan(A/B));
      // ------------------------------------------------
      // Walter 2.2.02: 
      // According to Winfried, the return value f, say,
      // should satisfy
      // diff(f,x) = diff(I*ln((A+I*B)/(A-I*B)),x)
      // This is wrong thinking: if the result is something
      // like  x*arctan(something), then it does *not*
      // suffice to replace ln by arctan *up to an additive constant*
      // in order to obtain x*arctan(..) = x*ln(..) + constant.
      // The constant must be correct, too!
      // ** Assuming A/B to be real **, the following
      // seems to be the correct answer:
      // return(2*arctan(A/B) - sign(B)*PI/2);
      // ------------------------------------------------
    elif degree(A,[x],x)<degree(B,[x],x) then
      // This is Winfried's original return value:
      return(intlib::lnToArctan(-B,A,x));
    else
      // This is Winfried's original return value:
      g:=gcdex(B,-A,x);
      return(2*arctan(normal((A*g[2]+B*g[3])/g[1]))
             +intlib::lnToArctan(g[2],g[3],x)
      );
    end_if;
  end_proc:

intlib::complexLnConversion:=
  proc(s,x)
    local J,sl,splitLnParts,lnPart,pos;
  begin
    assert(args(0)=2);
    // s - expression
    // J - variable representing the complex unit I
    // splits s into parts consisting of ln's of the form c*ln(A+I*B) and
    // returns a list of 3 lists: the first list includes all such ln's where
    // c is imaginary, the third list where c is real part and the second list
    // includes all other subexpressions.
    // Note: The third list could removed, when the simplifier will improved!
    splitLnParts:=proc(s,J)
                    local r,sl;
                  begin
                    r:=[[],[],[]];
                    map([op(expand(s))],
                        proc(e)
                          local f,A,B,h,cRe,cIm;
                        begin
                          if testtype(e,"_mult") then 
                           sl:=split([op(e)], testtype, "ln")
                          else
                            r[2]:=append(r[2],e);
                            return(r)
                          end_if;
                           case nops(sl[1])
                             of 1 do 
                               f:=_mult(op(sl[2]));
                               cRe:=coeff(f,[J],J,0);
                               cIm:=coeff(f,[J],J,1);
                               h:=op(op(sl[1]));
                               A:=coeff(h,[J],J,0);
                               B:=coeff(h,[J],J,1);
                               //-----------------------------------------
                               // Walter 6.2.02: Added the following 2 lines.
                               // Note that ln::expand(y) does not return 
                               // ln(expand(y)), but ln(normal(expand(y)))
                               // argument
                               //-----------------------------------------
                               A:= expand(A):
                               B:= expand(B):
                               if /*A=0 or*/ B=0 //or cIm=0
                                 //Note: the following two conditions are
                                 //due to the fact that e.g. poly(1/x,[x])
                                 //returns FAIL and thus would cause an error
                                 //in intlib::lnToArctan
                               /*
                                  //-----------------------------------------
                                  // Walter 6.2.02: deactivated the following 2 lines.
                                  // Non-Polys are now handled in lnToArctan
                                  //-----------------------------------------
                                  or not(testtype(A,Type::PolyExpr(x)))
                                  or not(testtype(B,Type::PolyExpr(x)))
                               */
                                  //-----------------------------------------
                                  // bij 24.4.03: inserted the following line.
                                  // Patch for errors caused by the upper deactivation
                                  // coeff returns FAIL possibly
                                  //-----------------------------------------
                                  or A = FAIL or B = FAIL
                                 then r[2]:=append(r[2],e); break;
                               else
                                 if cRe<>0 then
                                  r[3]:=append(r[3],table("A"=A,"B"=B,"c"=cRe))
                                 end_if;
                                 if cIm<>0 then 
                                  r[1]:=append(r[1],table("A"=A,"B"=B,"c"=cIm))
                                 end_if;
                               end_if; break;
                             otherwise r[2]:=append(r[2],e)
                           end_case
                        end_proc
                        );
                    r
                  end_proc:


    // it makes no sense to split the operands of sums and integrals
    if type(s) in {"sum", "product", "int"} then
      return(eval(subsop(s, 1 = intlib::complexLnConversion(op(s, 1), x))))
    end_if;
    
    J:=solvelib::getIdent("intlib::I", indets(s) union {x});
    lnPart:=0;
    sl:=splitLnParts(subs(s,I=J),J);
    if sl[1]=[] and sl[3]=[] then return(s)
    else
      // handle first list
      while sl[1]<>[] do
        pos:=contains(sl[1],table("A"=sl[1][1]["A"],"B"=-sl[1][1]["B"],
                                  "c"=-sl[1][1]["c"]));
        if pos>0 then
          if sl[1][1]["c"]=abs(sl[1][1]["c"]) then
            lnPart:=lnPart+sl[1][1]["c"]*
                     intlib::lnToArctan(sl[1][1]["A"],sl[1][1]["B"],x);
          else
            lnPart:=lnPart+sl[1][pos]["c"]*
                     intlib::lnToArctan(sl[1][pos]["A"],sl[1][pos]["B"],x);
          end_if;
          delete sl[1][pos];
          delete sl[1][1];
        else
          sl[2]:=append(sl[2],
                        sl[1][1]["c"]*I*ln(sl[1][1]["A"]+sl[1][1]["B"]*I));
          delete sl[1][1];
        end_if
      end_while;
      // handle third list
      while sl[3]<>[] do
        pos:=contains(sl[3],table("A"=sl[3][1]["A"],"B"=-sl[3][1]["B"],
                                  "c"=sl[3][1]["c"]));
        if pos>0 then
          lnPart:=lnPart+sl[3][1]["c"]*ln(sl[3][1]["A"]^2+sl[3][1]["B"]^2);
          delete sl[3][pos];
          delete sl[3][1];
        else
          sl[2]:=append(sl[2],
                        sl[3][1]["c"]*ln(sl[3][1]["A"]+sl[3][1]["B"]*I));
          delete sl[3][1];
        end_if
      end_while;
      if sl[2]=[] then lnPart
      else subs(_plus(lnPart,op(sl[2])),J=I)
      end_if
    end_if
  end_proc:

intlib::expandSum:=
  proc(f,x)
    local g,s,e;
  begin
    g:=f;
    if type(g)="sum" then
      g:=expand(g);
      if type(g)<>"sum" then return(map(g,intlib::expandSum,x)) end_if;
      if testtype(op(g, [2, 2]), RootOf) then
        s:=solvelib::discreteSolve(op(op(g,[2,2])),MaxDegree=4);
        if domtype(s)=DOM_SET and not has(s,RootOf) then
          g:=_plus(subs(op(g,1),op(g,[2,1])=e,EvalChanges) $ e in s);
        end_if;
      end_if;
    end_if;
    g;
  end_proc:
