// linalg::SSS -- solve a system a linear equations with 
//                coeffcient matrix A und right-hand-side b.
//
// Walter August 09: extended to rhs = matrix b with several columns

linalg::SSS:= proc(A, b) 
        local m, n, Mat, T, Rone, charind, rank, det, kernelCols, col, 
              dets, detA, sig, R, RHS, Rzero, addRows, detMult, f0, f1, 
              i, ind, j, kernelColVectors, l, nb, B, usepoly;
      begin 
        if args(0) < 2 then 
          error("expecting two arguments") 
        end_if;
        Mat:= A::dom;
        if Mat::hasProp(Cat::Matrix) <> TRUE then
          error("expecting a matrix of 'Cat::Matrix'");
        end_if;
        if Mat <> b::dom then 
          error("the 2nd argument must be of the same type as the 1st argument");
        end_if;
        if A::dom::matdim(A)[1] <> b::dom::matdim(b)[1] then 
          error("illegal operands");
        end_if;
        R:= A::dom::coeffRing;
        Rone:= R::one;
        Rzero:= R::zero;
        if Mat::constructor <> Dom::Matrix then 
          //==================================================
          // Convert the matrix into a matrix over Dom::Matrix
          //==================================================
          A:= Dom::Matrix(R)(A);
          b:= Dom::Matrix(R)(b);
        end_if;
        [m, n]:= A::dom::matdim(A):
        nb:= b::dom::matdim(b)[2];
        usepoly:= has([args()], "UsePoly"); 
        if usepoly then
           [T, rank, detA, charind, sig]:= linalg::ffG(A::dom::concatMatrix(A,b), "UsePoly");
        else
           [T, rank, detA, charind, sig]:= linalg::ffG(A::dom::concatMatrix(A,b));
        end_if:

        // linalg::linsolve turns the matrix entries into polynomials
        // and calls linalg::SSS with the option "UsePoly". This
        // allows to avoid normal in the call of linalg::ffG above.
        // However, before doing the backsubstitution further down below,
        // we better get rid of the poly containers:
        if usepoly then
           T:= map(T, op, 1):
           detA:= op(detA, 1);
        end_if:
        kernelCols:= {};
        //==============================================
        // Does there exist any solution at all? 
        // Check whether rank(A,B) = rank(A). 
        //==============================================
        if rank > 0 then
          for j from rank to n do
            if not iszero( T[rank,j] ) then 
              break;
            end_if
          end_for;
          if j > n then 
            return([]);
          end_if
        end_if;
        //==============================================
        // Positioning of the characteristic columns 
        // "along the main diagonal". Insert -1 on the 
        // main diagonal in zero rows. 
        //==============================================
        if m < n then 
          addRows:= Dom::Matrix(R)(n-m, n + nb);
          T:= T::dom::stackMatrix(T,addRows);
        end_if;
        for i from 1 to n do
          if iszero(T[i,i]) then 
            for l from i+1 to n do 
              if not iszero(T[i,l]) then 
                T:= T::dom::swapRow(T, i, l);
              end_if;
            end_for;
            T[i,i]:= Rone;
            kernelCols:= kernelCols union {i};
          end_if;
        end_for;
        //==============================================
        // Remove zero rows at the end of the matrix
        //==============================================
        for i from min(m,n) + 1 to m do  
          T:= T::dom::delRow(T, i);
        end_for;
        //=========================================
        // Compute the determinant of the matrix A
        //=========================================
        detA:= Rone; 
        for i from 1 to n do
          detA:= detA * T[i,i];
        end_for;
        //================================================
        // Now we need a special solution of the system.
        // This solution is computed via Cramer's rule.
        //================================================
        [T, B]:= [T[1..n, 1..n], T[1..n, n+1..n+nb]];
        for j from 1 to nb do 
          b:= B::dom::col(B,j);
          dets:= [Rzero $ n];
          det:= Rone;
          detMult:= Rone;
          //================================================
          // Note that in the case b = 0 we do not really 
          // have to compute all determinants. We simply
          // define them to be zero. 
          //================================================
          if not iszero(b) then 
            A:= T::dom::setCol(T, n, b);
            for i from 1 to n do 
              det:= det * A[i,i];          
            end_for;
            dets[n]:= det; 
            //================================================
            // else dets[n]:= Rzero; which is actually the 
            // the case already. 
            //================================================
          end_if;
          detMult:= Rone;
          if not iszero(b) then 
            for col from n-1 downto 1 do 
              A:= T::dom::setCol(T, col, b);
              det:= Rone;
              //============================
              // We are in the i-th row and 
              // the col-th column. 
              //============================
              if not iszero(A[col + 1,col]) then 
                f0:= A[col+1,col+1];
                f1:= -A[col+1,col];
                detMult:= detMult/f0;
                //==================================================
                // addCol(A,col,col+1,A[col+1,col+1],-A[col+1,col]);
                //==================================================
                for i from col downto 1 do
                  A[i,col]:= f0 * A[i,col] + f1 * A[i,col+1];
                end_for;
                A[col+1,col]:= Rzero;
              end_if;
              //==============================================
              // Now A should be in upper triangular form. 
              // We compute the determinant as the product 
              // of the main diagonal entries.
              //==============================================
              det:= _mult(A[i,i] $ i = 1..n) * detMult;
              //det:= _mult(A[i,i] $ i = 1..col) * det;
              b:= A::dom::col(A, col);
              dets[col]:= det;
            end_for;
            //================================================
            // else dets[col]:= Rzero; which is actually the 
            // the case already. 
            //================================================
          end_if;
          //==============================================
          // Compute a special solution of the system of 
          // linear equations. 
          //==============================================
          dets:= map(dets, ind -> ind/detA);
          b:= Dom::Matrix(R)(dets);
          B:= B::dom::setCol(B, j, b);
        end_for: // compute a solution for rhs = i-th column ob b

        //==============================================
        // Compute the kernel of the matrix A: 
        // kernelCols contains the indices of the  
        // columns, where we inserted 'Rone' on the 
        // main diagonal. 
        //================================================
        kernelColVectors:= [];
        for ind in kernelCols do 
          RHS:= Dom::Matrix(R)(n,1);
          RHS[ind,1]:= Rone; 
          det:= Rone;
          if ind = n then 
            A:= T::dom::setCol(T, n, RHS);
            det:= _mult(A[i,i] $ i = 1..n);
            dets[n]:= det; 
          else
            (dets[i]:= Rzero) $ i = ind + 1..n; 
          end_if;
          detMult:= Rone;
          for col from min(ind, n-1) downto 1 do 
            A:= T::dom::setCol(T, col, RHS);
            det:= Rone;
            //=========================================
            // We are in the i-th row and 
            // the col-th column. 
            //=========================================
            if not iszero(A[col + 1,col]) then 
              f0:= A[col+1,col+1];
              f1:= -A[col+1,col];
              detMult:= detMult/f0;
              //==================================================
              // addCol(A,col,col+1,A[col+1,col+1],-A[col+1,col]);
              //==================================================
              for i from col downto 1 do
                A[i,col]:= f0 * A[i,col] + f1 * A[i,col+1];
              end_for;
              A[col+1,col]:= Rzero;
            end_if;
            //==============================================
            // Now A is in upper triangular form. 
            // We compute the determinant as the product 
            // of the main diagonal entries.
            //==============================================
            det:= _mult(A[i,i] $ i = 1..n) * detMult;
          //det:= _mult(A[i,i] $ i = 1..col) * detMult;
            RHS:= A::dom::col(A, col);
            dets[col]:= det;
          end_for;
          dets:= map(dets, ind -> ind/detA);
          kernelColVectors:= append(kernelColVectors, Mat(dets));
        end_for;

        return([B, kernelColVectors, kernelCols]);
     end_proc:
