/* --------------------------------------------------------------- 
  SSS2 -- a special procedure to solve systems of linear equations 
           with many symbolic coefficients   
             
   PARAMETERS: 

   -- M, the coefficient matrix of a system a linear equations
   -- b, the right-hand-side of the system of linear equations

   It returns a list [v, [v1, v2, ...]], where v is a special
   solution of the system of linear equations M * x = b and 
   v1, v2, ...  span the kernel of the coeffcient matrix M, 
   i.e., M * v1 = 0, M * v2 = 0, ...

   REFERENCE:
             
   Efficient Gaussian Elimination Method for Symbolic
   Determinants and Linear Systems, T. Sasaki, H. Murao, 
   ACM Transactions on Mathematical Software, Vol. 8, 
   No. 3, September 1982, Pages 228-289. 
--------------------------------------------------------------- */

linalg::SSS2:= proc(M, b) 
        local m, n, nm2, A, X1X2ETC, R, Q, prod, sig, k, i, j, Ring,
              Mat, myexpand, RHS, Rone, Rzero, addRows, col, found, ind, 
              kernelCols, l, rank, det, detM, detMult, dets, f0, f1, 
              kernelColVectors, MM, isKernelCol, ncols, transformMatrix;
      begin
        Mat:= M::dom;

      //myexpand:= expand:
        myexpand:= b -> normal(b, Expand = FALSE):

     // M:= map(M, myexpand):
        
        //---------------
        // Type-checking:
        //---------------
        if args(0) <> 2 then 
          error("expecting one or two arguments") 
        end_if;
        if Mat::hasProp(Cat::Matrix) <> TRUE then
          error("expecting a matrix of 'Cat::Matrix'");
        end_if;
        if Mat <> b::dom then 
          error("second argument must be of the same type as the first argument");
        end_if;
        if M::dom::matdim(M)[1] <> b::dom::matdim(b)[1] then 
          error("illegal operands");
        end_if;        

        //---------------------------------------------
        // Special cases for M::dom::matdim(M)[1] <= 2:
        //---------------------------------------------
        // If the number of rows/cols of the coefficient matrix is 1 or 2 we 
        // use the procedure linalg::SSS for solving the symbolic system. 
        // linalg::SSS uses standard fraction free gaussian elimination to 
        // compute the solution. 
        if min(M::dom::matdim(M)) <= 2 then 
          return(linalg::SSS(M, b));
        end_if;

        Ring:= M::dom::coeffRing;
        Rone:= Ring::one;
        Rzero:= Ring::zero;
        // Special case for efficiency: the Ring is used in
        // calls to Dom::Matrix and poly. Make sure that
        // the Ring is not passed to poly (use the much
        // faster poly(..) = poly(.., Expr) instead)

        if Ring= Dom::ExpressionField() then
           Ring:= null();
        end_if:

        //---------------------------
        // Conversion to Dom::Matrix:
        //---------------------------
        if Mat::constructor <> Dom::Matrix then 
          // Convert the matrix into a matrix over Dom::Matrix
          M:= Dom::Matrix(Ring)(M);
          b:= Dom::Matrix(Ring)(b);
        end_if;

        M:= M::dom::concatMatrix(M,b); // compute the extended coefficient 
                                       // matrix (M | b)
        m:= M::dom::matdim(M)[1]; // number of rows of the matrix M
        n:= M::dom::matdim(M)[2]; // number of columns of the extended 
                                  // coefficient matrix (M | b), i.e. 
                                  // n = numberOfColumns(M) + 1
        
        nm2:= min(n - 2, m - 2);  
        
        isKernelCol:= FALSE;        
        kernelCols:= [];
        ncols:= min(m,n-1);
        A:= [M[k,k] $ k = 1..ncols];
        transformMatrix:= matrix::identity(n-1); // hier war ncols

        //------------
        // Elimination 
        //------------
        // We start the elimination phase here. 
        // First replace all main diagonal entries 
        // in columns 1 to n-2 by variables `#X`[k]. 
        // Then compute the upper triangular matrix 
        // with respect to reference paper mentioned 
        // above. 

//--------------------------------------------------------------------------------
// Bugfix by Walter, 18.10.06: 
// I changed the following lines:
//
//      for k from 1 to ncols do 
//        M[k,k]:= `#X`[k];
//      end_for;
//      M:= map(M, poly, [`#X`[j] $ j = 1..nm2]);
//
// Beware, you cannot map poly to M! If M is defined over some
// ring, why should it be able to deal with poly(.., Ring) ?
// Solution: map M to some new matrix (over Dom::ExpressionField),
// then do the elimination, then reconvert to a matrix over
// the ring (further down below).

        M:= matrix(m, n, [[poly(M[i,k], [`#X`[j] $ j = 1..nm2], Ring) $ k=1..n] $ i = 1..m]);
        for k from 1 to ncols do 
          M[k,k]:= poly(`#X`[k],  [`#X`[j] $ j = 1..nm2], Ring);
        end_for;

	assert(map({M::dom::nonZeroOperands(M)}, domtype) = {DOM_POLY});

//--------------------------------------------------------------------------------
// proceed with Kai's original code from here
//--------------------------------------------------------------------------------

//      for k from 1 to nm2+1 do // <-- Bug fix Walter, 18.10.06, Kais code
        for k from 1 to nm2+2 do // <-- Bug fix Walter, 18.10.06, my new code
          if k = 1 then 
            Q:= poly(1, [`#X`[j] $ j = 1..nm2], Ring);
          else
              X1X2ETC:= poly(_mult(`#X`[i] $ i = 1..k-1), [`#X`[j] $ j = 1..nm2], Ring);
              R:= M[k-1,k-1] - X1X2ETC;
              Q:= X1X2ETC - R;
              sig:= 1;
              prod:= R;
              while not iszero(prod) do
                prod:= linalg::otimes(prod, R, 1, k-1, nm2, Ring); 
                if iszero(prod) then
                   break;
                end_if;
                if sig = 1 then
                   Q:= Q + prod
                else
                   Q:= Q - prod
                end_if;
                sig:= -sig;
              end_while;
          end_if;
          if iszero(myexpand(subs(extop(M[k,k],1), [`#X`[j] = A[j] $ j = 1..ncols]))) then 
            // We have used a zero element as 'pivot' 
            // within the above elimination. 
            for j from k+1 to m do 
              if not iszero(myexpand(subs(extop(M[j,k],1), [`#X`[i] = A[i] $ i = 1..ncols]))) then 
                // In this case we have ignored a non-zero 
                // element in the i-th row within the 
                // elimination, which should have served as 
                // a pivot-element if we had the chance to 
                // interchange rows within the above algorithm.
                isKernelCol:= FALSE;

             // Bugfix by Walter, 18.10.06:
             // M:= linalg::addRow(M, j, k, Rone, Rone);
                M:= linalg::addRow(M, j, k, 1, 1);

                break; 
              else 
                isKernelCol:= TRUE
              end_if;
            end_for;
            if isKernelCol = TRUE then 
              for j from k+1 to ncols do 
                if not iszero(myexpand(subs(extop(M[k,j],1), [`#X`[i] = A[i] $ i = 1..ncols]))) then

               // Bugfix by Walter, 18.10.06:
               // M:= linalg::addCol(M, j, k, Rone, Rone); 
                  M:= linalg::addCol(M, j, k, 1, 1);

                  transformMatrix[j,k]:= 1;
                  isKernelCol:= FALSE;
                  break;
                else 
                  isKernelCol:= TRUE;
                end_if;
              end_for;
            end_if;
            if isKernelCol = TRUE then 
              for i from k+1 to m do 
                for j from k+1 to n-1 do 
                  if not iszero(myexpand(subs(extop(M[i,j],1), [`#X`[l] = A[l] $ l = 1..ncols]))) then

                 // Bugfix by Walter, 18.10.06:
                 // M:= linalg::addRow(M, i, k, Rone, Rone);
                 // M:= linalg::addCol(M, j, k, Rone, Rone);
                    M:= linalg::addRow(M, i, k, 1, 1);
                    M:= linalg::addCol(M, j, k, 1, 1);

                    transformMatrix[j,k]:= 1;
                    isKernelCol:= FALSE;
                    break;
                  end_if;
                end_for;
                if isKernelCol = FALSE then break; end_if;
              end_for;
            end_if;
            if isKernelCol = TRUE then 
              if not iszero(myexpand(subs(extop(M[k,n],1), [`#X`[l] = A[l] $ l = 1..ncols]))) then 
                return([]);
              end_if;
            end_if;
          end_if;
          for i from k+1 to m do 
            for j from k+1 to n do
              M[i,j]:= linalg::otimes( linalg::otimes(M[k,k], M[i,j], 1, k-1, nm2, Ring)
                                      -linalg::otimes(M[i,k], M[k,j], 1, k-1, nm2, Ring),
                                                                   Q, 1, k-1, nm2, Ring);
            end_for;
          end_for;
          assert(map({M::dom::nonZeroOperands(M)}, domtype) = {DOM_POLY});
        end_for;
        //-------------------
        // End of Elimination 
        //-------------------

        // The elimination has been finished. Note that M still 
        // contains the polynomials in the variables `#X`[i]. 

        M:= M::dom::assignElements(M, (([i,j] = 0) $ i= j+1 .. m) $ j=1..n);

        assert(_and((iszero(M[i,j]) $ i = j+1 .. m) $ j=1..n));

        // Here is the new code by Walter, 18.10.06:
        //  a) evaluate the polynomials by calling the polys as functions
        //     with `#X`[k] = original k-th diagonal element stored as A[k]
        //  b) substitute the remaining `X[k]` by the original diagonal elements
        //  c) expand the expressions
        //  d) insert Rzero's below the diagonal
        assert(M::dom::matdim(M) = [m, n]);
        M:= Dom::Matrix(Ring)(m,n, 
                [ [(myexpand(subs(M[i,j](A[k] $ k = 1..nm2),
                                [`#X`[k] = A[k] $ k = nm2 + 1 .. ncols])) $ j = 1..n)]
                  $ i = 1..m]);

        assert(_and((iszero(M[i,j]) $ i = j+1 .. m) $ j=1..n));

        // proceed with Kai's code:
        // Undo the effects caused by the addition of columns by 
        // multiplying with the inverse of transfromMatrix
        MM:= M::dom::delCol(M,n) * numeric::inverse(transformMatrix, Symbolic);
        M:= M::dom::concatMatrix(MM, M::dom::col(M,n));

        //----------------------------------------------
        // Start to solve the system of linear equations
        //----------------------------------------------

        // Now M is in upper triangular form. Does there exist any 
        // solution at all? We check if rank(A,B) = rank(A). 

        i:= 1;
        j:= 1;
        rank:= 0;
        while i <= min(m,n) do 
          found:= FALSE;
          if not iszero(M[i,j]) then 
            if j = n then 
              return([]); // the system of linear equations does 
                          // not have any solution
            end_if;
            rank:= rank + 1;
            i:= i + 1;
            j:= j + 1;
            found:= TRUE;
          else 
            for k from j+1 to n do 
              if not iszero(M[i,k]) then 
                rank:= rank + 1;
                i:= i + 1;
                j:= j + 1;
                found:= TRUE;
                if k = n then 
                  return([]); // the system of linear equations does 
                              // not have any solution
                end_if;
                break;
              end_if;
            end_for;
            if found = FALSE then 
              i:= i + 1;
            end_if;
          end_if;
        end_while;
        delete i,j;
        // From this point on we know that the system of linear 
        // equation has a solution. 

        //------------------------------------
        // Identifying possible kernel vectors
        //------------------------------------
        // Inserting 1's on the main diagonal:  
        // Positioning of the characteristic columns 
        // "along the main diagonal". Insert 1 on the 
        // main diagonal in zero rows. The set kernelCols
        // contains the column indices of those columns, 
        // where we insert 1 on the main diagonal. These 
        // columns contribute to the kernel of the 
        // coefficient matrix. 
        kernelCols:= {};
        if m < n-1 then 
          addRows:= Dom::Matrix(Ring)(n-1-m, n);
          M:= M::dom::stackMatrix(M,addRows);
        end_if;
        for i from 1 to n-1 do
          if iszero(M[i,i]) then 
            for l from i+1 to n-1 do 
              if not iszero(M[i,l]) then 
                M:= M::dom::swapRow(M, i, l);
              end_if;
            end_for;
            M[i,i]:= Rone;
            kernelCols:= kernelCols union {i};
          end_if;
        end_for;

        // Remove zero rows at the end of the matrix 
        for i from min(m,n-1) + 1 to m do  
          M:= M::dom::delRow(M, i);
        end_for;

        // The number of rows of M has changed and 
        // must be updated: 
        m:= M::dom::matdim(M)[1];
    
        //------------------------------------------
        // Computing the determinant of the matrix M 
        //------------------------------------------
        detM:= Rone; 
        for i from 1 to n-1 do
          detM:= detM * M[i,i];
        end_for;

        //-----------------------------
        // Computing a special solution 
        //-----------------------------
        // Now we need a special solution of the system.
        // This solution is computed via Cramer's rule.
        b:= M::dom::col(M,n); // the right-hand-side of 
                              // the system of linear 
                              // equations, which has 
                              // ben modified by the 
                              // elimination done above 
        M:= M::dom::delCol(M,n); // Delete the last column 
                                 // of M corresponding to 
                                 // the right-hand-side of 
                                 // the system of linear 
                                 // equations
        n:= n-1; // the 'new' number of columns 

        dets:= [Rzero $ n];
        det:= Rone;
        detMult:= Rone;
        // Note that in the case b = 0 we do not really 
        // have to compute all determinants. We simply
        // define them to be zero. 
        if not iszero(b) then 
          A:= M::dom::setCol(M, n, b);
          for i from 1 to n do 
            det:= det * A[i,i];          
          end_for;
          dets[n]:= det; 
          // else dets[n]:= Rzero; which is actually the 
          // the case already. 
        end_if;
        detMult:= Rone;
        if not iszero(b) then 
          for col from n-1 downto 1 do 
            A:= M::dom::setCol(M, col, b);
            det:= Rone;
            // We are in the i-th row and 
            // the col-th column. 
            if not iszero(A[col + 1,col]) then 
              f0:= A[col+1,col+1];
              f1:= -A[col+1,col];
              detMult:= detMult/f0;
              // addCol(A,col,col+1,A[col+1,col+1],-A[col+1,col]);
              for i from col downto 1 do
                A[i,col]:= f0 * A[i,col] + f1 * A[i,col+1];
              end_for;
              A[col+1,col]:= Rzero;
            end_if;
            // Now A should is in upper triangular form. 
            // We compute the determinant as the product 
            // of the main diagonal entries.
            det:= _mult(A[i,i] $ i = 1..n) * detMult;
            b:= A::dom::col(A, col);
            dets[col]:= det;
          end_for;
          // else dets[col]:= Rzero; which is actually the 
          // the case already. 
        end_if;
        // Compute a special solution of the system of 
        // linear equations by dividing all determinants 
        // by detM (Cramer's rule). 
        dets:= map(dets, ind -> ind/detM);
        b:= Dom::Matrix(Ring)(dets);

        //-----------------------------------
        // Compute the kernel of the matrix M 
        //-----------------------------------
        // kernelCols contains the indices of the  
        // columns, where we inserted 'Rone' on the 
        // main diagonal. 
        kernelColVectors:= [];
        for ind in kernelCols do 
          RHS:= Dom::Matrix(Ring)(n,1);
          RHS[ind,1]:= Rone; 
          det:= Rone;
          if ind = n then 
            A:= M::dom::setCol(M, n, RHS);
            det:= _mult(A[i,i] $ i = 1..n);
            dets[n]:= det; 
          else
            (dets[i]:= Rzero) $ i = ind + 1..n; 
          end_if;
          detMult:= Rone;
          for col from min(ind, n-1) downto 1 do 
            A:= M::dom::setCol(M, col, RHS);
            det:= Rone;
            // We are in the i-th row and 
            // the col-th column. 
            if not iszero(A[col + 1,col]) then 
              f0:= A[col+1,col+1];
              f1:= -A[col+1,col];
              detMult:= detMult/f0;
              for i from col downto 1 do
                A[i,col]:= f0 * A[i,col] + f1 * A[i,col+1];
              end_for;
              A[col+1,col]:= Rzero;
            end_if;
            // Now A is in upper triangular form. 
            // We compute the determinant as the product 
            // of the main diagonal entries.
            det:= _mult(A[i,i] $ i = 1..n) * detMult;
            RHS:= A::dom::col(A, col);
            dets[col]:= det;
          end_for;
          dets:= map(dets, ind -> ind/detM);
          kernelColVectors:= append(kernelColVectors, Mat(dets));
        end_for;

        return([b, kernelColVectors, kernelCols]);
      end_proc:
