/*++
  adjoint.mu

 linalg::adjoint(A) -- return the adjoint of the square matrix A

 Computes the adjoint of a matrix over a commutative ring.
 This is the matrix whose (i,j)-th entry is the (j,i)-th
 cofactor of A.

 The (j,i)-th cofactor of A is defined by 

     A'(i,j) = (-1)^(i+j)*det(A(i,j))

 where A(i,j) is the submatrix of A obtained from A by
 deleting the i-th row and j-th column.

++*/

linalg::adjoint := proc(A)
    local n, R, B, subst, symbolicity, i, j, a, Ai;
begin
  if args(0) < 1 then error("expecting one argument") end_if;
  if A::dom::hasProp( Cat::Matrix ) <> TRUE then
    error("first argument is not of 'Cat::Matrix'")
  end_if;
  if not (A::dom::coeffRing)::hasProp( Cat::CommutativeRing ) then
    error("expecting matrix over 'Cat::CommutativeRing'")
  end_if;
  n := A::dom::matdim(A);
  if n[1] <> n[2] then  
    error("not a square matrix") 
  end_if;
  n:= n[1];
  R:= A::dom::coeffRing;

  if n = 1 then 
     return(A::dom([[1]]));
  end_if:

  //====================
  // check 'symbolicity'
  //====================

  if R = Dom::ExpressionField() then
    [B, subst]:= [linalg::rationalizeMatrix(A)];
    symbolicity:= linalg::symbolicity(B, "MatrixIsRationalized");
    if symbolicity = 2 and n < 7 then
       B:= linalg::symbolicAdjoint_Fu(B, args(2..args(0)));
       return(B::dom::mapNonZeroes(B, eval@subs, subst));
    end_if:
  end_if:

  //============================
  // generic code for any ring,
  // based only on linalg::det
  //============================

  a:= array( 1..n,1..n );
  for i from 1 to n do
    Ai := A::dom::delCol( A,i );
    for j from 1 to n do
      a[i,j] := linalg::det(A::dom::delRow( Ai,j ));
      if i+j mod 2 = 1 then
        a[i,j] := -a[i,j]
      end_if
    end_for
  end_for;
  A::dom::create( a ) 
end_proc:

// end of file 
