/*
    linalg::angle(u,v) -- computes the angle between u and v where
    u and v are vectors over the same coeffRing.

    The coefficient ring of the vectors has to provide the method
    'norm', which result for any element has to be a number.
    'linalg::angle' returns FAIL if such a method does not exists.

    The angle x between two vectors u and v is defined by

                  ( u * v )
        x = arccos(-------),
                  (|u| |v|)
    
    where '*' meens the scalar product of two vectors defined by 
    linalg::scalarProduct and |u| = sqrt( u*u ).

    The user has to make sure that the argument of arccos is
    valid (see also: linalg::scalarProduct).
*/

linalg::angle := proc(u,v)
    local n1, n2;
begin
    if testargs() then
      if args(0) <> 2 then error("expecting 2 arguments") end_if;
      if not testtype( u,linalg::vectorOf(Type::AnyType) ) then
        error("expecting two vectors")
      end_if;
      if not testtype(v,linalg::vectorOf(u::dom::coeffRing,nops(u))) then
        error("expecting two vectors")
      end_if
    end_if;

    n1 := linalg::scalarProduct( u,u )^(1/2);
    n2 := linalg::scalarProduct( v,v )^(1/2);

    if iszero(n1) or iszero(n2) then
      error("angle is undefined")
    else
      return(arccos( linalg::scalarProduct(u,v) / n1 / n2 ))
    end_if
end_proc:

