
/*
    linalg::bezout(f, g, x)
   
    f, g : polynomials in x with degree(f) >= degree(g)
       x : variable

    Computes the Bezout matrix of two polynomials f and g. 
    The current definition is optimal for polylib::resultant; there various definitions of Bezout matrices in the literature.

    The current definition is the same as the definition of HBez in Kerber, Division-free computation of subresultants using Bezout matrices, 2006, up to swapping the sparse rows
    to the top of the matrix. 

    Due to this swapping, the determinant must be multiplied by (-1)^((degree(f) mod 4) div 2) to get the resultant of f and g.
    Note that if we switch to another definition from tze literature, this factor must be changed in polylib::resultant.

    See the RFAIN of g646619 (gcdex, partfrac, resultant) for further details

    Since the method is internal, we do little argument checking by now


*/


linalg::bezout:=
proc(f, g, x)
  local K, m, n, Pq, Pr, Qq, Qr, i, j, vars, R, M;
begin
  
  if type(f) <> DOM_POLY then
    f:= poly(f, [x]);
    g:= poly(g, [x])
  end_if;  
  
  [vars, R]:= [op(f, 2..3)];
  M:= Dom::Matrix(polylib::coeffRing(f));
  
  n:= degree(f, x);
  m:= degree(g, x);
  assert(n>=m);
  
  for i from 1 to m do
    [Pq, Pr]:= [divide(f, poly(x^(n-m+i), vars, R))];
    [Qq, Qr]:= [divide(g, poly(x^i, vars, R))];
    K[i]:= Pq*Qr*poly(x^(n-m), vars, R) - Qq*Pr
  end_for;  
  
  
  M(n, n, 
  revert(
  [revert([coeff(K[i], x, j) $j=0..n-1]) $i=1..m,
  [coeff(g, x, i-j) $j=1..i,  0 $n-i ] $i=m+1..n]
  ))
  
end_proc:
// end of file