/* linalg::companion -- the companion matrix for a univariate polynomial

Call:: linalg::companion(p, <x>)

Parameter:  p - a univariate polynomial expression or
                a univariate polynomial of domain type DOM_POLY.
                This polynomial must be monic and of degree>=1.
    
Option:     x - identifier or indexed identifier
                (the indeterminate of p)
            If p is DOM_POLY, then specifying <x> has no effect
            If p is an expression and contains only one symbolic
            indeterminate, then x need not be specified.
            If p is an expression and contains several symbolic
            indeterminates, then x must be specified and distinguishes
            the indeterminate x from the other symbolic parameters.

Returns:  The companion matrix C of p.
          If p is an expression, then domtype(C)=Dom::Matrix(Dom::ExpressionField()).
          If p=poly(..,[x], field) then domtype(C)=Dom::Matrix(Field).

          If field = Expr then Field=Dom::ExpressionField()
          If field = IntMod(q) then Field=Dom::IntegerMod(q).
          Otherwise Field=field.
       

Details:  The companion matrix of a degree n polynomial 
          is an n x n matrix C with charpoly(C,x)=p(x):

          >> linalg::companion(x^4+a3*x^3+a2*x^2+a1*x+a0,x);

                          +-              -+
                          |  0, 0, 0, -a0  |
                          |                |
                          |  1, 0, 0, -a1  |
                          |                |
                          |  0, 1, 0, -a2  |
                          |                |
                          |  0, 0, 1, -a3  |
                          +-              -+

Examples:
  >> p:= x^4+a3*x^3 +a2*x^2 +a1*x + a0: 
  >> linalg::companion(p);
     Error: multivariate expression [linalg::companion]
  >> linalg::companion(p,x);
                          +-              -+
                          |  0, 0, 0, -a0  |
                          |                |
                          |  1, 0, 0, -a1  |
                          |                |
                          |  0, 1, 0, -a2  |
                          |                |
                          |  0, 0, 1, -a3  |
                          +-              -+
                Domain: Dom::Matrix(Dom::ExpressionField(id, iszero))

  >> linalg::companion(p,a0);
                   +-          4    2       3   -+
                   | - x a1 - x  - x  a2 - x  a3 |
                   +-                           -+
                Domain: Dom::Matrix(Dom::ExpressionField(id, iszero))

  >> linalg::companion(p,a1);
     Error: polynomial is not monic [linalg::companion]
  >> p:= poly(x^2 +a1*x + a0,[x]): linalg::companion(p);
                             +-        -+
                             |  0, -a0  |
                             |          |
                             |  1, -a1  |
                             +-        -+
                Domain: Dom::Matrix(Dom::ExpressionField(id, iszero))

  >> p:= poly(x^2 +a1*x + a0,[x]): linalg::companion(p);
                             +-        -+
                             |  0, -a0  |
                             |          |
                             |  1, -a1  |
                             +-        -+
                Domain: Dom::Matrix(Dom::ExpressionField(id, iszero))
                                               
  >> p:= x^2+10*x+PI: linalg::companion(p);
                             +-        -+
                             |  0, -PI  |
                             |          |
                             |  1, -10  |
                             +-        -+
                 Dom::Matrix(Dom::ExpressionField(id, iszero))
  
  >> p:=poly(x^2+10*x+PI,[x]): linalg::companion(p);
                             +-        -+
                             |  0, -PI  |
                             |          |
                             |  1, -10  |
                             +-        -+
                Domain: Dom::Matrix(Dom::ExpressionField(id, iszero))

  >> p:=poly(x^2+10*x+7,[x],Dom::IntegerMod(3)): linalg::companion(p);
                        +-                  -+
                        |  0 mod 3, 2 mod 3  |
                        |                    |
                        |  1 mod 3, 2 mod 3  |
                        +-                  -+
                              Domain: Dom::Matrix(Dom::IntegerMod(3))
            
  >> p:=poly(x^2+10*x+7,[x],IntMod(3)): linalg::companion(p);
                        +-                  -+
                        |  0 mod 3, 2 mod 3  |
                        |                    |
                        |  1 mod 3, 2 mod 3  |
                        +-                  -+
                              Domain: Dom::Matrix(Dom::IntegerMod(3))
*/

linalg::companion := proc(p)
   local x, field, n, C, i; 
begin
   if args(0)<1 then error("expecting at least one argument") end_if;
   if args(0)>3 then error("expecting at most two arguments") end_if;
   p:= subs(p, float(0)=0);
   case domtype(p) 
   of DOM_POLY do
      x:= op(p,2);
      if nops(x)<>1 then error("not a univariate polynomial") end_if;
      field:= op(p, 3);
      if field = Expr then field:= Dom::ExpressionField(): end_if;
      if has(field,IntMod) then field:=Dom::IntegerMod(op(field,1)) end_if; 
      break;
   otherwise
      if args(0)=1 then
        x:=indets(float(p), PolyExpr);
        if nops(x)>1 then error("multivariate expression") end_if;
        if nops(x)<1 then error("poly has degree 0") end_if;
        x:=op(x);
        if not testtype(p,Type::PolyExpr(x)) then
           error("not a univariate polynomial") 
        end_if
      else x:= args(2);
      end_if;
      if p::dom::hasProp(Cat::Polynomial) = TRUE and p::dom::coeffRing <> FAIL then 
        field:= p::dom::coeffRing;        
      else 
        field:= Dom::ExpressionField();
        p:= poly(p, [x]);
      end_if;
      if p=FAIL then error("not a polynomial") end_if;
   end_case;
   n:= degree(p);
   if n=0 then error("poly has degree 0"); end_if;
 
   // !!!! lcoeff(poly(x,IntMod(p))) is 1 not 1 mod p !!!!
   // However, lcoeff(poly(x,Dom::IntegerMod(p))) is 1 mod p !!!!
   // So use field(lcoeff(p)) instead lcoeff(p):
   if field(lcoeff(p))<>field::one then error("polynomial is not monic") end_if;

   // Now we are ready to go:
   C:= array(1..n, 1..n, [[0 $ n] $ n]);
   (C[i+1,i]:= 1;) $ i=1..n-1;
   (C[i,n]:= -coeff(p,i-1);) $ i = 1..n;

   return(Dom::Matrix(field)(C));
end_proc:

/* end of file companion.mu */
