/*
        linalg::curl -- computes the curl of a vector over fields

        curl(x, m, <ogSystem, <c>>)

        x       : list or vector of expressions of length 3
        m       : list or vector of three variables
 
        ogSystem: (optional) name or list of the scale factors of an 
                  orthogonally curvilinear coordinate system
        c       : the parameter for the coordinate systems
                  'Torus' and 'EllipticCylindrical'

 This function computes the curl of x. If only two arguments 
 are given linalg::curl operates on the cartesian coordinate system.

 Otherwise linalg::curl operates on the orthogonally 
 curvilinear coordinate system specified by a list of scale 
 factors or by a name which scale factors are defined in table 
 linalg::ogCoordTab.

 If x is a vector over a Cat::Field R, then R has to provide a 
 method "diff". Otherwise an error message will be returned.
*/

linalg::curl := proc(x,m)
    local f, f1, f2, f3, g, i, R, rot, Rmult, Rplus, Rnegate, c;
begin
    if testargs() then
      if args(0) < 2 then
        error("expecting at least 2 arguments")
      end_if;
      if args(0) > 4 then
        error("expecting no more than 4 arguments")
      end_if;
      if testtype( x,linalg::vectorOf(Type::AnyType) ) then
        if not (x::dom::coeffRing)::hasProp( Cat::Field )
        or not testtype( x,linalg::vectorOf(Type::AnyType, 3) ) then
          error("expecting 3-dimensional vector over a 'Cat::Field'")
        end_if
      elif not testtype( x,Type::ListOf(Type::Arithmetical) ) then
        error("expecting list of arithmetical expressions")
      end_if;
      if not testtype( m,Type::ListOf(Type::Unknown,3,3 ) ) then
        error("expecting list of three (indexed) identifiers")
      end_if;

      if args(0) = 3 then
        if domtype(args(3)) <> DOM_LIST and 
           type( linalg::ogCoordTab[args(3),hold(Scales)] ) = "_index"
        then
           error("illegal or undefined coordinate system")
        end_if
      end_if
    end_if;

    if args(0) = 2 then 
      rot:= array(1..3, [
              diff(x[3], m[2]) - diff(x[2], m[3]),
              diff(x[1], m[3]) - diff(x[3], m[1]),
              diff(x[2], m[1]) - diff(x[1], m[2])
                        ] );
      if domtype(x) = DOM_LIST then
        return((Dom::Matrix())::coerce( rot ))
      else
        return(x::dom::coerce( rot ))
      end_if
    else
      if args(0) = 4 then
         c:= args(4);
      else
         c:= null();
      end_if;
      g:= args(3);
      if domtype(g) <> DOM_LIST then
        g:= linalg::ogCoordTab[g,hold(Scales)](m[i] $ i=1..3, c)
      end_if;
      if domtype(x) <> DOM_LIST then
        R:= x::dom::coeffRing;
        if not R::hasProp( Cat::Field ) then
          error("expecting vector over a Cat::Field")
        end_if;
        g:= map( g,R::coerce );
        if has(g,FAIL) then
          userinfo( 1,"unable to use given coordinate system over the coefficient domain" );
          return( FAIL )
        end_if;

        Rmult:= R::_mult;
        Rplus:= R::_plus;
        Rnegate:= R::_negate;

        f:=  map( [g[1],g[2],g[3]],R::_invert );
        f1:= Rmult(f[2],f[3]);
        f2:= Rmult(f[1],f[3]);
        f3:= Rmult(f[1],f[2]);

        x::dom::coerce( array(1..3,[
                  Rmult( f1,Rplus( diff(Rmult(g[3],x[3]),m[2]),
                  Rnegate(diff(Rmult(g[2],x[2]),m[3])))
                              ),
                  Rmult( f2,Rplus(
                    diff(Rmult(g[1],x[1]),m[3]),
                    Rnegate(diff(Rmult(g[3],x[3]),m[1])))
                  ),
                  Rmult( f3,Rplus(
                    diff(Rmult(g[2],x[2]),m[1]),
                    Rnegate(diff(Rmult(g[1],x[1]),m[2])))
                  )
                                  ]) )
        else
          f:=  [ 1/g[1], 1/g[2], 1/g[3] ];
          f1:= f[2]*f[3]; 
          f2:= f[1]*f[3]; 
          f3:= f[1]*f[2];
          (Dom::Matrix())::coerce( array(1..3,[
              f1*(diff(g[3]*x[3],m[2])-diff(g[2]*x[2],m[3])),
              f2*(diff(g[1]*x[1],m[3])-diff(g[3]*x[3],m[1])),
              f3*(diff(g[2]*x[2],m[1])-diff(g[1]*x[1],m[2]))
                                                   ]) )
        end_if
    end_if
end_proc:

linalg::curl := funcenv(linalg::curl):
linalg::curl::Content := (Out, data) ->
                         if nops(data) <> 2 then
                           return(Out::stdFunc(data));
                         else
                           Out::Capply(Out::Ccurl,
                                       map(op(op(data,1)), Out),
                                       map(op(op(data,2)), Out)):
                         end_if:
