/*
        linalg::det( A )
        linalg::det( A, MinorExpansion )

        A: square matrix

        Computes the determinant of a matrix over a commutative ring.
*/

linalg::det :=
proc(A)
    local R, Mat, Rnormal, Rmult, Rnegate, Rzero, i, j, n,
          subst, result, subdet, hasFloats, symbolicity;
begin
    if args(0) > 2 then
        error("expecting at most two arguments: a matrix of category 'Cat::Matrix' ".
              "and, optionally, the flag 'MinorExpansion'")
    end_if;

    Mat:= A::dom;
    if testargs() then
        if Mat::hasProp( Cat::Matrix ) <> TRUE then
            error("expecting a matrix of 'Cat::Matrix'")
        elif not (Mat::coeffRing)::hasProp( Cat::CommutativeRing ) then
            error("expecting matrix over 'Cat::CommutativeRing'")
        end_if;
        R:= Mat::matdim(A);
        if R[1] <> R[2] then
            error("not a square matrix")
        end_if
    end_if;

    R:= Mat::coeffRing;
    case op(Mat::matdim(A),1)
    of 1 do 
        return( A[1,1] );
    of 2 do
        userinfo(1,"use formula for the determinant of 2x2 matrices");

        Rmult:= R::_mult;
        if (Rnormal:= R::normal) = FAIL then
            if R::hasProp(Ax::systemRep) then 
               Rnormal:= normal 
            else 
               Rnormal:= () -> args(1); 
            end_if;
        end_if;
 
        return( Rnormal(R::_plus( 
           Rmult( A[1,1],A[2,2] ), R::_negate( Rmult( A[1,2],A[2,1] ) )
        )) )
    of 3 do
        userinfo(1,"use formula for the determinant of 3x3 matrices");

        Rmult:= R::_mult;
        if (Rnormal:= R::normal) = FAIL then
            if R::hasProp(Ax::systemRep) then 
               Rnormal:= normal 
            else 
               Rnormal:= () -> args(1); 
            end_if
        end_if;
 
        Rmult:= R::_mult;
        Rnegate:= R::_negate;
        return( Rnormal(R::_plus( 
            Rmult( A[1,1],A[2,2],A[3,3] ),
            Rnegate( Rmult( A[1,1],A[2,3],A[3,2] ) ),
            Rnegate( Rmult( A[2,1],A[1,2],A[3,3] ) ),
            Rmult( A[2,1],A[1,3],A[3,2] ),
            Rmult( A[3,1],A[1,2],A[2,3] ),
            Rnegate( Rmult( A[3,1],A[1,3],A[2,2] ) )
        )) )
    end_case;

    //---------------
    // MinorExpansion
    //---------------
    if has([args()], MinorExpansion) then
       n:= A::dom::matdim(A)[1];
       Rzero:= A::dom::coeffRing::zero;
       if Rzero = FAIL then
          return(FAIL);
       end_if;
       A:= A::dom::convert_to(A, DOM_ARRAY);

       // define local subroutine subdet = minor
       subdet:=proc(r,c,n) local result,i; option remember;
       begin
          if n=3 then return(A[r[1],c[1]]*A[r[2],c[2]]*A[r[3],c[3]]
                           + A[r[1],c[2]]*A[r[3],c[1]]*A[r[2],c[3]]
                           + A[r[2],c[1]]*A[r[1],c[3]]*A[r[3],c[2]]
                           - A[r[1],c[1]]*A[r[2],c[3]]*A[r[3],c[2]]
                           - A[r[1],c[2]]*A[r[2],c[1]]*A[r[3],c[3]]
                           - A[r[1],c[3]]*A[r[2],c[2]]*A[r[3],c[1]]
                            ) end_if;
          if n=2 then return(A[r[1],c[1]]*A[r[2],c[2]]
                            -A[r[1],c[2]]*A[r[2],c[1]]) 
          end_if;
          if n=1 then return(A[r[1],c[1]]) end_if;
          result:=Rzero;
          for i from 1 to n do
             if not iszero(A[r[i],c[1]]) then
                result:=result+(-1)^(1+i)*A[r[i],c[1]]
                *subdet(subsop(r,i=null()),subsop(c,1=null()),n-1);
             end_if;
          end_for;
          result;
      end_proc; //subdet
      result:= subdet([i$ i=1..n],[j$ j=1..n],n);
      return(result);
    end_if;

    //-----------------
    // other algorithms
    //-----------------
    if R = Dom::Float then
        userinfo(1,"call 'numeric::det'");
        return(numeric::det(A))
    end_if; 

  if R::hasProp(Cat::Field) and R::hasProp( Ax::systemRep ) then

      // ToDo: if there is a float in A and all elements
      // can be floated, then call numeric::det

      [A, subst]:= [linalg::rationalizeMatrix(A, ReturnType = matrix)];
      symbolicity:= linalg::symbolicity(A, "MatrixIsRationalized");

      // ToDo: use misc::maprec 

      hasFloats:= has(map({op(A)}, domtype@op, 1), DOM_FLOAT);
      if hasFloats then
         A:= map(A, numeric::rationalize);
      end_if:
      case symbolicity
      of 0 do
         userinfo(1,"call 'numeric::det' with option 'Symbolic'");
         result:= numeric::det(A,hold(Symbolic));
         break;
      of 1/2 do
      of 1 do
         result:= linalg::ffG(A)[3];
         break;
      of 2 do 
         result:= linalg::symbolicDet_Fu(A);
         break;
      end_case;
      result:= subs(result, subst, EvalChanges);
      if hasFloats then 
         result:= float(result);
      end_if:
      return(result);
  end_if;

  if R::hasProp( Cat::IntegralDomain ) then
      userinfo(1,"use Gaussian elimination");
      return( op(Mat::gaussElim(A),3) )
  end_if;

  // last resort
  userinfo(1,"use Berkowitz algorithm");
  return( linalg::detBerkowitz(A) )
end_proc:


linalg::det := funcenv(linalg::det):
linalg::det::Content := stdlib::genOutFunc("Cdeterminant", 1):

/*--
K. F. Gehrs, 2000/09/26

    linalg::detBerkowitz -- returns the determinant of the matrix A

    linalg::detBerkowitz( A )

    A: square matrix over a commutative ring
 
    Computes the the determinant of the matrix A
    The coefficient domain of A has to be a commutative ring.

    The algorithm is based on the "Berkowitz-Algorithm" to compute the 
    characterisitic polynomial. This algorithm was modified and is now
    implemented to replace the method of Laplace for computing the 
    determinant of a matrix over an ordinary commutative ring. 
            
    Reference: The Berkowitz Algorithm, Maple and Computing the 
               Characteristic Polynomial in an Arbitrary Commutative Ring
               by Jounaidi Abdeljaoued, May 1996
--*/
linalg::detBerkowitz:=
proc(A)
  local Rzero, Rmult, Rplus, Rnegate, Rone, Vect, i, j, r, C, S, Q, R, n, k;
begin
    R:= A::dom::coeffRing;
    n:= A::dom::matdim(A)[1];

    Rone:= R::one;
    Rzero:= R::zero;
    Rmult:= R::_mult;
    Rplus:= R::_plus;
    Rnegate:= R::_negate;

    Vect:= (S:= (Q:= array( 1..n+1 )));
    Vect[1]:= Rnegate(Rone); 
    Vect[2]:= A[1,1];
    C[1]:= Vect[1];

    for r from 2 to n do
        S:= assignElements(S, [i] = A[i,r] $ i=1..r-1);
        C[2]:= A[r,r];
        for i from 1 to r-2 do
            C[i+2]:= Rzero;
            ( C[i+2]:= Rplus( C[i+2], Rmult( A[r,j], S[j])) ) $ j=1..r-1;
            for j from 1 to r-1 do
                Q[j]:= Rzero;
                ( Q[j]:= Rplus( Q[j], Rmult( A[j,k], S[k])) ) $ k=1..r-1
            end_for;
            S:= assignElements(S, [j] = Q[j] $ j=1..r-1);
        end_for;
        C[r+1]:= Rzero;
        ( C[r+1]:= Rplus( C[r+1], Rmult( A[r,j], S[j])) ) $ j=1..r-1;
        for i from 1 to r+1 do
            Q[i]:= Rzero;
            ( Q[i]:= Rplus( Q[i], Rmult( C[i+1-j], Vect[j])) ) $ j=1..min(r,i)
        end_for;
        Vect:= assignElements(Vect, [i] = Q[i] $ i=1..r+1);
    end_for;

    if Vect[1] = Rone then 
        return( Vect[n+1] )
    elif n mod 2 = 1 then // (-1)^(n+1) = 1
        return( Vect[n+1] )
    else 
        return(Rnegate( Vect[n+1] ))
    end_if
end_proc:

