/*
        linalg::divergence -- computes the divergence of a vector function
        over fields

        divergence(v, u, <ogSystem, <c>>)

        v       : list or vector of arithmetical expressions 
                  (the vector function)
        u       : list of (indexed) identifiers
        ogSystem: (optional) name or list of the scale factors of an 
                  orthogonally curvilinear coordinate system
        c       : the parameter for the coordinate systems
                  'Torus' and 'EllipticCylindrical'

 This function computes the divergence of v in the variables given 
 in u. If only two arguments are given, linalg::divergence operates on
 the cartesian coordinate system. Hence the result is the sum
 diff(v[i], u[i]) as i ranges over the length of v .

 Otherwise, linalg::divergence operates on the orthogonally 
 curvilinear coordinate system specified by a list of scale 
 factors or by a name which scale factors are defined in table 
 linalg::ogCoordTab.
*/

linalg::divergence := proc(v, u)
    local g, i, R, Rmult, Rinvert, Rnormal, n, d, tmp, c;
begin
 if testargs() then
   if args(0) < 2 then
     error("expecting at least 2 arguments")
   end_if;
   if args(0) > 4 then
     error("expecting no more than 4 arguments")
   end_if;
   if testtype( v,linalg::vectorOf(Type::AnyType) ) then
     if not (v::dom::coeffRing)::hasProp( Cat::Field ) then
       error("expecting vectors over 'Cat::Field'")
     end_if
   elif not testtype( v,Type::ListOf(Type::Arithmetical) ) then
     error("expecting list of arithmetical expressions")
   end_if;

   if not testtype( u,Type::ListOf(Type::Unknown,nops(v) $ 2) ) then
     error("invalid number or type of variables")
   end_if;

   // n = the dimension
   n:= nops(u);

   if args(0) = 3 then
     if domtype(args(3)) <> DOM_LIST and 
        type(linalg::ogCoordTab[args(3),hold(Scales)]) = "_index"
     then
       error("illegal or undefined coordinate system")
     end_if;
     // presently only 3-dim transformations are installed
     // in ogCoordTab:
     if domtype(args(3)) <> DOM_LIST and n <> 3 then
       error("unknown orthogonal coordinate system")
     end_if;
   end_if
 end_if;

 // n = the dimension
 n:= nops(u);

 if args(0) = 2 then 
   return( _plus( diff(v[i], u[i]) $ i=1..n ) )
 else
   g:= args(3);
   if args(0) = 4 then
      c:= args(4);
   else
      c:= null();
   end_if;
   if domtype(g) <> DOM_LIST then
     g:= linalg::ogCoordTab[g,hold(Scales)](u[i] $ i=1..n, c)
   end_if;

   if domtype(v) <> DOM_LIST then
     R:= v::dom::coeffRing;
     for i from 1 to n do
       if (g[i]:= R::coerce(g[i])) = FAIL then
         userinfo(1,"unable to compute the divergence over ",R);
         return( FAIL )
       end_if
     end_for;
     Rmult:= R::_mult; 
     Rinvert:= R::_invert;
     Rnormal:= () -> args(1);
   else
     Rmult:= _mult;
     Rinvert:= () -> 1/args(1);
     Rnormal:= normal
   end_if;

   // The special case n = 3:
   /*
   if n = 3 then
      return(
             Rnormal(Rmult( Rinvert(g[i]) $ i=1..3, _plus( 
                 diff(Rmult(g[2],g[3],v[1]),u[1]),
                 diff(Rmult(g[3],g[1],v[2]),u[2]),
                 diff(Rmult(g[1],g[2],v[3]),u[3]) 
             ) ))
            );
   end_if;
   */
    
   // the general case (arbitrary n)
   d:= Rmult(g[i] $ i=1..n):
   if iszero(d) then
     error("the transformation is singular"):
   end_if:
        
   // div f = sum( diff(f.i/g.i, u[i]) 
   //            + f.i/g.i* sum(diff(g.k, u[i])/g.k, k=1..n),
   //             i=1..n); 
   //       = sum( diff(f.i/g.i, u[i]
   //            + f.i/g.i*   diff(g.1*g.2* .. *g.n, u[i])/(g.1*g.2* .. *g.n),
   //             i=1..n);
   Rnormal( _plus(
           (tmp:= Rmult(v[i], Rinvert(g[i]));
            diff(tmp, u[i]) + tmp* diff(d, u[i])/d;
           ) $ i = 1..n
                 )
          );
  end_if
end_proc:

linalg::divergence := funcenv(linalg::divergence):
linalg::divergence::Content :=
  proc(Out, data)
    local op1, op2;
  begin
    if nops(data) <> 2 then
      return(Out::stdFunc(data));
    end_if;
    op1 := op(data, 1):
    if domtype(op1) = DOM_LIST then
      op1 := op(op1);
    end_if;
    op2 := op(data, 2):
    if domtype(op2) = DOM_LIST then
      op2 := op(op2);
    end_if;
    Out::Capply(Out::Cdivergence, Out(op1), Out(op2)):
  end:
