/*
linalg::eigenvalues -- computes the eigenvalues of a matrix

linalg::eigenvalues(A [,Multiple])

A: square matrix
 
The call eigenvalues(A) returns a set of the eigenvalues of A.
If option Multiple is given, then a list of sublists 
is returned, where each sublist contains an eigenvalue and its 
algebraic multiplicity.

If the coefficient ring of A is Dom::Float then numeric::eigenvalues 
is used. In this case and for the option 'Multiple', the list e of 
eigenvalues is sorted which means that e[i]<=e[j] for i<j.
*/

linalg::eigenvalues :=
proc(A)
  local p, x, e, R, Afloated, res, B;
begin
  if testargs() then
    if args(0) < 1 or args(0) > 2 then
      error("expecting a matrix and optionally 'Multiple' as option")
    elif A::dom::hasProp( Cat::Matrix ) <> TRUE then
      error("first argument is not of 'Cat::Matrix'")
    elif args(0) = 2 and args(2) <> hold(Multiple) then
      error("expecting option 'Multiple'")
    end_if
  end_if;

  R:= A::dom::coeffRing;

  // Special code for treating diagonal, upper triangular and 
  // lower triangular matrices. 
  if A::dom::constructor <> Dom::Matrix then 
    B:= A::dom::convert_to(A,Dom::Matrix(R)) 
  else 
    B:= A;
  end_if;  
  if B <> FAIL then 
    if args(0) = 1 then 
      res:= B::dom::matrixShape(B,Diagonal);
      if type(res) = DOM_SET then 
        return(res)
      end_if;
    elif args(0) = 2 and args(2) = hold(Multiple) then 
      res:= B::dom::matrixShape(B,Multiple);
      if type(res) = DOM_LIST then 
        return(sort(res))
      end_if;
    end_if;  
  end_if;
  // End of special code for treating diagonal, upper triangular and 
  // lower triangular matrices.
  
  if R = Dom::Float then
    userinfo(1,"call 'numeric::eigenvalues'");
    p:= numeric::eigenvalues( A );
    if args(0) = 1 then
      p:= select( map({op(p)},R::coerce),_not@_equal,FAIL );
      return( p )
    else
      // numeric::eigenvalues already returns multiple eigenvalues,
            // hence collect multiple eigenvalues:
      x:= [];
      while p <> [] do
        p:= split( p,_equal,p[1] );
        if (e:= R::coerce(p[1][1])) <> FAIL then
          x:= append( x,[e,nops(p[1])] )
        end_if;
        p:= p[2]
      end_while;
      return( x )
    end_if
  // New branch in the if-construction: 
  // check whether the matrix contains floats and can be converted 
  // to 'float(Matrix)'. If this is the case, call the corresponding 
  // 'numeric'-routine as in the case above. 
  elif (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
    userinfo(1,"convert entries to floats and call 'numeric::eigenvalues'");
    p:= numeric::eigenvalues( Afloated );
    if args(0) = 1 then
      p:= select( map({op(p)},R::coerce),_not@_equal,FAIL );
      return( p )
    else
      // numeric::eigenvalues already returns multiple eigenvalues,
      // hence collect multiple eigenvalues:
      x:= [];
      while p <> [] do
        p:= split( p,_equal,p[1] );
        if (e:= R::coerce(p[1][1])) <> FAIL then
          x:= append( x,[e,nops(p[1])] )
        end_if;
        p:= p[2]
      end_while;
      return( x )
    end_if
  else
    x:= genident("x"):
    userinfo(1,"compute characteristic polynomial p(A,x)");
    p:= linalg::charpoly( A,x );
    
    userinfo(1,"compute zeros of p(A,x) in x");
    
    /* OLD version: 
    userinfo(1,"compute zeros of p(A,x) in x");
    if args(0) = 2 then
      // with option 'Multiple', return a list of lists:
		  res:= solve(p,x,hold(Multiple), hold(MaxDegree)=4);
			if type(res) = RootOf then 
        return(res) 
      else 
        return( sort([extop(res)] ))
      end_if;
    else
      return( solve(p,x, hold(MaxDegree)=4) )
    end_if
    */

    if args(0) = 2 then
      // with option 'Multiple', return a list of lists:
			res:= solve(p,x,hold(Multiple),hold(MaxDegree)=4);
			if type(res) = RootOf then 
        return(res) 
      elif type(res) = piecewise then 
        res:= piecewise::disregardPoints(res);
        if type(res) = piecewise then 
          warning("cannot find a simple representation of eigenvalues");
          warning("some cases are ignored");
          res:= res[1]; // sloppy workaround: if nothing helps to get rid of 
                        // piecewise, just choose the first branch
        end_if;
  			if type(res) = RootOf then 
          return(res) 
        else                   
          return(sort(res))
        end_if;  
      end_if;
      return( sort([op(res)] ))
    else 
      res:= solve(p,x, hold(MaxDegree)=4);
      if type(res) = piecewise then 
        res:= piecewise::disregardPoints(res);
        if type(res) = piecewise then 
          warning("cannot find a simple representation of eigenvalues");
          warning("some cases are ignored");
          res:= res[1]; // sloppy workaround: if nothing helps to get rid of 
                        // piecewise, just choose the first branch
        end_if;
      end_if;
      return(res)
    end_if;
  end_if
end_proc:

