/* --------------------------------------------------------------
Compute the set of eigenvectors of a matrix with rational entries
   
Calls:

   linalg::eigenVectorsRational(A <, maxdegree>)

Parameters:

   A - a matrix over some domain that is able to get by with 
       RootOfs as well as other symbolic expressions as entries
   maxdegree - an integer (default: 2)

Details:
   Entries of the eigenvectors are written as RootOf's 
   if their degree over the rationals is > maxdegree; 
   otherwise, they are written as expressions containing 
   (nested) fractional powers of integers, if possible 
   (i.e., if solve can handle the minimal polynomial)

Beware: before calling this procedure you must make
   sure that the matrix A only contains (real) integers
   or (real) rational entries!
------------------------------------------------------  */

linalg::eigenvectorsRational:=
proc(A, maxdegree = 2: DOM_INT)
  local f, z, l: DOM_LIST, S, i, n: DOM_INT,
  eigenspaces: DOM_PROC;
begin
  //---------------------------------------------------
  // eigenspaces - returns all eigenvectors corresp. to 
  // eigenvalues that are roots of g
  // m - multiplicity of the factor g
  eigenspaces:=
  proc(g, m: DOM_INT)
    local F, AA, M, lambdaEn, vlist;
  begin
    F:= Dom::AlgebraicExtension(Dom::Rational, g);
    M:= Dom::Matrix(F);
    AA:= coerce(A, M);
    lambdaEn:= M(n, n, [z$n], Diagonal);
    vlist:= map(linalg::nullspace(AA - lambdaEn), coerce, A::dom);
    if degree(g) > maxdegree then
      S:= RootOf(g, z)
    else
      S:= solve(g, z, MaxDegree = maxdegree);
      if type(S) <> DOM_SET then
        // could not solve -> use RootOf; should be the result of solve
        // anyway
        S:= RootOf(g, z)
      end_if
    end_if;
    [S[i], m , subs(vlist, z = S[i], EvalChanges)] $i=1..degree(g)
  end_proc;
  //---------------------------------------------------
  z:= solvelib::getIdent(C_);
  n:= A::dom::nrows(A);
  f:= linalg::charpoly(A, z);
  l:= [op(factor(f))];
  [eigenspaces(l[2*i], l[2*i+1]) $i=1..nops(l) div 2]
end_proc:

