/*++
  factorCholesky.mu

        linalg::factorCholesky -- computes the Cholesky decomposition

        factorCholesky(S [,NoCheck])

        S      : symmetric matrix
        NoCheck: (optional) identifier

        factorCholesky(S) computes the Cholesky decomposition for a 
        positive definite matrix S over a field.

        This is a decomposition in a product S=G*transpose(G) such that
        G is lower triangular and has positive entries on the main
        diagonal (the Cholesky factor of S).

        If S is not positive definite (i.e. such a decomposition does
        not exist), then an error occures. To suppress such errors
        use option 'NoCheck'. With this option it is even not checked
        if S is symmetric.

        For internal use only:
        If this function is called with option 'Test', then S is checked 
        to be positive definite. TRUE or FALSE is returned.
++*/

linalg::factorCholesky := proc(A)
    local R, Rplus, Rzero, Rmult, Rcoerce, R_divide, R_negate,
          t, i, j, k, n, checkOnly, noCheck, Afloated, Mat;
begin
    // check arguments:
    if testargs() then
        if args(0) < 1 or args(0) > 2 then
            error("expecting 1 or 2 arguments")
        end_if;
        if A::dom::hasProp( Cat::Matrix ) <> TRUE then
            error("first argument is not of 'Cat::Matrix'")
        end_if;
        n:= A::dom::matdim( A );
        if n[1] <> n[2] then
            error("not a square matrix")
        end_if;
        R:= A::dom::coeffRing;
        if not R::hasProp( Cat::Field ) then
            error("expecting a matrix over a 'Cat::Field'")
        end_if;
        if args(0) = 2 then
            if args(2) <> hold(NoCheck) and args(2) <> hold(Test) then
                error("option 'NoCheck' expected")
            end_if
        end_if;
    end_if;

    if args(0) = 2 then
        noCheck:= bool( args(2) = hold(NoCheck) );
        checkOnly:= bool( args(2) = hold(Test) );
    else
        noCheck:= FALSE;
        checkOnly:= FALSE
    end_if;

    if not checkOnly and testargs() and not noCheck then
        if A::dom::equal(A, A::dom::transpose(A)) <> TRUE then
            error("expecting a symmetric matrix")
        end_if
    end_if;

    // New cases: 
    // check whether the matrix contains floats and can be converted 
    // to 'float(Matrix)'. If this is the case, call the corresponding 
    // 'numeric'-routine as in the case above. 

    if (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
      Afloated:= numeric::factorCholesky(Afloated);
      Mat:= A::dom;
      if Mat::coerce(Afloated) <> FAIL then 
        return(Mat::coerce(Afloated));
      end_if;
      // There is no other chance than to skip to the 
      // symbolic computation
    end_if;

    // start computation for the symbolic case:

    R := A::dom::coeffRing;
    Rzero := R::zero;
    Rplus := R::_plus;
    Rmult := R::_mult;
    Rcoerce := R::coerce;
    R_negate := R::_negate;
    R_divide := R::_divide;

    n:= op(A::dom::matdim(A),1);
    for j from 1 to n do
        for k from 1 to j-1 do
            A[j,j]:= Rplus( A[j,j],R_negate(Rmult(A[j,k],A[j,k])) );
        end_for;

        k:= A[j,j];
        if not noCheck then
            t:= 0; // initialize (the following assignment
                   // may fail because of an error
            if traperror( (t:= bool(k > Rzero)) ) <> 0 then
                // t:= sign(k); // ???
                if t <> TRUE and t <> FALSE then
                    // try hard!
                    t:= is(k,Type::Positive);
                    if t = UNKNOWN then
                        userinfo(1,"Check if '".expr2text(k)."' > 0 ...");
                        error("cannot check whether matrix component is positive")
                    end_if
                end_if
            end_if;
            if t <> TRUE then
                // A[j,j] is not positive, hence A is not pos. def.
                if not checkOnly then
                    error("given matrix is not positive definite")
                else
                    A:= FAIL; 
                    break 
                end_if
            end_if
        end_if;

        // compute G:
        (A[j,k] := Rzero) $ k=j+1..n; // upper triangular of G is 0

        if (A[j,j]:= Rcoerce( sqrt( A[j,j] ) )) = FAIL then
            userinfo(1, "object: ", sqrt(A[j,j]), 
               " can not be converted into an object of type ", R);
            return( FAIL )
        end_if;

        for i from j+1 to n do
            for k from 1 to j-1 do
                A[i,j] := Rplus( A[i,j],R_negate(Rmult(A[i,k],A[j,k])) )
            end_for;
            A[i,j] := R_divide( A[i,j],A[j,j] )
        end_for
    end_for;

    if checkOnly then bool( A <> FAIL ) else A end_if
end_proc:

