/*
        linalg::factorQR  --  returns the QR-factorization of a matrix

        factorQR( A )

        A : matrix with linear independent column vectors

        Returns a list of two matrices Q and R such that
        A = Q * R, where Q is an orthogonal matrix and R is an upper
        triangular matrix. A has to be defined over a field.

        The Gram-Schmidt algorithm is used to compute the factorization.
*/

linalg::factorQR :=
proc(A)
  local s, n, m, i, j, r, k, e, R, T, M, Afloated;
begin
  if testargs() then
    if args(0) <> 1 then
      error("expecting 1 argument")
    end_if;
    if A::dom::hasProp( Cat::Matrix ) <> TRUE then
      error("first argument is not of 'Cat::Matrix'")
    end_if;
    if not (A::dom::coeffRing)::hasProp( Cat::Field ) then
      error("expecting matrix over 'Cat::Field'")
    end_if
  end_if;

  M:= A::dom;
  T:= M::coeffRing;
  if M::constructor <> Dom::DenseMatrix and M::constructor <> Dom::Matrix then
    M:= Dom::Matrix(T);
    A:= M::coerce(A);
    if A = FAIL then
      error("can't convert matrix to the domain 'Dom::Matrix'")
    end_if
  end_if;

  // New cases: 
  // check whether the matrix contains floats and can be converted 
  // to 'float(Matrix)'. If this is the case, call the corresponding 
  // 'numeric'-routine as in the case above. 

  if (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
    Afloated:= numeric::factorQR(Afloated);
    if M::coerce(Afloated[1]) <> FAIL and M::coerce(Afloated[1]) <> FAIL then 
      return(map(Afloated, M::coerce));
    end_if;
    // There is no other chance than to skip to the 
    // symbolic computation
  end_if;

  // start computation for the symbolic case:

  s:= M::matdim(A);
  n:= s[1]; m:= s[2];
  
  // initialize upper factor
  R:= M::create( n,m );
  // initialize orthogonal factor Q (= n x n submatrix of A):
  // if m > n, we are going to remove additional columns later
 
  if m < n then
    A:= M::concatMatrix( A,M::create( n,n-m ) )
  end_if;

  s:= [NIL $ n]; // s[j]:= col(Q,j) * col(Q,j)
  r:= s;         // r[j]:= |col(Q,j)|
  for j from 1 to n do
    if j <= m then
      ( A[i,j]:= T::_plus(A[i,j], 
                          T::_negate(T::_plus(T::_mult(A[i,k],
                                              T::_divide(R[k,j],s[k])) $ k=1..j-1 )
              )) ;) $ i=1..n;
    end_if; 

    e:= M::col(A,j);
    s[j]:= linalg::scalarProduct( e,e );
    r[j]:= T::coerce(M::norm( e,2 ));
    if r[j] = FAIL then
      userinfo(2,"can't normalize column vectors");
      return( FAIL )
    end_if;

    if iszero(s[j]) then
      // insert an arbitrary column. Try the standard columns
      // e[i] = (0,..,1,..,0)^t until e[i] is independent of the
      // first j-1 orthonormalized columns of Q:
      for k from 1 to n do
        e:= M::create(n,1); e[k]:= T::one;
        e:= e - _plus( M::col(A,i)/s[i] * conjugate(A[k,i]) $ i=1..j-1 );
        s[j]:= linalg::scalarProduct( e,e );
        if not iszero(s[j]) then
         if domtype(s[j]) = DOM_FLOAT or
           (domtype(s[j]) = DOM_COMPLEX and op(s[j],1) = DOM_FLOAT)
           then
             // If the matrix contains floats, we should
             // double check that this column can be accepted.
             if specfunc::abs(s[j]) < max(1/(m + n)/10^2, 10^(-DIGITS)) then
                next; // try the next value of k
             end_if;
          end_if;
          r[j]:= T::coerce(M::norm( e,2 ));
          if r[j] = FAIL then
            userinfo(2,"cannot normalize column vectors");
            return( FAIL )
          end_if;
          A:= M::setCol( A,j,e );
          break // k
        end_if
      end_for;
      if iszero(s[j]) then
        error("could not orthonormalize the columns")
      end_if;
      if j <= m then
        R[j,j]:= T::zero
      end_if
    elif j <= m then
      R[j,j]:= r[j]
    end_if;
    (R[j,k]:= linalg::scalarProduct( e,M::col(A,k) );) $ k=j+1..m // e=col(Q,j)
  end_for;

    // normalize columns of Q and R
  for j from 1 to n do
    (A[i,j]:= T::_divide( A[i,j],r[j] );) $ i=1..n;
    (R[j,i]:= T::_divide( R[j,i],r[j] );) $ i=j+1..m
  end_for;

  // throw away additional columns if any

  if m > n then
    for i from m downto n+1 do
      A:= M::delCol( A, i )
    end_for;
  end_if;
  
  return( [A,R] )
end_proc:

