// linalg::ffG -- computes the upper triangular form of the
//                matrix x, its rank, its determinant and
//                the set of characteristric column indices
//                using fraction free Gaussian elimination.
// Calls:  linalg::ffG(x)
//         linalg::ffG(x, "UsePoly")
//
// Parameter: x: matrix of category Cat::Matrix

linalg::ffG:=  proc(x)
            // option remember;
            local n, m, i, j, l, f, f0, f1, f2, r, k, p, l0, ps, sig, charind, 
                  R, Mat, Rnormal, Rzero, Rone, mydivide, vars, polyRing, opt;
begin
  if args(0) < 1 then error("expecting one argument") end_if;
  opt:= args(2 .. args(0));
  Mat:= x::dom;
  if Mat::hasProp(Cat::Matrix) <> TRUE then
    error("expecting a matrix of 'Cat::Matrix'");
  end_if;
  R:= x::dom::coeffRing;
  if Mat::constructor <> Dom::Matrix then 
    // Convert the matrix into a matrix over Dom::Matrix
    x:= Dom::Matrix(R)(x);
  end_if;
  m:= extop(x,1); // number of rows
  n:= extop(x,2); // number of columns
  Rnormal:= (if R = Dom::ExpressionField() then
                x -> normal(x, Expand = FALSE, opt);
             elif R::normal <> FAIL then 
                normal; // normal should be overloaded by R::normal
             else 
                () -> args(1); 
             end_if);
  mydivide:= (x, y) -> Rnormal(x/y);
  Rzero:= R::zero;
  Rone:= R::one;

  // Special case: with the flag "UsePoly", the exact divisions 
  // of the fraction free Gauss algo are done via divide(.., Quo) 
  // to avoid the use of Rnormal:
  if has([args()], "UsePoly") then
     vars:= [];
     polyRing:= Expr;
     for i from 1 to nops(x) do
       if domtype(op(x, i)) = DOM_POLY then
          vars:= op(x, [i, 2]);
          polyRing:= op(x, [i, 3]);
          break;
       end_if;
     end_for;
     if nops(vars) = 0 then 
       vars:= [genident("x")]; 
     end_if;
     Rnormal:= () -> args(1):
     mydivide:= (x, y) -> (if iszero(x) then 
                              return(x) 
                           else
                              return(x/y);
                           end_if):
     Rzero:= poly(Rzero, vars, polyRing);  
     Rone:= poly(Rone, vars, polyRing);
  end_if:

  ps:= R::pivotSize; // if ps = FAIL then no pivot strategy will be used,
                     // otherwise the pivot w.r.t. < will be choosen
  userinfo(1,"perform 2-step fraction free Gaussian elimination");
  sig:= 1;      // the sign of the determinant
  charind:= {k $ k = 1..n}; // the characteristic column indices
  f0:= Rone;  // the determinant
  k:= 2; 
  r:= 2;
  for k from 2 to 2*(n div 2) step 2 do
      if r > m then 
          break 
      end_if;
      // Start searching for a pivot element
      userinfo(3,"search for non-zero determinant");
      l0:= p:= FAIL; // initial value
      for i from r-1 to m do
          f1:= x[i,k];
          f2:= x[i,k-1];
          for j from i+1 to m do 
              f:= f2 * x[j,k] - f1 * x[j,k-1];
              if linalg::zeroTest(f) = FALSE  then
                  if ps = FAIL then
                      p:= [i,j,f];
                      break
                  elif l0 = FAIL then
                      l0:= ps(f);
                      p:= [i,j,f]
                  elif (l:= ps(f)) < l0 then
                      l0:= l;
                      p:= [i,j,f]
                  end_if
              end_if
          end_for;
          if j <= m and ps = FAIL then break end_if
      end_for;
      // Here, the pivot element p = [i, j, f] is found.
      if p = FAIL then
          userinfo(3,"no non-zero determinant found in step ",k);
          // initialize single step
          if r = 2 then f0:= Rone else f0:= x[r-2,k-2] end_if;
          break
      end_if;
      if p[1] <> r-1 then 
          i:= p[1];
          userinfo(3, "swap rows ", r-1, i);
          x:= x::dom::swapRow(x, r-1, i);
          sig := -sig
      end_if;
      if p[2] <> r then
          j := p[2];
          userinfo(3,"swap rows ",j,r);
          x:= x::dom::swapRow(x, j, r);
          sig := -sig
      end_if;

      if linalg::zeroTest(x[r-1,k-1]) <> FALSE then
          userinfo(3,"swap rows ",r-1,r);
          x:= x::dom::swapRow(x, r-1, r);
          p[3]:= -p[3];
          sig:= -sig
      end_if;

      // Do the elimination
      f := mydivide(p[3], f0);
      if f = FAIL then
          error("no exact division (numerical errors occurred?)")
      end_if;

      for i from r+1 to m do
          f1:= mydivide((x[r-1,k] * x[i,k-1] - x[r-1,k-1] * x[i, k ]), f0);
          f2:= mydivide((x[r,k-1] * x[i, k ] - x[ r , k ] * x[i,k-1]), f0);
          for j from k+1 to n do
              x[i,j]:= mydivide((f * x[i,j] + f1 * x[r,j] + f2 * x[r-1,j]), f0);
          end_for;
          x[i,k]:= Rzero; 
          x[i,k-1]:= Rzero
      end_for;

      for j from k+1 to n do
          x[r,j]:= mydivide((x[r-1,k-1] * x[r,j] - x[r-1,j] * x[r,k-1]), f0);
      end_for;

      x[r,k-1]:= Rzero;
      x[r,k] := f;
      // new divisor
      f0:= f; 
      // next step

      r:= r + 2
  end_for;

  r:= r - 2;

  if n mod 2 = 1 or r <= n then
      userinfo(2,"perform single-step algorithm");
      f1:= k - 1; 
      r:= r + 1;
      for k from f1 to n do
          if r > m then 
              break 
          end_if;
          userinfo(2,"searching for pivot element in row/column ",r,k);
          p:= m + 1; l0:= FAIL; // initial value
          for i from r to m do
              if linalg::zeroTest(x[i,k]) = FALSE then
                  if ps = FAIL then 
                    p:= i; 
                    break
                  elif l0 = FAIL then 
                    l0:= ps(x[i,k]); p:= i
                  elif (l:= ps(x[i,k])) < l0 then 
                    l0:= l; p:= i 
                  end_if
              end_if
          end_for;
          if p <= m then
              if r <> p then 
                  userinfo(3,"swap rows ",r,p);
                  x:= x::dom::swapRow(x, r, p);
                  sig:= -sig
              end_if;

              for i from r+1 to m do
                  f2:= -x[i,k];
                  for j from k+1 to n do
                      x[i,j]:= mydivide((x[r,k] * x[i,j] + x[r,j] * f2), f0);
                  end_for;
                  x[i,k]:= Rzero
              end_for;
              f0:= x[r,k]; 
              r:= r + 1
          else
              charind:= charind minus {k}
          end_if
      end_for;
      charind:= charind minus {l $ l = k..n};
      r:= r - 1
  end_if;
  if n = m then
     if r = m then 
        if sig = -1 then 
           f0:= -f0 
        end_if;
     else 
        f0:= Rzero
     end_if;

  else
//   f0 is the determinant of the left n x n submatrix.
//   It is used by linalg::inverseFFG.
//   f0:= FAIL
  end_if;
  return([x,r,f0,charind,sig])
end_proc:
