//----------------------------------------------------------------------
// This is the algorithm for computing the Frobenius form of a
// a matrix as published in the original paper by Ozello. It is
// 'wrong' in the sense that this is not really the Frobenius
// form. The blocks do not always have the right size. E.g.,
// for diagonal matrices, each single diagonal element is 
// regarded as a block -- which should not always be the case.
//
// A corrected form is available as linalg::frobeniusForm.
// However, this original form linalg::frobeniusFormOzello
// should survive in our library, because linalg::jordanForm
// (and things like exp(matrix), linalg::eigenvectors etc)
// first compute the Frobenius form and then proceed to do
// their special things.
// linalg::frobeniusFormOzello is much better than linalg::frobeniusForm
// for preprocessing the matrix to compute eigenvectors, exp etc. 
// linalgalg::frobeniusFromOzello tends to create much 'more diagonal' 
// matrices than linalg::frobenius, which tends to create larger
// 'companion blocks'. In fact, for a nearly diagonal 4x4 example,
// the proper frobenius form makes exp(matrix) get stuck due to a 4x4
// 'companion block', whereas frobeniusFromOzello creates a 2 x 2
// block plus two 1 x 1 blocks (which exp processes in a few seconds).
//
// linalg::frobeniusFormOzello is used by linalg::jordanForm
// (and consequently, by linalg::eigenvectors, matrix::exp etc)!
//----------------------------------------------------------------------

/* ------------------------------------------------------------------------
    linalg::frobeniusFormOzello 
         --  returns (in most cases) the Frobenius form of a matrix, 
             also called the Rational Canonical form of a matrix

    linalg::frobeniusFormOzello(A <,All>)
    
    A  : a square matrix over Cat::Field
    All: identifier (option)
 
    Computes the Rational Canonical Form of the matrix A,

                                +-------+
                                |B1    0|
                                |  .    |
    i.e., the unique matrix B = |   .   | with companion matrices B1,...,Br
                                |    .  |
                                | 0   Br|
                                +-------+
    
                               +---------------+
                               |0       -a(0)  | 
                     with Bi = |1 .      .     | for all i from 1 to r. 
                               |  . .    .     |         
                               |    1 0 -a(n-1)|
                               +---------------+
    
    The polynomial m(i):=X^n+a(n-1)*X^(n-1)+...+a(1)*X+a(0) thereby is the 
    minimal polynomial of the matrix Bi. 
    For the polynomials m(1),...,m(r) we receive: m(i) divides m(i+1) 
    for i from 1 to r-1. 

    Beware: for diagonal matrices, each single diagonal element is
            regarded as one block (the division property claimed above
            is not respected). See linalg::frobeniusForm for a
            corrected version.
    
    By giving the command 'linalg::frobeniusFormOzello(A,All)', the tranformation 
    matrix P is given, so that we get: (P^-1) * A * P = B
 

    Reference: Calcul exact des formes de Jordan et de Frobenius d'une 
               matrice, by Patrick Ozello, "Docteur de I'Universite 
               Scientifique", algorithms on pages 30 to 43.
------------------------------------------------------------------------ */
linalg::frobeniusFormOzello:= proc( B )
    local n, R, i1, i2, nargs, m, Mat, Riszero, Rnegate, Rinvert, Rmult, Rnormal,
          polyminel, zeroadroite, chercheco, unbloc, addRow, addCol;
begin
    /*---------------------------------------------------
        Procedure "polyminel"
        At first we determine an index k, so that b(j,k)<>0. 
        The first step of this procedure will turn the matrix B to a 
        matrix of the form:
            +---------------+
            |   C1  |   B1  |
            |       |       |
            |-------|-------|
            | 0...0 |       |
            | 0...0 |   B2  |
            +---------------+
    --------------------------------------------------*/
    polyminel:= proc(j)
        // global B, m, R, Riszero;
        local i,k,spalten,zeilen,chercheli,pivot;
    begin
        // Procedure "chercheli":
        // Returns an index k so that b(j,k) <> 0
        chercheli:= proc(j)
            // global B;
            local k;
        begin
            k:=j+1;
            while k<=n and Riszero(Rnormal(B[k,j])) do k:=k+1 end_while;
            return( k )
        end_proc;
      
        k:= chercheli(j);
        while k <= n do 
            if k <> j+1 then
                B:= Mat::swapRow(B,k,j+1);
                B:= Mat::swapCol(B,k,j+1);
            end_if;      

            pivot:= B[j+1,j];
            if pivot <> R::one then
                zeilen:= Mat::row(B,j+1);
                zeilen:= map(zeilen, ind -> Rnormal(Rmult(ind, Rinvert(pivot))));
                B:= Mat::setRow(B,j+1,zeilen);

                spalten:= Mat::col(B,j+1);
                spalten:= map(spalten, ind -> Rnormal(Rmult(ind,pivot)));
                B:= Mat::setCol(B,j+1,spalten);
            end_if;
    
            for i from 1 to n do
                if i <> j+1 then 
                    pivot:= B[i,j];

                    B:= addRow( B,1,n,i,j+1,Rnegate(pivot) );
                    B:= addCol( B,1,m,j+1,i,pivot );
                end_if;
            end_for;
            j:=j+1;

            k:= chercheli( j );
        end_while;

        return(j)
    end_proc:

    /*---------------------------------------------------
        Procedure "zeroadroite"
        Transforms the matrix to a matrix of the form
        
            +---------------+
            |   C1  |*******|
            |       |0.....0|
            |-------|-------|
            | 0...0 |       |
            | 0...0 |   B3  |
            +---------------+
    ---------------------------------------------------*/
    zeroadroite:= proc(i1,i2) 
        // global B, m; 
        // NOTE: B is changed!
        local i, j, pivot;
    begin
        for i from i2 downto i1+2 do
            for j from i2+1 to n do
                pivot:= Rnegate(B[i,j]);
                
                B:= addCol( B,1,m,j,i-1,pivot);
                B:= addRow( B,1,n,i-1,j,Rnegate(pivot) )
            end_for;
        end_for;
    end_proc:

    /*-------------------------------------------------
        Procedure "chercheco"
        compare to "cherceli"
    ---------------------------------------------------*/
    chercheco:= proc(i,j1) 
        // global B, m;
        local j;
    begin
        j:=j1;
        while j<=n and Riszero(Rnormal(B[i,j])) do j:=j+1 end_while;
        return(j)
    end_proc;

    /*---------------------------------------------------
        Procedure "unbloc"
        Transforms the given matrix into a matrix of the form

            +---------------+
            |F1.            |
            |   .           |
            |    Fi         |  <--- i-th line  (F1..Fi matrices)
            |      |--------|
            |      |   B1   |
            +---------------+
    
    ---------------------------------------------------*/
    unbloc:= proc(i1) 
        // global B,m,R,Riszero;
        // NOTE: B is changed!
        local j,i2,j1;
    begin
        repeat 
            i2:= polyminel(i1+1);
            zeroadroite(i1,i2);
            j:= chercheco(i1+1,i2+1);
            if j <= n then 
                //P(i1+1,j)
                B:= Mat::swapRow(B,i1+1,j);
                B:= Mat::swapCol(B,i1+1,j);
            end_if;
        until j>n end_repeat;
        return( i2 )
    end_proc:

    /*---------------------------------------------------
        addRow( A,c1,c2,r1,r2,p )

        c1,c2,r1,r2: positive integers
        p: any object

        Replace row r1 by row1 + p*row2, where row1 and row2 are
        the rows r1 and r2 of A, respectively.

        Act only on columns c1 to c2.
    ---------------------------------------------------*/
    addRow:= proc( A,c1,c2,r1,r2,p )
        local i, R, Rmult, Rplus;
    begin
        R:= A::dom::coeffRing;
        Rmult:= R::_mult;
        Rplus:= R::_plus;

        for i from c1 to c2 do
            A[r1,i]:= Rnormal(Rplus( A[r1,i], Rmult(p,A[r2,i]) ))
        end_for;
        return( A )
    end_proc;

    /*---------------------------------------------------
        addCol( A,r1,r2,c1,c2,p )

        c1,c2,r1,r2: positive integers
        p: any object

        Replace column c1 by col1 + p*col2, where col1 and col2 are
        the columns c1 and c2 of A, respectively.

        Act only on rows c1 to c2.
    ---------------------------------------------------*/
    addCol:= proc( A,r1,r2,c1,c2,p )
        local i, R, Rmult, Rplus;
    begin
        R:= A::dom::coeffRing;
        Rmult:= R::_mult;
        Rplus:= R::_plus;

        for i from r1 to r2 do
            A[i,c1]:= Rnormal(Rplus( A[i,c1], Rmult(p,A[i,c2]) ))
        end_for;
        return( A )
    end_proc;

    // *************************** MAIN *****************************
    nargs:= args(0);
    if nargs = 0 then error("no argument given") end_if;
    Mat:= B::dom;

    if testargs() then
        if Mat::hasProp( Cat::Matrix ) <> TRUE then
            error("argument is not of 'Cat::Matrix'")
        end_if;
       
        if not Mat::coeffRing::hasProp( Cat::Field ) then
            error("expecting matrix over 'Cat::Field'")
        end_if;      
        n:= Mat::matdim(B);
        if n[1] <> n[2] then
            error("not a square matrix")
        end_if;

        if nargs = 2 and args(2) <> hold(All) then
            error("2nd argument must be the option 'All'")
        elif nargs > 2 then
            error("too many arguments given")
        end_if
    end_if;

    R:= Mat::coeffRing;
    Riszero:= R::iszero;
    Rmult:= R::_mult;
    Rnegate:= R::_negate;
    Rinvert:= R::_invert;
    if R::normal = FAIL then
       Rnormal:= () -> args(1);
    else 
       Rnormal:= normal;
    end_if;
    n:= Mat::matdim(B)[2]; // number of columns
    if nargs = 2 then
        B:= Mat::stackMatrix( B,Mat::identity(n) );
        m:= 2*n; // number of rows
    else
        m:= n // number of rows
    end_if;
    
    // Calculating the rational canonical form 
    i1:= 0;
    repeat
        i2:= unbloc( i1 );
        i1:= i2
    until i2 = n end_repeat;

    if nargs = 2 then
        return( [B[1..n,1..n],B[n+1..m,1..n]] )
    else
        return( B )
    end_if
end_proc:

