/*
    linalg::gaussJordan  --  perform the Gauss-Jordan elimination

    gaussJordan(A [,All])

    gaussJordan(A) returns A reduced to the reduced-row-echolon form.
    With option 'All', the rank, determinant (or FAIL) and a set
    of the characteristic column indices of A are returned additionally
    in form of a list.
*/

linalg::gaussJordan := proc(A)
    local p, R, T, t, c, i, j, d, ns, _ns, de, r, charind,
          Rmult, Rdivide, Rplus, Rnegate, Rzero, Rone, Riszero, Rnormal;
begin
    if testargs() then
        if args(0) < 1 or args(0) > 2 then
           if args(0) = 3 and args(3) <> `#FloatCheck` then 
             error("expecting 1 or 2 arguments")
           end_if;
        end_if;
        if A::dom::hasProp( Cat::Matrix ) <> TRUE then
            error("first argument is not of 'Cat::Matrix'")
        end_if;
        if not (A::dom::coeffRing)::hasProp( Cat::IntegralDomain ) then
            error("expecting matrix over 'Cat::IntegralDomain'")
        end_if;
        if args(0) >= 2 then
            if args(2) <> hold(All) then
                error("expecting 'All' as option")
            end_if
        end_if
    end_if;
  
    userinfo(1,"compute Gauss-Jordan form");
    userinfo(2,"perform Gaussian elimination");

    if args(0) = 3 and args(3) = `#FloatCheck` then
      // print("In 'linalg::gaussJordan'");
      // At this time, the following case is only relevant, when we try to 
      // compute eigenvectors of the Matrix A and 'linalg::gaussJordan' is 
      // called from 'linalg::nullspace' which is again called from 
      // 'linalg::eigenvectors' (both with additional argument `#FloatCheck`). 
      // We proceed as follows: we call 'A::dom::gaussElim' with additional 
      // argument `#FloatCheck`. 
      // Then, 'A::dom::gaussElim' uses a special heuristic for zeroTest
      // and we will be better in finding 'eigenvectors' than in the general 
      // case where A perhaps contains symbolic entries. 
      T:= A::dom::gaussElim(A, `#FloatCheck`);
    else 
      T:= A::dom::gaussElim(A);
    end_if;

    A:= op(T,1);          // matrix in row echolon form
    charind:= op(T,4);    // the chararistic column indices

    R:= A::dom::coeffRing;

    Rplus:= R::_plus;
    Rnegate:= R::_negate;
    Rmult:= R::_mult;
    Rzero:= R::zero;

    d:= A::dom::matdim(A);
    r:= d[1]; c:= d[2];

    ns:= {j $ j=1..c} minus charind;

    userinfo(2,"compute row reduced form");

    // ds -- list of [i,j] whereby A[i,j] <> 0
    de:= [ NIL $ min(r,nops(charind)) ];
    i:= 1;
    for j from 1 to c do
        if contains( charind,j ) then
            de[i]:= [i,j];
            i:= i + 1;
            if i > r then break end_if
        end_if
    end_for;

    de:= revert( de );

    if R::hasProp(Cat::Field) then
        // -----------------------------------
        // Gauss-Jordan algorithm over fields:
        // -----------------------------------

        if (Rnormal:= R::normal) = FAIL then
            if R::hasProp(Ax::systemRep) then 
               Rnormal:= normal 
            else Rnormal:= () -> args(1); 
            end_if
        end_if;
        Rdivide:= R::_divide;
        Rone:= R::one;

        for t in de do
           // select kerVects belong to the right of A[.,t[2]]
            _ns:= select( ns,_not@_leequal,t[2] );

            p:= A[t[1],t[2]]; // the pivot

            for j in _ns do 
                A[t[1],j]:= Rnormal(Rdivide(A[t[1],j],p))
            end_for;

            A[t[1],t[2]]:= Rone;
            for i from t[1]-1 downto 1 do
                p:= Rnegate( A[i,t[2]] );
                for j in _ns do
                    A[i,j]:= Rnormal(Rplus( A[i,j], Rmult( p,A[t[1],j] ) ))
                end_for;
                A[i,t[2]]:= Rzero
            end_for;
        end_for;
    else
        // ---------------------------------------------
        // Gauss-Jordan algorithm over integral domains:
        // ---------------------------------------------
        for t in de do
            // select kerVects belong to the right of A[.,t[2]]:
            _ns:= select( ns,_not@_leequal,t[2] );

            p:= A[t[1],t[2]]; // the pivot

            delete de[1];
            for i from t[1]-1 downto 1 do
                // multiply each entry of row i by p:
                for j from i to c do
                    if j <> t[2] then A[i,j]:= Rmult( A[i,j],p ) end_if
                end_for;

                d:= Rnegate( A[i,t[2]] );
                for j in _ns do
                    A[i,j]:= Rplus( Rmult(A[t[1],j],d), A[i,j] )
                end_for;

                A[i,t[2]]:= Rzero
            end_for
        end_for;

        if (t:= R::gcd) <> FAIL then
            Rdivide:= R::_divide;
            Riszero:= R::iszero;
            Rone:= R::one;
            for i from 1 to r do
                d:= t( op(A::dom::row(A,i)) );
                if not Riszero(d) and d <> Rone then
                    ( A[i,j]:= Rdivide(A[i,j],d) ) $ j=1..c
                end_if
           end_for
        end_if
    end_if;

    if args(0) = 2 or args(0) = 3 then [A,T[2],T[3],charind] else A end_if
end_proc:

